
#include "conserver.h"
#include "meta.h"
#include "lockable.h"
#include "conn.h"
#include "rfc2553emu.h"
#include "acfg.h"


#include <sys/socket.h>
#include <sys/un.h>
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <errno.h>

#include <netdb.h>

#include <cstdio>
#include <list>
#include <map>
#include <iostream>

#include "debug.h"

using namespace MYSTD;

// for cygwin, incomplete ipv6 stuff
#ifndef AF_INET6
#define AF_INET6        23              /* IP version 6 */
#endif

namespace conserver
{

int sockunix(-1);
vector<int> vecSocks;

condition cond;

list<con*> /* qForHelper, */ qForWorker;
//map<con*,int> mConStatus;
//int nStandbyHelper(0);
//int nStandbyWorker(0);
int nSpareThreads(0);
bool bTerminationMode(false);
int nAllConThreadCount(0);

//#define TYPE_WORKER ((void*)0)
//#define TYPE_HELPER ((void*)1)
#define STANDBY_LIMIT acfg::tpoolsize
#define MAX_BACKLOG 200

void * ThreadAction(void *)
{
	list<con*> & myq = qForWorker;
	USRDBG(3, "Thread started");

	lockguard g(cond);
	nAllConThreadCount++;

	while (true)
	{
		if (bTerminationMode)
			break;

		if(myq.empty())
		{ // to be decreased by the pool client
			nSpareThreads++;
			while (myq.empty())
				cond.wait();
		}
				
		con *c=myq.front();
		myq.pop_front();
		g.unLock();
	
		c->WorkLoop();
		// should be dead and unreferenced now
#ifdef KILLABLE
		// should be already free, but better be safe...
		c->SignalStop(); 
#endif
		c->ShutDown(); // close file descriptors
		delete c;
		
		g.reLock();
		if (nSpareThreads>=STANDBY_LIMIT)
			return NULL;
	}
	
	nAllConThreadCount--;
	USRDBG(3, "Terminating thread");
	return NULL;
}

bool SpawnThread(void * type)
{
	USRDBG(3, "Creating " << (type? "helper" : "worker"));
	pthread_t thr;
	pthread_attr_t attr; // detached, to be initialized in Setup
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

	return (0==pthread_create(&thr, NULL, ThreadAction, type));
}

void SetupConAndGo(int fd, const char *szClientName=NULL)
{
	// protect from interference of con shutdown with OS on file descriptor 
	// management and also thread pool control
	lockguard g(cond);
	
	if(!szClientName)
		szClientName="";
	
	USRDBG(1, "Client name: " << szClientName);

	con *c(NULL);
	MYTRY
	{
		c=new con(fd, szClientName);
		if(!c)
		{
#ifdef NO_EXCEPTIONS
			USRDBG(4, "Out of memory");
#endif
			goto failure_mode;
		}
	}
	MYCATCH(bad_alloc)
	{
		USRDBG(4, "Out of memory");
		goto failure_mode;
	}
	
	// DOS prevention
	if (qForWorker.size() > MAX_BACKLOG)
	{
		USRDBG(4, "Worker queue overrun");
		goto failure_mode;
	}

	if (nSpareThreads>0)
		nSpareThreads--; // one is ours
	else if(!SpawnThread(0))
		goto failure_mode;
	
	qForWorker.push_back(c);
	cond.notifyAll();

	return;

	failure_mode: 
	if (c)
		delete c;
	shutdown(fd, SHUT_RDWR);
	USRDBG(3, "Connection setup error");
	close(fd);
}

void CreateUnixSocket() {
	string & sPath=acfg::fifopath;
	struct sockaddr_un addr_unx;
	memset(&addr_unx, 0, sizeof(addr_unx));
	
	int jo=1;
	size_t size = sPath.length()+1+offsetof(struct sockaddr_un, sun_path);
	
	if(sPath.length()>sizeof(addr_unx.sun_path))
	{
		errno=ENAMETOOLONG;
		goto unx_err;
	}
	
	addr_unx.sun_family = AF_UNIX;
	strncpy(addr_unx.sun_path, sPath.c_str(), sPath.length());
	
	mkbasedir(sPath);
	unlink(sPath.c_str());
	
	sockunix = socket(PF_UNIX, SOCK_STREAM, 0);
	setsockopt(sockunix, SOL_SOCKET, SO_REUSEADDR, &jo, sizeof(jo));

	if (sockunix<0)
		goto unx_err;
	
	if (bind(sockunix, (struct sockaddr *)&addr_unx, size) < 0)
		goto unx_err;
	
	if (0==listen(sockunix, SO_MAXCONN))
	{
		vecSocks.push_back(sockunix);
		return;
	}
	
	unx_err:
	cerr << "Error creating Unix Domain Socket, ";
	cerr.flush();
	perror(acfg::fifopath.c_str());
	cerr << "Check socket file and directory permissions" <<endl;
	exit(EXIT_FAILURE);

}

void Setup()
{
	using namespace acfg;
	
	if (fifopath.empty() && port.empty())
	{
		cerr << "Neither TCP nor UNIX interface configured, cannot proceed.\n";
		exit(EXIT_FAILURE);
	}
	
	if (atoi(port.c_str())>0)
	{
		struct addrinfo    hints;
		memset(&hints, 0, sizeof hints);
		hints.ai_socktype = SOCK_STREAM;
		hints.ai_flags = AI_PASSIVE;
		hints.ai_family = 0;
		int yes(1);
		
		tStrVec sAdds;
		if(bindaddr.empty())
			sAdds.push_back(""); // dummy, use any later
		else
			Tokenize(bindaddr, SPACECHARS, sAdds);
		for(tStrVec::iterator it=sAdds.begin(); it!=sAdds.end(); it++)
		{
			trimString(*it);
		    struct addrinfo *res, *p;
		    
		    if(0!=getaddrinfo(bindaddr.empty() ? NULL : it->c_str(),
				   port.c_str(), &hints, &res))
			   goto error_getaddrinfo;
		    
		    for(p=res; p; p=p->ai_next)
		    {
		    	int sockip = socket(p->ai_family, p->ai_socktype, p->ai_protocol);
				if (sockip<0)
					goto error_socket;


#ifdef IPV6_V6ONLY
		    	if(p->ai_family==AF_INET6)
		    		setsockopt(sockip, SOL_IPV6, IPV6_V6ONLY, &yes, sizeof(yes));
#endif		    	
				setsockopt(sockip, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(yes));
				
				
		    	if (bind(sockip, p->ai_addr, p->ai_addrlen))
		    		goto error_bind;
		    	if (listen(sockip, SO_MAXCONN))
		    		goto error_listen;
		    	
		    	vecSocks.push_back(sockip);
		    	
		    	continue;

				error_socket:
				
				if(EAFNOSUPPORT != errno &&
						EPFNOSUPPORT != errno &&
						ESOCKTNOSUPPORT != errno &&
						EPROTONOSUPPORT != errno)
				{
					perror("Error creating socket");
				}
				goto close_socket;

				error_listen:
				perror("Couldn't listen on socket");
				goto close_socket;

				error_bind:
				perror("Couldn't bind socket");
				goto close_socket;

				close_socket:
				close(sockip);
				sockip=-1;
		    }
		    freeaddrinfo(res);
		    continue;
		    
		    error_getaddrinfo:
		    perror("Error resolving address for binding");
		}
		if(vecSocks.empty())
		{
			cerr << "No socket(s) could be create/prepared. "
			"Check the network, check or unset the BindAddress directive.\n";
			exit(EXIT_FAILURE);
		}
	}
	else
		aclog::err("Not creating TCP listening socket, no valid port specified!");

	if ( !acfg::fifopath.empty() )
		CreateUnixSocket();
	else
		aclog::err("Not creating Unix Domain Socket, fifo_path not specified");
}

int Run()
{
	fd_set rfds, wfds;
	int maxfd(0);
	for(vector<int>::iterator it=vecSocks.begin(); it!=vecSocks.end(); it++)
		if(*it > maxfd)
			maxfd=*it;
	maxfd++;
	USRDBG(1, "Listening to incoming connections...");

	while (1)
	{ // main accept() loop

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		for(vector<int>::iterator it=vecSocks.begin(); it!=vecSocks.end(); it++)
			FD_SET(*it, &rfds);
		
		//cerr << "Polling..." <<endl;
		int nReady=select(maxfd, &rfds, &wfds, NULL, NULL);
		if (nReady<0)
		{
			if(errno == EINTR)
				continue;
			
			aclog::err("select", "failure");
			perror("Select died");
			exit(EXIT_FAILURE);
		}
		
		for(vector<int>::iterator it=vecSocks.begin(); it!=vecSocks.end(); it++)
		{
			if(!FD_ISSET(*it, &rfds))
			continue;
			if(sockunix == *it)
			{
				int fd = accept(sockunix, NULL, NULL);
				if (fd>=0)
				{
					set_nb(fd);
					USRDBG(1, "Detected incoming connection from the UNIX socket");
					SetupConAndGo(fd);
				}
			}
			else
			{
				struct sockaddr_storage addr;

				socklen_t addrlen = sizeof(addr);
				int fd=accept(*it,(struct sockaddr *)&addr, &addrlen);

				if (fd>=0)
				{
					set_nb(fd);
					char hbuf[NI_MAXHOST];
					USRDBG(1, "Detected incoming connection from the TCP socket");
					if (getnameinfo((struct sockaddr*) &addr, addrlen, hbuf, sizeof(hbuf),
									NULL, 0, NI_NUMERICHOST))
					{
						printf("ERROR: could not resolve hostname\n");
						return 1;
					}
					SetupConAndGo(fd, hbuf);
				}
			}
		}
	}
	return 0;
}

void Shutdown()
{
	lockguard g(cond);
	
	if(bTerminationMode)
		return; // double SIGWHATEVER? Prevent it.
	
	//for (map<con*,int>::iterator it=mConStatus.begin(); it !=mConStatus.end(); it++)
	//	it->first->SignalStop();
	// TODO: maybe call shutdown on all?
	//printf("Signaled stop to all cons\n");
	
	bTerminationMode=true;
	printf("Notifying waiting threads\n");
	cond.notifyAll();
	
	printf("Closing sockets\n");
	for(vector<int>::iterator it=vecSocks.begin(); it!=vecSocks.end(); it++)
	{
		if(*it>0)
		{
			shutdown(*it, SHUT_RDWR);
			close(*it);
			*it=-1;
		}
	}
}

}
