
#include "debug.h"

#include "dljob.h"
#include "dlcon.h"

#include <cstdio>

using namespace MYSTD;

tDlJob::tDlJob(dlcon *p, tFileItemPtr pFi, const string & sHost,
		const string & sPath) :
m_sHost(sHost), m_sPath(sPath), m_pHostiVec(NULL),
m_pStorage(pFi), m_parent(p)
{
	_Init();
}

tDlJob::tDlJob(dlcon *p, tFileItemPtr pFi, acfg::tHostiVec * pBackends,
		const MYSTD::string & sPath) :
	m_sPath(sPath), m_pHostiVec(pBackends), m_pStorage(pFi),
			m_parent(p)
{
	_Init();
}

tDlJob::~tDlJob()
{
	if(m_pStorage)
	{
		m_pStorage->DelDownloaderRef("567 Unknown download error occured");
		m_pStorage.reset();
	}
}

void tDlJob::_Init()
{
	m_nRest=0;
	m_bReconnectASAP=false;
	bSuggestReconnect=false;
	bRequestPrepared=false;
	m_DlState=STATE_GETHEADER;
	m_pBackend=NULL;
	
	if(m_pStorage)
		m_pStorage->AddDownloaderRef();
}


void tDlJob::MarkFailed(const string & sError)
{
	if(! m_pStorage.get())
		return;
	m_pStorage->DelDownloaderRef(sError);
	m_pStorage.reset();
}

bool tDlJob::FindConfig()
{
	// using backends? Find one which is not blacklisted
	
	if(m_pHostiVec)
	{
		// keep the existing one if possible
		if(m_pBackend && m_parent->m_MirrorHostBlacklist.find(m_pBackend->sHost) 
					== m_parent->m_MirrorHostBlacklist.end())
			return true;
		
		for(UINT i=0; i<m_pHostiVec->size(); i++)
		{
			tHttpUrl *beTest = & m_pHostiVec->at(i);
			if(beTest && m_parent->m_MirrorHostBlacklist.find(beTest->sHost) 
					== m_parent->m_MirrorHostBlacklist.end())
			{
				m_pBackend=beTest;
				return true;
			}
		}
		return false;
	}
	else
	{
		// not in blacklist -> OK
		return (m_parent->m_MirrorHostBlacklist.find(m_sHost)
				== m_parent->m_MirrorHostBlacklist.end());
	}
}

void tDlJob::BlacklistBackend()
{
	m_parent->m_MirrorHostBlacklist.insert(m_pBackend?m_pBackend->sHost:m_sHost);
}

// needs connectedHost, blacklist, output buffer from the parent, proxy mode?
bool tDlJob::MakeRequest(string sForThisHostOnly, string &head)
{
	if(! m_pStorage)
		return R_ERROR_LOCAL;
	
	const string sHost = GetHost();
	
	// just make sure that both are set or both unset
	if( (0==m_pBackend) != (0==m_pHostiVec))
		return false;
	
	// host filter was set and mismatched?
	if( ! m_parent->m_proxy 
			&& ! sForThisHostOnly.empty() 
			&& sForThisHostOnly != sHost)
	{
		bSuggestReconnect=true;
		return false;
	}

	head+="GET ";
	if (m_parent->m_proxy)
	{
		head+="http://";
		head+=sHost;
	}
	if(m_pBackend)
		head+=m_pBackend->sPath;
	head+= m_sPath +" HTTP/1.1\r\nHost: ";
	head+= sHost+"\r\n";//Accept: */*\r\n";
	head+= "Connection: keep-alive\r\n";
	if (m_parent->m_proxy) // also add authorization if there is any
		head+=m_parent->m_proxy->sPath+"Proxy-Connection: keep-alive\r\n";

	if (m_pStorage->m_nSizeSeen > 0)
	{
		bool bSetRange(false), bSetIfRange(false);
		
		lockguard g(m_pStorage.get());
		const header *pHead = m_pStorage->GetHeaderUnlocked();
		const char *p=pHead->h[header::LAST_MODIFIED];
		
		if (m_pStorage->m_bCheckFreshness)
		{
			if (p)
			{
				bSetIfRange=true;
				bSetRange=true;
			}
			// else no date available, cannot rely on anything
		}
		else
		{ 
			/////////////// this was protection against broken stuff in the pool ////
			// static file type, date does not matter. check known content length, not risking "range not satisfiable" result
			//
			//off_t nContLen=atol(h.get("Content-Length"));
			//if (nContLen>0 && j->m_pStorage->m_nFileSize < nContLen)
			bSetRange=true;
		}
		
		if (bSetRange)
		{
			/* use APT's trick - set the starting position one byte lower - this way the server has to
			 * send at least one byte if the assumed position is correct, and we never get a
			 * 416 error (one byte waste is acceptable). */
			char buf[50];
			sprintf(buf, "Range: bytes=%ld-\r\n", m_pStorage->m_nSizeSeen - 1 );
			head+=buf;
		}
		if (bSetIfRange)
		{
			head += "If-Range: ";
			head += p;
			head += "\r\n";
		}
	}
	// Debian Apt-Cacher/" ACVERSION 
	head+="User-Agent: ";
	head += acfg::agentname + "\r\n";
	
	if(acfg::exporigin)
		(head+="X-Forwarded-For: ")+=m_parent->m_sXForwardedFor+"\r\n";
	
	head += "\r\n";
			
	ldbg("Request cooked: " << head);
	
	bRequestPrepared=true;
	return true;
}


/*!
 * 
 * Process incoming traffic and write it down to disk/downloaders.
 */
tDlJob::tDlResult tDlJob::ProcessIncomming(acbuf & inBuf, string & sErrorRet)
{
	if(! m_pStorage)
		return R_ERROR_LOCAL;
	
	Reswitch:
	ldbg("switch: " << m_DlState);
	
	switch (m_DlState)
	{

	case (STATE_GETHEADER):
	{
		ldbg("STATE_GETHEADER");
		header h;
		if(inBuf.size()==0)
			return R_MOREINPUT;
		dbgline;
		int l=h.LoadFromBuf(inBuf.rptr(), inBuf.size());
		if (0==l)
			return R_MOREINPUT;
		else if(l>0)
		{
			dbgline;
			inBuf.drop(l);
			if (h.type!=header::ANSWER)
			{
				sErrorRet="500 Unexpected response type";
				return R_ERROR_REMOTE;
			}
		}
		else
		{
			dbgline;
			sErrorRet="500 Invalid header";
            return R_ERROR_REMOTE;
		}
		
		ldbg("GOT, parsed: " << h.frontLine);

		const char *p=h.h[header::CONNECTION] ? h.h[header::CONNECTION] : h.h[header::PROXY_CONNECTION];  
		if(p && 0==strcasecmp(p, "close") )
		{
			ldbg("Peer wants to close connection after request");
			m_bReconnectASAP=true;
		}
		p=h.h[header::TRANSFER_ENCODING];
		if (p && 0==strcasecmp(p, "chunked"))
			m_DlState=STATE_GETCHUNKHEAD;
		else
		{
			dbgline;
			p=h.h[header::CONTENT_LENGTH];
			if (!p)
            {
				sErrorRet="500 Missing Content-Length";
                return R_ERROR_REMOTE;
            }
			// may support such endless stuff in the future but that's too unreliable for now
			m_nRest=atol(p);
			m_DlState=STATE_GETDATA;
		}

        h.set(header::XORIG,
                string("http://", 7)
                +(GetHost()+'/')
                +m_sPath);

        if ( ! m_pStorage->DownloadStartedStoreHeader(h))
        {
            ldbg("Item dl'ed by others or in error state --> drop it, reconnect");
            m_DlState=STATE_GETDATA; // XXX maybe introduce a new error state, this is a kludge
            return R_SKIPITEM;
        }
		goto Reswitch;

	}
	case (STATE_GETDATA_CHUNKED): // fall through, just send it back to header parser hereafter
		ldbg("STATE_GETDATA_CHUNKED (to STATE_GETDATA)");
	case (STATE_GETDATA):
	{
		ldbg("STATE_GETDATA");
		
		while(true)
		{
			off_t nToStore = min((off_t)inBuf.size(), m_nRest);
			ldbg("To store: " <<nToStore);
			if(0==nToStore)
				break;

			if(!m_pStorage->StoreFileData(inBuf.rptr(),nToStore))
                return R_ERROR_LOCAL;

			m_nRest-=nToStore;
			inBuf.drop(nToStore);
		}
			
		ldbg("Rest: " << m_nRest );
		
		if (m_nRest==0)
			m_DlState = (STATE_GETDATA==m_DlState) ? STATE_FINISHJOB : STATE_GETCHUNKHEAD;
		else
			return R_MOREINPUT; // will come back

		goto Reswitch;
	}
	case (STATE_FINISHJOB):
	{
		ldbg("STATE_FINISHJOB");
		m_DlState=STATE_GETHEADER;
		m_pStorage->StoreFileData(NULL, 0);
        return m_bReconnectASAP ? R_SKIPITEM :  R_DONE;
	}
	case (STATE_GETCHUNKHEAD):
	{
		ldbg("STATE_GETCHUNKHEAD");
		char *p=inBuf.c_str();
		char *e=strstr(p, "\r\n");
		if(e==p)
		{ // came back from reading, drop remaining junk? 
			inBuf.drop(2);
			p+=2;
			e=strstr(p, "\r\n");
		}
		dbgline;
		if(!e)
		{
			inBuf.move();
			return R_MOREINPUT; // get more data
		}
		unsigned int len(0);
		int n = sscanf(p, "%x", &len); 
		
		unsigned int nCheadSize=e-p+2;
		if(n==1 && len>0)
		{
			ldbg("ok, skip " << nCheadSize <<" bytes, " <<p);
			inBuf.drop(nCheadSize);
			m_nRest=len;
			m_DlState=STATE_GETDATA_CHUNKED;
		}
		else if(n==1)
		{
			// skip the additional \r\n of the null-sized part here as well
			ldbg("looks like the end, but needs to get everything into buffer to change the state reliably");
			if(inBuf.size() < nCheadSize+2 )
			{
				inBuf.move();
				return R_MOREINPUT;
			}
			if( ! (e[2]=='\r' && e[3]=='\n'))
			{
				aclog::err(m_pStorage->m_sKey+" -- error in chunk format detected");
				return R_ERROR_REMOTE;
			}
			
			inBuf.drop(nCheadSize+2);
			m_DlState=STATE_FINISHJOB;
		}
		else
			return R_ERROR_REMOTE; // that's bad...
		goto Reswitch;
		break;
	}
	
	}
	return R_ERROR_REMOTE;
}
