
//#define LOCAL_DEBUG
#include "debug.h"

#include "expiration.h"
#include "lockable.h"
#include "acfg.h"
#include "meta.h"
#include "filereader.h"
#include "fileitem.h"
#include "dlcon.h"
#include "dirwalk.h"
#include "header.h"
#include "job.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <glob.h>

#include <map>
#include <string>
#include <iostream>

using namespace MYSTD;

#define ENABLED
#ifndef ENABLED
#warning defused
#endif

bool bSigTaskAbort=false;
pthread_mutex_t abortMx=PTHREAD_MUTEX_INITIALIZER;

expiration::expiration(int fd) :
	maintenance(fd), m_nTimeNow(0), m_nDropPrefLen(0), 
	m_bErrAbort(false), m_bVerbose(false), m_bForceDownload(false), 
	m_bScanInternals(false), m_nErrorCount(0), m_repFileStream(NULL),
	m_nProgIdx(0), m_nProgTold(1)
{
	m_szDecoFile="maint.html";

	m_nDropPrefLen=acfg::cachedir.size()+1;
	m_nTimeNow=time(NULL);

}

expiration::~expiration()
{
	if(m_repFileStream)
	{
		fclose(m_repFileStream);
		m_repFileStream=0;
	}
}
bool expiration::ProcessOthers(const string & sPath, const struct stat &)
{
	// NOOP
	return true;
}

bool expiration::ProcessRegular(const string & sPath, const struct stat &)
{

	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return false; // stopping the dirwalk
	}
	
	if (m_nProgIdx++>m_nProgTold)
	{
		m_nProgTold*=2;
		char buf[50];
		int n=snprintf(buf, sizeof(buf), "Scanned %u files...<BR>\n",
				m_nProgTold+1);
		maintenance::SendMsg(buf, n);
	}
	 
	if(0==sPath.compare(m_nDropPrefLen, 1, "_") && !m_bScanInternals)
		return true; // not for us

	tStrPos nSlashPos=sPath.rfind(sPathSep[0]);
	string sBasename(sPath, nSlashPos+1);
	
	// handle the head files separately
    if (endsWithSzAr(sPath, ".head"))
    {
        m_trashCandHeadSet.insert(sPath.substr(m_nDropPrefLen));
        return true;
    }

	if (rechecks::FILE_INDEX == m_rex.getFiletype(sPath.c_str()))
    {
        //cerr << "ifile: " <<sPath <<endl;
		m_volatileFiles.push_back(sPath);
    }
    string sDirnameAndSlash(sPath, m_nDropPrefLen, nSlashPos-m_nDropPrefLen+1);
    m_trashCandSet.insert(pair<string,tDiskFileInfo>(sBasename,
    		tDiskFileInfo(sDirnameAndSlash, m_nTimeNow)));
    return true;
}

void expiration::_LoadTrashMapFromFile(bool bForceInsert)
{

	filereader reader;
	reader.OpenFile(acfg::cachedir+sPathSep+"_expending_dat");
	
	string sLine;

#if 0
	// stuff for user info
	time_t now=time(NULL);
	time_t oldest(now), newest(0);
#endif
	
	while(reader.GetOneLine(sLine))
	{
		tStrVec elems;
		if(Tokenize(sLine, "\t", elems)==3)
		{
			time_t timestamp=atoll(elems[0].c_str());
			if(timestamp>m_nTimeNow)
				continue; // where is the DeLorean?
			
			if(elems[1].empty() || elems[2].empty())
				continue;
			
			if(elems[1].at(elems[1].size()-1)!=SZPATHSEP[0])
				elems[1].append(SZPATHSEP);
			
			if(bForceInsert)
			{
				m_trashCandSet.insert(
						pair<string, tDiskFileInfo>(elems[2], 
								tDiskFileInfo(elems[1], 0)));
				continue;
			}
			
			static pair<tS2DAT::iterator, tS2DAT::iterator> fromTo;
			fromTo=m_trashCandSet.equal_range(elems[2]);
			// file already gone?
			if(fromTo.first==fromTo.second)
				continue; // file already gone?
			
			// TODO: make global iterators for the other function too?
			static tS2DAT::iterator itmp, it;
			for(it=fromTo.first; it!=fromTo.second; it++)
				if(it->second.sDirname == elems[1]) // hit
					it->second.nLostAt=timestamp;										

#if 0
			if(timestamp < oldest)
				oldest=timestamp;
			if(timestamp > newest)
				newest=timestamp;
#endif
		}
	}
	
#if 0
	/*
	cout << "Unreferenced: ";
	for(trashmap_t::iterator it=m_trashCandidates.begin(); it!=m_trashCandidates.end(); it++)
		fprintf(stdout, "%lu\t%s\t%s\n",  it->second.first, it->second.second.c_str(), it->first.c_str());
	*/
	
	if(m_trashCandidates.size()>0)
	{
		// pretty printing for impatient users
		char buf[200];
		
		// map to to wait
		int nMax=acfg::extreshhold-((now-oldest)/86400);
		int nMin=acfg::extreshhold-((now-newest)/86400);
				
		snprintf(buf, _countof(buf), "Previously detected: %lu rotten package file(s), "
				"to be deleted in about %d-%d day(s)<br>\n", 
				(unsigned long) m_trashCandidates.size(),
				nMin, nMax==nMin?nMax+1:nMax); // cheat a bit for the sake of code simplicity
		SendMsg(buf);
	}
#endif

}

bool expiration::_DownloadOne(dlcon &dl, const MYSTD::string & sFilePath, string &sErr)
{
	tFileItemPtr fi;
	acfg::tHostiVec *pBackends(NULL);

	string sKeyPath=sFilePath.substr(acfg::cachedir.size()+1, stmiss);

	// keypath must represent a previosly resolved URL, being including
	// either repname or real host which works
	tHttpUrl url;
#ifdef WIN32
#error rewrite for the backslashes or change reliably in the walker to slashes
#endif
	if (!url.SetHttpUrl(sKeyPath))
	{
		sErr=sKeyPath+" does not contain a valid repository or host name.";
		goto rep_update_prob;
	}

	fi=fileitem::GetFileItem(sKeyPath);
	if (fi)
	{
		fi->Setup(true, m_bForceDownload);
	}
	else
	{
		sErr=" could not create file item handler.";
		goto rep_update_prob;
	}
	
	SendMsg(string("Checking/Updating ")+sKeyPath+"...\n");
	pBackends = acfg::GetBackendVec(&url.sHost);
	if (pBackends)
		url.sPath.erase(0, 1); // get only the path string residual
	else
	{ 
		// try something different, do we have the original URL? It's not reliable, though
		lockguard g(fi.get());
		const header *pHead=fi->GetHeaderUnlocked();
		// must be well formated to be valid
		if (pHead
		&& pHead->h[header::XORIG]
		&& 0==strncmp(pHead->h[header::XORIG], "http://", 7)
		&& url.SetHttpUrl(pHead->h[header::XORIG]))
		{
			pBackends = acfg::GetBackendVec(&url.sHost);
		}
	}
	if (pBackends)
		dl.AddJob(fi, pBackends, url.sPath);
	else
		// use as hostname, and pray
		dl.AddJob(fi, url);

	dl.WorkLoop();
	{
		
		if (fi->GetStatus() == FIST_ERROR)
			goto rep_update_prob;
	}

	SendMsg(string("<i>(")+ltos(fi->GetTransferCount()/1024)+("KiB)</i><br>\n"));
	
	if (fi)
	{
		fileitem::DelUser(sKeyPath);
		fi.reset();
	}
	
	return true;

	rep_update_prob:

	if (fi)
	{
		{
			lockguard g(*fi);
			const header *pErHead=fi->GetHeaderUnlocked();
			if (pErHead)
				sErr+=string("Reason: ")+pErHead->frontLine+"<br>";
		}
		fileitem::DelUser(sKeyPath);
		fi.reset();
	}
	SendMsg("<br>\n");
	
	if(_CanBeIgnored(sKeyPath))
	{
		SendMsg("<font color=\"orange\">Ignoring download failure as advised.</font><br>\n");
		return true;
	}
	return false;
}

void expiration::_LoadIgnoreList()
{
	filereader reader;
	string sTmp;
	if(reader.OpenFile(acfg::confdir+sPathSep+"ignore_list"))
	{
		while(reader.GetOneLine(sTmp))
		{
			trimLine(sTmp);
			if(!sTmp.empty() && sTmp[0]!='#')
				m_ignList.insert(sTmp);
		}
	}
}

bool expiration::_CanBeIgnored(const MYSTD::string & sCand)
{
	return ( m_ignList.find(sCand) != m_ignList.end()); 
}

void expiration::_UpdateVolatileFiles()
{
	_LoadIgnoreList();
	
	// TODO: implement pdiff reconstruction
	dlcon dl(true);
	tStrVec extraFiles;

	for(tStrVec::const_iterator it=m_volatileFiles.begin(); it!=m_volatileFiles.end(); it++)
	{	
		string sErr;
		if (!_DownloadOne(dl, *it, sErr))
		{
			m_nErrorCount++;
			SendMsg("Error while redownloading this file!<br> ");
			if (!sErr.empty())
				SendMsg(string("Reason: ")+sErr+"<br>");
		}
		
		// last but not least: some voodoo, make sure we have the full version
		string path=*it;
		tStrPos pos=path.find("Packages.diff");
		if(pos!=stmiss)
			path.replace(pos+9, path.length()-pos+9, "bz2");
		else if(pos=path.find("Sources.diff"), pos!=stmiss)
			path.replace(pos+8, path.length()-pos+8, "bz2");	
		else
			continue;
		// candidate of that already seen in some queue?
		string gzVersion=path;
		gzVersion.replace(path.length()-3, 3, "gz");
		for(tStrVec::const_iterator it=extraFiles.begin(); it!=extraFiles.end(); it++)
		{
			if(path==*it || gzVersion==*it)
				goto do_next_ifile;
		}
		for(tStrVec::const_iterator it=m_volatileFiles.begin(); it!=m_volatileFiles.end(); it++)
		{	
			if(path==*it || gzVersion==*it)
				goto do_next_ifile;
		}
		extraFiles.push_back(path);
		extraFiles.push_back(gzVersion);
		do_next_ifile:
		;
	}
	if(!extraFiles.empty())
		SendMsg("Trying to download index files from guessed locations...<br>");
	for(tStrVec::const_iterator it=extraFiles.begin(); it!=extraFiles.end(); it++)
	{
		if(it->empty())
			continue;
				
		string sErr;
		const string &path=*it;
		
		if(_DownloadOne(dl, path, sErr))
		{
			if(path[path.size()-1]=='2')
				it++; // skip .gz version if .bz2 was downloaded
			SendMsg(" ...OK!<br>\n");
			m_volatileFiles.push_back(path);
		}
		else
		{
			SendMsg("Error while redownloading this file!<br> ");
			if(!sErr.empty())
				SendMsg(string("Reason: ")+sErr+"<br>");
		}
			
	}
}

inline void expiration::_ParseVolatileFilesAndHandleEntries()
{
	for(tStrVec::const_iterator it=m_volatileFiles.begin(); it!=m_volatileFiles.end(); it++)
	{
		if(it->empty() || it->at(0)!=cPathSepUnix)
			continue;
		
		SendMsg(string("Parsing metadata in ")+it->substr(m_nDropPrefLen)+"<br>\n");
		filereader reader;
		if(!reader.OpenFile(*it))
		{
			SendMsg("<font color=red>ERROR: unable to open or read this file</font><br>");
			continue;
		}
		string sLine, key, val;
		const string sFile=reader.GetPureFilename();

		// pre calc relative base folders for later
		string sDirname("/");
		string sDebBaseDir("/"); // may differ from sDirname if the path looks like a Debian mirror path 
		tStrPos pos=it->rfind(sPathSep);
		if(stmiss!=pos)
		{
			sDirname.assign(*it, 0, pos+1);
			pos=sDirname.rfind("/dists/");
			if(stmiss!=pos)
				sDebBaseDir.assign(sDirname, 0, pos+1);
			else
				sDebBaseDir=sDirname;
		}
		
		tRemoteFileInfo info;
		info.SetInvalid();
		
		if (sFile=="Packages")
		{
			string csNeeded("MD5sum"), fnNeeded("Filename"), szNeeded("Size");
			
			for (bool bNoEof=true; bNoEof;)
			{
				bNoEof=reader.GetOneLine(sLine);
				trimBack(sLine);
				//cout << "file: " << *it << " line: "  << sLine<<endl;
				if (sLine.empty() || !bNoEof)
				{
					if(info.IsUsable())
						_HandlePkgEntry(info);
					info.SetInvalid();

					{
						lockguard g(&abortMx);
						if(bSigTaskAbort)
							return;
					}
					
					continue;
				}
				else if (_ParseLine(sLine, key, val))
				{
					// not looking for data we already have
					if(/*CSTYPE_INVALID == info.fpr.csType &&*/ 32==val.length() && key==csNeeded)
					{
						info.fpr.csType=CSTYPE_MD5;
						info.fpr.ReadCsFromString(val);
					}
					else if(/*0==info.fpr.size && */key==szNeeded)
						info.fpr.size=atol(val.c_str());
					else if(/*info.sFileName.empty() && */key==fnNeeded)
					{
						info.sDirectory=sDebBaseDir;
						tStrPos pos=val.rfind(sPathSepUnix);
						if(pos==stmiss)
							info.sFileName=val;
						else
						{
							info.sFileName=val.substr(pos+1);
							info.sDirectory.append(val, 0, pos+1);
						}
					}
				}
			}
		}
		else // sources-style formats
		{
			/* pickup all common information, then parse the remaining lines and
			 * assign the tokens to fields in each call of the callback function
			 */
			tStrVec vSumSizeName;
			string sStartMark;
			bool bUse(false);
			
			if (sFile=="Index")
			{
				info.fpr.csType = CSTYPE_SHA1;
				info.fpr.bUnpack = true;
				info.sDirectory=sDirname; // same for all
				sStartMark="SHA1-Patches:";
			}
			else if (sFile=="Sources")
			{
				info.fpr.csType = CSTYPE_MD5;
				info.fpr.bUnpack = false;
				sStartMark="Files:";
			}
			else if (sFile=="Release")
			{
				info.fpr.csType = CSTYPE_MD5;
				info.fpr.bUnpack = false;
				sStartMark="MD5Sum:";
			}
			else
				continue;
			const char typehint=sFile[0];
			
			for (bool bNoEof=true; bNoEof;)
			{
				bNoEof=reader.GetOneLine(sLine);
				trimBack(sLine);
				//if(sLine.find("unp_")!=stmiss)
				//	int nWtf=1;
				//cout << "file: " << *it << " line: "  << sLine<<endl;
				if (sLine.empty() || !bNoEof)
				{
					// optional, but better be sure
					info.sDirectory.clear();
					continue;
				}
				else if(startsWith(sLine, sStartMark))
					bUse=true;
				else if(startsWithSz(sLine, "Directory:"))
				{
					trimBack(sLine);
					tStrPos pos=sLine.find_first_not_of(SPACECHARS, 10);
					if(pos!=stmiss)
						info.sDirectory=sDebBaseDir+sLine.substr(pos)+sPathSepUnix;
				}
				else if (bUse && sLine[0]==' ' && 3==Tokenize(sLine,
						"\t ", vSumSizeName) && vSumSizeName[0].length()>30
						)
						
				{
				
					info.fpr.size = atol(vSumSizeName[1].c_str());
					if(info.fpr.size && info.fpr.ReadCsFromString(vSumSizeName[0]))
					{

						if (typehint=='S')//ources
							info.sFileName=vSumSizeName[2];
						else if (typehint=='I')//ndex
							// We cheat! Files there are compressed and thus the checksum is wrong, deal with that later
							info.sFileName=vSumSizeName[2]+".gz";
						else if (typehint=='R')//elease
						{
							// usually has subfolders
							pos=vSumSizeName[2].rfind(sPathSepUnix);
							if (stmiss!=pos)
							{
								info.sFileName=vSumSizeName[2].substr(pos+1);
								info.sDirectory=sDirname
										+vSumSizeName[2].substr(0, pos+1);
							}
							else // something new in main folder? unlikely...
							{
								info.sFileName=vSumSizeName[2];
								info.sDirectory=sDirname;
							}
						}
						// no else, caught above

						_HandlePkgEntry(info);

						{
							lockguard g(&abortMx);
							if(bSigTaskAbort)
								return;
						}
					}
				}
			}
		}

		if (!reader.CheckGoodState(false))
		{
			SendMsg("<font color=red>An error occured while reading this file, some contents may have been ignored.</font><br>");
			m_nErrorCount++;
			continue;
		}
		//		cout << "found package files: "<< m_trashCandidates.size()<<endl;
	}
}

void expiration::_HandlePkgEntry(const tRemoteFileInfo &entry)
{

	// singleton -> keep some things static, compiler optimization
	static pair<tS2DAT::iterator, tS2DAT::iterator> fromTo;
	// range of files having at least the same file names
	fromTo=m_trashCandSet.equal_range(entry.sFileName);
	
	if(fromTo.first==fromTo.second)
		return; // just no hits, make it quick
			
	static tS2DAT::iterator itmp, it;
	
	for(it=fromTo.first;
	it!=fromTo.second;
	/* erases inside -> step forward there */ )
	{
		
		tFingerprint & fprFile=it->second.fpr;
		string sFullPath(acfg::cachedir+sPathSep+it->second.sDirname+it->first);
		header h;
		
		// needs to match the exact file location
		if( (m_bByPath || m_bByChecksum)
				&& sFullPath != (entry.sDirectory+entry.sFileName))
		{
			goto package_mismatch;
		}
		
		if (0<h.LoadFromFile(sFullPath+".head"))
		{
			if (h.h[header::CONTENT_LENGTH])
			{
				// FIXME, LFS
				off_t len=atol(h.h[header::CONTENT_LENGTH]);
				struct stat stinfo;
				if (0==stat(sFullPath.c_str(), &stinfo) && len<stinfo.st_size)
				{
					SendMsg(string("<font color=red>WARNING, header file reported too small file size, invalidated ")
							+it->first+"</font><br>\n");
					goto package_mismatch;
				}
			}
		}
		else
			SendMsg(string("<font color=\"orange\">WARNING, header file missing for ")
					+sFullPath+"</font><br>\n");
		
		if(m_bByChecksum)
		{
			bool bSkipDataCheck=false;
			
			// scan file if not done before
			if(CSTYPE_INVALID == fprFile.csType)
			{
				// hints for checksumming to read what we need
				fprFile.csType=entry.fpr.csType;
				fprFile.bUnpack=entry.fpr.bUnpack;
				// finally get the physical file size and checksum of the contents
				if(!fprFile.ReadCsFromFile(sFullPath))
				{
					// IO error? better keep it for now
					aclog::err(string("An error occured while checksumming ")
							+sFullPath+", not touching it.");
					bSkipDataCheck=true;
				}
			}

			if ( !bSkipDataCheck)
			{
				if ( ! (fprFile == entry.fpr))
				{
					SendMsg(string("<font color=red>BAD: ")+sFullPath
					+"</font><br>\n");
					goto package_mismatch;
				}
			}
		}
		
		// ok, package matched, contents ok if checked, drop it from the removal list
		if (m_bVerbose)
			SendMsg(string("<font color=green>OK: ")+it->second.sDirname
					+it->first+"</font><br>\n");

		itmp=it;
		itmp++;

		m_trashCandHeadSet.erase(it->second.sDirname+it->first+".head");
		m_trashCandSet.erase(it);

		it=itmp;
		continue;

		package_mismatch:
		it++;
	}
}

inline void expiration::_RemoveAndStoreStatus(bool bPurgeNow)
{
	FILE *f(NULL);
    if(!bPurgeNow) 
    {
        string sDbFile=acfg::cachedir+sPathSep+"_expending_dat";

        f = fopen(sDbFile.c_str(), "w");
        if(!f)
        {
            SendMsg("Unable to open _expending.dat for writting, trying to recreate... ");
            ::unlink(sDbFile.c_str());
            f=::fopen(sDbFile.c_str(), "w");
            if(f)
                SendMsg("OK<br>");
            else
            {
                SendMsg("<font color=red>FAILED. ABORTING. Check filesystem and file permissions.");
                return;
            }
        }
    }
	
	int n(0);
	for(tS2DAT::iterator it=m_trashCandSet.begin(); 
            it!=m_trashCandSet.end(); it++)
	{
        if(m_rex.IsInWhitelist(it->first.c_str()))
        	continue;

		//cout << "Unreferenced: " << it->second.sDirname << it->first <<endl;

		string sWhat=acfg::cachedir+sPathSep+it->second.sDirname+it->first;
		
		// never purge headers in the remains cleanup below, only explicitely
	    m_trashCandHeadSet.erase(it->second.sDirname+it->first+".head");
	    //cout << "Took " << sWhatHead << " from the list" <<endl;

		if(bPurgeNow || (it->second.nLostAt < (m_nTimeNow-acfg::extreshhold*86400)))
		{
			SendMsg(string("Removing ")+it->second.sDirname+it->first+"<br>\n");

#ifdef ENABLED
			::unlink(sWhat.c_str());
			::unlink((sWhat+".head").c_str());
#endif
        }
		else if(f)
		{
			n++;
			fprintf(f, "%lu\t%s\t%s\n",  it->second.nLostAt,
					it->second.sDirname.c_str(), it->first.c_str());
		}
	}
    if(f)
        fclose(f);

    // now just kill dangling header files
	for(set<string>::iterator it=m_trashCandHeadSet.begin();
            it!=m_trashCandHeadSet.end(); it++)
	{
        // keep Release etc.
		if(m_rex.IsInWhitelist(it->c_str()))
            continue;
		
        cout << "Removing orphaned head file: " << *it << endl;
#ifdef ENABLED
        ::unlink((acfg::cachedir+sPathSep+*it).c_str());
#endif
    }
    if(n>0)
    {
        char buf[100];
        snprintf(buf, _countof(buf), "<br>%d package file(s) marked for removal in few days.<br><br>\n", n);
        SendMsg(buf);
    }
}



void expiration::Action(const string & cmd)
{
	
	if (cmd.find("justRemove")!=stmiss)
	{
		_LoadTrashMapFromFile(true);
		_RemoveAndStoreStatus(true);
		return;
	}
	if (cmd.find("justShow")!=stmiss)
	{
		_LoadTrashMapFromFile(true);
		for (tS2DAT::iterator it=m_trashCandSet.begin(); it
				!=m_trashCandSet.end(); it++)
		{
			SendMsg(it->second.sDirname+it->first+"<br>");
			SendMsg(it->second.sDirname+it->first+".head<br>");
		}
		return;
	}
	
	SendMsg("Locating potentially expired files in the cache...<br>\n");

	m_bErrAbort=(cmd.find("abortOnErrors=aOe")!=stmiss);
	m_bByPath=(cmd.find("byPath")!=stmiss);
	m_bByChecksum=(cmd.find("byChecksum")!=stmiss);
	m_bVerbose=(cmd.find("beVerbose")!=stmiss);
	m_bForceDownload=(cmd.find("forceRedownload")!=stmiss);
	

	DirectoryWalk(acfg::cachedir, this);
	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return;
	}
	//cout << "found package files: " << m_trashCandidates.size()<<endl;

	SendMsg("Redownloading index files...<br>\n");

	_UpdateVolatileFiles();
	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return;
	}
	
	if (m_bErrAbort && m_nErrorCount>0)
	{
		SendMsg("<font color=\"red\">Found errors during processing, aborting as requested.</font>"
			"<!-- TELL:THE:ADMIN -->");
		return;
	}

	SendMsg("Identifying fresh files...<br>\n");
	_ParseVolatileFilesAndHandleEntries();
	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return;
	}
	
	// update timestamps of pending removals
	_LoadTrashMapFromFile(false);
	
	_RemoveAndStoreStatus(cmd.find("purgeNow")!=stmiss);
	_PurgeMaintLogs();
	SendMsg("Done.");

}


time_t nMaintId(0), nLastId(0);
lockable idLock;

void expiration::DumpLog(time_t id)
{
	filereader reader;
	size_t size;
	const char *data(NULL);

	if (id<=0)
		goto log_na;

	char buf[PATH_MAX];
	snprintf(buf, sizeof(buf), "%s"SZPATHSEP"maint_%lu.log",
			acfg::logdir.c_str(), id);
	if (!reader.OpenFile(buf))
		goto log_na;

	data=reader.GetBuffer(size);
	maintenance::SendMsg(data, size);
	return;

	log_na:
	SendMsg(string("Log not available"));
}

/*
 *  TODO: this is kept in expiration class for historical reasons. Should be moved to some shared upper
 * class, like "detachedtask" or something like that
 */
void expiration::Run(const string &cmd)
{

	if (cmd.find("&sigabort")!=stmiss)
	{
		lockguard g(&abortMx);
		bSigTaskAbort=true;
		const char hs[]="HTTP/1.1 302 Redirect\r\nLocation: ";
		const char head[] = "\r\nConnection: close\r\n\r\n";
		tStrPos nQuest=cmd.find("?");
		if(nQuest==stmiss)
			return; // whatever
		_SendHelper(hs, sizeof(hs)-1, MSG_MORE);
		_SendHelper(cmd.c_str(), nQuest, MSG_MORE);
		_SendHelper(head, sizeof(head)-1, MSG_MORE);
		return;
	}
			
	SendDecoration(true, "200", m_szDecoFile);

	static lockable onlyOneCanRun;

	if(!onlyOneCanRun.tryLock())
	{
		if(cmd.find("&readonly")==stmiss)
		{
			string msg("<font color=blue>A maintenance task is already running!</font><br>"
					"<ul><li><a href=");
			msg+=cmd+"&readonly#bottom";
			msg+=">View current log messages</a></li><li><a href=/";
			msg+=acfg::reportpage+">Return to main page</a></li></ul>"+
			"<p>(<a href="+cmd+"&sigabort>Abort current task ASAP</a>)";
			
			SendMsg(msg);
		}
		else
		{
			time_t id(0);
			{
				lockguard g(idLock);
				id=nMaintId;
			}
			DumpLog(id);
			string msg("<br><a name=bottom>To be continued... (reload this page to see recent output)<br><a href=/");
			msg	+=acfg::reportpage+">Return to main page</a>\n";
			SendMsg(msg);
						
		}
	}
	else
	{
		// locking worked but not intended to act now
		if(cmd.find("&readonly")!=stmiss)
		{
			string msg("<font color=blue>Previous maintainenance task is completed!</font><br>"
						"<ul><li><a href=/");
			msg	+=acfg::reportpage+">Return to main page</a></li></ul><p>Last log contents:<br>\n";
			SendMsg(msg);

			DumpLog(nLastId);
		}
		else
		{
			/*****************************************************
			 * This is the worker part
			 *****************************************************/
			{
				lockguard g(idLock);
				nMaintId=time(NULL);
			}

			char buf[PATH_MAX];
			snprintf(buf, sizeof(buf), "%s"SZPATHSEP"maint_%lu.log",
					acfg::logdir.c_str(), nMaintId);
			m_repFileStream=fopen(buf, "w");

			{
				lockguard g(&abortMx);
				bSigTaskAbort=false;
			}
			
			Action(cmd);
			
			{
				lockguard g(&abortMx);
				bSigTaskAbort=false;
			}

			SendMsg(string("<br><a href=\"/")+acfg::reportpage
					+"\">Return to main page</a>");

			{
				lockguard g(idLock);
				nLastId=nMaintId;
				nMaintId=0;
			}

			if(m_repFileStream)
			{
				fclose(m_repFileStream);
				m_repFileStream=0;
			}
		}
		onlyOneCanRun.unlock();
	}
	SendDecoration(false, "", m_szDecoFile);
}

void expiration::SendMsg(const string &x)
{
	if (m_repFileStream)
	{
		fwrite(x.data(), sizeof(char), x.length(), m_repFileStream);
		fflush(m_repFileStream);
	}
	maintenance::SendMsg(x);
}


void expiration::_PurgeMaintLogs()
{
	glob_t globbuf;
	memset(&globbuf, 0, sizeof(glob_t));
	string pattern=acfg::logdir+SZPATHSEP"maint_*.log";
	glob(pattern.c_str(), GLOB_DOOFFS | GLOB_NOSORT, NULL, &globbuf);
	if (globbuf.gl_pathc>2)
	{
		SendMsg("Found required cleanup tasks: purging maintanence logs...<br>\n");
	}
	for (unsigned int i=0; i<globbuf.gl_pathc; i++)
	{
		//cerr << "Checking "<<globbuf.gl_pathv[i];
		// skip our log
		time_t id=atol(globbuf.gl_pathv[i]+acfg::logdir.size()+7);
		//cerr << "id ist: "<< id<<endl;
		if (id==nMaintId)
			continue;
		//cerr << "Remove: "<<globbuf.gl_pathv[i]<<endl;
#ifdef ENABLED
		::unlink(globbuf.gl_pathv[i]);
#endif
	}
	globfree(&globbuf);

}


