
#include "meta.h"
#include "caddrinfo.h"
#include "acfg.h"
#include "debug.h"
#include "lockable.h"

using namespace MYSTD;

static map<string, CAddrInfo::SPtr> mapDnsCache;
static pthread_mutex_t lockDnsCache= PTHREAD_MUTEX_INITIALIZER;

CAddrInfo::CAddrInfo() :
	m_nResTime(0), m_bestInfo(NULL), m_resolvedInfo(NULL)
{
}

bool CAddrInfo::Resolve(const string & sHostname, const string &sPort,
		string & sErrorBuf)
{
	LOGSTART2("CAddrInfo::Resolve", "Resolving " << sHostname);

	static struct addrinfo hints =
	{
	// we provide numbers, no resolution needed; only supported addresses
			AI_NUMERICSERV|AI_ADDRCONFIG, PF_UNSPEC, SOCK_STREAM, IPPROTO_TCP,
			0, NULL, NULL, NULL };
#if 0
	struct addrinfo hints;
	memset(&hints, 0, sizeof(hints));
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = 0;
#endif	

	if (m_resolvedInfo)
	{
		freeaddrinfo(m_resolvedInfo);
		m_resolvedInfo=NULL;
	}
	// TODO: allow arbitrary ports? Get them from hostname?
	int r=getaddrinfo(sHostname.c_str(), sPort.c_str(), &hints,
					&m_resolvedInfo);

	if (0!=r)
	{
		sErrorBuf=(tSS(250)<<"503 DNS error for hostname "<<sHostname<<": "<<gai_strerror(r)
				<<". If "<<sHostname<<" refers to a configured cache repository, "
				"please check the corresponding configuration file.");
		return false;
	}

	LOG("Host resolved: " << m_resolvedInfo->ai_canonname);

	for (struct addrinfo * pCur=m_resolvedInfo; pCur; pCur = pCur->ai_next)
	{
		if (pCur->ai_socktype == SOCK_STREAM&& pCur->ai_protocol == IPPROTO_TCP)
		{
			m_bestInfo=pCur;
			m_nResTime=time(NULL);
			return true;
		}
	}
	sErrorBuf="500 DNS resolution error";

	// TODO: remove me from the map
	return false;
}
;
CAddrInfo::~CAddrInfo()
{
	if (m_resolvedInfo)
		freeaddrinfo(m_resolvedInfo);
}
	

CAddrInfo::SPtr CAddrInfo::CachedResolve(const string & sHostname, const string &sPort, string &sErrorMsgBuf)
{
	SPtr p;
	time_t timeExpired=time(NULL)-acfg::dnscachetime;

	map<string, SPtr>::iterator it;
	{
		lockguard g(&lockDnsCache);
		it=mapDnsCache.find(sHostname);
		if (it != mapDnsCache.end())
		{
			if (it->second->m_nResTime > timeExpired)
				return it->second; // fresh enough
			p=it->second;
		}
	}

	bool bAddToCache(false); // don't readd, optimization
	if (!p)
	{
		p.reset(new CAddrInfo);
		bAddToCache=true;
	}

	if (p->Resolve(sHostname, sPort, sErrorMsgBuf))
	{
		if (bAddToCache)
		{
			lockguard g(&lockDnsCache);
			if (mapDnsCache.size()>300) // DOS attack?
				mapDnsCache.clear();
			mapDnsCache[sHostname]=p;
		}
	}
	else // error, return empty ptr
	{
		// better log here, often APT does not display stupid user errors
		aclog::err( (tSS()<<"Error resolving "<<sHostname<<": " <<sErrorMsgBuf).c_str());
		p.reset();
	}

	return p;
}

