
//#define LOCAL_DEBUG
#include "debug.h"

#include "expiration.h"
#include "lockable.h"
#include "acfg.h"
#include "meta.h"
#include "filereader.h"
#include "fileitem.h"
#include "dlcon.h"
#include "dirwalk.h"
#include "header.h"
#include "job.h"
#include "dljob.h"

#include "fileio.h"
#include <errno.h>
#include <unistd.h>
#include <dirent.h>

#include <map>
#include <string>
#include <iostream>

using namespace MYSTD;

#define ENABLED
#ifndef ENABLED
#warning defused
#endif

expiration::expiration(int fd) :
	bgtask(fd), m_nTimeNow(0), m_nDropPrefLen(0),
	m_bErrAbort(false), m_bVerbose(false), m_bForceDownload(false), 
	m_bScanInternals(false), m_nErrorCount(0),
	m_nProgIdx(0), m_nProgTold(1)
{
	m_szDecoFile="maint.html";

	m_nDropPrefLen=acfg::cachedir.size()+1;
	m_nTimeNow=time(NULL);

}

expiration::~expiration()
{

}
bool expiration::ProcessOthers(const string & sPath, const struct stat &)
{
	// NOOP
	return true;
}

void expiration::AddIFileCandidate(const string &sPath)
{
	if (rechecks::FILE_INDEX ==	rechecks::GetFiletype(sPath)
	// SUSE stuff, not volatile but also contains file index data
	|| endsWithSzAr(sPath, ".xml.gz") )
    {
		bool probeIdata=false;
		_ParseAndProcessIndexFile(sPath, sPathSep, sPathSep, &probeIdata);
		if(probeIdata)
			m_indexFiles.insert(sPath);
    }

}

bool expiration::ProcessRegular(const string & sPath, const struct stat &stinfo)
{

	if(CheckAbortCondition())
		return false;

	if (m_nProgIdx++>m_nProgTold)
	{
		m_nProgTold*=2;
		maintenance::SendChunk(tSS(50)<<"Scanning, found "<<m_nProgTold+1<<" files...<br />\n");
	}

	if(0==sPath.compare(m_nDropPrefLen, 1, "_") && !m_bScanInternals)
		return true; // not for us

	tStrPos nSlashPos=sPath.rfind(CPATHSEP);
	string sBasename(sPath, nSlashPos+1);
	
	// handle the head files separately
    if (endsWithSzAr(sPath, ".head"))
    {
        m_trashCandHeadSet.insert(sPath.substr(m_nDropPrefLen));
        return true;
    }

    AddIFileCandidate(sPath);

    string sDirnameAndSlash(sPath, m_nDropPrefLen, nSlashPos-m_nDropPrefLen+1);
    pair<string,tDiskFileInfo> fileDesc(sBasename,
        		tDiskFileInfo(sDirnameAndSlash, m_nTimeNow));
    fileDesc.second.fpr.size = stinfo.st_size;
    
    m_trashCandSet.insert(fileDesc);
    return true;
}

void expiration::_LoadTrashMapFromFile(bool bForceInsert)
{

	filereader reader;
	reader.OpenFile(acfg::cachedir+SZPATHSEP+"_expending_dat");
	
	string sLine;

#if 0
	// stuff for user info
	time_t now=time(NULL);
	time_t oldest(now), newest(0);
#endif
	
	while(reader.GetOneLine(sLine))
	{
		tStrVec elems;
		if(Tokenize(sLine, "\t", elems)==3)
		{
			time_t timestamp=(time_t) atoll(elems[0].c_str());
			if(timestamp>m_nTimeNow)
				continue; // where is the DeLorean?
			
			if(elems[1].empty() || elems[2].empty())
				continue;
			
			if(elems[1].at(elems[1].size()-1)!=SZPATHSEP[0])
				elems[1].append(SZPATHSEP);
			
			if(bForceInsert)
			{
				m_trashCandSet.insert(
						pair<string, tDiskFileInfo>(elems[2], 
								tDiskFileInfo(elems[1], 0)));
				continue;
			}
			
			static pair<tS2DAT::iterator, tS2DAT::iterator> fromTo;
			fromTo=m_trashCandSet.equal_range(elems[2]);
			// file already gone?
			if(fromTo.first==fromTo.second)
				continue; // file already gone?
			
			// TODO: make global iterators for the other function too?
			static tS2DAT::iterator itmp, it;
			for(it=fromTo.first; it!=fromTo.second; it++)
				if(it->second.sDirname == elems[1]) // hit
					it->second.nLostAt=timestamp;										

#if 0
			if(timestamp < oldest)
				oldest=timestamp;
			if(timestamp > newest)
				newest=timestamp;
#endif
		}
	}
	
#if 0
	/*
	cout << "Unreferenced: ";
	for(trashmap_t::iterator it=m_trashCandidates.begin(); it!=m_trashCandidates.end(); it++)
		fprintf(stdout, "%lu\t%s\t%s\n",  it->second.first, it->second.second.c_str(), it->first.c_str());
	*/
	
	if(m_trashCandidates.size()>0)
	{
		// pretty printing for impatient users
		char buf[200];
		
		// map to to wait
		int nMax=acfg::extreshhold-((now-oldest)/86400);
		int nMin=acfg::extreshhold-((now-newest)/86400);
				
		snprintf(buf, _countof(buf), "Previously detected: %lu rotten package file(s), "
				"to be deleted in about %d-%d day(s)<br>\n", 
				(unsigned long) m_trashCandidates.size(),
				nMin, nMax==nMin?nMax+1:nMax); // cheat a bit for the sake of code simplicity
		SendChunk(buf);
	}
#endif

}

class tDlJobHints
{	
public:
	const MYSTD::string *psTargetBasePath, *psSrcTmpPath;
	const tStrSet *pGoodChecklist;
};

class tDlJobBzip2Injector : public tDlJob
{
public:
	tDlJobHints *m_pHints;
	tDlJobBzip2Injector(dlcon *m, tFileItemPtr pFi, const tHttpUrl &url, tDlJobHints *pHints)
	: tDlJob(m, pFi, url.sHost, url.sPath), m_pHints(pHints)
	{
		m_sRequestCmd="HEAD ";
	}
	tDlJobBzip2Injector(dlcon *m, tFileItemPtr pFi, acfg::tHostiVec * pBackends,
			const MYSTD::string & sPath, tDlJobHints *pHints)
	: tDlJob(m, pFi, pBackends, sPath), m_pHints(pHints)
	{
		m_sRequestCmd="HEAD ";
	}
protected:
	virtual tDlResult NewDataHandler(acbuf & inBuf, string &sErrorRet)
	{
		string sBzFn, sBzTarget;
		// some mirrors don't use -9, and some might use the mt version
		const char *szCmds[] = { "bzip2 -k -9 '",
			"bzip2 -k '", "pbzip2 -k -9 '", "pbzip2 -k '" };

		if (!m_pHints || !m_pHints ->pGoodChecklist || !m_pHints->psSrcTmpPath
				|| !m_pHints->psTargetBasePath)
			goto repErrLocal;

		sBzFn = *(m_pHints->psSrcTmpPath) + ".bz2";
		sBzTarget = *(m_pHints->psTargetBasePath) + ".bz2";

		{ // just some sanity checks...

			lockguard g(m_pStorage.get());
			const header & h = m_pStorage->GetHeaderUnlocked();

			if (!h.h[header::CONTENT_LENGTH])
				goto repErrLocal;

			// .bz2 index and larger than 30mb? unlikely
			if (atoofft(h.h[header::CONTENT_LENGTH]) > 30000000)
				goto repErrLocal;
		}

		for (UINT i = 0; i < _countof(szCmds); i++)
		{
			::unlink(sBzFn.c_str());

			string cmd(szCmds[i]);
			cmd += *m_pHints->psSrcTmpPath + "'";
			if (0 != ::system(cmd.c_str()))
				continue;

			uint8_t cs[20];
			off_t nSize(0);
			filereader reader;

			if (reader.OpenFile(sBzFn, true) && reader.GetSha1Sum(cs, nSize))
			{
				// XXX lltos? For LFS?
				if (m_pHints->pGoodChecklist->find(ltos(nSize) + "_"
						+ CsBinToString(cs, 20))
						!= m_pHints->pGoodChecklist->end())
				{
					// hit! Looks okayish, try to pass data through

					if (!m_pStorage->StoreFileData(reader.GetBuffer(),
							reader.GetSize()))
					{
						goto repErrLocal;
					}

					m_DlState = STATE_FINISHJOB;
					return R_NEXTSTATE;
				}
			}

		}

		repErrLocal: sErrorRet = "567 Failed to reconstruct bzip'ed file";
		if (!sBzFn.empty())
			::unlink(sBzFn.c_str());

		return R_ERROR_LOCAL;
	}
};


void expiration::_InjectBz2ed(const MYSTD::string & sTargetBasePath, 
		const MYSTD::string & sSrcTmpPath, const tStrSet & goodChecklist)
{
	if(!acfg::recompbz2)
		return;
	
	class tDlJobHints hints;
	hints.pGoodChecklist=&goodChecklist;
	hints.psSrcTmpPath=&sSrcTmpPath;
	hints.psTargetBasePath=&sTargetBasePath;
	string junk;
	
	dlcon freshDler(true); // get a new connection, work around timeouts
	SendChunk("Attempting to inject bz2ed version...");
	if(_UpdateOne(freshDler, sTargetBasePath+".bz2", junk, &hints))
		SendChunk("succeeded<br>\n");
	else
		SendChunk("skipped<br>\n");
}


bool expiration::_UpdateOne(dlcon &dl, const MYSTD::string & sFilePath, string &sErr, tDlJobHints *pSpecialHints)
{
	sErr.clear();

	if(m_uptodateTags.find(sFilePath) != m_uptodateTags.end())
		return true; // this is recent enough
	
	tFileItemPtr fi;
	tHttpUrl url;
	acfg::tHostiVec *pBackends(NULL);

	string sKeyPath=sFilePath.substr(acfg::cachedir.size()+1, stmiss);
	
	if(!pSpecialHints) // running wrapped, STFU
		SendChunk(tSS()<<"Checking/Updating "<<sKeyPath<<"...\n");
	
	fi=fileitem::GetFileItem(sKeyPath);
	if (fi)
	{
		fi->Setup(true, m_bForceDownload);
	}
	else
	{
		sErr=" could not create file item handler.";
		goto rep_update_prob;
	}
	

#ifdef WIN32
#error rewrite for the backslashes or change reliably in the walker to slashes
#endif

	/* 
	 * Three ways to find the source to download from:
	 * 1. Interpret the base directory as repository name
	 * 2. Interpret the whole subpath as URL (host/path)
	 * 3. Use X-Original-Source (which might be broken)
	 */
	
	// abusing url class to extract base directory

	if (!url.SetHttpUrl(sKeyPath))
	{
		// should never happen, though
		sErr=sKeyPath+" does not seem to contain valid repository or host name.";
		goto rep_update_prob;
	}
	
	pBackends = acfg::GetBackendVec(&url.sHost);
	
	if (pBackends)
	{
		// HIT, control by backend scheme, strip leading slash from path
		
		if(pSpecialHints)
			dl.EnqJob(new tDlJobBzip2Injector(&dl, fi, pBackends, url.sPath.substr(1), pSpecialHints));
		else
			dl.AddJob(fi, pBackends, url.sPath.substr(1));
	}
	else
	{
		tHttpUrl urlOrig;
		string sOrig;
		{
			lockguard g(fi.get());
			const header & pHead=fi->GetHeaderUnlocked();
			if (pHead.h[header::XORIG])
				sOrig=pHead.h[header::XORIG];
		}
		if(startsWithSz(sOrig, "http://") && urlOrig.SetHttpUrl(sOrig))
		{
			// ok, looks valid, is it better than the one from the path?
			if(url != urlOrig)
			{
				if(urlOrig.sHost.find(".") != stmiss)
				{
					if(url.sHost.find(".") != stmiss)
					{
						// Both have dots, prefer directory as host
						//goto dl_from_url;
					}
					else
					{
						// dir has no dots, orig-url host has -> use orig url
						url=urlOrig;
						//goto dl_from_url;
					}
				}
				else // no dots in urlOrig host, most likely broken
				{
					/*
					 * if the directory has dots, use it and be quiet (unless verbosity is enabled).
					 * Otherwise, warn the user.
					 * */
					if (m_bVerbose || url.sHost.find(".") != stmiss)
					{
						SendChunk(tSS()<<"<font color=\"orange\">Code 520824! "
							"Read the manual about known bugs! Attempting to use "
								<< url.sHost << " as hostname</font>");
					}
				}
			}
		}
		if(pSpecialHints)
			dl.EnqJob(new tDlJobBzip2Injector(&dl, fi, url, pSpecialHints));
		else
			dl.AddJob(fi, url);
	}	

	dl.WorkLoop();

	if (fi->GetStatus() == FIST_ERROR)
		goto rep_update_prob;

	//if(!pSpecialHints)
		SendChunk(tSS() << "<i>(" << fi->GetTransferCount()/1024 << "KiB)</i><br>\n");
	
	if (fi)
	{
		fi->Unreg();
		fi.reset();
	}
	
	m_uptodateTags.insert(sFilePath);
	return true;

	rep_update_prob:

	if (fi)
	{
		sErr+=fi->GetHttpMsg();
		fi->Unreg();
		fi.reset();
	}
	SendChunk("<br>\n");
	
	if(_CanBeIgnored(sKeyPath))
	{
		SendChunk("<font color=\"orange\">Ignoring download failure as advised.</font><br>\n");
		return true;
	}
	return false;
}

void expiration::_LoadIgnoreList()
{
	filereader reader;
	string sTmp;
	if(reader.OpenFile(acfg::confdir+SZPATHSEP+"ignore_list"))
	{
		while(reader.GetOneLine(sTmp))
		{
			trimLine(sTmp);
			if(!sTmp.empty() && sTmp[0]!='#')
				m_ignList.insert(sTmp);
		}
	}
}

bool expiration::_CanBeIgnored(const MYSTD::string & sCand)
{
	return ( m_ignList.find(sCand) != m_ignList.end()); 
}

inline bool file_exists(const char *name, off_t size)
{
	struct stat stbuf;
	if(0!=stat(name, &stbuf))
		return false;
	return (stbuf.st_size == size);
}

void expiration::_UpdateWithDiffs(dlcon &dl, tStrSet & listGuessedLocationsToTry)
{
	
	string sErr;
	filereader reader;
	
	// quick-and-dirty, extracting a bunch of csums/sizes for simple later validity checks later
	tStrSet validCsums; // <size>_<sum>
	
	for (tStrSet::const_iterator it=m_indexFiles.begin(); it
			!=m_indexFiles.end(); it++)
	{
		if ( endsWithSzAr(*it, "Release")
		&& _UpdateOne(dl, *it, sErr)
		&& reader.OpenFile(*it))
		{
			tStrVec tmp;
			string sLine;
			while (reader.GetOneLine(sLine))
			{
				
// XXX vielleicht auch automatisch .diff/Index aufnehmen und bei Bedarf ins m_volfiles stecken?
				
				if (3==Tokenize(sLine, SPACECHARS, tmp) && tmp[0].size()>=32)
				{
					validCsums.insert(tmp[1]+"_"+tmp[0]);
				}
			}
		}
	}
	
	for(tStrSet::const_iterator it=m_indexFiles.begin(); it!=m_indexFiles.end(); it++)
	{

		if( !endsWithSzAr(*it, ".diff/Index") )
			continue;
		string sBaseFn=it->substr(0, it->length()-11);
		
		// just to be sure...
		if(sBaseFn.find('\'') != stmiss)
			continue;
		
		// as last resort later, try to download blindly from there later (comp.versions)
		listGuessedLocationsToTry.insert(sBaseFn);
		
		if(!_UpdateOne(dl, *it, sErr)
			|| !reader.OpenFile(*it) )
		{
			continue;
		}
		//SendMsg(*it + " offen<br>");
		tStrVec stateList, patchSums;
		string sLine;
		string sCurrentCs;
		off_t nCurrentSize(-1);
		bool bErrNext=false;
		enum { eCurLine, eHistory, ePatches} eSection;
		eSection=eCurLine;
		
		while(reader.GetOneLine(sLine))
		{
			//SendMsg(sLine + "<br>");
			if(startsWithSz(sLine, "SHA1-Current: "))
			{
				//SendMsg(sLine + " ist current");
				nCurrentSize=atoofft(sLine.c_str()+54);
				// detect obvious errors
				if(sLine[54] != ' ' || ! nCurrentSize)
				{
					//SendMsg(string(" error? <br>")+sLine[54]);
					bErrNext=true;
					break;
				}
				sCurrentCs=sLine.substr(14, 40);
				//SendMsg(sCurCs + " is curcs");
			}
			else if(startsWithSz(sLine, "SHA1-History:"))
				eSection=eHistory;
			else if(startsWithSz(sLine, "SHA1-Patches:"))
				eSection=ePatches;
			else if(eHistory == eSection || ePatches == eSection)
			{
				tStrVec & outList = (eHistory==eSection) ? stateList : patchSums;
				
				UINT slcount=outList.size();
				
				if(0==Tokenize(sLine, SPACECHARS, outList, true))
					continue;
				
				if(outList.size() != slcount+3)
				{
					bErrNext=true;
					break;
				}
			}
		}
		reader.Close();
		if(bErrNext || stateList.empty()) // XXX: report a recovery error?
			continue;
		// okay, got a list of states
		
		const char *suxe[] = { "", ".bz2", ".gz"};
		for(UINT si=0; si<_countof(suxe); si++)
		{
			uint8_t srcHexCs[20];
			off_t srcSize(0), tmpSize(0);
			uint8_t tmpHexCs[20];
			tSS sCmd;
			sCmd << " ( cat ";
			
			// the names are generated locally with good chars only, sBaseFn part checked before
			string sTmpDir=sBaseFn+".diff/_actmp";
			string sTmpFile=sTmpDir+"/tmp";
			string sMergedFn=sTmpFile+".merged";
			mkbasedir(sTmpFile);

			tStrDeq cleanSet;
			cleanSet.push_back(sTmpFile);
			string sGuessSrc = sBaseFn + suxe[si];
			UINT hit=0;
			if( ! filereader::GetSha1Sum(sGuessSrc, srcHexCs, suxe[si][0], // true when having suffix
					srcSize, sTmpFile.c_str()))
			{
				//SendMsg(sGuessSrc+": file missing or unpack/checksum error<br>");
				hit=UINT_MAX; // to be never found
			}
			
			// check current contents, does it hit the Release entry as-is?
			if(srcSize == nCurrentSize 
					&& CsEqual( (CUCHAR*) sCurrentCs.c_str(), srcHexCs, 20))
			{
				if(m_bVerbose)
					SendChunk(sGuessSrc+": content match, using this version<br>\n"); // Ignore all other files!
				::unlink(sTmpFile.c_str());
				
				// drop any other (older) version from the list, use only the new one
				m_indexFiles.erase(sBaseFn);
				m_indexFiles.erase(sBaseFn+".bz2");
				m_indexFiles.erase(sBaseFn+".gz");
				m_indexFiles.insert(sGuessSrc);
				
				// content is to be considered ok even if header is wrong/missing
				m_uptodateTags.insert(sGuessSrc);
				
				// feed the data set for the dupe filter
				string fpr=sCurrentCs+"_"+ltos(nCurrentSize);
				m_quickDupeHints[sBaseFn]=fpr;
				m_quickDupeHints[sBaseFn+".bz2"]=fpr;
				m_quickDupeHints[sBaseFn+".gz"]=fpr;
				
				goto clean_stuff_and_break;
			}
			
			// ok, then: does it hit some previous state we can patch from?
			for(; hit<stateList.size(); hit+=3)
			{
				//SendMsg(stateList[hit] + " <- testing<br>");				
				if(CsEqual((CUCHAR*) stateList[hit].c_str(),
						srcHexCs, 20)
					&& srcSize==atoofft(stateList[hit+1].c_str()) )
				{
					break; // GOTCHA! The index state to start patching
				}
			}
			if(hit>=stateList.size())
				goto clean_stuff_and_continue; // no hit, try other base file
			
			// Go fetch all patch files
			
			for(UINT pIdx=hit+2; pIdx<stateList.size(); pIdx+=3)
			{
				string sPatchFn=sBaseFn+".diff/"+stateList[pIdx]+".gz";

				string sPatchUncFn = sTmpFile+"."+ltos(pIdx);
				cleanSet.push_back(sPatchUncFn);

				// XXX: also check size/cs of the patches? needing to scan them above?
				if( ! _UpdateOne(dl, sPatchFn, sErr)
						|| ! filereader::GetSha1Sum(sPatchFn, tmpHexCs, true,
								tmpSize, sPatchUncFn.c_str()) )
				{
					aclog::err(sPatchFn+": failed to fetch or unpack the patch file");
					goto clean_stuff_and_continue;
					// XXX To consider: don't break here, keep pre-caching them all for later use. Pro: fetch earlier; Contra: maybe useless, client will never need them when it contatcs us only once a week
				}
				// double check it, just to be safe
				UINT i=0;
				for(; i<patchSums.size(); i+=3)
				{
					//aclog::err(sPatchFn+": "+stateList[pIdx]+" vs. "+ patchSums[i+2]);
					if(stateList[pIdx] == patchSums[i+2])
					{
						if(! CsEqual((CUCHAR*) patchSums[i].c_str(), tmpHexCs, 20)
								|| atoofft(patchSums[i+1].c_str()) != tmpSize)
						{
							aclog::err(sPatchFn+": bad checksum or filesize");
							goto clean_stuff_and_continue;
						}
						break;
					}
				}
				if(i>=patchSums.size())
				{
					aclog::err(sPatchFn+": patch not listed in the file history");
					goto clean_stuff_and_continue;
				}

				// double-check, catch any non-sense therein, should not happen though
				//assert(sPatchUncFn.find('\'') == stmiss);
				if(sPatchUncFn.find('\'') != stmiss)
					goto clean_stuff_and_continue;

				sCmd << " '" << sPatchUncFn + "'";
			}
			sCmd << " ; echo w tmp.merged ) | ( cd '"<<sBaseFn<< ".diff/_actmp/' ; red tmp ) ";
			//SendMsg(sCmd+"<br>");
			//string sMergeFn=sTmpFile+".merged";
			cleanSet.push_back(sMergedFn);
			
			if(0 == system(sCmd.c_str()))
			{
				if(filereader::GetSha1Sum(sMergedFn, tmpHexCs, true, tmpSize)
				&& validCsums.find( ltos(tmpSize)+"_" 
						+ CsBinToString(tmpHexCs, 20)) != validCsums.end())
				{
					// GOOD! Looks very valid
					SendChunk(string("Patching... OK, updating ")+sBaseFn+"<br>");
					::unlink(sBaseFn.c_str());
					::unlink((sBaseFn+".head").c_str());
					
					_InjectBz2ed(sBaseFn, sMergedFn, validCsums);
					
					if(0==::rename(sMergedFn.c_str(), sBaseFn.c_str()))
					{
						// fine, making the new stuff public and ignore the compressed versions instead
						m_indexFiles.erase(sBaseFn+".bz2");
						m_indexFiles.erase(sBaseFn+".gz");
						m_indexFiles.insert(sBaseFn);
						m_uptodateTags.insert(sBaseFn); // we have no .head but that's ok, don't refetch
						goto clean_stuff_and_break; // stop iteration over suffixes
					}
				}
				else
				{
					SendChunk(string("Patching... finished with errors, ")+sBaseFn+" will be redownloaded<br>\n");
				}
			}
			else
			{
				SendChunk("Patching... failed, is ed installed?<br>");
				if(acfg::debug>1)
					SendChunk(sCmd);
			}
			clean_stuff_and_continue:
			for(tStrDeq::iterator it = cleanSet.begin(); it!=cleanSet.end(); it++)
				::unlink(it->c_str());
			::rmdir(sTmpDir.c_str());
			continue;
			
			clean_stuff_and_break:
			// might have no .head but that's ok, don't refetch
			for(tStrDeq::iterator it = cleanSet.begin(); it!=cleanSet.end(); it++)
				::unlink(it->c_str());
			::rmdir(sTmpDir.c_str());
			break;
						
		}
	}

}

void expiration::UpdateIndexFiles()
{
	_LoadIgnoreList();

	// all possible versions based on data of Release files, might be downloadable as .gz or .bz2 variant
	tStrSet listGuessedLocationsToTry;

	dlcon dl(true);
	string sErr; // for downloader output

	_UpdateWithDiffs(dl, listGuessedLocationsToTry);

	/**
	 * Ok, what's still in the list?
	 * 1. ifiles without pdiffs (as found on the FS)
	 * 2. ifiles with pdiffs where reconstruction failed
	 * 3. other kinds of files, maybe having different variants, .bz2 and .gz or even uncompressed
	 *    
	 * Trying to find a compromise between user demands and transfer size. I.e.
	 * if only .gz version exists, refetch the .gz versions, if multiple version exists,
	 * try to get .bz2 and then .gz (on failure, or forget .gz on success).
	 */
	
	// Cleanup the list, remove semi-dupes
	// drop the current version if there is a good replacement somewhere
	// else in the queue
	
	m_indexFiles.insert(""); // make sure there is always a predecessor
	for(tStrSet::iterator it=m_indexFiles.begin();
	it!=m_indexFiles.end(); )
	{
		string sErr;
		string sFn=*it;
		
		// if the other version was downloaded before, skip the current one
		if(endsWithSzAr(sFn, ".bz2"))
		{
			string sBaseFn=sFn.substr(0, sFn.size()-4);
			if(m_uptodateTags.find(sBaseFn)!=m_uptodateTags.end()
					|| m_uptodateTags.find(sBaseFn+".gz")!=m_uptodateTags.end())
				goto drop_file;
		}
		else if(endsWithSzAr(sFn, ".gz"))
		{
			// if the other version was downloaded before, skip the current one
			string sBaseFn=sFn.substr(0, sFn.size()-3);
			if(m_uptodateTags.find(sBaseFn)!=m_uptodateTags.end()
					|| m_uptodateTags.find(sBaseFn+".bz2")!=m_uptodateTags.end())					
				goto drop_file;
			
			// try to get .bz2 version if seen in the cache
			// XXX: there is a small chance that .gz is better even though .bz2 is around
			// 		it might be checksummed and tested against checksum heap (above)
			if(m_indexFiles.find(sBaseFn+".bz2")!=m_indexFiles.end())
			{
				if(_UpdateOne(dl, sBaseFn+".bz2", sErr))
					goto drop_file;
			}
		}
		else
		{
			if(m_uptodateTags.find(sFn)!=m_uptodateTags.end())
				goto next_file; // this version == good, was downloaded/generated here before
			
			if(m_indexFiles.find(sFn+".bz2")!=m_indexFiles.end())
			{
				if(_UpdateOne(dl, sFn+".bz2", sErr))
				goto drop_file;
			}
			if(m_indexFiles.find(sFn+".gz")!=m_indexFiles.end())
			{
				if(_UpdateOne(dl, sFn+".gz", sErr))
				goto drop_file;
			}
		}

next_file:
                it++;
		continue;
		
drop_file:
		tStrSet::iterator dropIt = it;
		it++;
		m_indexFiles.erase(dropIt);
	}
	m_indexFiles.erase("");


	for(tStrSet::const_iterator it=m_indexFiles.begin(); it!=m_indexFiles.end(); it++)
	{	
		if (!_UpdateOne(dl, *it, sErr))
		{
			m_nErrorCount++;
			SendChunk("Error while redownloading this file!<br>\n ");
			if (!sErr.empty())
				SendChunk(string("Reason: ")+sErr+"<br>\n");
			sErr.clear();
		}
	}
	
	// download the guessed stuff unless fetched somehow before. Add to
	// vfiles on success, otherwise forget them
	for(tStrSet::const_iterator it=listGuessedLocationsToTry.begin();
	it!=listGuessedLocationsToTry.end(); it++)
	{
		string bVer((*it)+".bz2"), gVer((*it)+".gz");
		if(m_uptodateTags.find(*it) != m_uptodateTags.end()
				|| m_uptodateTags.find( bVer ) != m_uptodateTags.end()
				|| m_uptodateTags.find( gVer ) != m_uptodateTags.end() )
			continue;		
		if(_UpdateOne(dl, bVer, sErr))
			m_indexFiles.insert(bVer);
		else if(_UpdateOne(dl, gVer, sErr))
			m_indexFiles.insert(gVer);
		// otherwise just ignore it
	}
}

bool expiration::_ParseAndProcessIndexFile(const MYSTD::string &sPath,
		const string &sDirname, const string &sDebBaseDir,
		bool *pbProbeType)
{
	LOGSTART("expiration::_ParseAndProcessIndexFile");
	filereader reader;
	string sPureIfileName;

	if (reader.OpenFile(sPath))
		sPureIfileName = reader.GetPureFilename();
	else if(pbProbeType) // damn, unreadable, but we need to report its type
	{
			tStrPos pos=sPath.find(SZPATHSEP);
			sPureIfileName = stmiss==pos ? sPath : sPath.substr(pos+1);
			stripSuffix(sPureIfileName, ".gz");
			stripSuffix(sPureIfileName, ".bz2");
			stripSuffix(sPureIfileName, ".xz");
			stripSuffix(sPureIfileName, ".lzma");
	}
	else
	{
		SendChunk("<font color=orange>WARNING: unable to open or read this file</font><br>\n");
		return false;
	}

#define REPORTIFILE { if(pbProbeType) { *pbProbeType = true; return true; } }

	string sLine, key, val;
	tRemoteFileInfo info;
	info.SetInvalid();

	if (sPureIfileName=="Packages") // Debian's Packages file
	{
		LOG("filetype: Packages file");
		REPORTIFILE;
		static string sMD5sum("MD5sum"), sFilename("Filename"), sSize("Size");
		
		for (bool bNoEof=true; bNoEof;)
		{
			bNoEof=reader.GetOneLine(sLine);
			trimBack(sLine);
			//cout << "file: " << *it << " line: "  << sLine<<endl;
			if (sLine.empty() || !bNoEof)
			{
				if(info.IsUsable())
					_HandlePkgEntry(info);
				info.SetInvalid();

				if(CheckAbortCondition())
					return true; // XXX: should be rechecked by the caller ASAP!

				continue;
			}
			else if (_ParseLine(sLine, key, val))
			{
				// not looking for data we already have
				if(32==val.length() && key==sMD5sum)
				{
					info.fpr.csType=CSTYPE_MD5;
					info.fpr.ReadCsFromString(val);
				}
				else if(key==sSize)
					info.fpr.size=atoofft(val.c_str());
				else if(key==sFilename)
				{
					info.sDirectory=sDebBaseDir;
					tStrPos pos=val.rfind(SZPATHSEPUNIX);
					if(pos==stmiss)
						info.sFileName=val;
					else
					{
						info.sFileName=val.substr(pos+1);
						info.sDirectory.append(val, 0, pos+1);
					}
				}
			}
		}
	}
	else if (endsWithSzAr(sPureIfileName, ".db.tar"))
	{
		LOG("assuming Arch Linux package db");

		UINT nStep=0;
		enum tExpData { _fname, _csum, _csize, _nthng } typehint(_nthng);
		REPORTIFILE;

		while (reader.GetOneLine(sLine)) // last line doesn't matter, contains tar's padding
		{
			trimBack(sLine);

			if (nStep>=2)
			{
				if (info.IsUsable())
					_HandlePkgEntry(info);
				info.SetInvalid();
				nStep=0;

				if (CheckAbortCondition())
					return true;

				continue;
			}
			else if (endsWithSzAr(sLine, "%FILENAME%"))
				typehint=_fname;
			else if (endsWithSzAr(sLine, "%CSIZE%"))
				typehint=_csize;
			else if (endsWithSzAr(sLine, "%MD5SUM%"))
				typehint=_csum;
			else
			{
				switch (typehint)
				{
				case _fname:
					info.sDirectory = sDirname;
					info.sFileName = sLine;
					nStep=0;
					break;
				case _csum:
					info.fpr.csType = CSTYPE_MD5;
					info.fpr.ReadCsFromString(sLine);
					nStep++;
					break;
				case _csize:
					info.fpr.size = atoofft(sLine.c_str());
					nStep++;
					break;
				default:
					continue;
				}
				// next line is void for now
				typehint=_nthng;
			}
		}
	}
	else if(sPureIfileName=="setup")
	{
		LOG("assuming Cygwin package setup listing");
		REPORTIFILE;
		static const string keys[]={"install: ", "source: "};
		tStrVec vFileSizeSum;

		for (bool bNoEof=true; bNoEof;)
		{
			if(CheckAbortCondition())
				return true;

			bNoEof=reader.GetOneLine(sLine);
			trimBack(sLine);
			for(UINT i=0;i<_countof(keys);i++)
			{
				if(!startsWith(sLine, keys[i]))
					continue;
				//LOG("interesting line: " << sLine);
				if (3==Tokenize(sLine, "\t ", vFileSizeSum, false, keys[i].length())
						&& vFileSizeSum[2].length()>30)
				{
					//LOG("found tree items, checking");
					info.fpr.csType=CSTYPE_MD5;
					info.fpr.bUnpack=false;
					if( 0 < (info.fpr.size=atoofft(vFileSizeSum[1].c_str()))
							&& info.fpr.ReadCsFromString(vFileSizeSum[2]))
					{
						//LOG("looks ok");

						tStrPos pos = vFileSizeSum[0].rfind(SZPATHSEPUNIX);
						if (pos == stmiss)
						{
							info.sFileName = vFileSizeSum[0];
							info.sDirectory = sDirname;
						}
						else
						{
							info.sFileName
									= vFileSizeSum[0].substr(pos + 1);
							info.sDirectory = sDirname
									+vFileSizeSum[0].substr(0, pos + 1);
						}

						_HandlePkgEntry(info);
						info.SetInvalid();
					}
				}
			}
		}
	}
	else if(sPureIfileName=="repomd.xml")
	{
		LOG("SUSE index file, entry level");
		REPORTIFILE;
		tStrVec tokens;

		string sLine;
		while(reader.GetOneLine(sLine))
		{
			Tokenize(sLine, "\"/", tokens);
			for(UINT i=0;i<tokens.size();i++)
			{
				LOG("testing filename: " << tokens[i]);
				if(!endsWithSzAr(tokens[i], ".xml.gz"))
					continue;
				LOG("index basename: " << tokens[i]);
				info.sFileName = tokens[i];
				info.sDirectory = sDirname;
				_HandlePkgEntry(info);
				info.SetInvalid();
			}
		}
	}
	else if(sPureIfileName.length()>50 && endsWithSzAr(sPureIfileName, ".xml") && sPureIfileName[40] == '-')
	{
		LOG("SUSE list file, pickup any valid filename ending in .rpm");
		REPORTIFILE;
		tStrVec tokens;

		string sLine;
		while(reader.GetOneLine(sLine))
		{
			Tokenize(sLine, "\"'><=/", tokens);
			for(UINT i=0;i<tokens.size();i++)
			{
				LOG("testing token: " << tokens[i]);
				if(!endsWithSzAr(tokens[i], ".rpm"))
					continue;
				LOG("RPM basename: " << tokens[i]);
				info.sFileName = tokens[i];
				info.sDirectory = sDirname;
				_HandlePkgEntry(info);
				info.SetInvalid();
			}
		}
	}
	/* not used for now, just covered by wfilepat
	else if( (sPureIfileName == "MD5SUMS" ||
			sPureIfileName == "SHA1SUMS" ||
			sPureIfileName == "SHA256SUMS") && it->find("/installer-") != stmiss)
	{

	}
	*/
	else // maybe Debian sources-style formats
	{
		LOG("filetype: Sources file format or similar");
		/* pickup all common information, then parse the remaining lines and
		 * assign the tokens to fields in each call of the callback function
		 */
		tStrVec vSumSizeName;
		string sStartMark;
		bool bUse(false);
		enum tTypeHint { _index, _sources, _release } typehint;

		if (sPureIfileName=="Index")
		{
			REPORTIFILE;
			typehint = _index;
			info.fpr.csType = CSTYPE_SHA1;
			info.fpr.bUnpack = true;
			info.sDirectory=sDirname; // same for all
			sStartMark="SHA1-Patches:";
		}
		else if (sPureIfileName=="Sources")
		{
			REPORTIFILE;
			typehint = _sources;
			info.fpr.csType = CSTYPE_MD5;
			info.fpr.bUnpack = false;
			sStartMark="Files:";
		}
		else if (sPureIfileName=="Release")
		{
			REPORTIFILE;
			typehint = _release;
			info.fpr.csType = CSTYPE_MD5;
			info.fpr.bUnpack = false;
			sStartMark="MD5Sum:";
		}
		else
			return true;

		for (bool bNoEof=true; bNoEof;)
		{
			bNoEof=reader.GetOneLine(sLine);
			trimBack(sLine);
			//if(sLine.find("unp_")!=stmiss)
			//	int nWtf=1;
			//cout << "file: " << *it << " line: "  << sLine<<endl;
			if (sLine.empty() || !bNoEof)
			{
				// optional, but better be sure
				info.sDirectory.clear();
				continue;
			}
			else if(startsWith(sLine, sStartMark))
				bUse=true;
			else if(startsWithSz(sLine, "Directory:"))
			{
				trimBack(sLine);
				tStrPos pos=sLine.find_first_not_of(SPACECHARS, 10);
				if(pos!=stmiss)
					info.sDirectory=sDebBaseDir+sLine.substr(pos)+SZPATHSEP;
			}
			else if (bUse && sLine[0]==' ' && 3==Tokenize(sLine,
					"\t ", vSumSizeName) && vSumSizeName[0].length()>30
					)

			{
				info.fpr.size = atoofft(vSumSizeName[1].c_str());
				if(info.fpr.size && info.fpr.ReadCsFromString(vSumSizeName[0]))
				{

					if (_sources == typehint)
						info.sFileName=vSumSizeName[2];
					else if (_index == typehint)
					{
						// We cheat! Files there are compressed and thus the checksum is wrong, deal with that later
						info.sFileName=vSumSizeName[2]+".gz";
						//info.sDirectory=sDirname;
					}
					else if (_release ==typehint)
					{
						// usually has subfolders
						tStrPos pos=vSumSizeName[2].rfind(SZPATHSEPUNIX);
						if (stmiss!=pos)
						{
							info.sFileName=vSumSizeName[2].substr(pos+1);
							info.sDirectory=sDirname
									+vSumSizeName[2].substr(0, pos+1);
						}
						else // something new in main folder? unlikely...
						{
							info.sFileName=vSumSizeName[2];
							info.sDirectory=sDirname;
						}
					}
					// no else, caught above

					_HandlePkgEntry(info);

					if(CheckAbortCondition())
						return true;
				}
			}
		}
	}
	return reader.CheckGoodState(false);
}

void expiration::_ProcessSeenIndexFiles(tStrSet *pDupeCatcher)
{
	LOGSTART("expiration::_ParseVolatileFilesAndHandleEntries");
	for(tStrSet::const_iterator it=m_indexFiles.begin(); it!=m_indexFiles.end(); it++)
	{

		if(CheckAbortCondition())
			return;

		if(it->empty() || it->at(0)!=CPATHSEP)
			continue;

		string sSrcSig; // only set if pdupecatcher is set
		if(pDupeCatcher && ! m_bByPath)
		{
			map<string,string>::iterator spair = m_quickDupeHints.find(*it);
			if(spair != m_quickDupeHints.end())
			{
				sSrcSig=spair->second;
				if( ! sSrcSig.empty()
						&& pDupeCatcher->find(sSrcSig) != pDupeCatcher->end())
				{
					//SendMsg("found dupe in the catchup ;-)<br>");
					SendChunk(string("Ignoring ")
							+it->substr(m_nDropPrefLen)
							+" (equivalent file seen)<br>\n");
					continue;
				}
			}
		}

		SendChunk(string("Parsing metadata in ")+it->substr(m_nDropPrefLen)+"<br>\n");

		// pre calc relative base folders for later
		string sDirname(SZPATHSEP);
		string sDebBaseDir = sDirname; // may differ from sDirname if the path looks like a Debian mirror path
		tStrPos pos=it->rfind(CPATHSEP);
		if(stmiss!=pos)
		{
			sDirname.assign(*it, 0, pos+1);
			pos=sDirname.rfind("/dists/");
			if(stmiss!=pos)
				sDebBaseDir.assign(sDirname, 0, pos+1);
			else
				sDebBaseDir=sDirname;
		}

		if(_ParseAndProcessIndexFile(*it, sDirname, sDebBaseDir, NULL))
		{
			if( ! sSrcSig.empty() )
				pDupeCatcher->insert(sSrcSig);
		}
		else
		{
			SendChunk("<font color=red>An error occured while reading this file, some contents may have been ignored.</font><br>\n");
			m_nErrorCount++;
			continue;
		}
		//		cout << "found package files: "<< m_trashCandidates.size()<<endl;
	}
}

void expiration::_HandlePkgEntry(const tRemoteFileInfo &entry)
{
	LOGSTART2("expiration::_HandlePkgEntry:",
			"\ndir:" << entry.sDirectory <<
			"\nname: " << entry.sFileName <<
			"\nsize: " << entry.fpr.size <<
			"\ncsum: " << entry.fpr.GetCsAsString());
	
	// singleton -> keep some things static == optimization
	static pair<tS2DAT::iterator, tS2DAT::iterator> fromTo;

	// find a subset == range of files having at least the same file name
	fromTo=m_trashCandSet.equal_range(entry.sFileName);
	
	if(fromTo.first==fromTo.second)
		return; // just no hits, make it quick
			
	static tS2DAT::iterator dropIt, it;
	
	for(it=fromTo.first; it!=fromTo.second; /* erases inside -> step forward there */ )
	{
		tFingerprint & fprFile=it->second.fpr;
		string sShortPath(it->second.sDirname+it->first);
		string sFullLocalPath(acfg::cachedir+SZPATHSEP+sShortPath);
		header h;
		
		bool bCheckPath=m_bByPath;

		// debian-installer files need to include path checks
		if(sShortPath.find("/installer-")!=stmiss)
			bCheckPath=true;

		// needs to match the exact file location
		if(bCheckPath)
		{
			string sEntrPath=entry.sDirectory+entry.sFileName;
			LOG("Checking exact path: " << sEntrPath << " vs. " << sFullLocalPath);
			StrSubst(sEntrPath, "/./", "/");
			if(sFullLocalPath != sEntrPath)
				goto keep_in_trash;
		}
		
		// Basic header checks. Skip if the file was forcibly updated/reconstructed before.
		if ( ! m_bSkipHeaderChecks && m_uptodateTags.find(sFullLocalPath) == m_uptodateTags.end()
				&& ! it->second.bHeaderTestDone	) // do only once for each file
		{
			LOG("Doing basic header checks");
			it->second.bHeaderTestDone = true;
			
			if (0<h.LoadFromFile(sFullLocalPath+".head"))
			{
				if (h.h[header::CONTENT_LENGTH])
				{
					off_t len=atoofft(h.h[header::CONTENT_LENGTH]);
					struct stat stinfo;
					off_t lenInfo=0;
					
					// avoid duplicate stat call if the data from the includer is still there
					if(fprFile.bUnpack == false 
							&& fprFile.csType == CSTYPE_INVALID
							&& fprFile.size>0)
					{
						lenInfo=fprFile.size;
					}
					else if (0==stat(sFullLocalPath.c_str(), &stinfo))
						lenInfo = stinfo.st_size;
					else
					{
						SendChunk(sShortPath+ ": error reading attributes, ignoring");
						goto keep_in_trash;
					}
					
					if(len<lenInfo)
					{
						SendChunk(string("<font color=red>WARNING, header file of ")
								+ sShortPath
								+ " reported too small file size, invalidating "
								+it->first+"</font><br>\n");
						goto keep_in_trash;
					}
				}
				else
				{
					SendChunk(string("<font color=red>WARNING, header file of ")
							+ sShortPath + " does not contain content length");
					goto keep_in_trash;
				}
			}
			else
				SendChunk(string("<font color=\"orange\">WARNING, header file missing for ")
						+sFullLocalPath+"</font><br>\n");
		}
		
		if(m_bByChecksum)
		{
			bool bSkipDataCheck=false;
			
			// scan file if not done before
			if(CSTYPE_INVALID == fprFile.csType)
			{
				// hints for checksumming to read what we need
				fprFile.csType=entry.fpr.csType;
				fprFile.bUnpack=entry.fpr.bUnpack;
				// finally get the physical file size and checksum of the contents
				if(!fprFile.ReadCsFromFile(sFullLocalPath))
				{
					// IO error? better keep it for now
					aclog::err(string("An error occured while checksumming ")
							+sFullLocalPath+", not touching it.");
					bSkipDataCheck=true;
				}
			}

			if ( !bSkipDataCheck)
			{
				if ( ! (fprFile == entry.fpr))
				{
					SendChunk(string("<font color=red>BAD: ")+sFullLocalPath
					+"</font><br>\n");
					goto keep_in_trash;
				}
			}
		}
		
		// ok, package matched, contents ok if checked, drop it from the removal list
		if (m_bVerbose)
			SendChunk(string("<font color=green>OK: ")+it->second.sDirname
					+it->first+"</font><br>\n");

		dropIt=it;
		it++;

		m_trashCandHeadSet.erase(dropIt->second.sDirname+dropIt->first+".head");
		m_trashCandSet.erase(dropIt);

		continue;

		keep_in_trash:
		it++;
	}
}

inline void expiration::_RemoveAndStoreStatus(bool bPurgeNow)
{
	LOGSTART("expiration::_RemoveAndStoreStatus");
	FILE *f(NULL);
    if(!bPurgeNow) 
    {
        string sDbFile=acfg::cachedir+SZPATHSEP+"_expending_dat";

        f = fopen(sDbFile.c_str(), "w");
        if(!f)
        {
            SendChunk("Unable to open _expending.dat for writing, trying to recreate... ");
            ::unlink(sDbFile.c_str());
            f=::fopen(sDbFile.c_str(), "w");
            if(f)
                SendChunk("OK<br>\n");
            else
            {
                SendChunk("<font color=red>FAILED. ABORTING. Check filesystem and file permissions.");
                return;
            }
        }
    }
	
	int n(0);
	for(tS2DAT::iterator it=m_trashCandSet.begin(); 
            it!=m_trashCandSet.end(); it++)
	{
        if(rechecks::MatchWhitelist(it->first)
        || rechecks::MatchWhitelist(it->second.sDirname + it->first))
        {
        	LOG("File not to be removed, ignoring");
        	continue;
        }

		//cout << "Unreferenced: " << it->second.sDirname << it->first <<endl;

		string sWhat=acfg::cachedir+SZPATHSEP+it->second.sDirname+it->first;
		
		// purge head files here, not trying a second time in the code below
	    m_trashCandHeadSet.erase(it->second.sDirname+it->first+".head");

	    //cout << "Took " << sWhatHead << " from the list" <<endl;

		if(bPurgeNow || (it->second.nLostAt < (m_nTimeNow-acfg::extreshhold*86400)))
		{
			SendChunk(string("Removing ")+it->second.sDirname+it->first+"<br>\n");

#ifdef ENABLED

			::unlink(sWhat.c_str());
			::unlink((sWhat+".head").c_str());
			::rmdir(it->second.sDirname.c_str());
#endif
        }
		else if(f)
		{
			SendChunk(string("Tagging ")+it->second.sDirname+it->first+"<br>\n");

			n++;
			fprintf(f, "%lu\t%s\t%s\n",  it->second.nLostAt,
					it->second.sDirname.c_str(), it->first.c_str());
		}
	}
    if(f)
        fclose(f);

    // now just kill dangling header files
	for(set<string>::iterator it=m_trashCandHeadSet.begin();
            it!=m_trashCandHeadSet.end(); it++)
	{
        if(rechecks::MatchWhitelist(*it)) continue;
        if(endsWithSzAr(*it, ".head") &&
        		rechecks::MatchWhitelist(it->substr(0, it->length()-5)))
        	continue;
		
        cout << "Removing orphaned head file: " << *it << endl;
#ifdef ENABLED
        string sFullPath=acfg::cachedir+SZPATHSEP+*it;
        ::unlink(sFullPath.c_str());
        string::size_type pos=sFullPath.find_last_of(SZPATHSEPUNIX SZPATHSEPWIN);
        if(pos!=stmiss)
        	::rmdir(sFullPath.substr(0, pos).c_str());
#endif
    }
    if(n>0)
    {
    	tSS fmt; fmt << "<br>\n" << n <<" package file(s) marked "
    	"for removal in few days.<br>\n<br>\n";
    	SendChunk(fmt.rptr(), fmt.size());
    }

}



void expiration::Action(const string & cmd)
{
	
	if (cmd.find("justRemove")!=stmiss)
	{
		_LoadTrashMapFromFile(true);
		_RemoveAndStoreStatus(true);
		return;
	}
	if (cmd.find("justShow")!=stmiss)
	{
		_LoadTrashMapFromFile(true);
		for (tS2DAT::iterator it=m_trashCandSet.begin(); it
				!=m_trashCandSet.end(); it++)
		{
			SendChunk(it->second.sDirname+it->first+"<br>\n");
			SendChunk(it->second.sDirname+it->first+".head<br>\n");
		}
		return;
	}
	
	SendChunk("Locating potentially expired files in the cache...<br>\n");

	m_bErrAbort=(cmd.find("abortOnErrors=aOe")!=stmiss);
	m_bByPath=(cmd.find("byPath")!=stmiss);
	m_bByChecksum=(cmd.find("byChecksum")!=stmiss);
	m_bVerbose=(cmd.find("beVerbose")!=stmiss);
	m_bForceDownload=(cmd.find("forceRedownload")!=stmiss);
	m_bSkipHeaderChecks=(cmd.find("skipHeadChecks")!=stmiss);

	DirectoryWalk(acfg::cachedir, this);
	if(CheckAbortCondition())
		return;
	//cout << "found package files: " << m_trashCandidates.size()<<endl;

	SendChunk("Redownloading index files...<br>\n");

	UpdateIndexFiles();
	
	if(CheckAbortCondition())
		return;


	if (m_bErrAbort && m_nErrorCount>0)
	{
		SendChunk("<font color=\"red\">Found errors during processing, aborting as requested.</font>"
			"<!-- TELL:THE:ADMIN -->");
		return;
	}

	SendChunk("Identifying fresh files...<br>\n");
	tStrSet catchup;

	_ProcessSeenIndexFiles(&catchup);

	if(CheckAbortCondition())
			return;

	// update timestamps of pending removals
	_LoadTrashMapFromFile(false);
	
	_RemoveAndStoreStatus(cmd.find("purgeNow")!=stmiss);
	_PurgeMaintLogs();
	SendChunk("Done.");

}


void expiration::_PurgeMaintLogs()
{
	tStrDeq logs = ExpandFilePattern(acfg::logdir + SZPATHSEP"maint_*.log");
	if (logs.size() > 2)
		SendChunk(
				"Found required cleanup tasks: purging maintanence logs...<br>\n");
	for (tStrDeq::const_iterator it = logs.begin(); it != logs.end(); it++)
	{
		time_t id = atoofft(it->c_str() + acfg::logdir.size() + 7);
		//cerr << "id ist: "<< id<<endl;
		if (id == GetTaskId())
			continue;
		//cerr << "Remove: "<<globbuf.gl_pathv[i]<<endl;
#ifdef ENABLED
		::unlink(it->c_str());
#endif
	}
}


