#
# Copyright (C) 2006 Chris Halls <halls@debian.org>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of version 2.1 of the GNU Lesser General Public
# License as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""This module tests the client protocol itself"""

import os, time, urllib, sys
from twisted.trial import unittest
from twisted.internet import protocol, reactor, defer, base
from twisted.python import failure
from twisted import web
from twisted.web import http
from StringIO import StringIO

from apt_proxy.apt_proxy_conf import apConfig
from apt_proxy.cache import CacheEntry
from apt_proxy.apt_proxy import Factory
from apt_proxy.misc import log
from apt_proxy.fetchers import DownloadQueue
from apt_proxy.test.test_fetchers import RsyncServer
from apt_proxy.test.test_apt_proxy import apTestHelper

base.DelayedCall.debug = True

class uriData:
    """
    class containing test data for a request
    """
    def __init__(self, filename, expectedResponse, if_modified_since=None, expectedSize=None, filePath=None, abortTransfer=False, passOnDataRcvd=False):
        """
        Initialise data for client testing
        @param filename Name of file to be sent in request from apt client
        @param expectedResponse HTTP response expected from apt-proxy
        @param expectedSize Size of file that should be sent back to client. See also filePath
        @param filePath Path to source file - used to automatically calculate expectedSize instead of supplying expectedSize directly
        @param abortTransfer Close connection before transfer is complete to test error handling
        @param passOnDataRcvd Trigger deferred on first data packet received
        """
        self.filename = filename 
        self.expectedResponse = expectedResponse
        self.if_modified_since = if_modified_since
        self.filePath = filePath # Path to source file, used to calculate expected size
        self.abortTransfer = abortTransfer
        self.passOnDataRcvd = passOnDataRcvd

        if expectedSize is not None:
            self.expectedSize = expectedSize # If not none, the file sent should have this size
        elif filePath is not None:
            # Get size of actual file
            self.expectedSize = os.path.getsize(filePath)
        else:
            self.expectedSize = None

class uriRequester(http.HTTPClient):
    """
    class to request files and parse responses
    """
    class requestFactory(protocol.ClientFactory):
        """
        Helper factory to connect to apt-proxy and send
        HTTP requests using uriRequester
        """
        #def startedConnecting(self, connector):
        #    print 'Started to connect.'
        def __init__(self, request):
            self.request = request
        def buildProtocol(self, addr):
            "Pass incoming connection to our request"
            return self.request
        def clientConnectionLost(self, connector, reason):
            log.debug('Lost connection.  Reason:'+ str(reason), 'requestFactory')
        def clientConnectionFailed(self, connector, reason):
            log.err('Connection failed. Reason:', reason, 'requestFactory')
            self.request.failed(reason)

    def __init__(self, host, *testData):
        """
        Start test. *testData holds classes of uriData
        """
        self.factory = self.requestFactory(self)
        self.host = host
        self.deferred = defer.Deferred() # Deferred that returns result of test
        self.http_status = None
        self.received_len = 0
        self.tests=testData
        self.chunked = False

    def connectionMade(self):
        """
        Http connection made
        """
        log.debug("connection made to test apt-proxy server", 'uriRequester')
        for i in range(0,len(self.tests)):
            test = self.tests[i]
            url = urllib.quote(test.filename)
            log.debug("requesting: %s" %(url), 'uriRequester')
            #self.sendCommand("GET", test.filename)
            self.transport.write('%s %s HTTP/1.1\r\n' % ("GET", url))

            self.sendHeader('Host', self.host)
            if len(self.tests)>1 and i != len(self.tests):
                self.sendHeader('Connection','keep-alive')
            else:
                self.sendHeader('Connection','close')
            if test.if_modified_since is not None:
                datetime = http.datetimeToString(test.if_modified_since)
                self.sendHeader('if-modified-since', datetime)
            self.sendHeader("User-Agent", "apt-proxy test suite test_requests.py")

            self.endHeaders()
        self.getNextTest() # Start first test

    def getNextTest(self):
        # Ready for next status code
        self.firstLine = 1 
        #self.length = None
        self.__buffer = ''
        log.debug("getNextTest", 'uriRequester')

        if len(self.tests):
            self.nextTest = self.tests[0]
            self.tests = self.tests[1:]
            log.debug("waiting for test results for: " + self.nextTest.filename, 'uriRequester')
        else:
            log.debug('final test passed', 'uriRequester')
            self.connection.disconnect()
            self.allTestsDone()
            
    def handleStatus(self, version, code, message):
        log.debug('handleStatus: (%s) %s - %s, expected:%s' % 
                   (version, code, message, self.nextTest.expectedResponse), 'uriRequester')
        self.http_status = int(code)

    def handleHeader(self, key, value):
        if key.lower() == 'transfer-encoding' and value.lower() == 'chunked':
            # There doesn't seem to be any support for chunked mode in twisted-web's http
            # client, so use fromChunk ourselves
            log.debug("Chunked mode", 'uriRequester')
            self.chunked = True
    
    def dataReceived(self, data):
        self.received_len = self.received_len + len(data)
        log.debug("data received, len: %s" % (self.received_len), 'uriRequester')
        if self.nextTest.abortTransfer == True:
            log.debug("aborting transfer", 'uriRequester')
            self.disconnect()
            self.allTestsDone()
	elif self.nextTest.passOnDataRcvd == True:
            log.debug("returning early test passed", 'uriRequester')
	    self.passed()
        else:
            http.HTTPClient.dataReceived(self, data)

    class ResponseError(Exception):
        def __init__(self, received, expected):
            self.received = received
            self.expected = expected
        def __str__(self):
            return "ResponseError - received %s but expected %s" % (self.received,self.expected)
    class SizeError(Exception):
        def __init__(self, received, expected):
            self.received = received
            self.expected = expected
        def __str__(self):
            return "SizeError - received %s bytes but expected %s" % (self.received,self.expected)

    def handleResponse(self, buffer):
        if self.chunked:
            buffer, rest = http.fromChunk(buffer)
            while len(rest)>0:
                log.debug("buf size=%s rest size=%s" % (len(buffer), len(rest)), 'uriRequester')
                data, rest = http.fromChunk(rest)
                buffer += data
        received_len = len(buffer)
        log.debug('data received: %s bytes, expected:%s' % (received_len, self.nextTest.expectedSize), 'uriRequester')
        if self.http_status != self.nextTest.expectedResponse:
            log.debug('test FAILED: response code (%s) is not %s' % 
                       (self.http_status, self.nextTest.expectedResponse), 'uriRequester')
            raise self.ResponseError(self.http_status, self.nextTest.expectedResponse)
        elif self.nextTest.expectedSize is not None and received_len != self.nextTest.expectedSize:
            log.debug('test FAILED: received %s bytes, but expected %s' % 
                    (received_len, self.nextTest.expectedSize), 'uriRequester')
            raise self.SizeError(received_len, self.nextTest.expectedSize)
        else:
            log.debug("test passed", 'uriRequester')
            self.passed()

    def passed(self):
        self.getNextTest()
    def failed(self, data):
        log.debug('test failed', 'uriRequester')
        self.disconnect()
        reactor.runUntilCurrent()
        self.deferred.errback(data)
    def disconnect(self):
        reactor.callLater(0, self.connection.disconnect)
    def allTestsDone(self):
        # All done, just need to clean up pending timed calls
        log.debug("XXXXXXXXXXXXXXXXXX")
        reactor.runUntilCurrent()
        log.debug("XXXXXXXXXXXXXXXXXX")
        self.deferred.callback(None)

class TestRequestHelper(apTestHelper):
    def setUp(self, config):
        apTestHelper.setUp(self)
        config = self.config + '\n' + config
        log.debug("config:\n" + config, 'TestRequestHelper')
        self.c = apConfig(StringIO(config))
        self.factory = Factory(self.c)
        #self.factory.configurationChanged()
        self.port = reactor.listenTCP(0, self.factory, interface="127.0.0.1")

    def tearDown(self):
        self.port.stopListening()
        self.factory.stopFactory()
        del(self.factory)
        apTestHelper.tearDown(self)
        self.assertRaises(OSError, os.stat, self.cache_dir)
        
        # The ftp classes use callLater(0, ...) several times, so allow
        # those calls to complete
        reactor.runUntilCurrent()

    def doRequest(self, *data):
        portno = self.port.getHost().port
        host = "127.0.0.1:%s" % (portno)
        for d in data:
            log.debug("Starting test connection to %s, file:%s:" %(host, d.filename), 'uriRequesterTest')
        client = uriRequester(host, *data)
        connection = reactor.connectTCP("127.0.0.1", portno, client.factory)
        self.connection = connection
        client.connection = connection

        self.lastRequestFactory = client
        return client.deferred

class FileBackendTest(TestRequestHelper):
    def setUp(self):
        """
        Make a configuration with a single backend
        [files]
        backends=file:///<path to test packages directory>
        """
        self.filedir = os.path.normpath(os.getcwd()+"/../test_data/packages")
        config = ("dynamic_backends=off\n"+
                  "[files]\n" +
                  "backends=file://" + self.filedir)
        #print "config: " + config
        TestRequestHelper.setUp(self, config)

    def testNotFound(self):
        return self.doRequest(uriData('/files/test.gz', http.NOT_FOUND))
    testNotFound.timeout = 1
    def testPackagesFile(self):
        file = 'Packages.gz'
        d = uriData('/files/'+file, http.OK, filePath=self.filedir+os.sep+file)
        return self.doRequest(d).addCallback(self.PackagesFile2)
    testPackagesFile.timeout = 1
    def PackagesFile2(self, x):
        backend = self.factory.getBackend('files')
        # Check that request was deleted from backend
        reactor.runUntilCurrent() # Gzip decompression runs using reactor.callLater(..)
        self.assertEquals(len(backend.entries), 0)

    def testForbidden(self):
        d = self.doRequest(uriData('/notbackend/Release', http.NOT_FOUND))
        return d
    testForbidden.timeout = 1

class WebServer:
    def start(self):
        """
        Start web server, serving test data
        
        @ret port number that server listens on
        """
        root = web.static.File("../test_data")
        #self.application = service.Application('web')
        site = web.server.Site(root)
        #self.port = reactor.listenTCP(0, self.application, interface="127.0.0.1")
        self.port = reactor.listenTCP(0, site, interface="127.0.0.1")

        return self.port.getHost().port

    def stop(self):
        self.port.stopListening()

class BackendTestBase:
    """
    Class to perform a series of requests against a test backend.
    Derived classes should arrange for a local server to serve
    files from the test_data directory.
    """

    # Name of test backend
    backendName = 'test_data'
    packagesTestFile = '/packages/Packages'

    def init(self, config, debugName):
        """
        Make a configuration without starting a backend
        @param debugName Name of class to print in debug messages
        @param protocol Protocol of fetcher to be tested (http, ftp, rysnc etc)
        """
        self.debugname = debugName
        self.checkDest = True # Used to supress assertion when downloading the same file multiple times
        TestRequestHelper.setUp(self, config)
                
    def setUp(self, debugName, protocol, serverFactory, uriSuffix=None):
        """
        Make a configuration with a single backend
        
        @param debugName Name of class to print in debug messages
        @param protocol Protocol of fetcher to be tested (http, ftp, rysnc etc)
        @param serverFactory Class to provide start() and stop() methods for backend server
        @param uriSuffix This is added to uri requests if necessary
        """
        self.protocol = protocol
        self.server = serverFactory()
        self.uriSuffix = uriSuffix
        self.port = self.server.start()

        backend_uri = protocol + "://127.0.0.1:" + str(self.port)
        if self.uriSuffix:
            backend_uri += '/' + self.uriSuffix
        config = ("dynamic_backends=off\n" +
                  "[test_data]\n" +
                  "backends=" + str(backend_uri))
        self.init(config, debugName)
        self.testfilesdir = os.path.normpath(os.getcwd()+"/../test_data")
    def tearDown(self):
        log.debug("tearDown", self.debugname)
        #reactor.iterate(0.1)
        self.server.stop()
        #reactor.iterate(0.1)
        TestRequestHelper.tearDown(self)
    def downloadFile(self, file=packagesTestFile):
        """
        Download a file to cache
        self.backend is set to backend name
        self.file is set to last filename
        self.filepath is set to last physical filename
        """
        return self.downloadFiles(file)

    def calcFilePaths(self, file):
        """
        Given a filename, generate real filename and request path
        """
        filename = '/' + self.backendName + file
        if hasattr(self, 'testfilesdir'):
            sourcepath = self.testfilesdir+file
        else:
            sourcepath = None
        destpath = self.cache_dir + filename
        return filename, sourcepath, destpath

    def getFilePaths(self, file):
        """
        Given a filename, generate real filename and request path.
        Then check that file does not exist
        """
        filename, sourcepath, destpath = self.calcFilePaths(file)
        if self.checkDest:
            # File should not be in cache
            self.assertRaises(OSError, os.stat, destpath)
        return filename, sourcepath, destpath
        
    def downloadFiles(self, *files):
        """
        Download a number of files to cache
        """
        data = []
        self.filepaths = []
        for f in files:
            self.filename, sourcepath, destpath = self.getFilePaths(f)
            self.filepaths.append(destpath)
            data.append(uriData(self.filename, http.OK, filePath=sourcepath))
        d = self.doRequest(*data)
        def checkPath(x):
            # Check that files were really placed in cache
            for f in self.filepaths:
                log.debug("downloadFiles checkPath: %s" % (f))
                os.stat(f)
        d.addCallback(checkPath)
        return d

class BackendProtocolTests(BackendTestBase):
    """
    Class which implements tests which should be run for all tested protocols
    """
    def testNotFound(self):
        return self.doRequest(uriData('/test_data/NotHere.gz', http.NOT_FOUND))
    testNotFound.timeout = 2
    def testPackagesFile(self):
        return self.downloadFile().addCallback(self.PackagesFile2)
    def PackagesFile2(self, x):
        # Check access time datbase was updated
        filename = self.calcFilePaths(self.packagesTestFile)[0]
        self.assertApproximates(self.factory.access_times[filename], time.time(), 6)
    testPackagesFile.timeout = 2

    def testPackagesGzFile(self):
        return self.downloadFile(self.packagesTestFile+'.gz').addCallback(self.PackagesUncompressed)
    testPackagesGzFile.timeout = 2
    def PackagesUncompressed(self, x):
        # Check that Packages file was registered
        reactor.runUntilCurrent() # Gzip compression runs using reactor.callLater(..)
        filename = self.calcFilePaths(self.packagesTestFile)[0][1:] # Remove leading '/' from path
        backend = self.factory.getBackend(self.backendName)
        self.assertEquals(backend.get_packages_db().packages.get_files(), [filename])

    def testPackagesBz2File(self):
        return self.downloadFile(self.packagesTestFile+'.bz2').addCallback(self.PackagesUncompressed)
    testPackagesBz2File.timeout = 2

    def testNotModifiedGreater(self):
        "Check http not modified is sent for new file"
        d = self.downloadFile()
        self.testResult = defer.Deferred()
        d.addCallback(self.NotModifiedGreater2)
        d.addErrback(lambda x: self.testResult.errback(failure.Failure(self.UnknownFailure)))
        return self.testResult
    def NotModifiedGreater2(self, x):
        log.debug("testNotModifiedGreater: starting second client", self.debugname)
        d = self.doRequest(uriData(self.filename, http.NOT_MODIFIED, if_modified_since=time.time()))
        d.chainDeferred(self.testResult)
    testNotModifiedGreater.timeout = 3

    def testNotModifiedExact(self):
        d= self.downloadFile()
        self.testResult = defer.Deferred()
        d.addCallback(self.NotModifiedGreater2)
        d.addErrback(lambda x: self.testResult.errback(failure.Failure(self.UnknownFailure)))
        return self.testResult
    def NotModifiedExact2(self, x):
        d = self.doRequest(uriData(self.filename, http.NOT_MODIFIED, if_modified_since=os.path.getmtime(self.filepath)))
        d.chainDeferred(self.testResult)
    testNotModifiedExact.timeout = 2

    class UnknownFailure(Exception):
        pass

    def testNotModifiedUncached(self):
        "Check http not modified sent when file not in cache"
        # Using /packages/Packages as a file that is registered in the package database
        filename, sourcepath, self.destpath = self.getFilePaths('/packages/Packages')
        mtime = os.path.getmtime(sourcepath)
        self.testResult = defer.Deferred()
        d = self.doRequest(uriData(filename, http.NOT_MODIFIED, if_modified_since=mtime))
        d.addCallback(self.NotModifiedUncached2)
        d.addErrback(lambda x: self.testResult.errback(failure.Failure(self.UnknownFailure)))
        return self.testResult
    def NotModifiedUncached2(self, x):
        self.assertRaises(OSError, os.stat, self.destpath)
        self.testResult.callback(None)
    testNotModifiedUncached.timeout = 2

    def testCloseFetcherImmediately(self):
        DownloadQueue.closeTimeout = 0 # Close fetcher immediately
        return self.downloadFile().addCallback(self.CloseFetcherImmediately2)
    def CloseFetcherImmediately2(self, x):
        queues = self.factory.getBackend(self.backendName).queue.queues.values()
        self.assertEquals(len(queues), 0)
    testCloseFetcherImmediately.timeout = 2

    def testDownloadQueueException(self):
        self.testResult = defer.Deferred()
        self.fnsave = DownloadQueue.addFile
        def raiseException(fetcher, cacheentry):
            self.fnsave(fetcher,cacheentry)
            raise RuntimeError('Dummy')
        def restoreException(x):
            DownloadQueue.addFile = self.fnsave
        DownloadQueue.addFile = raiseException
        filename, sourcepath, self.destpath = self.getFilePaths('/packages/Packages')
        d = self.doRequest(uriData(filename, http.INTERNAL_SERVER_ERROR))
        d.addBoth(restoreException)
        d.addCallback(self.DownloadQueueException2)
        self.timer = reactor.callLater(2, self.DownloadQueueExceptionTimeout) # timeout will trigger errorBack if necessary
        return self.testResult
    def DownloadQueueException2(self, x):
        queues = self.factory.getBackend(self.backendName).queue.queues.values()
        self.assertEquals(len(queues), 0)
        log.debug("disconnecting")
        self.connection.disconnect()
        self.timer.cancel()
        log.debug("test result callback")
        self.testResult.callback(None)
    def DownloadQueueExceptionTimeout(self):
        DownloadQueue.addFile = self.fnsave
        log.debug("testDownloadQueueException: timed out")
        self.testResult.errback(failure.Failure(self.UnknownFailure))

    def testLeaveFetcherOpen(self):
        DownloadQueue.closeTimeout = 2 # 2 second delay to close
        return self.downloadFile().addCallback(self.LeaveFetcherOpen2)
    def LeaveFetcherOpen2(self, x):
        queues = self.factory.getBackend(self.backendName).queue.queues.values()
        self.assertNotEquals(len(queues), 0)
    testLeaveFetcherOpen.timeout = 4

    def testAutoCloseFetcher(self):
        DownloadQueue.closeTimeout = 0.1
        d = self.downloadFile()
        self.autoclosedeferred = defer.Deferred()
        d.addCallback(self.AutoCloseFetcher2)
        d.addErrback(lambda x: self.autoclosedeferred.errback(failure.Failure()))
        return self.autoclosedeferred
    def AutoCloseFetcher2(self, x):
        queues = self.factory.getBackend(self.backendName).queue.queues.values()
        self.assertEquals(len(queues), 0)
        self.autoclosedeferred.callback(None)
    testAutoCloseFetcher.timeout = 2

    def testCached(self):
        self.testResult = defer.Deferred()
        d = self.downloadFile()
        d.addCallback(self.Cached2)
        d.addErrback(self.CachedError)
        return self.testResult
    def Cached2(self, x):
        d = self.doRequest(uriData(self.filename, http.OK, filePath=self.filepaths[0]))
        d.addCallback(self.Cached3)
        d.addErrback(self.CachedError)
    def Cached3(self, x):
        log.debug("Downloading second copy", self.debugname)
        self.factory.config.min_refresh_delay = 0
        d = self.doRequest(uriData(self.filename, http.OK, filePath=self.filepaths[0]))
        d.addCallback(self.CachedPass)
        d.addErrback(self.CachedError)
    def CachedPass(self, x):
        self.testResult.callback(None)
    def CachedError(self, x):
        class CachedErrorException(Exception):
            pass
        log.debug("testCached ERROR", self.debugname)
        raise CachedErrorException()
    testCached.timeout = 2

    def testBwLimit(self):
        "Bandwidth limiting"
        b = self.factory.getBackend(self.backendName)
        b.config.bandwidth_limit = 10000000
        # We're not testing here that limiting is applied, just that the code runs
        return self.downloadFile(file='/packages/apt_0.0.1_test.deb')
    testBwLimit.timeout = 2

    def testAbort(self):
        "Abort with complete_clientless_downloads=off"
        self.testResult = defer.Deferred()
        b = self.factory.getBackend(self.backendName)
        b.config.bandwidth_limit = 10
        # We're not testing here that limiting is applied, just that the code runs
        filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
        d = self.doRequest(uriData(filename, http.OK, filePath=sourcepath, abortTransfer=True))
        #d.addCallback(lambda x: reactor.callLater(0.1, self.Abort2, None))
        d.addCallback(self.Abort2)
        d.addErrback(lambda x: self.testResult.errback(failure.Failure(sys.exc_info())))
        return self.testResult
    def Abort2(self, x):
        "Connection was aborted, check that fetchers were closed"
        log.debug("testAbort2", self.debugname)
        backendQueues = self.factory.getBackend(self.backendName).queue.queues
        self.assertEquals(len(backendQueues), 0)
        log.debug("testAbort2a", self.debugname)
        # Final result needs to be triggered after all callbacks have completed
        reactor.callLater(0.1, self.testResult.callback, None)
    testAbort.timeout = 2
    #testAbort.todo = "Test not yet working"
    def testClientDisconnect(self):
	"Test cleanup if client disconnects before data is sent"
        self.testResult = defer.Deferred()
        b = self.factory.getBackend(self.backendName)
        b.config.bandwidth_limit = 10
        filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
        d = self.doRequest(uriData(filename, http.OK, filePath=sourcepath, abortTransfer=True))
        d.addCallback(self.ClientDisconnect2)
        d.addErrback(lambda x: self.testResult.errback(failure.Failure(self.UnknownFailure)))
        return self.testResult
    def ClientDisconnect2(self, x):
        "Connection was aborted, check that fetchers were closed"
        log.debug("testClientDisconnect2", self.debugname)
        #backendQueues = self.factory.getBackend(self.backendName).queue.queues
        #self.assertEquals(len(backendQueues), 0)
        log.debug("testAbort2a", self.debugname)
        self.testResult.callback(None)
    testClientDisconnect.timeout = 2

    # This test does not work with current twisted http client :(
    #def testPipeline(self):
        #"Test pipelined GETs"
        #return self.downloadFiles('/packages/Packages.gz', '/packages/Packages', '/packages/Packages.bz2')
    #testPipeline.timeout = 2

    def testEmpty(self):
        "Test download of empty file"
        return self.downloadFiles('/packages/empty.txt')
    testEmpty.timeout = 2

    def testTilde(self):
        return self.downloadFiles('/packages/tilde~test.txt')
    testTilde.timeout = 2

    def testConnectionRefused(self):
        self.server.stop()
        filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
        d = self.doRequest(uriData(filename, http.SERVICE_UNAVAILABLE))
        d.addBoth(lambda x: self.server.start())
        return d
    testConnectionRefused.timeout = 2
        
    def testParallelFirstDownload(self):
        """
        same request from 3 clients (not in cache)
        """
        self.testResult = defer.Deferred()
        d1 = self.downloadFile()
        self.checkDest = False # Do not raise assertion when downloading the same file several times
        d2 = self.downloadFile()
        d3 = self.downloadFile()
        dl = defer.DeferredList([d1,d2,d3])
        #dl.addCallback(self.ParallelDownload2)
        return dl
    testParallelFirstDownload.timeout = 2
        
    def testParallelDownload(self):
        self.testResult = defer.Deferred()
        d = self.downloadFile(file='/packages/apt_0.0.1_test.deb')
        d.addCallback(self.ParallelDownload2)
        return self.testResult
    testParallelDownload.timeout = 2
    def ParallelDownload2(self,x):
        log.debug("testParallelDownload: First download passed, now requesting cached file from 2 separate clients")
        self.checkDest = False # Do not raise assertion when downloading the same file several times
        d1 = self.downloadFile(file='/packages/apt_0.0.1_test.deb') # Using immutable file
        d2 = self.downloadFile(file='/packages/apt_0.0.1_test.deb')
        dl = defer.DeferredList([d1,d2])
        dl.chainDeferred(self.testResult)
        
    def testDotDot(self):
        "Check path containing /../"
        #filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
        #return self.doRequest(uriData(filename, http.OK, filePath = os.path.normpath(destpath)))
        self.testResult = defer.Deferred()
        filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
        d=self.doRequest(uriData(filename.replace('/','/foo/../'), http.OK, filePath=sourcepath, abortTransfer=True))

        # Final result needs to be triggered after all callbacks have completed
        d.addCallback(lambda x:reactor.callLater(0.1, self.testResult.callback, None))
        return self.testResult
    testDotDot.timeout = 2

        
    #def testTimeout(self):
        #pass
    #testTimeout.todo = True

    #def setFileTime(self):
        #"cache file modification time is set to same time as server time"
        #pass
    #setFileTime.todo = True

    #def doubleDownload(self):
        #"download, delete from cache, re-request file"
        #pass
    #doubleDownload.todo = True

    # More TODO tests:
    # - file mtime is same as server mtime
    # - correct file path is entered in databases after download

class HttpBackendTest(TestRequestHelper, BackendProtocolTests):
    def setUp(self):
        """
        Make a configuration with a single backend
        [files]
        backends=file:///<path to test packages directory>
        """
        BackendTestBase.setUp(self, 'HttpBackendTest', 'http', WebServer)
    def tearDown(self):
        BackendTestBase.tearDown(self)

class FtpBackendTest(TestRequestHelper, BackendProtocolTests):
    def setUp(self):
        """
        Make a configuration with a single backend
        [files]
        backends=file:///<path to test packages directory>
        """
        import test_fetchers
        BackendTestBase.setUp(self, 'FtpBackendTest', 'ftp', test_fetchers.FtpServer)
    def tearDown(self):
        BackendTestBase.tearDown(self)

class RsyncBackendTest(TestRequestHelper, BackendProtocolTests):
    def setUp(self):
        """
        Make a configuration with a single backend
        [files]
        backends=file:///<path to test packages directory>
        """
        BackendTestBase.setUp(self, 'RsyncBackendTest', 'rsync', RsyncServer, uriSuffix='apt-proxy')
    def tearDown(self):
        BackendTestBase.tearDown(self)
    def testTempFile(self):
        "rysnc Tempfile is detected"
        b = self.factory.getBackend(self.backendName)
        b.config.bandwidth_limit = 100000
        filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
        self.doRequest(uriData(filename, None))
        reactor.callLater(0.5, self.TempFile2)
        self.testResult = defer.Deferred()
        return self.testResult
    def TempFile2(self):
        fetcher = self.factory.getBackend(self.backendName).queue.queues.values()[0].fetcher.fetcher
        fetcher.findRsyncTempFile()
        file = fetcher.rsyncTempFile
        log.debug("rsync TempFile is %s" % (file), self.debugname)
        fetcher.disconnect()
        self.connection.disconnect()
        if file is not None:
            self.testResult.callback("Tempfile is %s" %(file))
        else:
            raise UnknownFailure()
    testTempFile.timeout=2


# HangTest classes
#
# Define a protocol that allows a connection but does not respond
class HangProtocol(protocol.Protocol):
    pass
class HangFactory(protocol.ServerFactory):
    protocol = HangProtocol
class HangServer:
    def start(self):
        self.port = reactor.listenTCP(0, HangFactory(), interface="127.0.0.1")
        return self.port.getHost().port
    def stop(self):
        self.port.stopListening()

class ClientCleanupTest(BackendTestBase):
    def setUp(self):
	BackendTestBase.setUp(self, 'ClientCleanupTest', 'http', HangServer)
    def tearDown(self):
	BackendTestBase.tearDown(self)
    def testClientCleanup(self):
	b = self.factory.getBackend(self.backendName)
	b.config.timeout = 1
	filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
	d = self.doRequest(uriData(filename, http.SERVICE_UNAVAILABLE))

class HangTestBase(BackendTestBase):
    def setUp(self, protocol):
        BackendTestBase.setUp(self,protocol + 'HangTest', protocol, HangServer)
    def tearDown(self):
        BackendTestBase.tearDown(self)
    def testConnectionHang(self):
        b = self.factory.getBackend(self.backendName)
        b.config.timeout = 1
        filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
        d = self.doRequest(uriData(filename, http.SERVICE_UNAVAILABLE))
        def printTrace(x):
            import traceback
            log.debug("-------test errback------")
            print("test errback----")
            traceback.print_stack()
        def printOK(x):
            log.debug("---trace OK ---")
        d.addCallback(printOK)
        d.addErrback(printTrace)
        return d
    testConnectionHang.timeout = 5

class HttpHangTest(HangTestBase, TestRequestHelper):
    def setUp(self):
        HangTestBase.setUp(self, 'http')
class FtpHangTest(HangTestBase, TestRequestHelper):
    def setUp(self):
        HangTestBase.setUp(self, 'ftp')
class RsyncHangTest(HangTestBase, TestRequestHelper):
    def setUp(self):
        HangTestBase.setUp(self, 'rsync')

# Attempt a connection to a server that is not responding
dead_ip_address = '127.0.0.2:1'

class NoConnectionBase(BackendTestBase):
    def setUp(self, protocol):
        backend_uri = protocol + "://"+ dead_ip_address

        config = ("dynamic_backends=off\n" +
                  "[test_data]\n" +
                  "backends=" + backend_uri)
        BackendTestBase.init(self, config, protocol + 'HangTest')
    def tearDown(self):
        TestRequestHelper.tearDown(self)
    def testConnectionTimeout(self):
        b = self.factory.getBackend(self.backendName)
        b.config.timeout = 1
        filename, sourcepath, destpath = self.getFilePaths('/packages/apt_0.0.1_test.deb')
        d = self.doRequest(uriData(filename, http.SERVICE_UNAVAILABLE))
        return d
    testConnectionTimeout.timeout = 2

class HttpNoConnectionTest(NoConnectionBase, TestRequestHelper):
    def setUp(self):
        NoConnectionBase.setUp(self, 'http')
class FtpNoConnectionTest(NoConnectionBase, TestRequestHelper):
    def setUp(self):
        NoConnectionBase.setUp(self, 'http')
class HttpNoConnectionTest(NoConnectionBase, TestRequestHelper):
    def setUp(self):
        NoConnectionBase.setUp(self, 'http')
