/* filesel.c
 * Copyright (C) 2002-2004 Pascal Eberhard <pascal.ebo@netcourrier.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <stdio.h>      /* define FILENAME_MAX */
#include "common.h"
#include "gtkfilesel2.h"

typedef struct filesel_t
{
  GtkWidget  *filesel;
  char       *old_value;
  void       *user_data;
  void      (*compute_result) (const char *old_value, const char *new_value,
                               void *data);
} filesel_t;

// ----------------------------------------------------------------------------
static void     filesel_click_ok(filesel_t *filesel);
static void     filesel_click_cancel(filesel_t *filesel);
static gboolean filesel_delete_event(GtkWidget *widget, GdkEvent *event,
                                     filesel_t *filesel);

// ----------------------------------------------------------------------------
struct filesel_t* filesel_new(void)
{
  filesel_t *filesel;
  filesel = (filesel_t *) malloc(sizeof(filesel_t));
  g_assert(filesel != NULL);
  filesel->filesel = apwal_file_selection_new(NULL);
  gtk_widget_hide(filesel->filesel);
  g_signal_connect_swapped(G_OBJECT(APWAL_FILE_SELECTION(
                           filesel->filesel)->ok_button), "clicked",
                           G_CALLBACK(filesel_click_ok), filesel);
  g_signal_connect_swapped(G_OBJECT(APWAL_FILE_SELECTION(
                            filesel->filesel)->cancel_button), "clicked",
                            G_CALLBACK(filesel_click_cancel), filesel);
  g_signal_connect(G_OBJECT(GTK_DIALOG(filesel->filesel)), "delete_event",
                   G_CALLBACK(filesel_delete_event), filesel);

  filesel->old_value = NULL;
  return filesel;
}
// ----------------------------------------------------------------------------
void filesel_select(
         struct filesel_t *filesel, const char *old_value, gboolean dir_only,
         void (*compute_result) (const char *old_value, const char *new_value,
                                 void *data),
         void *data)
{
  char *absolute_path = NULL;
  g_assert(filesel != NULL);
  g_assert(compute_result != NULL);
  g_assert(old_value != NULL);
  filesel->compute_result = compute_result;
  filesel->user_data = data;

  if (!g_file_test(old_value, G_FILE_TEST_IS_EXECUTABLE))
    absolute_path = g_find_program_in_path(old_value);
  
  if (filesel->old_value != NULL)
    g_free(filesel->old_value);
  if (absolute_path == NULL)
    filesel->old_value = g_strdup(old_value);
  else
    filesel->old_value = absolute_path;

  apwal_file_selection_hide_fileop_buttons(APWAL_FILE_SELECTION(filesel->filesel));
  apwal_file_selection_set_dir_only(APWAL_FILE_SELECTION(filesel->filesel), dir_only);
  apwal_file_selection_set_filename(APWAL_FILE_SELECTION(filesel->filesel), old_value);
  gtk_grab_add(filesel->filesel);       //modal
  gtk_widget_show(filesel->filesel);
}
// ----------------------------------------------------------------------------
static void filesel_click_ok(filesel_t *filesel)
{
  const char *new_value;
  g_assert(filesel != NULL);
  new_value = apwal_file_selection_get_filename(
                                     APWAL_FILE_SELECTION(filesel->filesel));
  gtk_grab_remove(filesel->filesel);    //modal
  gtk_widget_hide(filesel->filesel);
  filesel->compute_result(filesel->old_value, new_value, filesel->user_data);
  if (filesel->old_value != NULL)
  {
    g_free(filesel->old_value);
    filesel->old_value = NULL;
  }
}
// ----------------------------------------------------------------------------
void filesel_click_cancel(filesel_t *filesel)
{
  g_assert(filesel != NULL);
  gtk_grab_remove(filesel->filesel); /*modal*/
  gtk_widget_hide(filesel->filesel);
  filesel->compute_result(filesel->old_value, NULL, filesel->user_data);
  if (filesel->old_value != NULL)
  {
    g_free(filesel->old_value);
    filesel->old_value = NULL;
  }
}
// ----------------------------------------------------------------------------
static gboolean filesel_delete_event(GtkWidget *widget, GdkEvent *event,
                                     filesel_t *filesel)
{
  g_assert(filesel != NULL);
  gtk_grab_remove(filesel->filesel); /*modal*/
  gtk_widget_hide(filesel->filesel);
  filesel->compute_result(filesel->old_value, NULL, filesel->user_data);
  if (filesel->old_value != NULL)
  {
    g_free(filesel->old_value);
    filesel->old_value = NULL;
  }
  return TRUE;/*TRUE to stop other handlers from being invoked for the event*/
}
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
