(* Main *)
(* $Id: main.ml,v 1.8 2003/06/10 21:46:32 berke Exp $ *)

open Ast
open Dpkg

let sf = Printf.sprintf

(*** Predicate construction *)

let hierarchical x y =
  let m = String.length x
  and n = String.length y
  in
  if m < n then
    -1
  else if m > n then
    1
  else
    compare x y

let predicate = function
| Exact(x) -> ((=) x)
| Lexicographic_le(x) -> fun y -> hierarchical y x <= 0
| Lexicographic_ge(x) -> fun y -> hierarchical y x >= 0
| Regular(x,o) ->
    let reg =
      let x' = if List.mem Word_boundary o then "\\<"^x^"\\>" else x in
      if List.mem Case_insensitive o then
        Str.regexp_case_fold x'
      else
        Str.regexp x'
    in
    fun y -> try ignore (Str.search_forward reg y 0); true with Not_found -> false

exception Yes

let rec eval db env = function
| And(qb1,qb2) -> IS.inter (eval db env qb1) (eval db env qb2)
| Or(qb1,qb2) -> IS.union (eval db env qb1) (eval db env qb2)
| Not(qb) -> IS.diff db.universe (eval db env qb)
| True -> db.universe
| False -> IS.empty
| Atom(x) -> eval_atom db env x
and eval_atom db env = function
| Matches(Some_field(fdpat),pat) ->
    let fdp = predicate fdpat
    and p = predicate pat
    in
    let rec loop i x =
      if i = db.m then
        x
      else
        if
          try
            SM.iter (fun fd v -> if fdp fd && p v then raise Yes) db.db.(i);
            false
          with
          | Yes -> true
        then
          loop (i + 1) (IS.add i x)
        else
          loop (i + 1) x
    in
    loop 0 IS.empty
| Matches(This_field(fd),pat) ->
    let p = predicate pat in
    let rec loop i x =
      if i = db.m then
        x
      else
        if
          try
            p (SM.find fd db.db.(i))
          with
          Not_found -> false
        then
          loop (i + 1) (IS.add i x)
        else
          loop (i + 1) x
    in
    loop 0 IS.empty

(* Predicate construction and evaluation ***)

(*** Dump styles *)

let bourbaki_dump db x =
  Format.printf "{ @[";
  List.iter (fun i ->
    try
      Format.printf "%s@ " (SM.find "Package" db.db.(i))
    with
    Not_found -> Format.printf "@ (%d)" i) x;
  Format.printf "@]}@."

let list_dump db x =
  List.iter (fun i ->
    try
      Printf.printf "%s\n" (SM.find "Package" db.db.(i))
    with
    Not_found -> Printf.printf "package-%d" i) x

let raw_dump db fd x =
  let f = match fd with
  | Opt.All -> fun _ -> true
  | Opt.These l ->
      let l' = List.map (fun (x,_) -> x) l in
      fun k -> List.mem k l'
  in
  List.iter (fun i ->
    let sm = db.db.(i) in
    (* Printf.printf "%05d Begin package number %d:\n" i i; *)
    SM.iter (fun k v ->
      if f k then Printf.printf "%05d %s: %s\n" i k v) sm;
    ) x

let table_dump db fd x =
  let borders = !Opt.borders in
  let headers = borders
  in
  let l =
  match fd with
  | Opt.All ->
      List.map (fun w -> (w,None))
        (let module SS = Set.Make(String) in
          SS.elements
          (List.fold_left
            (fun cols i ->
              SM.fold (fun key _ cols -> SS.add key cols) db.db.(i) cols)
            SS.empty x))
  | Opt.These(l) -> l
  in
  let a = Array.of_list l in
  let n = Array.length a in
  let b =
    if headers then
      Array.init n (fun i -> let (w,_) = a.(i) in String.length w)
    else
      Array.make n 0
  in
  (* compute maximum width *)
  List.iter (fun i->
    let sm = db.db.(i) in
    for j = 0 to n - 1 do
      try
        let (w,n) = a.(j) in
        let w = SM.find w sm in
        b.(j) <- max b.(j)
                   (match n with
                      None -> String.length w
                    | Some n -> min n (String.length w))
      with
      Not_found -> ()
    done) x;
  let total = Array.fold_left (+) 0 b in
  let dashes () =
    begin
      for i = 0 to n - 1 do
        print_string "+--";
        for j = 0 to b.(i) - 1 do
          print_char '-'
        done;
      done;
      print_char '+';
      print_char '\n'
    end
  in
  let spaces n =
    for i = 1 to n do
      print_char ' '
    done
  in
  if headers then
    begin
      if borders then dashes ();
      if borders then print_string "| ";
      for i = 0 to n - 1 do
        if i > 0 then
          print_char ' ';
        let (w,_) = a.(i) in
        print_string w;
        spaces (b.(i) - String.length w);
        if borders then print_string " |"
      done;
      print_char '\n';
    end;
  if borders then dashes ();
  List.iter (fun i ->
    if borders then print_string "| ";
    for j = 0 to n - 1 do
      let (w,p) = a.(j) in
      let w = try SM.find w db.db.(i) with Not_found -> "" in
      if j > 0 then
        print_char ' ';
      begin
        match p with
          None ->
            print_string w;
            if borders or j < n - 1 then spaces (b.(j) - (String.length w));
        | Some(p) ->
            print_string (Utils.limit p w);
            if borders or j < n - 1 then spaces (b.(j) - (min p (String.length w)));
      end;
      if borders then print_string " |"
    done;
    print_char '\n') x;
  if borders then dashes ()

(* Dump styles ***)

(*** Error display *)

let columns = 75

let put_arrows i j =
  for h = 1 to i do
    prerr_char ' '
  done;
  for h = i to j do
    prerr_char '^';
  done;
  prerr_char '\n'

let show_highlighted out w i j n =
  let m = String.length w in
  let j = max 0 (min (m - 1) j) in
  let b = min (n / 3) (j - i + 1) in
  let ps = min (m - b) (n - b) in
  let s = min (m - j - 1) ((ps + 1) / 2) in
  let p = min i (ps - s) in
  let s = min (m - j - 1) (ps - p) in
  let p_i = i - p in
  let hi_i =
    if p_i > 0 then
      begin
        out "...";
        out (String.sub w p_i p);
        p + 3
      end
    else
      begin
        out (String.sub w 0 p);
        p
      end
  in
  if b < j - i + 1 then
    begin
      let b' = b - 3 in
      let bl = b' / 2 in
      let br = b' - bl
      in
      out (String.sub w i bl);
      out "...";
      out (String.sub w (j - br) br)
    end
  else
    out (String.sub w i b);
  if j + 1 + s < m then
    begin
      out (String.sub w (j + 1) s);
      out "..."
    end
  else
    out (String.sub w (j + 1) s);
  out "\n";
  put_arrows hi_i (hi_i + b - 1)

let escape_and_record_indexes w l =
  let m = String.length w in
  let b = Buffer.create m in
  let r = ref [] in
  for i = 0 to m - 1 do
    if List.mem i l then
      r := (i,Buffer.length b)::!r;
    Buffer.add_string b (String.escaped (String.make 1 w.[i]))
  done;
  if List.mem m l then
    r := (m,Buffer.length b)::!r;
  (Buffer.contents b,!r)

let lower_half x = x / 2
let upper_half x = x - (x / 2)

let show_parse_error i j x w =
  let m = String.length w in
  if m = 0 then
    Printf.eprintf "Error: Syntax error -- Empty query.\n"
  else
    begin
        Printf.eprintf "Error: Syntax error %s of query --- %s:\n"
          (if i = j then 
            if i >= m - 1 then
              "end"
            else
              sf "at character %d" (i + 1)
           else "between "^
           (if i = 0 then "beginning" else sf "character %d" (i + 1))^
           " and "^
           (if j >= m - 1 then "end" else sf "character %d" (j + 1)))
          x;
        let (w',z) = escape_and_record_indexes w [i;j] in
        let m = String.length w'
        and i' = List.assoc i z
        and j' = List.assoc j z
        in
        (* show string w' highlighting i' to j' on columns columns *)
        let w' = if j' >= m - 1 then w'^" " else w' in
        show_highlighted prerr_string w' i' j' columns
      end
(* Error display ***)

(*** Process and interactive *)

exception Parse_error of int * int * string

let query_of_string w =
  let l = Lexing.from_string w in
  try
    Syntax.quantified Lexic.token l
  with
  | Parsing.Parse_error ->
      raise (Parse_error(Lexing.lexeme_start l,Lexing.lexeme_end l,"Parse error"))
  | Failure x ->
      raise (Parse_error(Lexing.lexeme_start l,Lexing.lexeme_end l,"Failure: "^x))
  | Lexic.Parse_error(i,j,x) ->
      raise (Parse_error(i,j,x))

let process db (style,fields,q) =
  let x = eval db [] q in
  let name_of i = SM.find "Package" db.db.(i) in
  let xl = List.sort (fun i j -> compare (name_of i) (name_of j)) (IS.elements x) in
  (match style with
  | Opt.Bourbaki -> bourbaki_dump db
  | Opt.List -> list_dump db
  | Opt.Raw -> raw_dump db fields
  | Opt.Table -> table_dump db fields) xl

let rec interactive db =
  let continue = ref true in
  while !continue do
    print_string "> ";
    flush stdout;
    let w = input_line stdin in
    if w <> "" then
      let q =
        try
          Some(query_of_string w)
        with
        | End_of_file -> continue := false; None
        | Parse_error(i,j,x) ->
          show_parse_error i j x w;
          flush stderr;
          None
        | x -> Printf.printf "Error: %s.\n" (Printexc.to_string x); None
      in
      match q with
      | None -> ()
      | Some(q) -> process db (Opt.List,Opt.All,q)
    else
      ()
  done

(* Process and interactive ***)

(*** Main *)

let _ =
  Arg.parse Opt.specs
    (fun w -> Opt.queries := (Opt.Raw,!Opt.fields,w)::!Opt.queries)
    (sf "Usage: %s [-db path] <options or queries ...>" Sys.argv.(0));
  let queries' =
    List.map (fun (style,fields,w) ->
      try
      let q = query_of_string w in
      if !Opt.ast then
        begin
          Ast.dump Format.err_formatter q;
          Format.fprintf Format.err_formatter "@.";
        end;
      (style,fields,query_of_string w)
      with
      | Parse_error(i,j,x) ->
          show_parse_error i j x w;
          exit 1) !Opt.queries
  in
  if queries' = [] && not !Opt.interactive then
    begin
      prerr_string "Error: No queries given and -interactive option not set ; -help for help and -examples for examples.\n";
      exit 1
    end
  else
    let db = Dpkg.load !Opt.dbfn in
    List.iter (process db) queries';
    if !Opt.interactive then interactive db

(* Main ***)
