(* Main *)
(* $Id: old.ml,v 1.1.1.1 2002/12/03 00:21:24 berke Exp $ *)

module SM = Map.Make(String)
module IM = Map.Make(struct type t = int let compare = compare end)
module IS = Set.Make(struct type t = int let compare = compare end)

module type BOOLEAN_ALGEBRA =
  sig
    type t
    val min : t list -> t
    val max : t list -> t
    val complement : t -> t
    val zero : t
    val one : t
  end

module Query(B:BOOLEAN_ALGEBRA) =
  struct
    type 'a t =
      And of 'a t list
    | Or of 'a t list
    | Not of 'a t
    | True
    | False
    | Atom of 'a

    let rec evaluate f = function
      And(l) -> B.min (List.map (evaluate f) l)
    | Or(l) -> B.max (List.map (evaluate f) l)
    | Not(e) -> B.complement (evaluate f e)
    | True -> B.one
    | False -> B.zero
    | Atom(a) -> f a
  end

module Dpkg =
  struct
    exception Malformed_line

    let decompose_line l =
      try
        let i = String.index l ':' in
        if i + 1 = String.length l then
          raise Malformed_line
        else
          let tag = String.sub l 0 i
          and value = String.sub l (i + 2) (String.length l - i - 2)
          in
          (tag,value)
      with
      Not_found -> raise Malformed_line

    let read_tags ic =
      let b = Buffer.create 16 in
      let f sm = function
        None -> sm
        |Some(x') ->
           let sm' = SM.add x' (Buffer.contents b) sm in
           Buffer.clear b;
           sm'
      in
      let rec loop sm x =
       match try Some(input_line ic) with End_of_file -> None
       with
         None -> sm
       | Some(l) ->
           if String.length l = 0 then
             f sm x
           else
             match l.[0] with
               (' '|'\t') ->
                 if l = " ." then
                   Buffer.add_string b " "
                 else
                   Buffer.add_string b l;
                 loop sm x
             | _ ->
                 let sm = f sm x in
                 let (x,y) = decompose_line l in
                 Buffer.add_string b y;
                 loop sm (Some x)
       in
       loop SM.empty None

    let load fn =
      let ic = open_in fn in
      let rec loop db m =
        let sm = read_tags ic in
        if SM.empty = sm then
          begin
            close_in ic;
            (db,m)
          end
        else
          loop (IM.add m sm db) (m + 1)
      in
      loop IM.empty 0

    let dump db i oc =
      let sm = IM.find i db in
      Printf.fprintf oc "Package number %d:\n" i;
      SM.iter (fun k v ->
        Printf.fprintf oc "  %s = %S\n" k v) sm;
      Printf.fprintf oc "End of package number %d.\n" i
  end

let split_at c s =
  let m = String.length s in
  try
    let i = String.index s c in
    (String.sub s 0 i,
    String.sub s (i + 1) (m - i - 1))
  with
    Not_found -> (s,"")

let _ =
  let (db,m) = Dpkg.load Sys.argv.(1) in
  Printf.printf "Total %d packages available.\n" m;
  let s = Stack.create () in
  let stay = ref true in
  let universe = 
    let rec loop i u =
      if i = m then
        u
      else
        loop (i + 1) (IS.add i u)
    in
    loop 0 IS.empty
  in
  let index =
    let rec loop i idx =
      if i = m then
        idx
      else
        let idx' =
          try
            SM.add (SM.find "Package" (IM.find i db)) i idx
          with
            Not_found -> idx
        in
        loop (i + 1) idx'
    in
    loop 0 SM.empty
  in
  let ifne f =
    if Stack.is_empty s then
      Printf.printf "Stack is empty.\n"
    else
      f ()
  in
  let memory = ref SM.empty in
  while !stay do
    Stack.iter (fun x -> Printf.printf "%d " (IS.cardinal x)) s;
    Printf.printf "> ";
    flush stdout;
    let w = input_line stdin in
    let (cmd,args) = split_at ' ' w in
    match cmd with
        "" -> ()
      | "?"|"help" ->
          Printf.printf
          "  q|quit - quit system
             l|list - list names of packages on top of stack
             d|dump - exhaustively dump packages on top of stack
             u|universe - push the set of all packages on top of stack
             p|pop - remove the set of packages on top of stack
             f|field F R - replace the topmost package set with its
                subset of packages having a field F matching regular
                expression R\n"
      | "q"|"quit" -> stay := false
      | "l"|"list" ->
          ifne (fun () ->
            Printf.printf "{";
            IS.iter (fun i -> Printf.printf " %s" (SM.find "Package" (IM.find i
            db))) (Stack.top s);
            Printf.printf " }\n")
      | "d"|"dump" ->
          ifne (fun () ->
            if args = "" then
              IS.iter (fun i -> Dpkg.dump db i stdout) (Stack.top s)
            else
              let fn = args in
              try
                let oc = open_out fn in
                try
                  IS.iter (fun i -> Dpkg.dump db i oc) (Stack.top s);
                  flush oc;
                  close_out oc
                with
                x -> close_out oc; raise x
              with
              Sys_error(x) ->
                Printf.printf "Error: Could not open file %S: %s.\n" fn x)
      | "d|"|"dumptoproc" ->
          ifne (fun () ->
            let pager =
              if args = "" then
                try Sys.getenv "PAGER" with Not_found -> "/usr/bin/less"
              else
                args
            in
            let oc = Unix.open_process_out pager in
            try
              IS.iter (fun i -> Dpkg.dump db i oc) (Stack.top s);
              flush oc;
              ignore (Unix.close_process_out oc)
            with
            x -> close_out oc; raise x)
      | "u"|"universe" -> Stack.push universe s
      | "p"|"pop" -> ifne (fun () -> ignore (Stack.pop s))
      | "."|"dup" ->
          let n =
            try
              Scanf.sscanf args "%d" (fun x -> x)
            with
              Scanf.Scan_failure _|End_of_file -> 1
          in
          if Stack.length s < n then
            Printf.printf "Error: Stack too short for duplicating %d elements.\n" n
          else
            let rec loop i l =
              if i = n then
                begin
                  List.iter (fun x -> Stack.push x s) l;
                  List.iter (fun x -> Stack.push x s) l
                end
              else
                loop (i + 1) ((Stack.pop s)::l)
          in
          loop 0 []
      | "->"|"store" ->
          if args = "" then
            Printf.printf "Error: Store requires non-empty variable name."
          else
            if Stack.is_empty s then
              Printf.printf "Error: Store requires non-empty stack."
            else
              memory := SM.add args (Stack.top s) !memory
      | "..."|"memory" ->
          Printf.printf "Memory contents:\n";
          SM.iter
            (fun w x ->
              Printf.printf "  %S cardinality %d\n" w (IS.cardinal x)) !memory
      | "!"|"load" ->
          begin
            if args = "" then
              Printf.printf "Error: Load requires non-empty variable name."
            else
              try
                Stack.push (SM.find args !memory) s
              with
              Not_found -> Printf.printf "Error: Memory has no contents associated to word %S.\n" args
          end
      | "&"|"intersection" ->
          if Stack.length s >= 2 then
            let s2 = Stack.pop s
            and s1 = Stack.pop s
            in
            Stack.push (IS.inter s1 s2) s
          else
            Printf.printf "Error: Intersection requires two sets."
      | "<>"|"swap" ->
          if Stack.length s >= 2 then
            let s1 = Stack.pop s
            and s2 = Stack.pop s
            in
            Stack.push s1 s;
            Stack.push s2 s
          else
            Printf.printf "Error: Swapping two sets.\n"
      | "r"|"read" ->
          begin
            let fn = args in
            if fn = "" then
              Printf.printf "Error: Need a filename.\n"
            else
              try
                let ic = open_in fn in
                let rs = ref IS.empty in
                try
                  while true do
                    Scanf.fscanf ic "%d" (fun i -> rs := IS.add i !rs)
                  done
                with
                Scanf.Scan_failure(x) ->
                  close_in ic;
                  Printf.printf "Error: Scan failure: %s.\n" x
                |End_of_file ->
                  close_in ic;
                  Stack.push !rs s
                |x -> close_in ic; raise x
              with
              Sys_error(x) ->
                Printf.printf "Error: Could not open file %S: %s.\n" fn x
          end
      | "w"|"write" ->
          ifne (fun () ->
            let fn = args in
            if fn = "" then
              Printf.printf "Error: Need a filename.\n"
            else
              try
                let oc = open_out fn in
                try
                  IS.iter (fun i -> Printf.fprintf oc "%d " i) (Stack.top s);
                  close_out oc
                with
                Sys_error(x) ->
                  close_out oc;
                  Printf.printf "Error: While writing to file %S: %s.\n" fn x
                |x ->
                  close_out oc;
                  raise x
              with
              Sys_error(x) ->
                Printf.printf "Error: Could not open file %S: %s.\n" fn x)
      | "1"|"first" ->
          ifne (fun () ->
            let s1 = Stack.top s in
            if IS.is_empty s1 then
              Printf.printf "Error: Empty set.\n"
            else
              Stack.push (IS.singleton (IS.choose s1)) s)
      | "|"|"union" ->
          if Stack.length s >= 2 then
            let s2 = Stack.pop s
            and s1 = Stack.pop s
            in
            Stack.push (IS.union s1 s2) s
          else
            Printf.printf "Error: Union requires two sets.\n"
      | "&~"|"diff" ->
          if Stack.length s >= 2 then
            let s2 = Stack.pop s
            and s1 = Stack.pop s
            in
            Stack.push (IS.diff s1 s2) s
          else
            Printf.printf "Error: Diff requires two sets.\n"
      | "&~f"|"&~field" ->
          ifne (fun () ->
            let (field,regexp) = split_at ' ' args in
            let reg =  Str.regexp regexp in
            Stack.push (IS.filter (fun i ->
              let sm = IM.find i db in
              try
                let w = SM.find field sm in
                not (try ignore (Str.search_forward reg w 0); true with
                Not_found -> false)
              with
              Not_found -> true) (Stack.pop s)) s)
      | "c"|"clear" -> Stack.clear s
      | "&f"|"&field" ->
          ifne (fun () ->
            let (field,regexp) = split_at ' ' args in
            let reg = Str.regexp regexp in
            Stack.push (IS.filter (fun i ->
              let sm = IM.find i db in
              try
                let w = SM.find field sm in
                (try ignore (Str.search_forward reg w 0); true with
                Not_found -> false)
              with
              Not_found -> false) (Stack.pop s)) s)
      | ":"|"singleton" ->
          begin
            try
              Stack.push (IS.singleton (SM.find args index)) s
            with
            Not_found -> Printf.printf "Error: Package %S not found.\n" args
          end
      | x -> Printf.printf "Unknown command %S. Type ? for help.\n" x
  done

