# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman, Enno Cramer
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.006;
use strict;
use warnings;

package ArchWay::Widget::Inventory::Entry;

use Glib qw(TRUE FALSE);
use Gtk2;

use base qw(Gtk2::Frame);

use Arch::Inventory qw(:category :type);
use POSIX qw(strftime);

use constant FIELDS => qw(type mode size mtime path category id);
use constant LABELS => ('Type', 'Mode', 'Size', 'Last Modified', 'Full Path', 'Category', 'Inventory ID');

my %TYPES = (
	FILE()      => 'File',
	DIRECTORY() => 'Directory',
	SYMLINK()   => 'Symlink',
);

my %CATEGORIES = (
	TREE()         => 'Arch Tree Root',
	SOURCE()       => 'Source',
	PRECIOUS()     => 'Precious',
	BACKUP()       => 'Backup',
	JUNK()         => 'Junk',
	UNRECOGNIZED() => 'Unrecognized',
);

sub new ($$) {
	my $class     = shift;
	my $inventory = shift;

	my $self = bless Gtk2::Frame->new(' File Details '), $class;

	$self->_init();
	$self->show($inventory) if ($inventory);

	return $self;
}

sub _init ($) {
	my $self = shift;

	$self->set_border_width(3);

	my $table = Gtk2::Table->new(scalar(FIELDS), 2);
	$table->set_row_spacings(2);
	$table->set_col_spacings(10);

	for (my $i = 0; $i < scalar(LABELS); ++$i) {
		my $lbl = Gtk2::Label->new();
		$lbl->set_markup('<b>'.(LABELS)[$i].':</b>');
		$lbl->set_alignment(0.0, 0.0);
		$table->attach($lbl, 0, 1, $i, $i+1, [ 'fill' ], [ 'fill' ], 0, 0);
	}

	for (my $i = 0; $i < scalar(FIELDS); ++$i) {
		my $lbl = Gtk2::Label->new();
		$lbl->set_alignment(0.0, 0.0);
		$table->attach($lbl, 1, 2, $i, $i+1, [ 'expand', 'fill' ], [ 'fill' ], 0, 0);

		$self->{(FIELDS)[$i]} = $lbl;
	}

	$self->add($table);
}

sub show ($$) {
	my $self      = shift;
	my $inventory = shift;

	if (defined $inventory) {
		$self->{type}->set_text($TYPES{$inventory->{type}});
		$self->{path}->set_text($inventory->{path});
		$self->{category}->set_text($CATEGORIES{$inventory->{category}});
		$self->{id}->set_text($inventory->{id});

		if (exists $inventory->{stat}) {
			$self->{mode}->set_text(mode_string($inventory->{stat}->[2]));
			$self->{size}->set_text($inventory->{stat}->[7]);
			$self->{mtime}->set_text(
				strftime(
					'%c',
					localtime($inventory->{stat}->[9])
				)
			);
		} else {
			foreach my $name (qw(mode size mtime)) {
				$self->{$name}->set_markup('<i>unknown</i>');
			}
		}
	} else {
		foreach my $field (FIELDS) {
			$self->{$field}->set_text('');
		}
	}
}

sub mode_string ($) {
	my $mode = shift;

	my @modes = qw(--- --x -w- -wx r-- r-x rw- rwx);

	return join(' ', map { $modes[($mode >> 3*$_) & 7] } (2, 1, 0));
}

1;

__END__
