# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package ArchZoom::TemplateEngine::Native;

use base 'ArchZoom::TemplateEngine';

use ArchZoom::Util;
no strict 'refs';

# accept filters :asis :urlize :htmlize :break
sub escape ($;$) {
	my $str = shift;
	my $filter_str = shift || '';

	$filter_str = ":htmlize$filter_str"
		unless $filter_str =~ /:(asis|htmlize)\b/;
	die "Invalid variable filter string ($filter_str)\n"
		unless $filter_str =~ s/^:(\w+(:\w+)*)$/$1/;
	my @filters = split(':', $filter_str);

	foreach (@filters) {
		if (/^urlize|htmlize|break$/) {
			$str = &$_($str);
		}
		elsif (/^lc|uc|lcfirst|ucfirst$/) {
			$str = eval "$_(\$str)";
		}
	}
	return $str;
}

sub expand_stash ($$) {
	my $stash = shift;
	$_[0] =~ s{(&\w+(?::\w+)*)\(([^\)]*)\)}{
		my ($prefix, $arg) = ($1, $2);
		$arg =~ s/,\s*/\01/g;
		"$prefix\0$arg\0"
	}sge;

	$_[0] =~ s{\$(?:(\w+)\.)?(\w+)((?::\w+)*)}{
		my $value = $1? do {
			my $this = $stash->{$1};
			ref($this) eq 'HASH'? $this->{$2}: $this->$2()
		}: $stash->{$2};
		defined $value? escape($value, $3): $&;
	}sge;

	$_[0] =~ s{\$#(\w+)}{
		my $value = $stash->{$1};
		ref($value) eq 'ARRAY'? scalar(@$value):
			defined $value? length $value: $&;
	}sge;

	$_[0] =~ s{&(\w+)((?::\w+)*)\0([^\0]*)\0}{
		my $match = $&;
		my ($func, $arg) = ("main::$1", $3);
		my $filter_str = !$2 && substr($func, 6, 7) eq 'selfurl' ? ':urlize' : $2;
		# check whether we are ready to expand the function (not perfect)
		if ($arg =~ /(?:^|\01)\$/) {
			$match
		} else {
			my @args = split(/\01/, $arg);
			my $value = &$func(@args);
			escape($value, $filter_str)
		}
	}sge;
	return undef;
}

sub parse ($$;$$) {
	my $self = shift;
	my $file_name = shift || die "No template to parse";
	my $stash = shift || {};
	$stash = { %{$self->{default_stash}}, %$stash };
	my $outer_label = shift || "";

	my $last_if_condition = 1;
	my $index_total = 0;

	my $content;
	if (ref($file_name) eq 'SCALAR') {
		$content = $$file_name;
		$file_name = "(inline)";
	} else {
		my $dir = $self->{set_dir};
		$file_name .= ".html" unless $file_name =~ /\./;
		$file_name = "$dir/$file_name" unless $file_name =~ /^\//;

		open TMPL, "<$file_name" or die "Can't open template $file_name: $!\n";
		$content = join("", <TMPL>) || die "Can't read from $file_name: $!";
		close TMPL;
	}
	die "Internal error, undefined template content" unless defined $content;

	my $error = sub ($) { die "${outer_label}Template error in $file_name: $_[0]\n" };

	my $result = "";

	#/<\[\s*(#|\w+)\s((?:<\[.*?\]>|.)*?)\]>/
	foreach my $chunk (split /(<\[\s*(?:#|\w[\w-]+)\s(?:(?:<\[(?:<\[(?:<\[(?:<\[(?:<\[(?:<\[.*?\]>|.)*?\]>|.)*?\]>|.)*?\]>|.)*?\]>|.)*?\]>|.)*?)\]>)/s, $content) {
		unless ($chunk =~          /<\[\s*(#|\w[\w-]+)\s((?:<\[(?:<\[(?:<\[(?:<\[(?:<\[(?:<\[.*?\]>|.)*?\]>|.)*?\]>|.)*?\]>|.)*?\]>|.)*?\]>|.)*?)\]>/s) {
			expand_stash($stash, $chunk);
			next;
		}
		my ($op, $rest) = (uc($1), $2);
		my $label = "From $op in $file_name: ";
		if ($op eq "#") {
			$chunk = "";  # ignore comment
		}
		elsif ($op eq "INCLUDE" || $op eq "INCLUDE-MEMOIZE") {
			my $substash = { %$stash };
			expand_stash($stash, $rest);
			while ($rest =~ /^\s*(\w+)=(['"]?)(.*?)\2\s+(.*)$/s) {
				$substash->{$1} = $3;
				$rest = $4;
			}
			$rest =~ /^([^\s]+)(?:\s+(s[^\w].*?))?\s*$/
				or &$error("$op: no name ($rest)");
			my ($tname, $filter) = ($1, $2);
			if ($op eq "INCLUDE-MEMOIZE" && defined $ArchZoom::_TemplateIncludeCache{$tname}) {
				$chunk = $ArchZoom::_TemplateIncludeCache{$tname};
				goto FINISH_INCLUDE;
			}
			if ($tname =~ m!^http://!) {
				$tname =~ s!__SERVER__!$ENV{SERVER_NAME} || "localhost"!e;
				require Arch::LiteWeb;
				my $web = Arch::LiteWeb->new;
				$chunk = $web->get($tname);
				unless (defined $chunk) {
					my $msg = $web->error . " while processing " . $web->request_url . "\n";
					$substash->{die_on_error}? die $msg: warn $msg;
					$chunk = $substash->{include_error}? "<p>$msg</p>": "";
				}
				$chunk = $self->parse(\$chunk, $substash, $label)
					if $chunk && !$substash->{dont_parse};
			} else {
				$chunk = $self->parse($tname, $substash, $label);
			}
			$ArchZoom::_TemplateIncludeCache{$tname} = $chunk
				if $op eq "INCLUDE-MEMOIZE";
			FINISH_INCLUDE:
			eval qq(\$chunk =~ $filter; 1) || warn $@ if $filter;
		}
		elsif ($op eq "FOREACH") {
			$rest =~ /^\s*\$(\w+(?:\s+\$\w+)*)\s+\@(\w+)(!?)\s(.*)$/s
				or &$error("FOREACH: incorrect syntax in ($rest)");
			my ($var_str, $array_name, $strict, $subcontent) = ($1, $2, $3, $4);
			my @var_names = split(/[\s\$]+/, $var_str);
			my $array = $stash->{$array_name};
			$array ||= [] unless $strict;
			&$error("FOREACH: No array $array_name is defined")
				unless ref($array) eq 'ARRAY';

			$chunk = "";
			my $index = 0;
			foreach my $value (@$array) {
				my $substash = {
					%$stash,
					index => $index,
					count => $index + 1,
					zebra0 => $index % 2,
					zebra => $index_total % 2,
					index_total => $index_total,
					count_total => $index_total + 1,
				};
				if (@var_names == 1) {
					$substash->{$var_names[0]} = $value;
				} else {
					if (ref($value) eq 'HASH') {
						my $new_value = [];
						foreach (@var_names) {
							&$error("FOREACH: \$${array_name}->[$index]->{$_} no such key")
								unless exists $value->{$_};
							push @$new_value, $value->{$_};
						}
						$value = $new_value;
					}
					&$error("FOREACH: \$${array_name}->[$index] is not ARRAY")
						unless ref($value) eq 'ARRAY';
					&$error("FOREACH: \$${array_name}->[$index] is ARRAY of "
						. @$value . " not " . @var_names)
						unless @$value == @var_names;
					my $i = 0;
					$substash->{$var_names[$i++]} = $_ foreach @$value;
				}
				$chunk .= $self->parse(\$subcontent, $substash, $label);
				$index++; $index_total++;
			}
		}
		elsif ($op eq "ELSIF" && $last_if_condition) {
			$chunk = "";
		}
		elsif ($op eq "IF" || $op eq "ELSIF") {
			$rest =~ /^\s*(.*?)\s+THEN\b\s?(.*)$/s
				or &$error("$op: missing proper THEN syntax in ($rest)");
			my ($code, $subcontent) = ($1, $2);
			expand_stash($stash, $code);
			$last_if_condition = eval("package main; $code");
			$chunk = $last_if_condition?
				$self->parse(\$subcontent, $stash, $label): "";
		}
		elsif ($op eq "ELSE") {
			$chunk = $last_if_condition?
				"": $self->parse(\$rest, $stash, $label);
		}
		elsif ($op =~ /^ELS(IF-DEFINED|IF-ENABLED)$/ && $last_if_condition) {
			$chunk = "";
		}
		elsif ($op =~ /^(ELS)?(IF-DEFINED|IF-ENABLED)$/) {
			my $do_defined = $2 eq 'IF-DEFINED';
			$rest =~ /^\s*(!?(?:\w+\.)?\w+(?:,\s*!?(?:\w+\.)?\w+)*)\s?(.*)$/s
				or &$error("$op: missing variable name(s) in ($rest)");
			my ($var_string, $subcontent) = ($1, $2);
			$last_if_condition = 1;
			my @vars = split(/,\s*/, $var_string);
			foreach (@vars) {
				my $not = s/^!//;
				/^(?:(\w+)\.)?(\w+)$/ or die;
				my $value = $1? do {
					my $this = $stash->{$1};
					ref($this) eq 'HASH'? $this->{$2}: $this->$2()
				}: $stash->{$2};
				$last_if_condition &&= ($do_defined? defined $value: $value? 1: 0) ^ $not;
			}
			$chunk = $last_if_condition?
				$self->parse(\$subcontent, $stash, $label): "";
		}
		elsif ($op eq "SET") {
			while ($rest =~ s/^\s*(\w+)=(?:(['"`])(.*?)\2|(\S+))//s) {
				my ($name, $value) = ($1, defined $3? $3: $4);
				$stash->{$name} = $self->parse(\$value, $stash, $label);
			}
			$chunk = "";
		}
	} continue {
		$result .= $chunk;
	}

	return $result;
}

1;
