/*
    Copyright (C) 2004 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: crossfade_edit.cc,v 1.1 2004/02/29 23:33:54 pauld Exp $
*/

#include <cmath>

#include <sigc++/bind.h>

#include <gtk--/frame.h>
#include <gtk--/pixmap.h>
#include <gtk--/scrolledwindow.h>
#include <ardour/automation_event.h>
#include <ardour/curve.h>
#include <ardour/crossfade.h>
#include <ardour/session.h>
#include <ardour/auditioner.h>
#include <ardour/audioplaylist.h>
#include <ardour/playlist_templates.h>

#include <gtkmmext/gtk_ui.h>

#include "crossfade_edit.h"
#include "rgb_macros.h"
#include "canvas-simplerect.h"
#include "canvas-waveview.h"
#include "keyboard.h"
#include "utils.h"

using namespace ARDOUR;
using namespace Gtk;
using namespace SigC;

#include "i18n.h"

const int CrossfadeEditor::Point::size = 7;
const double CrossfadeEditor::canvas_border = 10;
CrossfadeEditor::Presets* CrossfadeEditor::presets = 0;
CrossfadeEditor::Preset* CrossfadeEditor::s_in_preset = 0;
CrossfadeEditor::Preset* CrossfadeEditor::s_out_preset = 0;
CrossfadeEditor::Preset* CrossfadeEditor::midpoint_in_preset = 0;
CrossfadeEditor::Preset* CrossfadeEditor::midpoint_out_preset = 0;

static const gchar * s_in_xpm[] = {
"25 25 2 1",
" 	c None",
".	c #000000",
"                         ",
"                         ",
"                    .... ",
"                   ..    ",
"                  .      ",
"                 ..      ",
"                 .       ",
"                .        ",
"               .         ",
"               .         ",
"              .          ",
"             ..          ",
"            ..           ",
"                         ",
"           .             ",
"           .             ",
"          .              ",
"         ..              ",
"        ..               ",
"      ...                ",
"  .....                  ",
"                         ",
"                         ",
"                         ",
"                         "};

static const gchar * s_out_xpm[] = {
"25 25 2 1",
" 	c None",
".	c #000000",
"                         ",
"  .....                  ",
"      ...                ",
"        ..               ",
"         ..              ",
"          .              ",
"           .             ",
"           .             ",
"           .             ",
"           ..            ",
"            ..           ",
"             ..          ",
"               .         ",
"               .         ",
"                .        ",
"                 .       ",
"                  .      ",
"                   ..    ",
"                    .... ",
"                         ",
"                         ",
"                         ",
"                         "};

static const gchar * midstep_in_xpm[] = {
"25 25 2 1",
" 	c None",
".	c #000000",
"                         ",
"          .............. ",
"          ..            ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"  ..........             ",
"                         ",
"                         ",
"                         ",
"                         "};


static const gchar * midstep_out_xpm[] = {
"25 25 2 1",
" 	c None",
".	c #000000",
"                         ",
"  ..........             ",
"          ..            ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          ..             ",
"          .............. ",
"                         ",
"                         ",
"                         ",
"                         "};

CrossfadeEditor::Half::Half ()
	: line (0), 
	  normative_curve (0.0, 1.0, 1.0, true),
	  gain_curve (0.0, 2.0, 1.0, true)
{
}

CrossfadeEditor::CrossfadeEditor (Session& s, Crossfade& xf, double my, double mxy)
	: Window (GTK_WINDOW_TOPLEVEL),
	  cancel_button (_("cancel")),
	  ok_button (_("apply")),
	  xfade (xf),
	  session (s),
	  clear_button (_("clear")),
	  revert_button (_("reset")),
	  audition_button (_("audition")),

	  miny (my),
	  maxy (mxy),

	  select_in_button (_("Fade In")),
	  select_out_button (_("Fade Out"))
{
	set_wmclass ("ardour_automationedit", "Ardour");
	set_name ("CrossfadeEditWindow");
	set_title (_("ardour: autoedit"));
	set_position (GTK_WIN_POS_MOUSE);

	add (vpacker);
	add_events (GDK_KEY_PRESS_MASK|GDK_KEY_RELEASE_MASK|GDK_POINTER_MOTION_MASK);

	select_out_button.set_group (select_in_button.group());
	select_out_button.set_mode (false);
	select_in_button.set_mode (false);

	if (presets == 0) {
		build_presets ();
	}

	point_grabbed = false;
	toplevel = 0;

	_canvas = gtk_canvas_new_aa ();

	canvas = wrap (_canvas);
	canvas->size_allocate.connect (slot (*this, &CrossfadeEditor::canvas_allocation));
	canvas->set_usize (300, 200);

	toplevel = gtk_canvas_item_new (gtk_canvas_root (GTK_CANVAS(_canvas)),
					gtk_canvas_simplerect_get_type(),
					"x1", 0.0,
					"y1", 0.0,
					"x2", 10.0,
					"y2", 10.0,
					"fill", (gboolean) TRUE,
					"fill_color_rgba", (guint32) RGBA_TO_UINT (40, 46, 74, 255),
					"outline_pixels", 0,
					NULL);

	gtk_signal_connect (GTK_OBJECT(toplevel),
			    "event",
			    (GtkSignalFunc) _canvas_event,
			    this);

	fade[Out].line = gtk_canvas_item_new (gtk_canvas_root (GTK_CANVAS(_canvas)),
					      gtk_canvas_line_get_type (),
					      "width_pixels", (guint) 1,
					      "fill_color_rgba", RGBA_TO_UINT (0, 220, 220, 255),
					      NULL);
	
	fade[In].line = gtk_canvas_item_new (gtk_canvas_root (GTK_CANVAS(_canvas)),
					     gtk_canvas_line_get_type (),
					     "width_pixels", (guint) 1,
					     "fill_color_rgba", RGBA_TO_UINT (0, 220, 220, 120),
					     NULL);
	
	// cerr << "autoedit curve @ " << curve << endl;

	gtk_signal_connect (GTK_OBJECT(fade[In].line),
			    "event",
			    (GtkSignalFunc) _curve_event,
			    this);

	gtk_signal_connect (GTK_OBJECT(fade[Out].line),
			    "event",
			    (GtkSignalFunc) _curve_event,
			    this);


	select_in_button.set_name (X_("CrossfadeEditCurveButton"));
	select_out_button.set_name (X_("CrossfadeEditCurveButton"));

	select_in_button.clicked.connect (bind (slot (this, &CrossfadeEditor::curve_select_clicked), In));
	select_out_button.clicked.connect (bind (slot (this, &CrossfadeEditor::curve_select_clicked), Out));

	Frame* frame = manage (new Frame);
	frame->add (*canvas);
	frame->set_shadow_type (GTK_SHADOW_IN);

	preset_upper_box.set_spacing (5);
	preset_upper_box.set_homogeneous (false);

	preset_lower_box.set_spacing (5);
	preset_lower_box.set_homogeneous (false);

	Pixmap *pxmap;
	Button* pbutton;

	pxmap = manage (new Pixmap (s_in_xpm));
	pbutton = manage (new Button);
	
	pbutton->add (*pxmap);
	pbutton->set_name ("CrossfadeEditorButton");
	pbutton->clicked.connect (bind (slot (*this, &CrossfadeEditor::apply_preset),
					s_in_preset));
		
	preset_upper_box.pack_start (*pbutton, false, false);

	pxmap = manage (new Pixmap (midstep_in_xpm));
	pbutton = manage (new Button);
	
	pbutton->add (*pxmap);
	pbutton->set_name ("CrossfadeEditorButton");
	pbutton->clicked.connect (bind (slot (*this, &CrossfadeEditor::apply_preset),
					midpoint_in_preset));

	preset_upper_box.pack_start (*pbutton, false, false);

	pxmap = manage (new Pixmap (s_out_xpm));
	pbutton = manage (new Button);
	pbutton->add (*pxmap);
	pbutton->set_name ("CrossfadeEditorButton");
	pbutton->clicked.connect (bind (slot (*this, &CrossfadeEditor::apply_preset),
					s_out_preset));

	preset_lower_box.pack_start (*pbutton, false, false);

	pxmap = manage (new Pixmap (midstep_out_xpm));
	pbutton = manage (new Button);
	pbutton->add (*pxmap);
	pbutton->set_name ("CrossfadeEditorButton");
	pbutton->clicked.connect (bind (slot (*this, &CrossfadeEditor::apply_preset),
					midpoint_out_preset));

	preset_lower_box.pack_start (*pbutton, false, false);
	
	vpacker2.set_border_width (12);
	vpacker2.set_spacing (5);
	vpacker2.pack_start (*frame, true, true);
	vpacker2.pack_start (preset_upper_box, false, false);
	vpacker2.pack_start (preset_lower_box, false, false);

	clear_button.set_name ("CrossfadeEditButton");
	revert_button.set_name ("CrossfadeEditButton");
	ok_button.set_name ("CrossfadeEditButton");
	cancel_button.set_name ("CrossfadeEditButton");
	audition_button.set_name ("CrossfadeEditAuditionButton");

	clear_button.clicked.connect (slot (*this, &CrossfadeEditor::clear));
	revert_button.clicked.connect (slot (*this, &CrossfadeEditor::reset));
	audition_button.toggled.connect (slot (*this, &CrossfadeEditor::audition_toggled));

	action_box.set_border_width (12);
	action_box.set_spacing (5);
	action_box.set_homogeneous (false);
	action_box.pack_end (cancel_button, false, false);
	action_box.pack_end (ok_button, false, false);
	action_box.pack_end (revert_button, false, false);
	action_box.pack_end (clear_button, false, false);
	action_box.pack_end (audition_button, false, false);

	Gtk::Label* label = manage (new Gtk::Label (_("Select active curve: ")));
	label->set_name (X_("CrossfadeEditLabel"));

	curve_button_box.set_spacing (7);
	curve_button_box.pack_start (*label, false, false);
	curve_button_box.pack_start (select_in_button, false, true);
	curve_button_box.pack_start (select_out_button, false, true);

	vpacker.set_border_width (12);
	vpacker.set_spacing (12);
	vpacker.pack_start (curve_button_box, false, true);
	vpacker.pack_start (vpacker2, true, true);
	vpacker.pack_start (action_box, false, false);

	current = In;

	set (xfade.fade_in(), In);
	set (xfade.fade_out(), Out);

	curve_select_clicked (In);

	session.AuditionActive.connect (slot (*this, &CrossfadeEditor::audition_state_changed));


}

CrossfadeEditor::~CrossfadeEditor()
{
	/* most objects will be destroyed when the toplevel window is. */

	for (list<Point*>::iterator i = fade[In].points.begin(); i != fade[In].points.end(); ++i) {
		delete *i;
	}

	for (list<Point*>::iterator i = fade[Out].points.begin(); i != fade[Out].points.end(); ++i) {
		delete *i;
	}
}

void
CrossfadeEditor::audition_state_changed (bool yn)
{
	if (!Gtkmmext::UI::instance()->caller_is_gui_thread()) {
		Gtkmmext::UI::instance()->call_slot (bind (slot (*this, &CrossfadeEditor::audition_state_changed), yn));
		return;
	}
	
	if (audition_button.get_active() != yn) {
		audition_button.set_active (yn);
	}
}

void
CrossfadeEditor::set (const ARDOUR::Curve& curve, WhichFade which)
{
	double firstx, endx;
	ARDOUR::Curve::const_iterator the_end;

	for (list<Point*>::iterator i = fade[which].points.begin(); i != fade[which].points.end(); ++i) {
			delete *i;
	}
	
	fade[which].points.clear ();
	fade[which].gain_curve.clear ();
	fade[which].normative_curve.clear ();

	if (curve.empty()) {
		goto out;
	}
	
	the_end = curve.const_end();
	--the_end;
	
	firstx = (*curve.const_begin())->when;
	endx = (*the_end)->when;
	
	for (ARDOUR::Curve::const_iterator i = curve.const_begin(); i != curve.const_end(); ++i) {
		
		double xfract = ((*i)->when - firstx) / (endx - firstx);
		double yfract = ((*i)->value - miny) / (maxy - miny);
		
		Point* p = make_point ();
		
		p->move_to (x_coordinate (xfract), y_coordinate (yfract),
			    xfract, yfract);
		
		fade[which].points.push_back (p);
	}

	/* no need to sort because curve is already time-ordered */

  out:
	
	swap (which, current);
	redraw ();
	swap (which, current);
}

gint		     
CrossfadeEditor::_canvas_event (GtkCanvasItem* item, GdkEvent* event, gpointer data)
{
	CrossfadeEditor* ed = static_cast<CrossfadeEditor*> (data);
	return ed->canvas_event (item, event);
}

gint
CrossfadeEditor::canvas_event (GtkCanvasItem* item, GdkEvent* event)
{
	switch (event->type) {
	case GDK_BUTTON_PRESS:
		add_control_point ((event->button.x - canvas_border)/effective_width(),
				   1.0 - ((event->button.y - canvas_border)/effective_height()));
		return TRUE;
		break;
	default:
		break;
	}
	return FALSE;
}

CrossfadeEditor::Point::~Point()
{
	gtk_object_destroy (GTK_OBJECT(box));
}

CrossfadeEditor::Point*
CrossfadeEditor::make_point ()
{
	Point* p = new Point;

	p->box = gtk_canvas_item_new (gtk_canvas_root (GTK_CANVAS(_canvas)),
				      gtk_canvas_simplerect_get_type(),
				      "fill", (gboolean) TRUE,
				      "fill_color_rgba", RGBA_TO_UINT (0,255,0,255),
				      "outline_color_rgba", RGBA_TO_UINT (0,0,255,255),
				      "outline_pixels", (gint) 1,
				      NULL);

	p->curve = fade[current].line;

	gtk_object_set_data (GTK_OBJECT(p->box), "point", p);
	gtk_signal_connect (GTK_OBJECT(p->box), "event", 
			    (GtkSignalFunc) _point_event, 
			    this);
	
	return p;
}

void
CrossfadeEditor::add_control_point (double x, double y)
{
	PointSorter cmp;

	/* enforce end point x location */
	
	if (fade[current].points.empty()) {
		x = 0.0;
	} else if (fade[current].points.size() == 1) {
		x = 1.0;
	} 

	Point* p = make_point ();

	p->move_to (x_coordinate (x), y_coordinate (y), x, y);

	fade[current].points.push_back (p);
	fade[current].points.sort (cmp);

	redraw ();
}

void
CrossfadeEditor::Point::move_to (double nx, double ny, double xfract, double yfract)
{
	const double half_size = rint(size/2.0);
	double x1 = nx - half_size;
	double x2 = nx + half_size;

	gtk_canvas_item_set (box,
			     "x1", x1,
			     "x2", x2,
			     "y1", ny - half_size,
			     "y2", ny + half_size,
			     NULL);
	x = xfract;
	y = yfract;
}

void
CrossfadeEditor::canvas_allocation (GtkAllocation *alloc)
{
	if (toplevel) {
		gtk_canvas_item_set (toplevel,
				     "x1", 0.0,
				     "y1", 0.0,
				     "x2", (double) _canvas->allocation.width + canvas_border,
				     "y2", (double) _canvas->allocation.height + canvas_border,
				     NULL);
	}

	gtk_canvas_set_scroll_region (GTK_CANVAS(_canvas), 0.0, 0.0,
				      _canvas->allocation.width,
				      _canvas->allocation.height);

	Point* end = make_point ();
	PointSorter cmp;

	if (fade[In].points.size() > 1) {
		Point* old_end = fade[In].points.back();
		fade[In].points.pop_back ();
		end->move_to (x_coordinate (old_end->x),
			      y_coordinate (old_end->y),
			      old_end->x, old_end->y);
		delete old_end;
	} else {
		double x = 1.0;
		double y = 0.5;
		end->move_to (x_coordinate (x), y_coordinate (y), x, y);

	}

	fade[In].points.push_back (end);
	fade[In].points.sort (cmp);

	for (list<Point*>::iterator i = fade[In].points.begin(); i != fade[In].points.end(); ++i) {
		(*i)->move_to (x_coordinate((*i)->x), y_coordinate((*i)->y),
			       (*i)->x, (*i)->y);
	}
	
	end = make_point ();
	
	if (fade[Out].points.size() > 1) {
		Point* old_end = fade[Out].points.back();
		fade[Out].points.pop_back ();
		end->move_to (x_coordinate (old_end->x),
			      y_coordinate (old_end->y),
			      old_end->x, old_end->y);
		delete old_end;
	} else {
		double x = 1.0;
		double y = 0.5;
		end->move_to (x_coordinate (x), y_coordinate (y), x, y);

	}

	fade[Out].points.push_back (end);
	fade[Out].points.sort (cmp);

	for (list<Point*>::iterator i = fade[Out].points.begin(); i != fade[Out].points.end(); ++i) {
		(*i)->move_to (x_coordinate ((*i)->x),
			       y_coordinate ((*i)->y),
			       (*i)->x, (*i)->y);
	}
	
	WhichFade old_current = current;
	current = In;
	redraw ();
	current = Out;
	redraw ();
	current = old_current;

	double spu = xfade.length() / (double) _canvas->allocation.width;

	if (fade[In].waves.empty()) {
		make_waves (xfade.left(), In);
	}

	if (fade[Out].waves.empty()) {
		make_waves (xfade.right(), Out);
	}

	double ht;
	vector<GtkCanvasItem*>::iterator i;
	unsigned int n;

	ht = _canvas->allocation.height / xfade.right().n_channels();

	for (n = 0, i = fade[In].waves.begin(); i != fade[In].waves.end(); ++i, ++n) {
		double yoff;

		yoff = n * ht;

		gtk_canvas_item_set ((*i),
				     "y", yoff,
				     "height", ht,
				     "samples_per_unit", spu,
				     NULL);
	}

	ht = _canvas->allocation.height / xfade.left().n_channels();

	for (n = 0, i = fade[Out].waves.begin(); i != fade[Out].waves.end(); ++i, ++n) {
		double yoff;

		yoff = n * ht;

		gtk_canvas_item_set ((*i),
				     "y", yoff,
				     "height", ht,
				     "samples_per_unit", spu,
				     NULL);
	}

}

gint
CrossfadeEditor::_point_event (GtkCanvasItem* item, GdkEvent* event, gpointer data)
{
	CrossfadeEditor* ed = static_cast<CrossfadeEditor*> (data);
	return ed->point_event (item, event);
}

gint
CrossfadeEditor::point_event (GtkCanvasItem* item, GdkEvent* event)
{
	Point* point = static_cast<Point*> (gtk_object_get_data (GTK_OBJECT (item), "point"));

	if (point->curve != fade[current].line) {
		return FALSE;
	}

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		point_grabbed = true;
		break;
	case GDK_BUTTON_RELEASE:
		point_grabbed = false;

		if (Keyboard::is_delete_event (&event->button)) {
			fade[current].points.remove (point);
			delete point;
		}

		redraw ();
		break;

	case GDK_MOTION_NOTIFY:
		if (point_grabbed) {
			double new_x, new_y;

			/* can't drag first or last points horizontally */

			if (point == fade[current].points.front() || point == fade[current].points.back()) {
				new_x = point->x;
			} else {
				new_x = (event->motion.x - canvas_border)/effective_width();
			}

			new_y = 1.0 - ((event->motion.y - canvas_border)/effective_height());
			point->move_to (x_coordinate (new_x), y_coordinate (new_y), 
					new_x, new_y);
			redraw ();
		}
		break;
	default:
		break;
	}
	return TRUE;
}

gint
CrossfadeEditor::_curve_event (GtkCanvasItem* item, GdkEvent* event, gpointer data)
{
	CrossfadeEditor* ed = static_cast<CrossfadeEditor*> (data);
	return ed->curve_event (item, event);
}

gint
CrossfadeEditor::curve_event (GtkCanvasItem* item, GdkEvent* event)
{
	/* treat it like a toplevel event */

	return canvas_event (item, event);
}

void
CrossfadeEditor::redraw ()
{
	if (_canvas->allocation.width < 2) {
		return;
	}

	jack_nframes_t len = xfade.length ();

	fade[current].normative_curve.clear ();
	fade[current].gain_curve.clear ();

	for (list<Point*>::iterator i = fade[current].points.begin(); i != fade[current].points.end(); ++i) {
		fade[current].normative_curve.add ((*i)->x, (*i)->y);
		fade[current].gain_curve.add (((*i)->x * len), (*i)->y);
	}

	size_t npoints = (size_t) effective_width();
	float vec[npoints];

	fade[current].normative_curve.get_vector (0, 1.0, vec, npoints);
	
	GtkCanvasPoints* pts = gtk_canvas_points_new (npoints);
	
	for (size_t i = 0; i < npoints; ++i) {

		double y = vec[i];

		pts->coords[i*2] = canvas_border + i;
		pts->coords[(i*2)+1] = y_coordinate (y);
	}

	gtk_canvas_item_set (fade[current].line, "points", pts, NULL);
	gtk_canvas_points_unref (pts);

	for (vector<GtkCanvasItem*>::iterator i = fade[current].waves.begin(); i != fade[current].waves.end(); ++i) {
		gtk_canvas_item_set ((*i), "gain_src", &fade[current].gain_curve, NULL);
	}
}

void
CrossfadeEditor::apply_preset (Preset *preset)
{
	for (list<Point*>::iterator i = fade[current].points.begin(); i != fade[current].points.end(); ++i) {
		delete *i;
	}

	fade[current].points.clear ();

	for (Preset::iterator i = preset->begin(); i != preset->end(); ++i) {
		Point* p = make_point ();
		p->move_to (x_coordinate ((*i).x), y_coordinate ((*i).y),
			    (*i).x, (*i).y);
		fade[current].points.push_back (p);
	}

	redraw ();
}

void
CrossfadeEditor::apply ()
{
	_apply_to (&xfade);
}

void
CrossfadeEditor::_apply_to (Crossfade* xf)
{
	ARDOUR::Curve& in (xf->fade_in());
	ARDOUR::Curve& out (xf->fade_out());

	/* IN */


	ARDOUR::Curve::const_iterator the_end = in.const_end();
	--the_end;

	double firstx = (*in.begin())->when;
	double endx = (*the_end)->when;
	double miny = in.get_min_y ();
	double maxy = in.get_max_y ();

	in.freeze ();
	in.clear ();

	for (list<Point*>::iterator i = fade[In].points.begin(); i != fade[In].points.end(); ++i) {

		double when = firstx + ((*i)->x * (endx - firstx));
		double value = (*i)->y; // miny + ((*i)->y * (maxy - miny));
		in.add (when, value);
	}

	/* OUT */

	the_end = out.const_end();
	--the_end;

	firstx = (*out.begin())->when;
	endx = (*the_end)->when;
	miny = out.get_min_y ();
	maxy = out.get_max_y ();

	out.freeze ();
	out.clear ();

	for (list<Point*>::iterator i = fade[Out].points.begin(); i != fade[Out].points.end(); ++i) {

		double when = firstx + ((*i)->x * (endx - firstx));
		double value = (*i)->y; // miny + ((*i)->y * (maxy - miny));
		out.add (when, value);
	}
	
	in.thaw ();
	out.thaw ();
}

void
CrossfadeEditor::setup (Crossfade* xfade)
{
	_apply_to (xfade);
	xfade->set_active (true);
	xfade->fade_in().solve ();
	xfade->fade_out().solve ();
}

void
CrossfadeEditor::clear ()
{
	for (list<Point*>::iterator i = fade[current].points.begin(); i != fade[current].points.end(); ++i) {
		delete *i;
	}

	fade[current].points.clear ();

	redraw ();
}

void
CrossfadeEditor::reset ()
{
	set (xfade.fade_in(),  In);
	set (xfade.fade_out(), Out);
}

void
CrossfadeEditor::build_presets ()
{
	presets = new Presets;

	/* S-fade out */

	s_out_preset = new Preset;

	s_out_preset->push_back (PresetPoint (0.0, 1.0));
	s_out_preset->push_back (PresetPoint (0.2, 0.99));
	s_out_preset->push_back (PresetPoint (0.35, 0.97));
	s_out_preset->push_back (PresetPoint (0.5, 0.8));
	s_out_preset->push_back (PresetPoint (0.6, 0.5));
	s_out_preset->push_back (PresetPoint (0.8, 0.03));
	s_out_preset->push_back (PresetPoint (0.9, 0.01));
	s_out_preset->push_back (PresetPoint (1.0, 0.0));

	presets->push_back (s_out_preset);

	/* S-fade in */

	s_in_preset = new Preset;

	s_in_preset->push_back (PresetPoint (0.0, 0.0));
	s_in_preset->push_back (PresetPoint (0.2, 0.03));
	s_in_preset->push_back (PresetPoint (0.4, 0.3));
	s_in_preset->push_back (PresetPoint (0.5, 0.5));
	s_in_preset->push_back (PresetPoint (0.65, 0.8));
	s_in_preset->push_back (PresetPoint (0.8, 0.97));
	s_in_preset->push_back (PresetPoint (0.9, 0.99));
	s_in_preset->push_back (PresetPoint (1.0, 1.0));

	presets->push_back (s_in_preset);
	
	/* midpoint jump in */

	midpoint_in_preset = new Preset;

	midpoint_in_preset->push_back (PresetPoint (0.0, 0.0));
	midpoint_in_preset->push_back (PresetPoint (0.48, 0.0));
	midpoint_in_preset->push_back (PresetPoint (0.52, 0.98));
	midpoint_in_preset->push_back (PresetPoint (1.0, 1.0));

	presets->push_back (midpoint_in_preset);

	/* midpoint jump out */

	midpoint_out_preset = new Preset;

	midpoint_out_preset->push_back (PresetPoint (0.0, 1.0));
	midpoint_out_preset->push_back (PresetPoint (0.48, 0.98));
	midpoint_out_preset->push_back (PresetPoint (0.52, 0.0));
	midpoint_out_preset->push_back (PresetPoint (1.0, 0.0));

	presets->push_back (midpoint_out_preset);
}

void
CrossfadeEditor::curve_select_clicked (WhichFade wf)
{
	current = wf;

	if (wf == In) {

		gtk_canvas_item_set (fade[In].line, "fill_color_rgba", RGBA_TO_UINT (0, 220, 220, 255), NULL);
		gtk_canvas_item_set (fade[Out].line, "fill_color_rgba", RGBA_TO_UINT (0, 240, 240, 60), NULL);

		for (list<Point*>::iterator i = fade[Out].points.begin(); i != fade[Out].points.end(); ++i) {
			gtk_canvas_item_hide ((*i)->box);
		}

		for (list<Point*>::iterator i = fade[In].points.begin(); i != fade[In].points.end(); ++i) {
			gtk_canvas_item_show ((*i)->box);
		}

	} else {

		gtk_canvas_item_set (fade[Out].line, "fill_color_rgba", RGBA_TO_UINT (0, 220, 220, 255), NULL);
		gtk_canvas_item_set (fade[In].line, "fill_color_rgba", RGBA_TO_UINT (0, 240, 240, 60), NULL);

		for (list<Point*>::iterator i = fade[In].points.begin(); i != fade[In].points.end(); ++i) {
			gtk_canvas_item_hide ((*i)->box);
		}
		
		for (list<Point*>::iterator i = fade[Out].points.begin(); i != fade[Out].points.end(); ++i) {
			gtk_canvas_item_show ((*i)->box);
		}

	}
}

double 
CrossfadeEditor::x_coordinate (double& xfract) const
{
	xfract = min (1.0, xfract);
	xfract = max (0.0, xfract);
    
	return canvas_border + (xfract * effective_width());
}

double
CrossfadeEditor::y_coordinate (double& yfract) const
{
	yfract = min (1.0, yfract);
	yfract = max (0.0, yfract);

	return (_canvas->allocation.height - (canvas_border)) - (yfract * effective_height());
}

void
CrossfadeEditor::make_waves (AudioRegion& region, WhichFade which)
{
	gdouble ht;
	unsigned int nchans = region.n_channels();
	guint32 color;

	if (which == In) {
		color = RGBA_TO_UINT (252,236,20,85);
	} else {
		color = RGBA_TO_UINT (255,255,255,60);
	}

	ht = _canvas->allocation.height / (double) nchans;

	for (unsigned int n = 0; n < nchans; ++n) {
		
		gdouble yoff = n * ht;
		
		if (region.source(n).peaks_ready (bind (slot (*this, &CrossfadeEditor::peaks_ready), &region, which))) {
			
			GtkCanvasItem *wave = gtk_canvas_item_new (gtk_canvas_root (GTK_CANVAS(_canvas)),
								   gtk_canvas_waveview_get_type (),
								   "data_src", (gpointer) &region,
								   "cache_updater", (gboolean) TRUE,
								   "cache", gtk_canvas_waveview_cache_new (),
								   "channel", (guint32) n,
								   "length_function", (gpointer) region_length_from_c,
								   "peak_function", (gpointer) region_read_peaks_from_c,
								   "gain_function", (gpointer) curve_get_vector_from_c,
								   "gain_src", &fade[which].gain_curve,
								   "y", yoff,
								   "height", ht,
								   "samples_per_unit", 2048.0,
								   "amplitude_above_axis", 2.0,
								   "wave_color", color,
								   NULL);
			
			gtk_canvas_item_lower_to_bottom (wave);
			fade[which].waves.push_back (wave);
		}
	}

	gtk_canvas_item_lower_to_bottom (toplevel);
}

void
CrossfadeEditor::peaks_ready (AudioRegion* r, WhichFade which)
{
	/* this should never be called, because the peak files for an xfade
	   will be ready by the time we want them. but our API forces us
	   to provide this, so ..
	*/

	make_waves (*r, which);
}

void
CrossfadeEditor::audition ()
{
	AudioPlaylist& pl (session.the_auditioner().prepare_playlist());

	AudioRegion* left = new AudioRegion (xfade.left(), xfade.left().length() - xfade.length(), xfade.length(), "xfade left", 
					     0, Region::DefaultFlags, false);
	AudioRegion* right = new AudioRegion (xfade.right(), 0, xfade.length(), "xfade right", 
					      0, Region::DefaultFlags, false);

	pl.add_region (*left, 0);
	pl.add_region (*right, 1);

	/* there is only one ... */

	pl.foreach_crossfade (this, &CrossfadeEditor::setup);

	session.audition_playlist ();
}
	
void
CrossfadeEditor::cancel_audition ()
{
	session.cancel_audition ();
}

void
CrossfadeEditor::audition_toggled ()
{
	bool x;

	if ((x = audition_button.get_active ()) != session.is_auditioning()) {

		if (x) {
			audition ();
		} else {
			cancel_audition ();
		}
	}
}
