/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: plugin_manager.cc,v 1.22 2004/01/15 01:56:10 pbd Exp $
*/

#include <gtk--/table.h>
#include <gtk--/button.h>
#include <ardour/ladspa_plugin.h>

#include <ardour_ui.h>
#include "plugin_manager.h"

#include "i18n.h"

using namespace ARDOUR;

static const gchar *i_titles[] = {
	N_("Available LADSPA plugins"), 
	N_("Type"),
	N_("# Inputs"), 
	N_("# Outputs"),
	0
};

static const gchar *o_titles[] = {
	N_("To be added"),
	0
};

PluginManager::PluginManager (LADSPA::Manager *ladspap)
	: i_selector (_input_refiller, this, internationalize (i_titles), false, true),
	  o_selector (_output_refiller, this, internationalize (o_titles), false, true)
{
	set_position (GTK_WIN_POS_MOUSE);
	set_name ("PluginManagerWindow");
	set_title (_("ardour: plugins"));
	set_modal(true);
	add_events (GDK_KEY_PRESS_MASK|GDK_KEY_RELEASE_MASK);

	ladspa = ladspap;
	session = 0;

	Gtk::Button *btn_add = manage(new Gtk::Button(_("Add")));
	ARDOUR_UI::instance()->tooltips().set_tip(*btn_add, _("Add a plugin to the effect list"));
	Gtk::Button *btn_remove = manage(new Gtk::Button(_("Remove")));
	ARDOUR_UI::instance()->tooltips().set_tip(*btn_remove, _("Remove a plugin from the effect list"));
	Gtk::Button *btn_ok = manage(new Gtk::Button(_("OK")));
	Gtk::Button *btn_cancel = manage(new Gtk::Button(_("Cancel")));

	Gtk::Button *btn_update = manage(new Gtk::Button(_("Update")));
	ARDOUR_UI::instance()->tooltips().set_tip(*btn_update, _("Update available plugins"));

	btn_ok->set_name("PluginManagerButton");
	btn_cancel->set_name("PluginManagerButton");
	btn_add->set_name("PluginManagerButton");
	btn_remove->set_name("PluginManagerButton");
	
	Gtk::Table *table = manage(new Gtk::Table(7, 10));
	table->set_usize(530, 400);
	table->attach(i_selector, 0, 7, 0, 5);

	table->attach(*btn_add, 1, 2, 5, 6, GTK_FILL, 0, 5, 5);
	table->attach(*btn_remove, 3, 4, 5, 6, GTK_FILL, 0, 5, 5);
	table->attach(*btn_update, 5, 6, 5, 6, GTK_FILL, 0, 5, 5);

	table->attach(o_selector, 0, 7, 7, 9);
	table->attach(*btn_ok, 1, 3, 9, 10, GTK_FILL, 0, 5, 5);
	table->attach(*btn_cancel, 3, 4, 9, 10, GTK_FILL, 0, 5, 5);
	add (*table);

	table->set_name("PluginManagerTable");
	i_selector.set_name("PluginManagerList");
	o_selector.set_name("PluginManagerList");
	
	i_selector.clist().column(0).set_auto_resize (true);
	i_selector.clist().column(1).set_auto_resize (true);
	o_selector.clist().column(0).set_auto_resize (true);

	
	i_selector.selection_made.connect(slot(*this, &PluginManager::i_plugin_selected));
	i_selector.choice_made.connect(slot(*this, &PluginManager::i_plugin_chosen));
	o_selector.selection_made.connect(slot(*this, &PluginManager::o_plugin_selected));
	o_selector.choice_made.connect(slot(*this, &PluginManager::o_plugin_chosen));
	btn_update->clicked.connect (slot(*this, &PluginManager::btn_update_clicked));
	btn_add->clicked.connect(slot(*this, &PluginManager::btn_add_clicked));
	btn_remove->clicked.connect(slot(*this, &PluginManager::btn_remove_clicked));
	btn_ok->clicked.connect(slot(*this, &PluginManager::btn_ok_clicked));
	btn_cancel->clicked.connect(slot(*this, &PluginManager::btn_cancel_clicked));
	delete_event.connect (slot (*this, &PluginManager::wm_close));
}

void
PluginManager::set_session (Session* s)
{
	session = s;

	if (session) {
		session->going_away.connect (bind (slot (*this, &PluginManager::set_session), static_cast<Session*> (0)));
	}
}

void
PluginManager::_input_refiller (Gtk::CList &list, void *arg)

{
	((PluginManager *) arg)->input_refiller (list);
}

void
PluginManager::_output_refiller (Gtk::CList &list, void *arg)
{
	((PluginManager *) arg)->output_refiller (list);
}

int compare(const void *left, const void *right)
{
  return strcmp(*((char**)left), *((char**)right));
}

void
PluginManager::input_refiller (Gtk::CList &clist)

{
	const gchar *rowdata[4];
	guint row;
	list<LADSPA::Info *> &plugs = ladspa->plugin_info ();
	list<LADSPA::Info *>::iterator i;
	char ibuf[16], obuf[16];
	
	// Insert into GTK list
	for (row = 0, i=plugs.begin(); i != plugs.end(); ++i, ++row) {
		rowdata[0] = (*i)->name.c_str();

		rowdata[1] = (*i)->uri.c_str();

		snprintf (ibuf, sizeof(ibuf)-1, "%d", (*i)->n_inputs);
		snprintf (obuf, sizeof(obuf)-1, "%d", (*i)->n_outputs);		
		rowdata[2] = ibuf;
		rowdata[3] = obuf;
		
		clist.insert_row (row, rowdata);
	}

 	clist.set_sort_column (0);
 	clist.sort ();
}

void
PluginManager::output_refiller (Gtk::CList &clist)
{
	const gchar *rowdata[2];
	guint row;
	list<string>::iterator i;
	
	// Insert into GTK list
	for (row=0,i=added_plugins.begin(); i != added_plugins.end(); ++i, ++row){
		rowdata[0] = (*i).c_str();
		clist.insert_row (row, rowdata);
	}
}

void
PluginManager::i_plugin_chosen (Gtkmmext::Selector *selector,
			      Gtkmmext::SelectionResult *res)
{
	if (res) {
		// get text for name column (0)
		i_selected_plug = i_selector.clist().row(res->row)[0].get_text();
		//i_selected_plug = *res->text;
	} else {
		i_selected_plug = "";
	}
}

void
PluginManager::i_plugin_selected (Gtkmmext::Selector *selector,
				Gtkmmext::SelectionResult *res)

{
	if (res) {
		added_plugins.push_back(i_selector.clist().row(res->row)[0].get_text());
		//added_plugins.push_back(*(res->text));
		o_selector.rescan();
	}
}

void
PluginManager::o_plugin_chosen (Gtkmmext::Selector *selector,
			      Gtkmmext::SelectionResult *res)
{
	if (res && res->text) {
		o_selected_plug = res->row;
	} else {
		o_selected_plug = -1;
	}

}

void
PluginManager::o_plugin_selected (Gtkmmext::Selector *selector,
				Gtkmmext::SelectionResult *res)
{
	if(res && res->text){
		gint row = 0;
		list<string>::iterator i = added_plugins.begin();
		while(row < res->row){
			i++;
			row++;
		}
		added_plugins.erase(i);
		o_selector.rescan();
		o_selected_plug = -1;
	}
}

void
PluginManager::use_plugin (string name)
{
	list<LADSPA::Info *>::iterator i;
	list<LADSPA::Info *> &plugs = ladspa->plugin_info();

	if (name.length() == 0 || session == 0) {
		return;
	}

	for (i = plugs.begin(); i != plugs.end(); ++i) {
		if ((*i)->name == name) {
			break;
		}
	}

	if (i == plugs.end()) {
		return;
	}

	LADSPA::Plugin *plugin = ladspa->load (*session, *i);
	PluginCreated (plugin);
}

void
PluginManager::btn_add_clicked()
{
	if (i_selected_plug.length()) {
		added_plugins.push_back(i_selected_plug);
		o_selector.rescan();
	}
}

void
PluginManager::btn_remove_clicked()
{
	if (o_selected_plug > -1){
		gint row = 0;
		list<string>::iterator i = added_plugins.begin();
		while(row < o_selected_plug){
			i++;
			row++;
		}
		added_plugins.erase(i);
		o_selector.rescan();
		o_selected_plug = -1;
	}
}

// Adds a plugin, and closes the window.
void 
PluginManager::btn_ok_clicked()
{
	using namespace Gtk::CList_Helpers;

	list<string>::iterator i;

	for (i = added_plugins.begin(); i != added_plugins.end(); ++i){
		use_plugin ((*i).c_str());
	}

	hide();
	added_plugins.clear();
	o_selector.rescan();
	i_selected_plug = "";
	o_selected_plug = -1;

	SelectionList s_list = i_selector.clist().selection();
	SelectionList::iterator s = s_list.begin();
	if (s != s_list.end()) {
		(*s).unselect();
	}
}

void
PluginManager::btn_cancel_clicked()
{
	hide();
	added_plugins.clear();
	o_selector.rescan();
	i_selected_plug = "";
	o_selected_plug = -1;
}

void
PluginManager::btn_update_clicked()
{
	ladspa->refresh ();
	i_selector.rescan ();
}

gint
PluginManager::wm_close(GdkEventAny* ev)
{
	btn_cancel_clicked();
	return TRUE;
}
