/*
    Copyright (C) 2002-2003 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: redirect_automation_line.cc,v 1.24 2004/02/22 18:18:41 pauld Exp $
*/

#include <ardour/curve.h>

#include "public_editor.h"
#include "redirect_automation_line.h"
#include "audio_time_axis.h"
#include "utils.h"

#include <ardour/session.h>
#include <ardour/ladspa_plugin.h>
#include <ardour/insert.h>

#include "i18n.h"

using namespace ARDOUR;

RedirectAutomationLine::RedirectAutomationLine (string name, Redirect& rd, unsigned long port, Session& s,
						TimeAxisView& tv, GtkCanvasItem* parent,
						AutomationList& l,
						gint (*point_callback)(GtkCanvasItem*, GdkEvent*, gpointer),
						gint (*line_callback)(GtkCanvasItem*, GdkEvent*, gpointer))

	: AutomationLine (name, tv, parent, l, point_callback, line_callback),
	  session (s),
	  _redirect (rd),
	  _port (port)
{
	set_verbose_cursor_uses_gain_mapping (false);

	PluginInsert *pi;

	if ((pi  = dynamic_cast<PluginInsert*>(&_redirect)) == 0) {
		fatal << _("redirect automation created for non-plugin") << endmsg;
		/*NOTREACHED*/
	}
	
	LADSPA_PortRangeHint prh;

	prh  = pi->plugin().port_range_hints()[_port];
	
	if (LADSPA_IS_HINT_TOGGLED(prh.HintDescriptor)) {
		no_draw = true;
		return;
	}

	no_draw = false;

	if (LADSPA_IS_HINT_BOUNDED_BELOW(prh.HintDescriptor)) {
		if (LADSPA_IS_HINT_SAMPLE_RATE(prh.HintDescriptor)) {
			lower = prh.LowerBound * session.frame_rate();
		} else {
			lower = prh.LowerBound;
		}
	} else {
		lower = 0;
	}

	if (LADSPA_IS_HINT_BOUNDED_ABOVE(prh.HintDescriptor)) {
		if (LADSPA_IS_HINT_SAMPLE_RATE(prh.HintDescriptor)) {
			upper = prh.UpperBound * session.frame_rate();
		} else {
			upper = prh.UpperBound;
		}
	} else {
		upper = 4.0; /* see plugin_ui.cc */
	}

	/* XXX logarithmic ... ??? */

	range = upper - lower;

	/* XXX set min/max for underlying curve ??? */
}

string
RedirectAutomationLine::get_verbose_cursor_string (float fraction)
{
	char buf[32];

	snprintf (buf, sizeof (buf), "%.2f", lower + (fraction * range));
	return buf;
}

void
RedirectAutomationLine::view_to_model_y (double& y)
{
	y = lower + (y * range);
}

void
RedirectAutomationLine::model_to_view_y (double& y)
{
	y = (y - lower) / range;
}

