/*
    Copyright (C) 2001 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: regionview.cc,v 1.98 2004/02/29 23:33:55 pauld Exp $
*/

#include <cmath>

#include <gtk--.h>

#include <gtkmmext/gtk_ui.h>

#include <ardour/playlist.h>
#include <ardour/audioregion.h>
#include <ardour/sndfilesource.h>
#include <ardour/diskstream.h>

#include "streamview.h"
#include "regionview.h"
#include "audio_time_axis.h"
#include "canvas-simplerect.h"
#include "canvas-simpleline.h"
#include "canvas-waveview.h"
#include "public_editor.h"
#include "region_editor.h"
#include "region_gain_line.h"
#include "ghostregion.h"
#include "audio_time_axis.h"
#include "utils.h"
#include "rgb_macros.h"

using namespace SigC;
using namespace ARDOUR;

SigC::Signal1<void,AudioRegionView*> AudioRegionView::AudioRegionViewGoingAway;

AudioRegionView::AudioRegionView (GtkCanvasGroup *parent, AudioTimeAxisView &tv, 
				  AudioRegion& r, 
				  double spu, 
				  double amplitude_above_axis,
				  GdkColor& basic_color)

	: TimeAxisViewItem (r.name(), parent, tv, spu, basic_color, r.position(), r.length(),
			    TimeAxisViewItem::Visibility (TimeAxisViewItem::ShowNameText|
				                          TimeAxisViewItem::ShowNameHighlight|
							  TimeAxisViewItem::ShowFrame)),
	  region (r)
{
	GtkCanvasPoints *shape;
	XMLNode *node;

	editor = 0;
	valid = true;
	current_visible_sync_position = 0;
	_amplitude_above_axis = amplitude_above_axis;
	zero_line = 0;

	_flags = 0;

	if ((node = region.extra_xml ("GUI")) != 0) {
		set_flags (node);
	} else {
		_flags = WaveformVisible;
		store_flags ();
	}

	compute_colors (basic_color);

	create_wave ();

	gtk_object_set_data (GTK_OBJECT(name_highlight), "regionview", this);
	gtk_object_set_data (GTK_OBJECT(name_text), "regionview", this);

	gain_line = new AudioRegionGainLine ("gain", tv.session(), *this, group, region.envelope(),
					     PublicEditor::canvas_control_point_event,
					     PublicEditor::canvas_line_event);

	if (!(_flags & EnvelopeVisible)) {
		gain_line->hide ();
	} else {
		gain_line->show ();
	}

	shape = gtk_canvas_points_new (4);

	/* a 6x6 equilateral triangle */

	shape->coords[0] = 0;
	shape->coords[1] = 1;

	shape->coords[2] = 5;
	shape->coords[3] = 1;

	shape->coords[4] = 2;
	shape->coords[5] = 6;

	shape->coords[6] = 0;
	shape->coords[7] = 1;

	sync_mark = gtk_canvas_item_new (GTK_CANVAS_GROUP(group),
					 gtk_canvas_polygon_get_type(),
					 "points", shape,
					 "fill_color_rgba", fill_color,
					 NULL);
	gtk_canvas_item_hide (sync_mark);

	gtk_canvas_points_unref (shape);

	reset_width_dependent_items ((double) region.length() / samples_per_unit);

	gain_line->reset ();

	region_muted ();
	region_sync_changed ();
	region_resized (BoundsChanged);
	region_locked ();

	region.StateChanged.connect (slot (*this, &AudioRegionView::region_changed));

	gtk_signal_connect (GTK_OBJECT(group), "event",
			    (GtkSignalFunc) PublicEditor::canvas_region_view_event,
			    this);
	gtk_signal_connect (GTK_OBJECT(name_highlight), "event",
			    (GtkSignalFunc) PublicEditor::canvas_region_view_name_highlight_event,
			    this);

	gtk_signal_connect (GTK_OBJECT(name_text), "event",
			    (GtkSignalFunc) PublicEditor::canvas_region_view_name_event,
			    this);

	gtk_signal_connect (GTK_OBJECT(lock_mark), "event",
			    (GtkSignalFunc) _lock_toggle,
			    this);

	/* XXX sync mark drag? */

	set_height (trackview.height - 2) ;

}

AudioRegionView::~AudioRegionView ()
{
	 AudioRegionViewGoingAway (this); /* EMIT_SIGNAL */

	for (vector<GtkCanvasItem *>::iterator wave = waves.begin(); wave != waves.end() ; ++wave) {
		gtk_object_destroy (GTK_OBJECT(*wave));
	}

	for (vector<GhostRegion*>::iterator g = ghosts.begin(); g != ghosts.end(); ++g) {
		delete *g;
	}
	
	if (editor) {
		delete editor;
	}

	delete gain_line;
}

gint
AudioRegionView::_lock_toggle (GtkCanvasItem* item, GdkEvent* ev, void* arg)
{
	switch (ev->type) {
	case GDK_BUTTON_RELEASE:
		static_cast<AudioRegionView*>(arg)->lock_toggle ();
		break;
	default:
		break;
	} 
	return TRUE;
}

void
AudioRegionView::lock_toggle ()
{
	region.set_locked (!region.locked());
}

void
AudioRegionView::region_changed (Change what_changed)
{
	if (what_changed & BoundsChanged) {
		region_resized (what_changed);
	}
	if (what_changed & Region::MuteChanged) {
		region_muted ();
	}
	if (what_changed & Region::OpacityChanged) {
		region_opacity ();
	}
	if (what_changed & ARDOUR::NameChanged) {
		region_renamed ();
	}
	if (what_changed & Region::SyncOffsetChanged) {
		region_sync_changed ();
	}
	if (what_changed & Region::LayerChanged) {
		region_layered ();
	}
	if (what_changed & Region::LockChanged) {
		region_locked ();
	}
}

void
AudioRegionView::region_locked ()
{
	show_locked (region.locked());
}

void
AudioRegionView::region_resized (Change what_changed)
{
	double unit_length;

	if (what_changed & ARDOUR::PositionChanged) {
		set_position (region.position(), 0);
	}


	if (what_changed & Change (StartChanged|LengthChanged)) {

		set_duration (region.length(), 0);

		unit_length = region.length() / samples_per_unit;
		
		reset_width_dependent_items (unit_length);
		
		for (unsigned int n = 0; n < waves.size(); ++n) {
			// TODO: something else to let it know the channel
			gtk_canvas_item_set (waves[n], "data_src", &region, NULL);
		}
		
		for (vector<GhostRegion*>::iterator i = ghosts.begin(); i != ghosts.end(); ++i) {

			(*i)->set_duration (unit_length);

			for (vector<GtkCanvasItem*>::iterator w = (*i)->waves.begin(); w != (*i)->waves.end(); ++w) {
				gtk_canvas_item_set ((*w), "data_src", &region, NULL);
			}
		}
	}
}

void
AudioRegionView::reset_width_dependent_items (double pixel_width)
{
	TimeAxisViewItem::reset_width_dependent_items (pixel_width);
	_pixel_width = pixel_width;

	if (pixel_width >= 1.0) {
		if (zero_line) {
			gtk_canvas_item_set (zero_line, "x2", pixel_width, NULL);
		}
	}
}

void
AudioRegionView::region_layered ()
{
	AudioTimeAxisView *atv = dynamic_cast<AudioTimeAxisView*> (&get_time_axis_view());
	atv->view->region_layered (this);
}
	
void
AudioRegionView::region_muted ()
{
	set_frame_color ();

	for (unsigned int n=0; n < waves.size(); ++n) {
		if (region.muted()) {
			gtk_canvas_item_set (waves[n], "wave_color", muted_wave_color, NULL);
		}
		else {
			gtk_canvas_item_set (waves[n], "wave_color", wave_color, NULL);
		}
	}
}

void
AudioRegionView::region_opacity ()
{
	set_frame_color ();
}

void
AudioRegionView::raise ()
{
	region.raise ();
}

void
AudioRegionView::raise_to_top ()
{
	region.raise_to_top ();
}

void
AudioRegionView::lower ()
{
	region.lower ();
}

void
AudioRegionView::lower_to_bottom ()
{
	region.lower_to_bottom ();
}

bool
AudioRegionView::set_position (jack_nframes_t pos, void* src, double* ignored)
{
	double delta;
	bool ret;

	if (!(ret = TimeAxisViewItem::set_position (pos, this, &delta))) {
		return false;
	}

	if (ignored) {
		*ignored = delta;
	}

	if (delta) {
		for (vector<GhostRegion*>::iterator i = ghosts.begin(); i != ghosts.end(); ++i) {
			gtk_canvas_item_move ((*i)->group, delta, 0.0);
		}
	}

	return ret;
}

void
AudioRegionView::set_height (gdouble height)
{
	unsigned int wcnt = waves.size();

	TimeAxisViewItem::set_height (height);

	_height = height;
	
	for (unsigned int n=0; n < wcnt; ++n) {
		gdouble ht;

		if ((height) < NAME_HIGHLIGHT_THRESH) {
			ht = ((height-2*wcnt) / (double) wcnt);
		} else {
			ht = (((height-2*wcnt) - NAME_HIGHLIGHT_SIZE) / (double) wcnt);
		}
		
		gdouble yoff = n * (ht+1);
		
		gtk_canvas_item_set (waves[n], "height", ht, NULL);
		gtk_canvas_item_set (waves[n], "y", yoff + 2, NULL);
	}

	if ((height/wcnt) < NAME_HIGHLIGHT_SIZE) {
		gain_line->hide ();
	} else {
		if (_flags & EnvelopeVisible) {
			gain_line->show ();
		}
	}

#if 0
  use for fade grab handles
	if (height < NAME_HIGHLIGHT_THRESH && _pixel_width >= GRAB_HANDLE_LENGTH) {
		gtk_canvas_item_raise_to_top (frame_handle_start);
		gtk_canvas_item_raise_to_top (frame_handle_end);
	}
	else {
		gtk_canvas_item_hide (frame_handle_start);
		gtk_canvas_item_hide (frame_handle_end);
	}
#endif
	
	if (height >= 100) {
		gdouble wave_midpoint = (height - NAME_HIGHLIGHT_SIZE) / 2.0;
		gtk_canvas_item_set (zero_line, "y1", wave_midpoint, "y2", wave_midpoint, NULL);
		gtk_canvas_item_show (zero_line);
	} else {
		gtk_canvas_item_hide (zero_line);
	}

	gain_line->set_height ((unsigned int) rint (height));

	gtk_canvas_item_raise_to_top(name_text) ;

}

void
AudioRegionView::set_samples_per_unit (gdouble spu)
{
	TimeAxisViewItem::set_samples_per_unit (spu);

	for (unsigned int n=0; n < waves.size(); ++n) {
		gtk_canvas_item_set (waves[n], "samples_per_unit", spu, NULL);
	}

	for (vector<GhostRegion*>::iterator i = ghosts.begin(); i != ghosts.end(); ++i) {
		(*i)->set_samples_per_unit (spu);
	}

	gain_line->reset ();
	region_sync_changed ();
}

void
AudioRegionView::set_amplitude_above_axis (gdouble spp)
{
	for (unsigned int n=0; n < waves.size(); ++n) {
		gtk_canvas_item_set (waves[n], "amplitude_above_axis", spp, NULL);
	}
}

void
AudioRegionView::compute_colors (GdkColor& basic_color)
{
	TimeAxisViewItem::compute_colors (basic_color);
	wave_color = RGBA_TO_UINT(110,32,155,255);
	muted_wave_color = RGBA_TO_UINT(160,82,205,255);
	gain_color  = RGBA_TO_UINT(0,255,0,255);
}

void
AudioRegionView::set_colors ()
{

	TimeAxisViewItem::set_colors ();

	gain_line->set_line_color (gain_color);
	gtk_canvas_item_set (sync_mark, "fill_color_rgba", fill_color, NULL);
	for (unsigned int n=0; n < waves.size(); ++n) {
		if (region.muted()) {
			gtk_canvas_item_set (waves[n], "wave_color", muted_wave_color, NULL);
		} else {
			gtk_canvas_item_set (waves[n], "wave_color", wave_color, NULL);
		}
	}
}

void
AudioRegionView::set_frame_color ()
{
	if (region.opaque()) {
		fill_opacity = 220;
	} else {
		fill_opacity = 100;
	}

	TimeAxisViewItem::set_frame_color ();
}

void
AudioRegionView::show_region_editor ()
{
	if (editor == 0) {
		editor = new AudioRegionEditor (trackview.session(), region, *this);
		editor->realize ();
		trackview.editor.ensure_float (*editor);
	} 

	editor->show_all ();
	editor->get_window().raise();
}

void
AudioRegionView::hide_region_editor()
{
	if (editor) {
		editor->hide_all ();
	}
}

void
AudioRegionView::region_renamed ()
{
	set_item_name (region.name(), this);
}

void
AudioRegionView::region_sync_changed ()
{
	/* this has to handle both a genuine change of position and a change of samples_per_unit
	 */

	if (region.sync_offset() == 0) {
		gtk_canvas_item_hide (sync_mark);
		current_visible_sync_position = 0;
	} else {
		double new_position = region.sync_offset() / samples_per_unit;
		double delta = new_position - current_visible_sync_position;
		gtk_canvas_item_show (sync_mark);
		gtk_canvas_item_move (sync_mark, delta, 0);
		current_visible_sync_position = new_position;
	}
}

void
AudioRegionView::set_waveform_visible (bool yn)
{
	if (((_flags & WaveformVisible) != yn)) {
		if (yn) {
			for (unsigned int n=0; n < waves.size(); ++n) {
				gtk_canvas_item_show (waves[n]);
			}
			_flags |= WaveformVisible;
		} else {
			for (unsigned int n=0; n < waves.size(); ++n) {
				gtk_canvas_item_hide (waves[n]);
			}
			_flags &= ~WaveformVisible;
		}
		store_flags ();
	}
}

void
AudioRegionView::set_envelope_visible (bool yn)
{
	if ((_flags & EnvelopeVisible) != yn) {
		if (yn) {
			cerr << "gain line should be visible\n";
			gain_line->show ();
			_flags |= EnvelopeVisible;
		} else {
			cerr << "gain line should be hidden\n";
			gain_line->hide ();
			_flags &= ~EnvelopeVisible;
		}
		store_flags ();
	}
}

void
AudioRegionView::create_wave ()
{
	bool create_zero = false;
	AudioTimeAxisView& atv (*(dynamic_cast<AudioTimeAxisView*>(&trackview))); // ick

	if (waves.size() == 0 && atv.get_diskstream()) {

		unsigned int nchans = atv.get_diskstream()->n_channels();

		for (unsigned int n = 0; n < nchans; ++n) {

			if (n >= region.n_channels()) {
				break;
			}
			
			gdouble ht;
			if (trackview.height < NAME_HIGHLIGHT_SIZE) {
				ht = ((trackview.height) / (double) nchans);
			} else {
				ht = ((trackview.height - NAME_HIGHLIGHT_SIZE) / (double) nchans);
			}
			gdouble yoff = n * ht;

			wave_caches.push_back (gtk_canvas_waveview_cache_new ());

			if (region.source(n).peaks_ready (slot (*this, &AudioRegionView::peaks_ready_handler))) {
				
				create_zero = true;

				GtkCanvasItem *wave = gtk_canvas_item_new (GTK_CANVAS_GROUP(group),
									   gtk_canvas_waveview_get_type (),
									   "data_src", (gpointer) &region,
									   "cache_updater", (gboolean) TRUE,
									   "cache", wave_caches[n],
									   "channel", (guint32) n,
									   "length_function", (gpointer) region_length_from_c,
									   "peak_function", (gpointer) region_read_peaks_from_c,
									   "x", 0.0,
									   "y", yoff,
									   "height", (double) ht,
									   "samples_per_unit", samples_per_unit,
									   "amplitude_above_axis", _amplitude_above_axis,
									   "wave_color", (guint32) wave_color,
									   NULL);
				
				if (!(_flags & WaveformVisible)) {
					gtk_canvas_item_hide (wave);
				}

				waves.push_back(wave);
			}
		}
	}

	if (create_zero) {
		zero_line = gtk_canvas_item_new (GTK_CANVAS_GROUP(group),
						 gtk_canvas_simpleline_get_type(),
						 "x1", (gdouble) 0.0,
						 "x2", (gdouble) region.length() / samples_per_unit,
						 "color_rgba", (guint) RGBA_TO_UINT(181,181,181,255),
						 NULL);
		// cerr << region.name() << " zero line @ " << zero_line << endl;
	}
}

void
AudioRegionView::peaks_ready_handler ()
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &AudioRegionView::create_wave));
}

void
AudioRegionView::add_gain_point_event (GtkCanvasItem *item, GdkEvent *ev)
{
	double x, y;

	/* don't create points that can't be seen */

	set_envelope_visible (true);
	
	x = ev->button.x;
	y = ev->button.y;

	gtk_canvas_item_w2i (item, &x, &y);

	jack_nframes_t fx = trackview.editor.pixel_to_frame (x);

	if (fx > region.last_frame()) {
		return;
	}

	/* compute vertical fractional position */

	y = 1.0 - (y / trackview.height);
	
	/* map using gain line */

	gain_line->view_to_model_y (y);

	trackview.session().begin_reversible_command ("add gain control point");
	trackview.session().add_undo (region.playlist()->get_memento());
	region.envelope().add (fx, y);
	trackview.session().add_redo_no_execute (region.playlist()->get_memento());
	trackview.session().commit_reversible_command ();
}

void
AudioRegionView::remove_gain_point_event (GtkCanvasItem *item, GdkEvent *ev)
{
	ControlPoint *cp = reinterpret_cast<ControlPoint *> (gtk_object_get_data(GTK_OBJECT(item), "control_point"));
	region.envelope().erase (cp->model);
}

void
AudioRegionView::store_flags()
{
	XMLNode *node = new XMLNode ("GUI");

	node->add_property ("waveform-visible", (_flags & WaveformVisible) ? "yes" : "no");
	node->add_property ("envelope-visible", (_flags & EnvelopeVisible) ? "yes" : "no");

	region.add_extra_xml (*node);
}

void
AudioRegionView::set_flags (XMLNode* node)
{
	XMLProperty *prop;

	if ((prop = node->property ("waveform-visible")) != 0) {
		if (prop->value() == "yes") {
			_flags |= WaveformVisible;
		}
	}

	if ((prop = node->property ("envelope-visible")) != 0) {
		if (prop->value() == "yes") {
			_flags |= EnvelopeVisible;
		}
	}
}
	
void
AudioRegionView::set_waveform_shape (WaveformShape shape)
{
	bool yn;

	/* this slightly odd approach is to leave the door open to 
	   other "shapes" such as spectral displays, etc.
	*/

	switch (shape) {
	case Rectified:
		yn = true;
		break;

	default:
		yn = false;
		break;
	}

	if (yn != (bool) (_flags & WaveformRectified)) {
		for (vector<GtkCanvasItem *>::iterator wave = waves.begin(); wave != waves.end() ; ++wave) {
			gtk_canvas_item_set ((*wave), "rectified", (gboolean) yn, NULL);
		}

		if (yn) {
			gtk_canvas_item_hide (zero_line);
		} else {
			gtk_canvas_item_show (zero_line);
		}
		
		if (yn) {
			_flags |= WaveformRectified;
		} else {
			_flags &= ~WaveformRectified;
		}
	}
}

std::string
AudioRegionView::get_item_name ()
{
	return region.name();
}

void
AudioRegionView::move (double x_delta, double y_delta)
{
	if (region.locked()) {
		return;
	}

	gtk_canvas_item_move (get_canvas_group(), x_delta, y_delta);

	/* note: ghosts never leave their tracks so y_delta for them is always zero */

	for (vector<GhostRegion*>::iterator i = ghosts.begin(); i != ghosts.end(); ++i) {
		gtk_canvas_item_move ((*i)->group, x_delta, 0.0);
	}
}

GhostRegion*
AudioRegionView::add_ghost (AutomationTimeAxisView& atv)
{
	AudioTimeAxisView& myatv (*(dynamic_cast<AudioTimeAxisView*>(&trackview))); // ick
	double unit_position = region.position () / samples_per_unit;
	GhostRegion* ghost = new GhostRegion (atv, unit_position);
	unsigned int nchans;
	
	nchans = myatv.get_diskstream()->n_channels();

	for (unsigned int n = 0; n < nchans; ++n) {
		
		if (n >= region.n_channels()) {
			break;
		}
		
		GtkCanvasItem *wave = gtk_canvas_item_new (GTK_CANVAS_GROUP(ghost->group),
							   gtk_canvas_waveview_get_type (),
							   "data_src", (gpointer) &region,
							   "cache_updater", (gboolean) FALSE,
							   "cache", wave_caches[n],
							   "channel", (guint32) n,
							   "length_function", (gpointer) region_length_from_c,
							   "peak_function", (gpointer) region_read_peaks_from_c,
							   "x", 0.0,
							   "samples_per_unit", samples_per_unit,
							   "amplitude_above_axis", _amplitude_above_axis,
							   "wave_color", RGBA_TO_UINT (108,116,104,255),
							   NULL);

			
		ghost->waves.push_back(wave);
	}

	ghost->set_height ();
	ghost->set_duration (region.length() / samples_per_unit);
	ghosts.push_back (ghost);

	return ghost;
}
