/*
    Copyright (C) 2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: audioengine.h,v 1.27 2003/12/02 21:01:18 pbd Exp $
*/

#ifndef __ardour_audioengine_h__
#define __ardour_audioengine_h__

#include <set>

#include <sigc++/signal_system.h>
#include <pthread.h>
#include <ardour/ardour.h>
#include <jack/jack.h>

using std::set;

namespace ARDOUR {

class Session;
class Port;

class AudioEngine : public SigC::Object 
{
   public:
	AudioEngine (jack_client_t *jack);
	virtual ~AudioEngine ();
	
	int stop ();
	int start ();
	bool running() const { return _running; }

	PBD::NonBlockingLock& process_lock() { return _process_lock; }

	jack_nframes_t frame_rate();
	jack_nframes_t frames_per_cycle();
	jack_nframes_t frames_since_cycle_start () {
		if (!_running) return 0;
		return jack_frames_since_cycle_start (_jack);
	}
	jack_nframes_t frame_time () {
		if (!_running) return 0;
		return jack_frame_time (_jack);
	}
	jack_nframes_t transport_frame () const {
		return jack_get_current_transport_frame (_jack);
	}
	
	jack_nframes_t set_monitor_check_interval (jack_nframes_t);

	float get_cpu_load() { 
		if (!_running) return 0;
		return jack_cpu_load (_jack);
	}

	void set_session (Session *);
	void remove_session ();

	Port *register_audio_input_port (const string& portname);
	Port *register_audio_output_port (const string& portname);
	int   unregister_port (Port *);
	
	int connect (const string& source, const string& destination);
	int disconnect (const string& source, const string& destination);
	int disconnect (Port *);
	
	const char ** get_ports (const string& port_name_pattern, const string& type_name_pattern, unsigned long flags);

	unsigned long n_physical_outputs () const;
	unsigned long n_physical_inputs () const;

	string get_nth_physical_output (unsigned long n) {
		return get_nth_physical (n, JackPortIsInput);
	}

	string get_nth_physical_input (unsigned long n) {
		return get_nth_physical (n, JackPortIsOutput);
	}

	jack_nframes_t get_port_total_latency (const Port&);

	/* the caller may not delete the object pointed to by 
	   the return value
	*/

	Port *get_port_by_name (const string& name, bool keep = true);

	void transport_start ();
	void transport_stop ();
	void transport_locate (jack_nframes_t);
	int  reset_timebase ();

	/* start/stop freewheeling */

	int freewheel (bool onoff);
	bool freewheeling() const { return _freewheeling; }

	/* this signal is sent for every process() cycle while freewheeling.
	   the regular process() call to session->process() is not made.
	*/

	SigC::Signal1<int,jack_nframes_t> Freewheel;

	SigC::Signal0<void> Xrun;

	/* this signal is if JACK notifies us of a graph order event */

	SigC::Signal0<void> GraphReordered;

	/* this signal is emitted if the sample rate changes */

	SigC::Signal1<void,jack_nframes_t> SampleRateChanged;

	/* this signal is sent if JACK ever disconnects us */

	SigC::Signal0<void> Halted;

	/* these two are emitted when the engine itself is
	   started and stopped
	*/

	SigC::Signal0<void> Running;
	SigC::Signal0<void> Stopped;

  private:
	ARDOUR::Session      *session;
	jack_client_t       *_jack;
	PBD::NonBlockingLock  port_lock;
	PBD::NonBlockingLock _process_lock;
	PBD::Lock             session_remove_lock;
	pthread_cond_t        session_removed;
	bool                  session_remove_pending;
	bool                 _running;
	bool                 _has_run;
	jack_nframes_t        monitor_check_interval;
	jack_nframes_t        last_monitor_check;
	jack_nframes_t       _processed_frames;
	bool                 _freewheeling;
	SigC::Slot1<int,jack_nframes_t>  freewheel_action;

	typedef set<Port *> Ports;
	Ports ports;

	int    process_callback (jack_nframes_t nframes);
	void   clear_ports ();
	string get_nth_physical (unsigned long which, int flags);

	static int _xrun_callback (void *arg);
	static int _graph_order_callback (void *arg);
	static int _process_callback (jack_nframes_t nframes, void *arg);
	static int  sample_rate_callback (jack_nframes_t nframes, void *arg);
	static void _jack_timebase_callback (jack_transport_state_t, jack_nframes_t, jack_position_t*, int, void*);
	static int  _jack_sync_callback (jack_transport_state_t, jack_position_t*, void *arg);
	static void _freewheel_callback (int , void *arg);
	void jack_timebase_callback (jack_transport_state_t, jack_nframes_t, jack_position_t*, int);
	int  jack_sync_callback (jack_transport_state_t, jack_position_t*);
	
	static void halted (void *);
	static void meter (Port *port, jack_nframes_t nframes);
};

}; /* namespace ARDOUR */

#endif /* __ardour_audioengine_h__ */
