/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: filesource.h,v 1.22 2004/02/07 20:16:37 pauld Exp $
*/

#ifndef __playlist_file_buffer_h__ 
#define __playlist_file_buffer_h__

#include <vector>
#include <string>

#include <ardour/source.h>

struct tm;

using std::string;

namespace ARDOUR {

class FileSource : public Source {
  public:
	FileSource (string path, jack_nframes_t rate);
	FileSource (const XMLNode&, jack_nframes_t rate);
	~FileSource ();

	jack_nframes_t length() const { return _length; }
	jack_nframes_t read (Sample *dst, jack_nframes_t start, jack_nframes_t cnt) const;
	jack_nframes_t write (Sample *src, jack_nframes_t cnt);
	void           mark_for_remove();
	string         peak_path(string audio_path);

	int update_header (jack_nframes_t when, struct tm&, time_t);

	int move_to_trash (const string trash_dir_name);

	static bool is_empty (string path);
	void mark_streaming_write_completed ();

	void   mark_take (string);
	string take_id() const { return _take_id; }

	static void set_bwf_country_code (string x);
	static void set_bwf_organization_code (string x);
	static void set_bwf_serial_number (int);
	
	static void set_search_path (string);

  private:
	int            fd;
	string        _path;
	bool           remove_at_unref;
	bool           is_bwf;
	off_t          data_offset;
	string        _take_id;

	static char bwf_country_code[3];
	static char bwf_organization_code[4];
	static char bwf_serial_number[13];

	struct GenericChunk {
	    char    id[4];
	    gint32  size; 
	};

	struct WAVEChunk : public GenericChunk {
	    char    text[4];      /* wave pseudo-chunk id "WAVE" */
	};

	struct FMTChunk : public GenericChunk {
	    gint16   formatTag;           /* format tag; currently pcm   */
	    gint16   nChannels;           /* number of channels         */
	    guint32  nSamplesPerSec;      /* sample rate in hz          */
	    gint32   nAvgBytesPerSec;     /* average bytes per second   */
	    gint16   nBlockAlign;         /* number of bytes per sample */
	    gint16   nBitsPerSample;      /* number of bits in a sample */
	};

	struct BroadcastChunk : public GenericChunk {
	    char   description[256];
	    char   originator[32];
	    char   originator_reference[32];
	    char   origination_date[10];
	    char   origination_time[8];
	    gint32 time_reference_low;
	    gint32 time_reference_high;
	    gint16 version;               /* 1.0 because we have umid and 190 bytes of "reserved" */
	    char   umid[64];
	    char   reserved[190];
	    /* we don't treat coding history as part of the struct */
	};

	struct ChunkInfo {
	    string        name;
	    unsigned long size;
	    off_t         offset;
	    
	    ChunkInfo (string s, unsigned long sz, off_t o) 
		    : name (s), size (sz), offset (o) {}
	};

	vector<ChunkInfo> chunk_info;
	
	struct {
	    WAVEChunk               wave;
	    FMTChunk                format;
	    GenericChunk            data;
	    BroadcastChunk          bext;
	    vector<string>          coding_history;
	} header;

	int init (string, bool must_exist, jack_nframes_t);
	jack_nframes_t read_unlocked (Sample *dst, jack_nframes_t start, jack_nframes_t cnt) const;

	int discover_chunks (bool silent);
	ChunkInfo* lookup_chunk (string name);

	int write_header ();
	int read_header (bool silent);

	int check_header (jack_nframes_t rate, bool silent);
	int fill_header (jack_nframes_t rate);
	
	int read_broadcast_data (ChunkInfo&);
	void compute_header_size ();
	
	static const ssize_t wave_header_size = sizeof (WAVEChunk) + sizeof (FMTChunk) + sizeof (GenericChunk);
	static const ssize_t bwf_header_size = wave_header_size + sizeof (BroadcastChunk);

	static string search_path;
};

}

#endif /* __playlist_file_buffer_h__ */
