/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: io.h,v 1.42 2004/02/29 23:33:57 pauld Exp $
*/

#ifndef __ardour_io_h__
#define __ardour_io_h__

#include <string>
#include <vector>
#include <cmath>
#include <sigc++/signal_system.h>
#include <glib.h>
#include <jack/jack.h>

#include <pbd/lockmonitor.h>
#include <pbd/fastlog.h>
#include <pbd/undo.h>
#include <pbd/atomic.h>
#include <midi++/controllable.h>

#include <ardour/ardour.h>
#include <ardour/stateful.h>
#include <ardour/utils.h>
#include <ardour/state_manager.h>
#include <ardour/curve.h>

class XMLNode;

namespace ARDOUR {

class Session;
class AudioEngine;
class Port;
class Connection;

class IO : public Stateful, public ARDOUR::StateManager
{

  public:
	static const string state_node_name;

	IO (Session&, const string& name, 
	    int input_min = -1, int input_max = -1, 
	    int output_min = -1, int output_max = -1);
	virtual ~IO();

	int input_minimum() const { return _input_minimum; }
	int input_maximum() const { return _input_maximum; }
	int output_minimum() const { return _output_minimum; }
	int output_maximum() const { return _output_maximum; }

	void set_input_minimum (int n);
	void set_input_maximum (int n);
	void set_output_minimum (int n);
	void set_output_maximum (int n);

	const string& name() const { return _name; }
	virtual void set_name (string str, void *src);
	
	void silence  (jack_nframes_t, jack_nframes_t offset);
	void pan (vector<Sample*>& bufs, jack_nframes_t nframes, jack_nframes_t offset, pan_t* panbuffer);
	void collect_input  (vector<Sample*>&, jack_nframes_t nframes, jack_nframes_t offset);
	void deliver_output (vector<Sample *>&, jack_nframes_t nframes, jack_nframes_t offset);

	virtual void   set_gain (gain_t g, void *src);
	void   inc_gain (gain_t delta, void *src) {
		set_gain (gain() + delta, src);
	}
	gain_t         gain () const                      { return _desired_gain; }
	virtual gain_t effective_gain () const;

	const vector<pan_t>&  pans() const { return _pans; }
	virtual pan_t  effective_stereo_pan() const;

	int use_input_connection (Connection&, void *src);
	int use_output_connection (Connection&, void *src);

	Connection *input_connection() const { return _input_connection; }
	Connection *output_connection() const { return _output_connection; }

	int add_input_port (string source, void *src);
	int add_output_port (string destination, void *src);

	int remove_input_port (Port *, void *src);
	int remove_output_port (Port *, void *src);

	int set_input (Port *, void *src);

	int connect_input (Port *our_port, string other_port, void *src);
	int connect_output (Port *our_port, string other_port, void *src);

	int disconnect_input (Port *our_port, string other_port, void *src);
	int disconnect_output (Port *our_port, string other_port, void *src);

	int disconnect_inputs (void *src);
	int disconnect_outputs (void *src);

	int set_pans (const vector<pan_t>& pans, void *src);

	jack_nframes_t output_latency() const;
	jack_nframes_t input_latency() const;
	void           set_port_latency (jack_nframes_t);

	Port *output (unsigned long n) const {
		if (n < _noutputs) {
			return _outputs[n];
		} else {
			return 0;
		}
	}

	Port *input (unsigned long n) const {
		if (n < _ninputs) {
			return _inputs[n];
		} else {
			return 0;
		}
	}

	unsigned long n_inputs () const { return _ninputs; }
	unsigned long n_outputs () const { return _noutputs; }

	SigC::Signal1<void,void*> input_configuration_changed;
	SigC::Signal1<void,void*> output_configuration_changed;
	SigC::Signal1<void,void*> input_changed;
	SigC::Signal1<void,void*> output_changed;
	SigC::Signal1<void,void*> pan_changed;
	SigC::Signal1<void,void*> gain_changed;
	SigC::Signal1<void,void*> name_changed;

	XMLNode& get_state (void);
	int set_state (const XMLNode&);

	virtual UndoAction get_memento() const;

	static int  disable_connecting (void);
	static int  enable_connecting (void);
	static int  disable_ports (void);
	static int  enable_ports (void);

	static SigC::Signal0<int> PortsLegal;
	static SigC::Signal0<int> ConnectingLegal;
	static SigC::Signal1<void,unsigned long> MoreOutputs;
	static SigC::Signal0<int> PortsCreated;

	/* MIDI control */

	void set_midi_to_gain_function (gain_t (*function)(double val)) {
		_midi_gain_control.midi_to_gain = function;
	}

	MIDI::Controllable& midi_gain_control() {
		return _midi_gain_control;
	}

	MIDI::Controllable& midi_pan_control() {
		return _midi_pan_control;
	}

	virtual void reset_midi_control (MIDI::Port*, bool on);

	/* Peak metering */

	float peak_input_power (unsigned long n) { 
		if (n < max(_ninputs, _noutputs)) {
			float x = _stored_peak_power[n];
			if(x > 0.0) {
				return 20 * fast_log10(x);
			} else {
				return minus_infinity();
			}
		} else {
			return minus_infinity();
		}
	}

	static SigC::Signal0<void> GrabPeakPower;

	/* automation */

	void clear_automation ();

	virtual void set_automation_recording_mode (AutomationList::Mode, void *src);
	virtual void set_automation_recording_style (AutomationList::Style, void *src);
	virtual void set_automation_playback (bool yn, void *src);

	AutomationList::Mode automation_mode() const;
	AutomationList::Style automation_style () const;

	bool automation_recording() const { return _automation_recording; }
	bool automation_playback() const { return  _automation_playback; }

	void set_automation_recording_bits (unsigned long flags);
	void set_automation_playback_bits (unsigned long flags);
	unsigned long automation_recording_bits() { return _automation_recording_bits; }
	unsigned long automation_playback_bits() { return _automation_playback_bits; }

	SigC::Signal1<void,void*> automation_recording_changed;
	SigC::Signal1<void,void*> automation_playback_changed;

	static void set_automation_interval (jack_nframes_t frames) {
		_automation_interval = frames;
	}

	static jack_nframes_t automation_interval() { 
		return _automation_interval;
	}

	ARDOUR::Curve& gain_automation_curve () { return _gain_automation_curve; }
	ARDOUR::Curve& pan_automation_curve () { return _pan_automation_curve; }

	void start_gain_touch ();
	void end_gain_touch ();

	id_t id() const { return _id; }

  private:
	mutable PBD::Lock io_lock;

  protected:
	Session&            _session;
	vector<pan_t>       _pans;
	vector<pan_t>       _desired_pans;
	gain_t              _gain;
	gain_t              _effective_gain;
	pan_t               _effective_stereo_pan;
	gain_t              _desired_gain;
	PBD::NonBlockingLock declick_lock;
	PBD::NonBlockingLock pan_lock;
	vector<Port*>       _outputs;
	vector<Port*>       _inputs;
	unsigned int		_unique_input_id;
	unsigned int		_unique_output_id;
	vector<float>       _peak_power;
	vector<float>       _stored_peak_power;
	string              _name;
	Connection*         _input_connection;
	Connection*         _output_connection;
	id_t                _id;

	void reset_peak_meters();
	
	static void apply_declick (vector<Sample*>&, jack_nframes_t nframes, 
				   gain_t initial, gain_t target, bool invert_polarity);

	void manage_gain (vector<Sample*>& bufs, jack_nframes_t nframes) {
		TentativeLockMonitor dm (declick_lock, __LINE__, __FILE__);
		if (dm.locked()) {
			if (_gain != _desired_gain) {
				apply_declick (bufs, nframes, _gain, _desired_gain, false);
				_gain = _desired_gain;
			}
		}
	}

	struct MIDIGainControl : public MIDI::Controllable {
	    MIDIGainControl (IO&, MIDI::Port *);
	    void set_value (float);

	    IO& io;
	    gain_t (*midi_to_gain) (double val);
	};

	struct MIDIPanControl : public MIDI::Controllable {
	    MIDIPanControl (IO&, MIDI::Port *);
	    void set_value (float);

	    IO& io;
	};

	MIDIGainControl _midi_gain_control;
	MIDIPanControl  _midi_pan_control;

	static void compute_peak (Sample *buf, unsigned long n, jack_nframes_t nsamples, float& peak, gain_t gain) {
		float p = peak;
		
		for (jack_nframes_t n = 0; n < nsamples; ++n) {
			p = f_max (p, buf[n]);
		}
		
		peak = p * gain;
	}	

	/* state management */

	Change               restore_state (State&);
	StateManager::State* state_factory (std::string why) const;
	void                 send_state_changed();

	/* automation */

	jack_nframes_t last_automation_snapshot;
	static jack_nframes_t _automation_interval;

	void automation_snapshot (jack_nframes_t now) {
		if (_automation_recording) {
			if (last_automation_snapshot > now || now - last_automation_snapshot > _automation_interval) {
				if (_automation_recording_bits & GainAutomation) {
					_gain_automation_curve.rt_add (now, gain());
				}
				if (_automation_recording_bits & PanAutomation) {
					/* we only automate stereo pan right now */
					if (n_outputs() == 2) {
						_pan_automation_curve.rt_add (now, pans()[0]);
					}
				}
				last_automation_snapshot = now;
			}
		}
	}

	bool    _automation_recording;
	unsigned long _automation_recording_bits;
	bool    _automation_playback;
	unsigned long _automation_playback_bits;

	bool     apply_gain_automation;
	bool     apply_pan_automation;

	Curve    _gain_automation_curve;
	Curve    _pan_automation_curve;
	
	gain_t *gain_automation_buffer;
	pan_t  *pan_automation_buffer;

	int  save_automation (const string&);
	int  load_automation (const string&);
	
	PBD::NonBlockingLock automation_lock;

  private:

	unsigned long _ninputs;
	unsigned long _noutputs;

	/* are these the best variable names ever, or what? */

	SigC::Connection input_connection_configuration_connection;
	SigC::Connection output_connection_configuration_connection;
	SigC::Connection input_connection_connection_connection;
	SigC::Connection output_connection_connection_connection;

	static bool connecting_legal;
	static bool ports_legal;
	
	XMLNode *pending_state_node;
	int ports_became_legal ();
	int connecting_became_legal ();
	SigC::Connection connection_legal_c;
	SigC::Connection port_legal_c;

	int _input_minimum;
	int _input_maximum;
	int _output_minimum;
	int _output_maximum;

	static int parse_io_string (const string&, vector<string>& chns);
	static int parse_gain_string (const string&, vector<string>& chns);
	static int parse_pan_string (const string&, vector<pan_t>& pans);
	
	int set_sources (vector<string>&, void *src, bool add);
	int set_destinations (vector<string>&, void *src, bool add);

	int set_inputs (const string& str);
	int set_outputs (const string& str);
	int set_pans (const string& str);

	int ensure_inputs (unsigned int, bool clear, void *src);
	int ensure_outputs (unsigned int, bool clear, void *src);

	void reset_pans (void *src);
	
	void drop_input_connection ();
	void drop_output_connection ();

	void input_connection_configuration_changed ();
	void input_connection_connection_changed (int);
	void output_connection_configuration_changed ();
	void output_connection_connection_changed (int);

	int create_ports (const XMLNode&);
	int make_connections (const XMLNode&);

	void setup_peak_meters ();
	void grab_peak_power ();
};

}; /* namespace ARDOUR */

#endif /*__ardour_io_h__ */
