/*
    Copyright (C) 2000-2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: route.h,v 1.56 2004/02/29 23:33:57 pauld Exp $
*/

#ifndef __ardour_route_h__
#define __ardour_route_h__

#include <cmath>
#include <list>
#include <set>
#include <map>

#include <pthread.h>
#include <glib.h>


#include <pbd/atomic.h>
#include <pbd/fastlog.h>
#include <pbd/lockmonitor.h>
#include <pbd/xml++.h>
#include <pbd/undo.h>

#include <ardour/ardour.h>
#include <ardour/stateful.h>
#include <ardour/io.h>
#include <ardour/session.h>
#include <ardour/redirect.h>

namespace ARDOUR {

class Insert;
class Send;
class RouteGroup;

enum mute_type {
    PRE_FADER =    0x1,
    POST_FADER =   0x2,
    CONTROL_OUTS = 0x4,
    MAIN_OUTS =    0x8
};

class Route : public IO
{
  private:
	typedef list<Redirect *> RedirectList;

  public:

	enum Flag {
		Hidden = 0x1,
		MasterOut = 0x2,
		ControlOut = 0x4,
	};

	Route (Session&, const string& name, int input_min, int input_max, int output_min, int output_max, Flag flags = Flag(0));
	Route (Session&, const XMLNode&);
	virtual ~Route();

	const string& comment() { return _comment; }
	void set_comment (const string &str, void *src);

	long order_key(string name) const;
	void set_order_key (string name, long n);

	bool hidden() const { return _flags & Hidden; }
	bool master() const { return _flags & MasterOut; }
	bool control() const { return _flags & ControlOut; }

	/* these are the core of the API of a Route. see the protected sections as well */

	virtual int  roll (jack_nframes_t nframes, jack_nframes_t end_frame, jack_nframes_t offset, int declick,
			   bool can_record, bool rec_monitors_input);
	virtual int  no_roll (jack_nframes_t nframes, jack_nframes_t end_frame, jack_nframes_t offset, bool state_changing,
			      bool can_record, bool rec_monitors_input);
	virtual void toggle_monitor_input ();
	virtual bool can_record() const { return false; }
	virtual void set_record_enable (bool yn, void *src) {}
	virtual bool record_enabled() const { return false; }
	virtual void transport_stopped (bool abort, bool did_locate, bool flush_redirects);
	virtual IO& input_io() { return *this; }
	virtual IO& output_io(){ return *this; }

	/* end of vfunc-based API */

	/* override IO::set_gain() to provide group control */

	void set_gain (gain_t val, void *src);
	void inc_gain (gain_t delta, void *src);

	bool active() const { return _active; }
	void set_active (bool yn);

	void set_solo (bool yn, void *src);
	bool soloed() const { return _soloed; }

	void set_solo_safe (bool yn, void *src);
	bool solo_safe() const { return _solo_safe; }

	void set_mute (bool yn, void *src);
	bool muted() const { return _muted; }

	void set_mute_config (mute_type, bool, void *src);
	bool get_mute_config (mute_type);

	void set_phase_invert (bool yn, void *src);
	bool phase_invert() const { return _phase_invert; }
	
	void       set_edit_group (RouteGroup *, void *);
	RouteGroup *edit_group () { return _edit_group; }

	void       set_mix_group (RouteGroup *, void *);
	RouteGroup *mix_group () { return _mix_group; }

	void  set_meter_pre_fader (bool yn, void *src);
	bool  meter_pre_fader() const { return _meter_pre_fader; }

	/* Redirects */

	void flush_redirects ();

	template<class T> void foreach_redirect (T *obj, void (T::*func)(Redirect *)) {
		LockMonitor lm (redirect_lock, __LINE__, __FILE__);
		for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
			(obj->*func) (*i);
		}
	}

	Redirect *nth_redirect (unsigned long n) {
		LockMonitor lm (redirect_lock, __LINE__, __FILE__);
		RedirectList::iterator i;
		for (i = _redirects.begin(); i != _redirects.end() && n; ++i, --n);
		if (i == _redirects.end()) {
			return 0;
		} else {
			return *i;
		}
	}
		
	void add_redirect (Redirect *, void *src);
	void move_redirect (Redirect *, void *src);
	void remove_redirect (Redirect *, void *src);
	void clear_redirects (void *src);
	void sort_redirects ();
	void copy_redirects (const Route&, Redirect::Placement);
	void all_redirects_flip();
	void all_redirects_active (bool state);

	virtual jack_nframes_t update_total_latency();
	jack_nframes_t signal_latency() const { return _own_latency; }
	void set_latency_delay (jack_nframes_t);

	SigC::Signal1<void,void*> solo_changed;
	SigC::Signal1<void,void*> solo_safe_changed;
	SigC::Signal1<void,void*> comment_changed;
	SigC::Signal1<void,void*> mute_changed;
	SigC::Signal1<void,void*> pre_fader_changed;
	SigC::Signal1<void,void*> post_fader_changed;
	SigC::Signal1<void,void*> control_outs_changed;
	SigC::Signal1<void,void*> main_outs_changed;
	SigC::Signal1<void,void*> redirects_changed;
	SigC::Signal1<void,void*> record_enable_changed;
	SigC::Signal1<void,void*> edit_group_changed;
	SigC::Signal1<void,void*> mix_group_changed;
	SigC::Signal0<void>       active_changed;
	SigC::Signal1<void,void*> meter_change;

	SigC::Signal0<void> GoingAway;

	/* gui's call this for their own purposes. */

	SigC::Signal2<void,string,void*> gui_changed;

	/* stateful */

	XMLNode& get_state();
	int set_state(const XMLNode& node);
	XMLNode& get_template();

	SigC::Signal1<void,void*> SelectedChanged;

	/* undo */

	UndoAction get_memento() const;
	void set_state (state_id_t);

	int set_control_outs (const vector<string>& ports);
	IO* control_outs() { return _control_outs; }

	bool feeds (Route *);
	set<Route *> fed_by;

	void reset_midi_control (MIDI::Port*, bool);
	
  protected:
	friend class Session;

	void set_solo_mute (bool yn);
	void set_block_size (jack_nframes_t nframes);
	bool has_external_redirects() const;
	void curve_reallocate ();

  protected:
	unsigned char _flags;

	/* tight cache-line access here is more important than sheer speed of
	   access.
	*/

	bool                     _muted : 1;
	bool                     _soloed : 1;
	bool                     _solo_muted : 1;
	bool                     _solo_safe : 1;
	bool                     _phase_invert : 1;
	bool                     _recordable : 1;
	bool                     _active : 1;
	bool                     _mute_affects_pre_fader : 1;
	bool                     _mute_affects_post_fader : 1;
	bool                     _mute_affects_control_outs : 1;
	bool                     _mute_affects_main_outs : 1;
	bool                     _silent : 1;
	bool                     _meter_pre_fader;

	gain_t                    solo_gain;
	gain_t                    mute_gain;
	gain_t                    desired_solo_gain;
	gain_t                    desired_mute_gain;

	jack_nframes_t            check_initial_delay (jack_nframes_t, jack_nframes_t&);

	jack_nframes_t           _initial_delay;
	jack_nframes_t           _roll_delay;
	jack_nframes_t           _own_latency;
	RedirectList             _redirects;
	PBD::NonBlockingLock      redirect_lock;
	IO                      *_control_outs;
	PBD::NonBlockingLock      control_outs_lock;
	RouteGroup              *_edit_group;
	RouteGroup              *_mix_group;
	string                   _comment;
	jack_nframes_t            block_size;
	
	void get_process_buffers (vector<Sample*>& bufs, jack_nframes_t nframes, jack_nframes_t offset);
	virtual void passthru     (jack_nframes_t nframes, jack_nframes_t offset, int declick);
	void process_output_buffers (vector<Sample*>& bufs, jack_nframes_t nframes, jack_nframes_t offset, bool with_redirects, int declick);
	
  protected:
	/* for derived classes */

	virtual XMLNode& state(bool);

	void silence (jack_nframes_t nframes, jack_nframes_t offset);
	SigC::Connection input_signal_connection;

	state_id_t _current_state_id;

  private:
	void init ();

	static unsigned long order_key_cnt;
	typedef std::map<std::string,long> OrderKeys;
	OrderKeys order_keys;

	void input_change_handler (void *src);
	void output_change_handler (void *src);

	void reset_plugin_counts ();
};

}; /* namespace ARDOUR*/

#endif /* __ardour_route_h__ */
