/*
    Copyright (C) 2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: audio_track.cc,v 1.35 2004/02/29 23:33:56 pauld Exp $
*/

#include <ardour/audio_track.h>
#include <ardour/diskstream.h>
#include <ardour/session.h>
#include <ardour/redirect.h>
#include <ardour/route_group_specialized.h>
#include <ardour/insert.h>
#include <ardour/audioplaylist.h>
#include "i18n.h"

using namespace SigC;
using namespace ARDOUR;

AudioTrack::AudioTrack (Session& sess, string name, Route::Flag flag)
	: Route (sess, name, 0, 0, -1, -1, flag),
	  diskstream (0)
{
	DiskStream::Flag dflags = DiskStream::Flag (0);

	if (_flags & Hidden) {
		dflags = DiskStream::Flag (dflags | DiskStream::Hidden);
	} else {
		dflags = DiskStream::Flag (dflags | DiskStream::Recordable);
	}

	DiskStream* ds = new DiskStream (_session, name, dflags);
	set_diskstream (*ds, this);

	_freeze_state = NoFreeze;

	output_changed.connect (slot (*this, &AudioTrack::output_change_handler));
}

AudioTrack::AudioTrack (Session& sess, const XMLNode& node)
	: Route (sess, "to be renamed", 0, 0, -1, -1),
	  diskstream (0)
{
	set_state (node);
	_freeze_state = NoFreeze;
	output_changed.connect (slot (*this, &AudioTrack::output_change_handler));
}

AudioTrack::~AudioTrack ()
{
	if (diskstream) {
		diskstream->unref();
	}
}

IO&
AudioTrack::input_io ()
{
	return *diskstream;
}

int
AudioTrack::set_diskstream (DiskStream& ds, void *src)
{
	if (diskstream) {
		diskstream->unref();
	}

	diskstream = &ds.ref();

	diskstream->set_record_enabled (false, this);
	diskstream->monitor_input (false);

	/* act as a proxy for the diskstream's i/o changes */

	diskstream->input_configuration_changed.connect (input_configuration_changed.slot());
	diskstream->input_changed.connect (input_changed.slot());

	 diskstream_changed (src); /* EMIT SIGNAL */

	return 0;
}	

int 
AudioTrack::use_diskstream (string name)
{
	DiskStream *dstream;

	if ((dstream = _session.diskstream_by_name (name)) == 0) {
		error << compose(_("AudioTrack: diskstream \"%1\" not known by session"), name) << endmsg;
		return -1;
	}
	
	return set_diskstream (*dstream, this);
}

int 
AudioTrack::use_diskstream (id_t id)
{
	DiskStream *dstream;

	if ((dstream = _session.diskstream_by_id (id)) == 0) {
		error << compose(_("AudioTrack: diskstream \"%1\" not known by session"), id) << endmsg;
		return -1;
	}
	
	return set_diskstream (*dstream, this);
}

bool
AudioTrack::record_enabled () const
{
	return diskstream->record_enabled();
}

void
AudioTrack::set_record_enable (bool yn, void *src)
{
	if (_mix_group && src != _mix_group && _mix_group->is_active()) {
		_mix_group->apply (&AudioTrack::set_record_enable, yn, _mix_group);
		return;
	}

	diskstream->set_record_enabled (yn, src);
}

XMLNode&
AudioTrack::state(bool full_state)
{
	XMLNode& track_state (Route::state (full_state));
	char buf[64];

	/* we don't return diskstream state because we don't
	   own the diskstream exclusively. control of the diskstream
	   state is ceded to the Session, even if we create the
	   diskstream.
	*/

	snprintf (buf, sizeof (buf), "%Lu", diskstream->id());
	track_state.add_property ("diskstream-id", buf);

	return track_state;
}

int
AudioTrack::set_state (const XMLNode& node)
{
	const XMLProperty *prop;

	if (Route::set_state (node)) {
		return -1;
	}

	if ((prop = node.property ("diskstream-id")) == 0) {
		
		/* some old sessions use the diskstream name rather than the ID */

		if ((prop = node.property ("diskstream")) == 0) {
			fatal << _("programming error: AudioTrack given state without diskstream!") << endmsg;
			/*NOTREACHED*/
			return -1;
		}

		return use_diskstream (prop->value());
	}

 	id_t id = strtoull (prop->value().c_str(), 0, 10);
	return use_diskstream (id);
}

void
AudioTrack::passthru (jack_nframes_t nframes, jack_nframes_t offset, int declick)
{
	/* we don't have our own inputs, so we have to use the diskstream's */

	Port*   port = diskstream->input(0);

	if (port) {

		Sample* ibuf = port->get_buffer (nframes) + offset;
		vector<Sample*> bufs;

		_silent = false;
		get_process_buffers (bufs, nframes, offset);

		unsigned int n=1;
		for (vector<Sample*>::iterator o = bufs.begin(); o != bufs.end(); ++o, ++n) {

			memcpy ((*o), ibuf, sizeof (Sample) * nframes);

			if (n < diskstream->n_inputs()) {
				ibuf = diskstream->input(n)->get_buffer (nframes) + offset;
			}
		}

		process_output_buffers (bufs, nframes, offset, true, declick);

 	} else {

		silence (nframes, offset);
	}
}

void
AudioTrack::passthru_silence (jack_nframes_t nframes, jack_nframes_t offset, int declick)
{
	vector<Sample*> bufs;

	get_process_buffers (bufs, nframes, offset);

	for (vector<Sample*>::iterator o = bufs.begin(); o != bufs.end(); ++o) {
		memset ((*o), 0, sizeof (Sample) * nframes);
	}
	
	process_output_buffers (bufs, nframes, offset, true, declick);
}

int 
AudioTrack::no_roll (jack_nframes_t nframes, jack_nframes_t end_frame, jack_nframes_t offset, 
		bool session_state_changing, bool can_record, bool rec_monitors_input)
{
	if (n_outputs() == 0) {
		return 0;
	}

	if (!_active) {
		silence (nframes, offset);
	}

	if (session_state_changing) {

		/* XXX is this safe to do against transport state changes? */

		passthru_silence (nframes, offset, 0);
		return 0;
	}
	
	if (_session.get_auto_input()) {
		if (Config->get_no_sw_monitoring()) {
			passthru_silence (nframes, offset, 0);
		} else {
			passthru (nframes, offset, 0);
		}
	} else {
		if (diskstream->record_enabled()) {
			passthru (nframes, offset, 0);
		} else {
			passthru_silence (nframes, offset, 0);
		}
	}

	return 0;
}

int
AudioTrack::roll (jack_nframes_t nframes, jack_nframes_t end_frame, jack_nframes_t offset, int declick,
		  bool can_record, bool rec_monitors_input)
{
	jack_nframes_t start_frame;
	int dret;
	Sample* b;
	Sample* tmpb;

	automation_snapshot (_session.transport_frame());

	if (n_outputs() == 0 && _redirects.empty()) {
		return 0;
	}

	if (!_active) {
		silence (nframes, offset);
		return 0;
	}

	if ((nframes = check_initial_delay (nframes, offset)) == 0) {
		/* need to do this so that the diskstream sets its
		   playback distance to zero, thus causing diskstream::commit
		   to do nothing.
		*/
		return diskstream->process (0, 0, false, false);
	}

	_silent = false;
	apply_gain_automation = false;
	apply_pan_automation = false;
	
	if ((dret = diskstream->process (nframes, offset, can_record, rec_monitors_input)) != 0) {

		silence (nframes, offset);

		/* positive error code means that the diskstream lock could not be taken */

		if (dret < 0) {
			error << compose(_("%1: disk stream error at frame %2 (%3)"), diskstream->name(), 
					 _session.transport_frame(), dret) << endmsg;
		}

		return dret;
	}

	if (diskstream->record_enabled() && !can_record && !_session.get_auto_input()) {

		/* not actually recording, but we want to hear the input material anyway */
		passthru (nframes, offset, 0);

	} else if ((b = diskstream->playback_buffer(0)) != 0) {

		/*
		  XXX is it true that the earlier test on n_outputs()
		  means that we can avoid checking it again here? i think
		  so, because changing the i/o configuration of an IO
		  requires holding the AudioEngine lock, which we hold
		  while in the process() tree.
		*/

		/* copy the diskstream data to all output buffers */
		
		vector<Sample*> bufs;
		
		get_process_buffers (bufs, nframes, offset);

		unsigned int n=1;
		for (vector<Sample*>::iterator i = bufs.begin(); i != bufs.end(); ++i, ++n) {
			memcpy ((*i), b, sizeof (Sample) * nframes); 
			if (n < diskstream->n_channels()) {
				tmpb = diskstream->playback_buffer(n);
				if (tmpb!=0) {
					b = tmpb;
				}
			}
		}

		/* don't waste time with automation if we're recording */

		if (!diskstream->record_enabled()) {

			start_frame = end_frame - nframes;
			
			if (_automation_playback && (_automation_playback_bits & GainAutomation)) {
				apply_gain_automation = _gain_automation_curve.rt_safe_get_vector (start_frame, end_frame, gain_automation_buffer, nframes);
			}
			
			if (_automation_playback && (_automation_playback_bits & PanAutomation)) {
				apply_pan_automation = _pan_automation_curve.rt_safe_get_vector (start_frame, end_frame, pan_automation_buffer, nframes);
			}
		}

		process_output_buffers (bufs, nframes, offset, (!_session.get_record_enabled() || _session.get_recording_plugins()), declick);
		
	} else {
		/* problem with the diskstream; just be quiet for a bit */
		silence (nframes, offset);
	}

	return 0;
}

void
AudioTrack::toggle_monitor_input ()
{
	Port *port;
	
	if ((port = diskstream->input_port()) != 0) {
		port->request_monitor_input (!port->monitoring_input());
	}
}

void
AudioTrack::output_change_handler (void *src)
{
}

void
AudioTrack::set_name (string str, void *src)
{
	diskstream->set_name (str, src);
	IO::set_name (str, src);
}

int
AudioTrack::export_stuff (vector<Sample*>& buffers, jack_nframes_t start, jack_nframes_t nframes)
{
	gain_t  gain_automation[nframes];
	gain_t  gain_buffer[nframes];
	float   mix_buffer[nframes];
	RedirectList::iterator i;
	bool post_fader_work = false;
	gain_t this_gain = _gain;
	vector<Sample*>::iterator bi;
	Sample * b;
	
	LockMonitor rlock (redirect_lock, __LINE__, __FILE__);
		
	if (diskstream->playlist()->read (buffers[0], mix_buffer, gain_buffer, start, nframes) != nframes) {
		return -1;
	}

	unsigned int n=1;
	bi = buffers.begin();
	b = buffers[0];
	++bi;
	for (; bi != buffers.end(); ++bi, ++n) {
		if (n < diskstream->n_channels()) {
			if (diskstream->playlist()->read ((*bi), mix_buffer, gain_buffer, start, nframes, n) != nframes) {
				return -1;
			}
			b = (*bi);
		}
		else {
			/* duplicate last across remaining buffers */
			memcpy ((*bi), b, sizeof (Sample) * nframes); 
		}
	}


	/* note: only run inserts during export. other layers in the machinery
	   will already have checked that there are no external port inserts.
	*/
	
	for (i = _redirects.begin(); i != _redirects.end(); ++i) {
		Insert *insert;
		
		if ((insert = dynamic_cast<Insert*>(*i)) != 0) {
			switch (insert->placement()) {
			case Redirect::PreFader:
				insert->run (buffers, nframes, 0);
				break;
			case Redirect::PostFader:
				post_fader_work = true;
				break;
			}
		}
	}
	
	if (_automation_playback && _automation_playback_bits & GainAutomation) {
		
		_gain_automation_curve.get_vector (start, start + nframes, gain_automation, nframes);

		for (bi = buffers.begin(); bi != buffers.end(); ++bi) {
			Sample *b = *bi;
			for (jack_nframes_t n = 0; n < nframes; ++n) {
				b[n] *= gain_automation[n];
			}
		}

	} else {

		for (bi = buffers.begin(); bi != buffers.end(); ++bi) {
			Sample *b = *bi;
			for (jack_nframes_t n = 0; n < nframes; ++n) {
				b[n] *= this_gain;
			}
		}
	}

	if (post_fader_work) {

		for (i = _redirects.begin(); i != _redirects.end(); ++i) {
			PluginInsert *insert;
			
			if ((insert = dynamic_cast<PluginInsert*>(*i)) != 0) {
				switch ((*i)->placement()) {
				case Redirect::PreFader:
					break;
				case Redirect::PostFader:
					insert->run (buffers, nframes, 0);
					break;
				}
			}
		}
	} 

	return 0;
}

jack_nframes_t
AudioTrack::update_total_latency ()
{
	_own_latency = 0;

	for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
		if ((*i)->active ()) {
			_own_latency += (*i)->latency ();
		}
	}

	set_port_latency (_own_latency);

	return _own_latency;
}

void
AudioTrack::freeze (UndoCommand fc)
{
	_freeze_cmd = fc;
	_freeze_state = Frozen;
}

void
AudioTrack::unfreeze ()
{
	if (_freeze_state == Frozen) {
		_freeze_cmd.undo ();
		_freeze_state = UnFrozen;
	}
}

void
AudioTrack::refreeze ()
{
	if (_freeze_state == UnFrozen) {
		_freeze_cmd.redo ();
		_freeze_state = Frozen;
	}
}
