/*
    Copyright (C) 2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: audioengine.cc,v 1.47 2004/02/29 23:33:56 pauld Exp $
*/

#include <ardour/audioengine.h>
#include <ardour/port.h>
#include <ardour/session.h>
#include <ardour/cycle_timer.h>
#include <ardour/utils.h>

#include <ardour/timestamps.h>

#include "i18n.h"

using namespace ARDOUR;

jack_nframes_t Port::short_over_length = 2;
jack_nframes_t Port::long_over_length = 10;

AudioEngine::AudioEngine (jack_client_t *jack) 
	: _jack (jack) 
{
	jack_init_timestamps (512);

	pthread_cond_init (&session_removed, 0);
	session = 0;
	session_remove_pending = false;
	_running = false;
	_has_run = false;
	last_monitor_check = 0;
	monitor_check_interval = max_frames;
	_processed_frames = 0;
	_freewheeling = false;
}

AudioEngine::~AudioEngine ()
{
	if (_running) {
		jack_client_close (_jack);
	}
}

int
AudioEngine::start ()
{
	if (!_running) {

		if (session) {
			jack_nframes_t blocksize = jack_get_buffer_size (_jack);

			session->set_block_size (blocksize);
			session->set_frame_rate (jack_get_sample_rate (_jack));

			/* page in as much of the session process code as we
			   can before we really start running.
			*/

			session->process (blocksize);
			session->process (blocksize);
			session->process (blocksize);
			session->process (blocksize);
			session->process (blocksize);
			session->process (blocksize);
			session->process (blocksize);
			session->process (blocksize);
		}

		_processed_frames = 0;
		last_monitor_check = 0;

		jack_on_shutdown (_jack, halted, this);
		jack_set_graph_order_callback (_jack, _graph_order_callback, this);
		jack_set_process_callback (_jack, _process_callback, this);
		jack_set_sample_rate_callback (_jack, sample_rate_callback, this);
		jack_set_xrun_callback (_jack, _xrun_callback, this);
		jack_set_sync_callback (_jack, _jack_sync_callback, this);
		jack_set_freewheel_callback (_jack, _freewheel_callback, this);

		if (Config->get_jack_time_master()) {
			jack_set_timebase_callback (_jack, 0, _jack_timebase_callback, this);
		}

		if (jack_activate (_jack) == 0) {
			_running = true;
			_has_run = true;
			 Running(); /* EMIT SIGNAL */
		} else {
			error << _("cannot activate JACK client") << endmsg;
		}
	}

	return _running ? 0 : -1;
}

int
AudioEngine::stop ()
{
	if (_running) {
		_running = false;
		jack_deactivate (_jack);
		 Stopped(); /* EMIT SIGNAL */
	}

	return _running ? -1 : 0;
}


void
AudioEngine::_jack_timebase_callback (jack_transport_state_t state, jack_nframes_t nframes,
									  jack_position_t* pos, int new_position, void *arg)
{
	static_cast<AudioEngine*> (arg)->jack_timebase_callback (state, nframes, pos, new_position);
}

void
AudioEngine::jack_timebase_callback (jack_transport_state_t state, jack_nframes_t nframes,
									 jack_position_t* pos, int new_position)
{
	if (session) {
		session->jack_timebase_callback (state, nframes, pos, new_position);
	}
}

int
AudioEngine::_jack_sync_callback (jack_transport_state_t state, jack_position_t* pos, void* arg)
{
	return static_cast<AudioEngine*> (arg)->jack_sync_callback (state, pos);
}

int
AudioEngine::jack_sync_callback (jack_transport_state_t state, jack_position_t* pos)
{
	if (session) {
		return session->jack_sync_callback (state, pos);
	} else {
		return TRUE;
	}
}

int
AudioEngine::_xrun_callback (void *arg)
{
	 static_cast<AudioEngine *>(arg)->Xrun (); /* EMIT SIGNAL */
	return 0;
}

int
AudioEngine::_graph_order_callback (void *arg)
{
	 static_cast<AudioEngine *>(arg)->GraphReordered (); /* EMIT SIGNAL */
	return 0;
}

int
AudioEngine::_process_callback (jack_nframes_t nframes, void *arg)
{
	return static_cast<AudioEngine *> (arg)->process_callback (nframes);
}

void
AudioEngine::_freewheel_callback (int onoff, void *arg)
{
	static_cast<AudioEngine*>(arg)->_freewheeling = onoff;
}

void 
AudioEngine::meter (Port *port, jack_nframes_t nframes)
{
	double peak;
	unsigned long overlen;
	jack_default_audio_sample_t *buf;
	
	buf = port->get_buffer (nframes);
	peak = port->_peak;
	overlen = port->overlen;
	
	{
		for (jack_nframes_t n = 0; n < nframes; ++n) {
			
			/* 1) peak metering */
			
			peak = f_max (peak, buf[n]);
			
			/* 2) clip/over metering */
			
			if (buf[n] >= 1.0) {
				overlen++;
			} else if (overlen) {
				if (overlen > Port::short_over_length) {
					port->_short_overs++;
				}
				if (overlen > Port::long_over_length) {
					port->_long_overs++;
				}
				overlen = 0;
			}
		}
	}
	
        /* post-loop check on the final status of overlen */
	
	if (overlen > Port::short_over_length) {
		port->_short_overs++;
	}
	if (overlen > Port::long_over_length) {
		port->_short_overs++;
	}

	if (peak > 0.0) {
		port->_peak_db= 20 * fast_log10 (peak);
	} else {
		port->_peak_db = minus_infinity();
	}
	
	port->_peak = peak;
	port->overlen = overlen;
}

int
AudioEngine::process_callback (jack_nframes_t nframes)
{
	TentativeLockMonitor tm (_process_lock, __LINE__, __FILE__);
	jack_nframes_t next_processed_frames;

#ifdef RT_MALLOC_DEBUG
	extern void malloc_set_rt_thread (pthread_t);
	malloc_set_rt_thread (pthread_self());
#endif

	/* handle wrap around of total frames counter */
	
	if (max_frames - _processed_frames < nframes) {
		next_processed_frames = nframes - (max_frames - _processed_frames);
	} else {
		next_processed_frames = _processed_frames + nframes;
	}
	
	if (!tm.locked() || session == 0) {
		_processed_frames = next_processed_frames;
		return 0;
	}

	if (session_remove_pending) {

		session = 0;
		session_remove_pending = false;
		pthread_cond_signal (&session_removed);
		_processed_frames = next_processed_frames;
		return 0;
	}

	if (_freewheeling) {
		if (Freewheel (nframes)) {
			_freewheeling = false;
			jack_set_freewheel (_jack, false);
		}
		return 0;
	}

	/* do input peak metering */

	for (Ports::iterator i = ports.begin(); i != ports.end(); ++i) {
		if ((*i)->metering) {
			meter ((*i), nframes);
		}
	}
	
	session->process (nframes);

	if (!_running) {
		/* we were zombified, maybe because a ladspa plugin took
		   too long, or jackd exited, or something like that.
		*/
		
		_processed_frames = next_processed_frames;
		return 0;
	}
		
	if (last_monitor_check + monitor_check_interval < next_processed_frames) {
		for (Ports::iterator i = ports.begin(); i != ports.end(); ++i) {
			
			Port *port = *i;
			bool x;
			
			if (port->last_monitor != (x = port->monitoring_input ())) {
				port->last_monitor = x;
				/* XXX I think this is dangerous, due to 
				   a likely mutex in the signal handlers ...
				*/
				 port->MonitorInputChanged (x); /* EMIT SIGNAL */
			}
		}
		last_monitor_check = next_processed_frames;
	}

	_processed_frames = next_processed_frames;
	return 0;
}

int
AudioEngine::sample_rate_callback (jack_nframes_t nframes, void *arg)
{
	AudioEngine *ae = reinterpret_cast<AudioEngine *> (arg);

	if (ae->session) {
		ae->session->set_frame_rate (nframes);
	}

	/* check for monitor input change every 1/10th of second */

	ae->monitor_check_interval = nframes / 10;
	ae->last_monitor_check = 0;

	 ae->SampleRateChanged (nframes); /* EMIT SIGNAL */

	return 0;
}

void 
AudioEngine::set_session (Session *s)
{
	if (!session) {
		s->set_block_size (jack_get_buffer_size (_jack));
		s->set_frame_rate (jack_get_sample_rate (_jack));
		session = s;
	}
}

void 
AudioEngine::remove_session ()
{
	if (_running) {
		LockMonitor lm (_process_lock, __LINE__, __FILE__);
		session_remove_pending = true;
		pthread_cond_wait (&session_removed, _process_lock.mutex());
	} else {
		session = 0;
	}
}

Port *
AudioEngine::register_audio_input_port (const string& portname)
{
	if (!_running) {
		if (!_has_run) {
			fatal << _("register audio input port called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return 0;
		}
	}

	jack_port_t *p = jack_port_register (_jack, portname.c_str(), JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);

	if (p) {
		Port *newport;
		if ((newport = new Port (p)) != 0) {
			ports.insert (ports.begin(), newport);
		}
		return newport;
	} 

	error << compose (_("could not register an input port called \"%1\""), portname)
	      << endmsg;

	return 0;
}

Port *
AudioEngine::register_audio_output_port (const string& portname)
{
	if (!_running) {
		if (!_has_run) {
			fatal << _("register audio output port called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return 0;
		}
	}

	jack_port_t *p;

	if ((p = jack_port_register (_jack, portname.c_str(), JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0)) != 0) {
		Port *newport = new Port (p);
		ports.insert (ports.begin(), newport);
		return newport;
	} else {
		error << compose (_("could not register an output port called \"%1\""), portname)
		      << endmsg;
	}

	return 0;
}

int          
AudioEngine::unregister_port (Port *port)
{
	if (!_running) { 
		/* probably happening when the engine has been halted by JACK,
		   in which case, there is nothing we can do here.
		*/
		return 0;
	}

	if (port) {

		int ret = jack_port_unregister (_jack, port->port);
		
		if (ret == 0) {
			Ports::iterator i = ports.find (port);
			
			if (i != ports.end()) {
				ports.erase (i);
			}
		}

		return ret;

	} else {
		return -1;
	}
}

int 
AudioEngine::connect (const string& source, const string& destination)
{
	if (!_running) {
		if (!_has_run) {
			fatal << _("connect called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return -1;
		}
	}
	return jack_connect (_jack, source.c_str(), destination.c_str());
}

int 
AudioEngine::disconnect (const string& source, const string& destination)
{
	if (!_running) {
		if (!_has_run) {
			fatal << _("disconnect called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return -1;
		}
	}
	return jack_disconnect (_jack, source.c_str(), destination.c_str());
}

int
AudioEngine::disconnect (Port *port)
{
	if (!_running) {
		if (!_has_run) {
			fatal << _("disconnect called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return -1;
		}
	}
	return jack_port_disconnect (_jack, port->port);
}

jack_nframes_t
AudioEngine::frame_rate ()
{
	return jack_get_sample_rate (_jack);
}

jack_nframes_t
AudioEngine::frames_per_cycle ()
{
	return jack_get_buffer_size (_jack);
}

Port *
AudioEngine::get_port_by_name (const string& portname, bool keep)
{
	LockMonitor lm (_process_lock, __LINE__, __FILE__);

	if (!_running) {
		if (!_has_run) {
			fatal << _("get_port_by_name() called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return 0;
		}
	}
	
	/* check to see if we have a Port for this name already */

	if (keep) {
		for (Ports::iterator i = ports.begin(); i != ports.end(); ++i) {
			if (portname == (*i)->name()) {
				return *i;
			}
		}
	}

	jack_port_t *p;

	if ((p = jack_port_by_name (_jack, portname.c_str())) != 0) {
		Port *newport = new Port (p);
		if (keep) {
			ports.insert (ports.begin(), newport);
		}
		return newport;
	} else {
		return 0;
	}
}

const char **
AudioEngine::get_ports (const string& port_name_pattern, const string& type_name_pattern, unsigned long flags)
{
	if (!_running) {
		if (!_has_run) {
			fatal << _("get_ports called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return 0;
		}
	}
	return jack_get_ports (_jack, port_name_pattern.c_str(), type_name_pattern.c_str(), flags);
}

void
AudioEngine::halted (void *arg)
{
	AudioEngine *ae = reinterpret_cast<AudioEngine *> (arg);
	ae->_running = false;
	cerr << "JACK has shutdown Ardour.\n";
	 ae->Halted(); /* EMIT SIGNAL */
}

unsigned long
AudioEngine::n_physical_outputs () const
{
	const char ** ports;
	unsigned long i = 0;

	if ((ports = jack_get_ports (_jack, NULL, NULL, JackPortIsPhysical|JackPortIsInput)) == 0) {
		return 0;
	}

	if (ports) {
		for (i = 0; ports[i]; ++i);
		free (ports);
	}
	return i;
}

unsigned long
AudioEngine::n_physical_inputs () const
{
	const char ** ports;
	unsigned long i = 0;
	
	if ((ports = jack_get_ports (_jack, NULL, NULL, JackPortIsPhysical|JackPortIsOutput)) == 0) {
		return 0;
	}

	if (ports) {
		for (i = 0; ports[i]; ++i);
		free (ports);
	}
	return i;
}

string
AudioEngine::get_nth_physical (unsigned long n, int flag)
{
	const char ** ports;
	unsigned long i;
	string ret;

	if (!_running) {
		if (!_has_run) {
			fatal << _("get_nth_physical called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return "";
		}
	}

	ports = jack_get_ports (_jack, NULL, NULL, JackPortIsPhysical|flag);
	
	if (ports == 0) {
		return "";
	}

	for (i = 0; i < n && ports[i]; ++i);

	if (ports[i]) {
		ret = ports[i];
	}

	free ((char *) ports);

	return ret;
}

jack_nframes_t
AudioEngine::get_port_total_latency (const Port& port)
{
	if (!_running) {
		if (!_has_run) {
			fatal << _("get_port_total_latency() called before engine was started") << endmsg;
			/*NOTREACHED*/
		} else {
			return max_frames;
		}
	}
	return jack_port_get_total_latency (_jack, port.port);
}

void
AudioEngine::transport_stop ()
{
	if (Config->get_jack_transport_master()) {
		jack_transport_stop (_jack);
	}
}

void
AudioEngine::transport_start ()
{
	if (Config->get_jack_transport_master()) {
		jack_transport_start (_jack);
	}
}

void
AudioEngine::transport_locate (jack_nframes_t where)
{
	if (Config->get_jack_transport_master()) {
		jack_transport_locate (_jack, where);
	}
}

int
AudioEngine::reset_timebase ()
{
	if (Config->get_jack_time_master()) {
		return jack_set_timebase_callback (_jack, 0, _jack_timebase_callback, this);
	} else {
		return jack_release_timebase (_jack);
	}
}

int
AudioEngine::freewheel (bool onoff)
{
	return jack_set_freewheel (_jack, onoff);
}
