/*
    Copyright (C) 2000-2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: ladspa_plugin.cc,v 1.47 2004/02/29 23:33:56 pauld Exp $
*/

#include <vector>
#include <string>

#include <cstdlib>
#include <cstdio> // so libraptor doesn't complain
#include <cmath>
#include <dirent.h>
#include <sys/stat.h>
#include <cerrno>

#include <lrdf.h>

#include <pbd/compose.h>
#include <pbd/error.h>
#include <pbd/pathscanner.h>
#include <pbd/xml++.h>

#include <midi++/manager.h>

#include <ardour/ardour.h>
#include <ardour/session.h>
#include <ardour/audioengine.h>
#include <ardour/ladspa_plugin.h>

#include <pbd/stl_delete.h>

#include "i18n.h"
#include <locale.h>

using namespace LADSPA;
using namespace ARDOUR;

Manager::Manager (AudioEngine& e)
	: _engine (e)
{
	char* s;

	string lrdf_path;
	if ((s = getenv ("LADSPA_RDF_PATH"))){
		lrdf_path = s;
	}

	if (lrdf_path.length() == 0) {
		lrdf_path = "/usr/local/share/ladspa/rdf:/usr/share/ladspa/rdf";
	}
	add_lrdf_data(lrdf_path);

	descend_lrdf_classes (LADSPA_BASE "Plugin", "", "Generic");

	if ((s = getenv ("LADSPA_PATH"))) {
		search_path = s;
	}

	refresh ();
}

void
Manager::refresh ()
{
	for (list<Info*>::iterator i = _plugin_info.begin(); i != _plugin_info.end(); ++i) {
		delete *i;
	}

	_plugin_info.clear ();

	if (search_path.length() == 0) {
		search_path = "/usr/local/lib/ladspa:/usr/lib/ladspa";
	}
	discover_from_path (search_path);
}

int
Manager::add_directory (string path)
{
	if (discover_from_path (path) == 0) {
		search_path += ':';
		search_path += path;
		return 0;
	} 
	return -1;
}

static bool plugin_filter (const string& str, void *arg)
{
	/* Not a dotfile, has a prefix before a period, suffix is "so" */
	
	return str[0] != '.' && (str.length() > 3 && str.find (".so") == (str.length() - 3));
}

int
Manager::discover_from_path (string path)
{
	PathScanner scanner;
	vector<string *> *plugin_objects;
	vector<string *>::iterator x;
	int ret = 0;

	plugin_objects = scanner (search_path, plugin_filter, 0, true, true);

	if (plugin_objects) {
		for (x = plugin_objects->begin(); x != plugin_objects->end (); ++x) {
			discover (**x);
		}
	}

	vector_delete (plugin_objects);
	return ret;
}

static bool lrdf_filter (const string &str, void *arg)
{
//	return str[0] != '.' && (str.length() > 3 && str.find (".so") == (str.length() - 3));

	return (str.find(".rdf") == (str.length() -4)) ||
			(str.find(".rdfs") == (str.length() - 5));
}

void
Manager::add_lrdf_data (const string &path)
{
	PathScanner scanner;
	vector<string *>* rdf_files;
	vector<string *>::iterator x;
	string uri;

	rdf_files = scanner (path, lrdf_filter, 0, true, true);

	if (rdf_files) {
		for (x = rdf_files->begin(); x != rdf_files->end (); ++x) {
			uri = "file://" + **x;

			if (lrdf_read_file(uri.c_str())) {
				warning << "Could not parse rdf file: " << uri << endmsg;
			}
		}
	}

	vector_delete (rdf_files);
}

int 
Manager::discover (string path)
{
	Info *info;
	void *module;
	const LADSPA_Descriptor *descriptor;
	LADSPA_Descriptor_Function dfunc;
	char *errstr;

	if ((module = dlopen (path.c_str(), RTLD_NOW)) == 0) {
		error << compose(_("LADSPA: cannot load module \"%1\" (%2)"), path, dlerror()) << endmsg;
		return -1;
	}

	dfunc = (LADSPA_Descriptor_Function) dlsym (module, "ladspa_descriptor");

	if ((errstr = dlerror()) != 0) {
		error << compose(_("LADSPA: module \"%1\" has no descriptor function."), path) << endmsg;
		dlclose (module);
		return -1;
	}

	for (unsigned long i = 0; ; ++i) {
		if ((descriptor = dfunc (i)) == 0) {
			break;
		}

		info = new Info;
		info->name = descriptor->Name;
		info->path = path;
		info->index = i;
		info->n_inputs = 0;
		info->n_outputs = 0;

		if (rdf_type.find(descriptor->UniqueID) != rdf_type.end()) {
			info->uri = rdf_type[descriptor->UniqueID];
		} else {
			info->uri = "Unknown";
		}
		
		for (unsigned long n=0; n < descriptor->PortCount; ++n) {
			if ( LADSPA_IS_PORT_AUDIO (descriptor->PortDescriptors[n]) ) {
				if ( LADSPA_IS_PORT_INPUT (descriptor->PortDescriptors[n]) ) {
					info->n_inputs++;
				}
				else if ( LADSPA_IS_PORT_OUTPUT (descriptor->PortDescriptors[n]) ) {
					info->n_outputs++;
				}
			}
		}
		
		_plugin_info.push_back (info);
	}

// GDB WILL NOT LIKE YOU IF YOU DO THIS
//	dlclose (module);

	return 0;
}

Plugin *
Manager::load (Session& session, LADSPA::Info *info)
{
	void *module;
	Plugin *plugin = 0;

	if ((module = dlopen (info->path.c_str(), RTLD_NOW)) == 0) {
		error << compose(_("LADSPA: cannot load module from \"%1\""), info->path) << endmsg;
		return 0;
	}

	try {
		plugin = new Plugin (module, _engine, info->index, session.frame_rate());
	}

	catch (failed_constructor &err) {
		plugin = 0;
	}

	MIDI::Controllable *mcontrol;

	for (unsigned long i = 0; i < plugin->port_count(); ++i) {
		if (LADSPA_IS_PORT_INPUT(plugin->port_descriptors()[i]) &&
		    LADSPA_IS_PORT_CONTROL(plugin->port_descriptors()[i])) {
			if ((mcontrol = plugin->get_nth_midi_control (i)) != 0) {
				mcontrol->midi_rebind (session.mmc_port(), 0);
			}
		}
	}

	plugin->set_info(*info);
	
	return plugin;
}

void
Manager::descend_lrdf_classes (string uri, string base, string label)
{
	typedef map<unsigned long, string> ClassMap;

	lrdf_uris* uris = lrdf_get_subclasses(uri.c_str());
	if (uris){
		for (int i = 0; i < uris->count; ++i) {
			string newlabel = lrdf_get_label(uris->items[i]);
			string newbase = compose("%1/%2", base, newlabel);
			descend_lrdf_classes (uris->items[i], newbase, newlabel);
		}
		lrdf_free_uris(uris);
	}

	uris = lrdf_get_instances (uri.c_str());
	if (uris) {
		for (int i = 0; i < uris->count; ++i) {
			rdf_type.insert(
				ClassMap::value_type(lrdf_get_uid(uris->items[i]), label));
		}
		lrdf_free_uris(uris);
	}
}

Plugin *
LADSPA::find_plugin(Session& session, string name)
{
	LADSPA::Manager *ladspa = LADSPA::manager;
	list<LADSPA::Info *>::iterator i;
	list<LADSPA::Info *> &plugs = ladspa->plugin_info();

	for (i = plugs.begin(); i != plugs.end(); ++i) {
		if ((*i)->name == name) {
			break;
		}
	}
	
	if (i == plugs.end()) {
		return 0;
	}
	
	LADSPA::Plugin *plugin = ladspa->load (session, *i);
	return plugin;
}

const string LADSPA::Plugin::state_node_name = "Ladspa";


Plugin::Plugin (void *mod, AudioEngine& e, unsigned long index, jack_nframes_t rate)
	: _engine (e)
{
	init (mod, index, rate);
}

Plugin::Plugin (const Plugin &other)
	: _engine(other._engine), _info(other._info)
{
	init (other.module, other._index, other.sample_rate);

	for (unsigned int i = 0; i < port_count(); ++i) {
		control_data[i] = other.control_data[i];
	}
}

void
Plugin::init (void *mod, unsigned long index, jack_nframes_t rate)
{
	LADSPA_Descriptor_Function dfunc;
	unsigned long i, port_cnt;
	char *errstr;

	module = mod;
	control_data = 0;
	latency_control_port = 0;

	dfunc = (LADSPA_Descriptor_Function) dlsym (module, "ladspa_descriptor");

	if ((errstr = dlerror()) != NULL) {
		error << _("LADSPA: module has no descriptor function.") << endmsg;
		throw failed_constructor();
	}

	if ((descriptor = dfunc (index)) == 0) {
		error << _("LADSPA: plugin has gone away since discovery!") << endmsg;
		throw failed_constructor();
	}

	_index = index;

	if (LADSPA_IS_INPLACE_BROKEN(descriptor->Properties)) {
		error << compose(_("LADSPA: \"%1\" cannot be used, since it cannot do inplace processing"), descriptor->Name) << endmsg;
		throw failed_constructor();
	}
	
	sample_rate = rate;

	if (descriptor->instantiate == 0) {
		throw failed_constructor();
	}

	if ((handle = descriptor->instantiate (descriptor, rate)) == 0) {
		throw failed_constructor();
	}

	port_cnt = port_count();

	control_data = new LADSPA_Data[port_cnt];

	for (i = 0; i < port_cnt; ++i) {
		if (LADSPA_IS_PORT_CONTROL(port_descriptors()[i])) {
			connect_port (i, &control_data[i]);
			
			if (LADSPA_IS_PORT_OUTPUT(port_descriptors()[i]) &&
			    strcmp (port_names()[i], X_("latency")) == 0) {
				latency_control_port = &control_data[i];
				*latency_control_port = 0;
			}

			if (!LADSPA_IS_PORT_INPUT(port_descriptors()[i])) {
				continue;
			}
		
			control_data[i] = default_value (i);
		}
	}

	/* set up a vector of null pointers for the MIDI controls.
	   we'll fill this in on an as-needed basis.
	*/

	for (i = 0; i < port_cnt; ++i) {
		midi_controls.push_back (0);
	}

	if (latency_control_port) {

		/* we need to run the plugin so that it can set its latency
		   parameter.
		*/

		activate ();

		unsigned long port_index = 0;
		unsigned long in_index = 0;
		unsigned long out_index = 0;
		const jack_nframes_t bufsize = 1024;
		LADSPA_Data buffer[bufsize];

		/* Note that we've already required that plugins
		   be able to handle in-place processing.
		*/
		
		port_index = 0;
		
		while (port_index < port_count()) {
			if (LADSPA_IS_PORT_AUDIO (port_descriptors()[port_index])) {
				if (LADSPA_IS_PORT_INPUT (port_descriptors()[port_index])) {
					connect_port (port_index, buffer);
					in_index++;
				} else if (LADSPA_IS_PORT_OUTPUT (port_descriptors()[port_index])) {
					connect_port (port_index, buffer);
					out_index++;
				}
			}
			port_index++;
		}
	
		run (bufsize);
		deactivate ();
	}
}

Plugin::~Plugin ()
{
	deactivate ();

	 GoingAway (this); /* EMIT SIGNAL */

	for (unsigned long n = 0; n < descriptor->PortCount; ++n) {
		if (midi_controls[n]) {
			delete midi_controls[n];
		}
	}
	
	/* XXX who should close a plugin? */

        // dlclose (module);

	if (control_data) {
		delete [] control_data;
	}
}

void
Plugin::store_state (PluginState& state)
{
	state.parameters.clear ();
	
	for (guint i = 0; i < port_count(); ++i){

		if (LADSPA_IS_PORT_INPUT(port_descriptors()[i]) && 
		    LADSPA_IS_PORT_CONTROL(port_descriptors()[i])){
			pair<unsigned long,float> datum;

			datum.first = i;
			datum.second = control_data[i];

			state.parameters.insert (datum);
		}
	}
}

void
Plugin::restore_state (PluginState& state)
{
	for (map<unsigned long,float>::iterator i = state.parameters.begin(); i != state.parameters.end(); ++i) {
		set_control_port (i->first, i->second);
	}
}

float
Plugin::default_value (unsigned long port)
{
	const LADSPA_PortRangeHint *prh = port_range_hints();
	
	/* defaults - case 1 */
	
	if (LADSPA_IS_HINT_HAS_DEFAULT(prh[port].HintDescriptor)) {
		if (LADSPA_IS_HINT_DEFAULT_MINIMUM(prh[port].HintDescriptor)) {
			return prh[port].LowerBound;
		}
		
		/* FIXME: add support for logarithmic defaults */
		
		else if (LADSPA_IS_HINT_DEFAULT_LOW(prh[port].HintDescriptor)) {
			return prh[port].LowerBound * 0.75f + prh[port].UpperBound * 0.25f;
		}
		else if (LADSPA_IS_HINT_DEFAULT_MIDDLE(prh[port].HintDescriptor)) {
			return prh[port].LowerBound * 0.50f + prh[port].UpperBound * 0.50f;
		}
		else if (LADSPA_IS_HINT_DEFAULT_HIGH(prh[port].HintDescriptor)) {
			return prh[port].LowerBound * 0.25f + prh[port].UpperBound * 0.75f;
		}
		else if (LADSPA_IS_HINT_DEFAULT_MAXIMUM(prh[port].HintDescriptor)) {
			return prh[port].UpperBound;
		}
		else if (LADSPA_IS_HINT_DEFAULT_0(prh[port].HintDescriptor)) {
			return 0.0f;
		}
		else if (LADSPA_IS_HINT_DEFAULT_1(prh[port].HintDescriptor)) {
			return 1.0f;
		}
		else if (LADSPA_IS_HINT_DEFAULT_100(prh[port].HintDescriptor)) {
			return 100.0f;
		}
		else if (LADSPA_IS_HINT_DEFAULT_440(prh[port].HintDescriptor)) {
			return 440.0f;
		}
		else {
			/* no hint found */
			return 0.0f;
		}
	}
	
	/* defaults - case 2 */
	else if (LADSPA_IS_HINT_BOUNDED_BELOW(prh[port].HintDescriptor) &&
		 !LADSPA_IS_HINT_BOUNDED_ABOVE(prh[port].HintDescriptor)) {
		
		if (prh[port].LowerBound < 0) return 0.0f;
		else return prh[port].LowerBound;
	}
	
	/* defaults - case 3 */
	else if (!LADSPA_IS_HINT_BOUNDED_BELOW(prh[port].HintDescriptor) &&
		 LADSPA_IS_HINT_BOUNDED_ABOVE(prh[port].HintDescriptor)) {
		
		if (prh[port].UpperBound > 0) return 0.0f;
		else return prh[port].UpperBound;
	}
	
	/* defaults - case 4 */
	else if (LADSPA_IS_HINT_BOUNDED_BELOW(prh[port].HintDescriptor) &&
		 LADSPA_IS_HINT_BOUNDED_ABOVE(prh[port].HintDescriptor)) {
		
		if (prh[port].LowerBound < 0 && prh[port].UpperBound > 0) return 0.0f;
		else if (prh[port].LowerBound < 0 && prh[port].UpperBound < 0) return prh[port].UpperBound;
		else return prh[port].LowerBound;
	}
	
	/* defaults - case 5 */
		
	if (LADSPA_IS_HINT_SAMPLE_RATE(prh[port].HintDescriptor)) {
		return sample_rate;
	}

	return 0.0f;
}	

MIDI::Controllable *
Plugin::get_nth_midi_control (unsigned long n)
{
	if (n >= port_count()) {
		return 0;
	}

	if (midi_controls[n] == 0) {
		midi_controls[n] = new MIDIPortControl (*this, n, 0);
	}
	
	return midi_controls[n];
}

void
Plugin::set_control_port (unsigned long port, LADSPA_Data val)
{
	/* XXX bounds checking */
	control_data[port] = val;
	 ControlPortChanged (port, val); /* EMIT SIGNAL */
}

LADSPA_Data
Plugin::get_control_port(unsigned long port) const
{
	return control_data[port];
}

unsigned long
Plugin::nth_control_port(unsigned long n, bool& ok) const
{
	unsigned long x, c;

	ok = false;

	for (c = 0, x = 0; x < descriptor->PortCount; ++x) {
		if (LADSPA_IS_PORT_CONTROL (descriptor->PortDescriptors[x])) 
			if (c++ == n) {
				ok = true;
				return x;
			}
	}
	return 0;
}

XMLNode&
Plugin::get_state()
{
	XMLNode *root = new XMLNode(state_node_name);
	XMLNode *child;
	gchar buf[16];

	setlocale (LC_NUMERIC, "POSIX");
	
	for (guint i = 0; i < port_count(); ++i){

		if (LADSPA_IS_PORT_INPUT(port_descriptors()[i]) && 
		    LADSPA_IS_PORT_CONTROL(port_descriptors()[i])){

			child = new XMLNode("port");
			snprintf(buf, sizeof(buf), "%u", i);
			child->add_property("number", string(buf));
			snprintf(buf, sizeof(buf), "%+f", control_data[i]);
			child->add_property("value", string(buf));
			root->add_child_nocopy (*child);

			MIDI::Controllable *pcontrol = get_nth_midi_control (i);
			
			if (pcontrol) {

				MIDI::eventType ev;
				MIDI::byte      additional;
				MIDI::channel_t chn;
				XMLNode*        midi_node;

				if (pcontrol->get_control_info (chn, ev, additional)) {

					midi_node = child->add_child ("midi-control");
		
					snprintf (buf, sizeof(buf), "0x%x", ev);
					midi_node->add_property ("event", buf);
					snprintf (buf, sizeof(buf), "%d", chn);
					midi_node->add_property ("channel", buf);
					snprintf (buf, sizeof(buf), "0x%x", additional);
					midi_node->add_property ("additional", buf);
				}
			}
		}
	}

	setlocale (LC_NUMERIC, "");

	return *root;
}

int
Plugin::set_state(string entry)
{
	string presets_path = Config->get_user_ardour_path();
	string path(compose("%1/presets/%2/%3.xml", presets_path, unique_id(), entry));

	if (path.length() > PATH_MAX) {
		warning << compose(_("%1 : longer than maximum path limit.  Trunicating"), path) << endmsg;
		path = path.substr(0, PATH_MAX);
	}

	XMLTree tree(path);
	XMLNode* root = tree.root();

	if (root != 0){
		return set_state(*root);
	} else {
		error << compose(_("LADSPA Plugin: could not open preset file \"%1\" (%2)"), path, strerror (errno)) << endmsg;
		return 0;
	}
}

int
Plugin::set_state(const XMLNode& node)
{
	XMLNodeList nodes;
	XMLProperty *prop;
	XMLNodeConstIterator iter;
	XMLNode *child;
	const char *port;
	const char *data;
	unsigned long port_id;

	if(node.name() != state_node_name) {
		error << _("Bad node send to Plugin::set_state") << endmsg;
		return 0;
	}

	nodes = node.children ("port");

	setlocale (LC_NUMERIC, "POSIX");
	
	for(iter = nodes.begin(); iter != nodes.end(); ++iter){

		child = *iter;

		if ((prop = child->property("number")) != 0) {
			port = prop->value().c_str();
		} else {
			warning << _("LADSPA: no ladspa port number") << endmsg;
			continue;
		}
		if ((prop = child->property("value")) != 0) {
			data = prop->value().c_str();
		} else {
			warning << _("LADSPA: no ladspa port data") << endmsg;
			continue;
		}

		sscanf (port, "%lu", &port_id);
		set_control_port (port_id, atof(data));

		XMLNodeList midi_kids;
		XMLNodeConstIterator iter;
		
		midi_kids = child->children ("midi-control");
		
		for (iter = midi_kids.begin(); iter != midi_kids.end(); ++iter) {
			
			child = *iter;

			MIDI::eventType ev = MIDI::on; /* initialize to keep gcc happy */
			MIDI::byte additional = 0; /* initialize to keep gcc happy */
			MIDI::channel_t chn = 0; /* initialize to keep gcc happy */
			bool ok = true;
			int xx;
			
			if ((prop = child->property ("event")) != 0) {
				sscanf (prop->value().c_str(), "0x%x", &xx);
				ev = (MIDI::eventType) xx;
			} else {
				ok = false;
			}
			
			if (ok && ((prop = child->property ("channel")) != 0)) {
				sscanf (prop->value().c_str(), "%d", &xx);
				chn = (MIDI::channel_t) xx;
			} else {
				ok = false;
			}
			
			if (ok && ((prop = child->property ("additional")) != 0)) {
				sscanf (prop->value().c_str(), "0x%x", &xx);
				additional = (MIDI::byte) xx;
			}
			
			if (ok) {
				MIDI::Controllable* pcontrol = get_nth_midi_control (port_id);

				if (pcontrol) {
					pcontrol->set_control_type (chn, ev, additional);
				}

			} else {
				error << compose(_("LADSPA Plugin MIDI control specification for port %1 is incomplete, so it has been ignored"), port) << endl;
			}
		}
	}

	setlocale (LC_NUMERIC, "");

	return -1;
}

Plugin::MIDIPortControl::MIDIPortControl (Plugin& p, unsigned long port_id, MIDI::Port *port)
	: MIDI::Controllable (port, 0), plugin (p), absolute_port (port_id)
{
	LADSPA_PortRangeHint prh;

	prh  = plugin.port_range_hints()[absolute_port];

	toggled = LADSPA_IS_HINT_TOGGLED(prh.HintDescriptor);
	logarithmic = LADSPA_IS_HINT_LOGARITHMIC(prh.HintDescriptor);

	if (LADSPA_IS_HINT_BOUNDED_BELOW(prh.HintDescriptor)) {
		if (LADSPA_IS_HINT_SAMPLE_RATE(prh.HintDescriptor)) {
			lower = prh.LowerBound * plugin.engine().frame_rate();
		} else {
			lower = prh.LowerBound;
		}
	} else {
		lower = 0;
	}

	if (LADSPA_IS_HINT_BOUNDED_ABOVE(prh.HintDescriptor)) {
		if (LADSPA_IS_HINT_SAMPLE_RATE(prh.HintDescriptor)) {
			upper = prh.UpperBound * plugin.engine().frame_rate();
		} else {
			upper = prh.UpperBound;
		}
	} else {
		upper = 4; /* completely arbitrary */
	}
	
	range = upper - lower;
}

void
Plugin::MIDIPortControl::set_value (float value)
{
	if (toggled) {
		if (value > 0.5) {
			value = 1.0;
		} else {
			value = 0.0;
		}
	} else {
		value = lower + (range * value);
		
		if (logarithmic) {
			value = exp(value);
		}
	}

	plugin.set_control_port (absolute_port, value);
}

list<string>
Plugin::get_ladspa_settings()
{
	string presets_path = Config->get_user_ardour_path();
	string path(compose("%1/presets/%2/", presets_path, unique_id()));

	if (path.length() > PATH_MAX) {
		warning << compose(_("%1 : longer than maximum path limit.  Trunicating"), path) << endmsg;
		path = path.substr(0, PATH_MAX);
	}

	DIR* dp;
	list<string> settings;

	if ((dp = opendir(path.c_str()))){
		string file;
		struct dirent* ep;

		while ((ep = readdir(dp))){
			if(ep->d_name[0] == '.')	// skip dot files (., .., .file) 
				continue;
			file = ep->d_name;
			settings.push_back(string(ep->d_name, 0, file.length() - 4));
		}
		closedir (dp);
	}

	return settings;
}

bool
Plugin::save_preset (string name)
{
	DIR* dp;

	string presets_path = Config->get_user_ardour_path();
	string path(compose("%1/presets/", presets_path));

	if (path.length() > PATH_MAX) {
		warning << compose(_("%1 : longer than maximum path limit.  Trunicating"), path) << endmsg;
		path = path.substr(0, PATH_MAX);
	}

	if ((dp = opendir (path.c_str()))){
		closedir (dp);
	} else {
		if(mkdir (path.c_str(), 
					S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)<0) {
			error << compose(_("Could not create LADSPA presets directory \"%1\" (%2)"), path, strerror (errno)) << endmsg;
			return false;
		}
	}

	path = compose("%1/presets/%2/", presets_path, unique_id());
	if (path.length() > PATH_MAX) {
		warning << compose(_("%1 : longer than maximum path limit.  Trunicating"), path) << endmsg;
		path = path.substr(0, PATH_MAX);
	}

	if ((dp = opendir (path.c_str()))) {
		closedir (dp);
	} else {
		if(mkdir(path.c_str(), 
					S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)<0){
			error << compose(_("LADSPA Plugin: Could not create preset directory \"%1\" (%2)"), path, strerror (errno)) << endmsg;
			return false;
		}
	}
	
	path = compose("%1/presets/%2/%3.xml", presets_path, unique_id(), name);
	if (path.length() > PATH_MAX) {
		warning << compose(_("%1 : longer than maximum path limit.  Trunicating"), path) << endmsg;
		path = path.substr(0, PATH_MAX);
	}

	XMLTree tree (path);
	tree.set_root(&get_state());

	if (tree.write()){
		return true;
	} else {
		error << compose(_("LADSPA Plugin: preset file not saved! (%1)"), strerror (errno)) << endmsg;
		return false;
	}
}

string
Plugin::describe_parameter (unsigned long which)
{
	if (which < port_count()) {
		return port_names()[which];
	} else {
		return "??";
	}
}

void
Plugin::reset_midi_control (MIDI::Port* port, bool on)
{
	MIDI::channel_t chn;
	MIDI::eventType ev;
	MIDI::byte extra;

	if (!on) {
		port = 0;
	}

	for (vector<MIDIPortControl*>::iterator i = midi_controls.begin(); i != midi_controls.end(); ++i) {
		if (*i == 0)
			continue;
		(*i)->get_control_info (chn, ev, extra);
		(*i)->midi_rebind (port, chn);
	}
}

jack_nframes_t
Plugin::latency () const
{
	if (latency_control_port) {
		return (jack_nframes_t) floor (*latency_control_port);
	} else {
		return 0;
	}
}

set<unsigned long>
Plugin::automatable () const
{
	set<unsigned long> ret;

	for (unsigned long i = 0; i < port_count(); ++i){
		if (LADSPA_IS_PORT_INPUT(port_descriptors()[i]) && 
		    LADSPA_IS_PORT_CONTROL(port_descriptors()[i])){
			
			ret.insert (ret.end(), i);
		}
	}

	return ret;
}
