/*
    Copyright (C) 2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: mtc_slave.cc,v 1.7 2003/08/06 16:05:22 pbd Exp $
*/

#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <unistd.h>
#include <pbd/error.h>
#include <pbd/failed_constructor.h>
#include <pbd/pthread_utils.h>

#include <midi++/port.h>
#include <ardour/slave.h>
#include <ardour/session.h>
#include <ardour/audioengine.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;
using namespace MIDI;

MTC_Slave::MTC_Slave (Session& s, MIDI::Port& p) 
	: session (s),
	  port (p)
{
	last_inbound_mtc_frame = 0;
	mtc_speed_counter = 0;
	memset (mtc_speed_accumulator, 0, sizeof (mtc_speed_accumulator));

	port.input()->mtc_time.connect (slot (*this, &MTC_Slave::update_mtc_time));
	port.input()->mtc_status.connect (slot (*this, &MTC_Slave::update_mtc_status));

	/* if this is not the same port as the MMC port, or we're not
	   using MMC control, then we need a thread to watch for 
	   data arriving there.
	*/

	if (session.mmc_port() != &port || !session.get_mmc_control()) {

		pthread_mutex_init (&thread_lock, 0);
		pthread_cond_init (&thread_running, 0);
		
		pthread_mutex_lock (&thread_lock);
		thread_ok = false;
		
		if (pthread_create_and_store ("MTC slave", &thread, 0, _thread_work, this)) {
			error << _("MTC Slave: cannot start thread") << endmsg;
			throw failed_constructor();
		}
		
		pthread_cond_wait (&thread_running, &thread_lock);
		pthread_mutex_unlock (&thread_lock);
	}
}

MTC_Slave::~MTC_Slave()
{
	pthread_cancel_one (thread);
}

void *
MTC_Slave::_thread_work (void *arg)
{
	return reinterpret_cast<MTC_Slave *> (arg)->thread_work ();
}

void *
MTC_Slave::thread_work ()
{
	struct pollfd pfd[1];

	/* XXX it would be good for this to be SCHED_FIFO */

	pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, 0);
	pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, 0);

	pthread_mutex_lock (&thread_lock);
	thread_ok = true;
	pthread_cond_signal (&thread_running);
	pthread_mutex_unlock (&thread_lock);

	while (1) {
		
		pfd[0].fd = port.selectable();
		pfd[0].events = POLLIN|POLLHUP|POLLERR;
		
		if (poll (pfd, 1, 1000) < 0) {
			error << compose(_("MTC Slave: poll error on MIDI port (%1)"), strerror (errno)) << endmsg;
			break;
		}

		if (pfd[0].revents & ~(POLLIN)) {
			error << _("MTC Slave: MIDI port is unreadable.") << endmsg;
			break;
		}

		if (pfd[0].revents & POLLIN) {
			byte buf[512];
	
			/* reading from the MIDI port activates the Parser
			   that in turn generates signals that we care
			   about. the port is already set to NONBLOCK so that
			   can read freely here.
			*/
			
			while (1) {
				
				int nread = port.read (buf, sizeof (buf));
				
				if (nread > 0) {
					if ((size_t) nread < sizeof (buf)) {
						break;
					} else {
						continue;
					}
				} else if (nread == 0) {
					break;
				} else if (errno == EAGAIN) {
					break;
				} else {
					fatal << compose(_("Error reading from MIDI Timecode port %1"), port.name()) << endmsg;
					/*NOTREACHED*/
				}
			}
		}
	}

	thread_ok = false;
	return (void *) -1;
}

void
MTC_Slave::update_mtc_time (const byte *msg, bool was_full)
{
	jack_nframes_t now = session.engine().frame_time();
	jack_nframes_t frame_rate = session.frame_rate();
	jack_nframes_t mtc_frame;

	mtc_frame = (msg[3] * 60 * 60 * frame_rate) + 
		(msg[2] * 60 * frame_rate) +
		(msg[1] * frame_rate) +
		(msg[0] * session.frames_per_smpte_frame());

	// fprintf (stderr, "MTC: %02d:%02d:%02d:%02d\n", msg[3], msg[2], msg[1], msg[0]);
	
	if (was_full) {

		mtc_speed_counter = 0;
		memset (mtc_speed_accumulator, 0, sizeof (mtc_speed_accumulator));

		current.guard1++;
		current.timestamp = 0;
		current.guard2++;

		session.request_locate (mtc_frame, false);

	} else {

		if (current.timestamp) {

			jack_nframes_t mtc_delta;
			jack_nframes_t wall_clock_frames;
			float mtc_apparent_speed;

			if (mtc_frame > last_inbound_mtc_frame) {
				mtc_delta = mtc_frame - last_inbound_mtc_frame;
			} else {
				mtc_delta = last_inbound_mtc_frame - mtc_frame;
			}
			
			wall_clock_frames = now - current.timestamp;
			
			if ((mtc_apparent_speed = (float) mtc_delta / wall_clock_frames) > 0.0f) {
				
				/* stuff the current speed into the MTC speed accumulator.
				   we'll average later.
				*/
				
				mtc_speed_accumulator[mtc_speed_counter++] = mtc_apparent_speed;
				
				if (mtc_speed_counter >= mtc_accumulator_size) {
					
					float total = 0;
					
					for (int i = 0; i < mtc_accumulator_size; ++i) {
						total += mtc_speed_accumulator[i];
					}
					
					mtc_speed = total / mtc_accumulator_size;
					
					cerr << "MTC Speed: " << mtc_speed << endl;
					
					mtc_speed_counter = 0;
					memset (mtc_speed_accumulator, 0, sizeof (mtc_speed_accumulator));
				}
			}
		}
		
		current.guard1++;
		// XXX need memory barrier here
		current.mtc.frames = msg[0];
		current.mtc.seconds = msg[1];
		current.mtc.minutes = msg[2];
		current.mtc.hours = msg[3];
		current.timestamp = now;
		// XXX need memory barrier here
		current.guard2++;
		
		last_inbound_mtc_frame = mtc_frame;
	}
}

void
MTC_Slave::update_mtc_status (MIDI::Parser::MTC_Status status)
{
	cerr << "received new MTC status\n";

	switch (status) {
	case MTC_Stopped:
		cerr << "MTC stopped ...\n";
		session.request_stop ();
		break;

	case MTC_Forward:
		cerr << "MTC starts forward ...\n";
		session.request_roll ();
		break;

	case MTC_Backward:
		/* backward motion not supported yet */
		break;
	}
}

void
MTC_Slave::read_current (SafeTime *st) const
{
	int tries = 0;
	do {
		if (tries == 10) {
			error << _("MTC Slave: atomic read of current time failed, sleeping!") << endmsg;
			usleep (20);
			tries = 0;
		}

		*st = current;
		tries++;

	} while (st->guard1 != st->guard2);
}

jack_nframes_t
MTC_Slave::current_audio_frame () const
{
	SafeTime last;
	jack_nframes_t now = session.engine().frame_time();
	jack_nframes_t frame_rate = session.frame_rate();
	unsigned long elapsed;

	read_current (&last);

	/* scale elapsed time by the current MTC speed */

	elapsed = (unsigned long) floor (mtc_speed * (now - last.timestamp));

	/* now add the most recent timecode value plus the estimated elapsed interval */

	return elapsed +(last.mtc.hours * 60 * 60 * frame_rate) + 
		(last.mtc.minutes * 60 * frame_rate) +
		(last.mtc.seconds * frame_rate) +
		(last.mtc.frames * session.frames_per_smpte_frame());
}

bool
MTC_Slave::locked () const
{
	return port.input()->mtc_locked();
}

bool
MTC_Slave::ok() const
{
	return thread_ok;
}

