/*
    Copyright (C) 2004 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: normalize.cc,v 1.4 2004/03/01 02:39:17 pauld Exp $
*/

#include <cmath>

#include <pbd/basename.h>

#include <ardour/types.h>
#include <ardour/normalize.h>
#include <ardour/filesource.h>
#include <ardour/session.h>
#include <ardour/audioregion.h>

#include "i18n.h"

using namespace ARDOUR;

Normalize::Normalize (Session& s)
	: AudioFilter (s)
{
}

Normalize::~Normalize ()
{
}

int
Normalize::run (AudioRegion& region)
{
	AudioRegion::SourceList nsrcs;
	AudioRegion::SourceList::iterator si;
	const jack_nframes_t blocksize = 256 * 1048;
	Sample buf[blocksize];
	jack_nframes_t fpos;
	jack_nframes_t fend;
	jack_nframes_t to_read;
	int ret = -1;
	float scale;
	double maxamp = 0;

	/* create new sources */

	if (make_new_sources (region, nsrcs)) {
		goto out;
	}

	fpos = region.start();
	fend = region.start() + region.length();

	/* first pass: find max amplitude */

	while (fpos < fend) {

		unsigned int n;

		to_read = min (region.source(0).length() - fpos, blocksize);

		for (n = 0, si = nsrcs.begin(); n < region.n_channels(); ++n, ++si) {

			/* read it in */
			
			if (region.source (n).read (buf, fpos, to_read) != to_read) {
				goto out;
			}
			
			/* check maxamp */
			
			for (jack_nframes_t i = 0; i < to_read; ++i) {
				maxamp = max (maxamp, (double) fabs (buf[i]));
			}
		}

		fpos += to_read;
	};

	/* compute scale factor */

	scale = 1.0f / maxamp;

	/* second pass: scale and write out */

	fpos = region.start();
	fend = region.start() + region.length();

	while (fpos < fend) {

		unsigned int n;

		to_read = min (region.source(0).length() - fpos, blocksize);

		for (n = 0, si = nsrcs.begin(); n < region.n_channels(); ++n, ++si) {

			/* read it in */
			
			if (region.source (n).read (buf, fpos, to_read) != to_read) {
				goto out;
			}
			
			/* scale */
			
			for (jack_nframes_t i = 0; i < to_read; ++i) {
				buf[i] *= scale;
			}

			if ((*si)->write (buf, to_read) != to_read) {
				goto out;
			}
		}

		fpos += to_read;
	};

	ret = finish (region, nsrcs);

  out:

	if (ret) {
		for (si = nsrcs.begin(); si != nsrcs.end(); ++si) {
			(*si)->mark_for_remove ();
			delete *si;
		}
	}

	return ret;
}
