/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: send.cc,v 1.25 2004/02/29 23:33:56 pauld Exp $
*/

#include <algorithm>

#include <pbd/xml++.h>

#include <ardour/send.h>
#include <ardour/session.h>

#include "i18n.h"

using namespace ARDOUR;

Send::Send (Session& s, Redirect::Placement p)
	: Redirect (s, s.next_send_name(), p)
{
	_metering = false;
	save_state (_("initial state"));
	 RedirectCreated (this); /* EMIT SIGNAL */
}

Send::Send (Session& s, const XMLNode& node)
	: Redirect (s,  "send", Redirect::PreFader)
{
	_metering = false;

	if (set_state (node)) {
		throw failed_constructor();
	}

	save_state (_("initial state"));
	 RedirectCreated (this); /* EMIT SIGNAL */
}

Send::Send (const Send& other)
	: Redirect (other._session, other._session.next_send_name(), other.placement())
{
	_metering = false;
	save_state (_("initial state"));
	 RedirectCreated (this); /* EMIT SIGNAL */
}

Send::~Send ()
{
	GoingAway (this);
}

XMLNode&
Send::get_state(void)
{
	XMLNode *node = new XMLNode("Send");
	node->add_child_nocopy (Redirect::get_state());
	return *node;
}

int
Send::set_state(const XMLNode& node)
{
	XMLNodeList nlist = node.children();
	XMLNodeIterator niter;
	
	for (niter = nlist.begin(); niter != nlist.end(); ++niter) {
		if ((*niter)->name() == Redirect::state_node_name) {
			Redirect::set_state (**niter);
			break;
		}
	}

	if (niter == nlist.end()) {
		error << _("XML node describing a send is missing a Redirect node") << endmsg;
		return -1;
	}

	return 0;
}

void
Send::run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset)
{
	if (active()) {

		if (_metering) {
			unsigned long n;

			if (_gain == 0) {

				for (n = 0; n < n_outputs(); ++n) {
					_peak_power[n] = 0;
				} 

			} else {

				vector<Sample*>::iterator bufiter;
				size_t limit = 0;

				/* pans and peak_power arrays are always sized to match
				   the number of outputs.
				*/

				limit = min ((size_t) n_outputs(), bufs.size());
				
				for (n = 0, bufiter = bufs.begin(); n < limit; ++bufiter, ++n) {
					compute_peak (*bufiter, n, nframes, _peak_power[n], _pans[n] * _gain);
				}
			}
		}

		IO::deliver_output (bufs, nframes, offset);

	} else {
		silence (nframes, offset);
	}
}

void
Send::set_metering (bool yn)
{
	_metering = yn;

	if (!_metering) {
		/* XXX possible thread hazard here */
		reset_peak_meters ();
	}
}
