/*
    Copyright (C) 1999-2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: session_butler.cc,v 1.14 2004/02/20 03:14:33 pauld Exp $
*/

#include <algorithm>
#include <string>

#include <cmath>
#include <cerrno>
#include <unistd.h>

#include <ardour/timestamps.h>

#include <pbd/error.h>
#include <pbd/lockmonitor.h>
#include <pbd/pthread_utils.h>

#include <ardour/configuration.h>
#include <ardour/audioengine.h>
#include <ardour/session.h>
#include <ardour/diskstream.h>
#include <ardour/crossfade.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;

static float _read_data_rate;
static float _write_data_rate;

/* XXX put this in the right place */

static inline unsigned int next_power_of_two (unsigned int n)
{
	--n;
	n |= n >> 16;
	n |= n >> 8;
	n |= n >> 4;
	n |= n >> 2;
	n |= n >> 1;
	++n;
	return n;
}

/*---------------------------------------------------------------------------
 BUTLER THREAD 
 ---------------------------------------------------------------------------*/

int
Session::start_butler_thread ()
{
	/* Get a large enough buffer, and ensure that it is an exact
	   multiple of the frames_per_cycle size. This will guarantee
	   that we never, ever have to deal with a split buffer when
	   reading or writing into it from the audio thread (unless
	   running with varispeed, and thats handled as a special
	   case.
	*/

	dstream_buffer_size = 3 * (Config->get_minimum_disk_io() / sizeof (Sample));
	dstream_buffer_size = next_power_of_two (dstream_buffer_size);

	Crossfade::set_buffer_size (dstream_buffer_size);

	pthread_cond_init (&butler_request, 0);
	pthread_cond_init (&butler_paused, 0);
	
	butler_should_run = false;

	if (pthread_create_and_store ("disk butler", &butler_thread, 0, _butler_thread_work, this)) {
		error << _("Session: could not create butler thread") << endmsg;
		return -1;
	}

	pthread_detach (butler_thread);

	return 0;
}

void
Session::terminate_butler_thread ()
{
	pthread_cancel_one (butler_thread);
}

void
Session::schedule_butler_transport_work ()
{
	atomic_inc (&butler_should_do_transport_work);
	summon_butler ();
}

void
Session::schedule_curve_reallocation ()
{
	post_transport_work = PostTransportWork (post_transport_work | PostTransportCurveRealloc);
	schedule_butler_transport_work ();
}

void
Session::summon_butler ()
{
	LockMonitor lm (butler_request_lock, __LINE__, __FILE__);
	butler_should_run = true;
	pthread_cond_signal (&butler_request);
}

void
Session::stop_butler ()
{
	LockMonitor lm (butler_request_lock, __LINE__, __FILE__);
	butler_should_run = false;
	pthread_cond_signal (&butler_request);
	pthread_cond_wait (&butler_paused, butler_request_lock.mutex());
	butler_should_run = true;
}

void
Session::wait_till_butler_finished ()
{
	LockMonitor lm (butler_request_lock, __LINE__, __FILE__);
	pthread_cond_signal (&butler_request);
	pthread_cond_wait (&butler_paused, butler_request_lock.mutex());
}

void *
Session::_butler_thread_work (void* arg)
{
	pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, 0);
	pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, 0);

	((Session *) arg)->butler_thread_work ();
	return 0;
}

#define transport_work_requested() atomic_read(&butler_should_do_transport_work)

void
Session::butler_thread_work ()
{
	bool more_work;
	unsigned int err = 0;
	size_t bytes;
	bool compute_io;
	struct timeval begin, end;

	butler_mixdown_buffer = new Sample[DiskStream::disk_io_frames()];
	butler_gain_buffer = new gain_t[DiskStream::disk_io_frames()];

	/* wait for the first request to run */

	{ 
		LockMonitor lm (butler_request_lock, __LINE__, __FILE__);
		
		while (!butler_should_run) {
			pthread_cond_signal (&butler_paused);
			pthread_cond_wait (&butler_request, butler_request_lock.mutex());
		}
	}

	/* now loop forever. note: we do not hold the lock while doing this - check the scope of lm above */

	while (1) {

//		for (DiskStreamList::iterator i = diskstreams.begin(); !transport_work_requested() && butler_should_run && i != diskstreams.end(); ++i) {
//			cerr << "BEFORE " << (*i)->name() << ": pb = " << (*i)->playback_buffer_load() << " cp = " << (*i)->capture_buffer_load() << endl;
//		}

		if (transport_work_requested()) {
			butler_transport_work ();
		}

		more_work = false;
		bytes = 0;
		compute_io = true;

		gettimeofday (&begin, 0);

		for (DiskStreamList::iterator i = diskstreams.begin(); !transport_work_requested() && butler_should_run && i != diskstreams.end(); ++i) {
			
			// cerr << "rah for " << (*i)->name () << endl;

			switch ((*i)->do_refill (butler_mixdown_buffer, butler_gain_buffer)) {
			case 0:
				bytes += (*i)->read_data_count();
				break;
			case 1:
				bytes += (*i)->read_data_count();
				more_work = true;
				break;
				
			default:
				compute_io = false;
				error << compose(_("Butler read ahead failure on dstream %1"), (*i)->name()) << endmsg;
				break;
			}
		}
		
		if (!err && transport_work_requested()) {
			continue;
		}

		if (compute_io) {
			gettimeofday (&end, 0);
			
			double b = begin.tv_sec  + (begin.tv_usec/1000000.0);
			double e = end.tv_sec + (end.tv_usec / 1000000.0);
			
			_read_data_rate = bytes / (e - b);
		}

		bytes = 0;
		compute_io = true;
		gettimeofday (&begin, 0);

		for (DiskStreamList::iterator i = diskstreams.begin(); !transport_work_requested() && butler_should_run && i != diskstreams.end(); ++i) {
			
			// cerr << "write behind for " << (*i)->name () << endl;
			
			switch ((*i)->do_flush ()) {
			case 0:
				bytes += (*i)->write_data_count();
				break;
			case 1:
				bytes += (*i)->write_data_count();
				more_work = true;
				break;
				
			default:
				err++;
				compute_io = false;
				error << compose(_("Butler write-behind failure on dstream %1"), (*i)->name()) << endmsg;
				/* don't break - try to flush all streams in case they 
				   are split across disks.
				*/
			}
		}

		if (err && actively_recording()) {
			/* stop the transport and try to catch as much possible
			   captured state as we can.
			*/
			request_stop ();
		}

		if (!err && transport_work_requested()) {
			continue;
		}


		if (compute_io) {
			gettimeofday (&end, 0);
			
			double b = begin.tv_sec  + (begin.tv_usec/1000000.0);
			double e = end.tv_sec + (end.tv_usec / 1000000.0);
			
			_write_data_rate = bytes / (e - b);
		}
		
		if (!more_work) {
			refresh_disk_space ();
		}

		{
			LockMonitor lm (butler_request_lock, __LINE__, __FILE__);

			if (butler_should_run && (more_work || transport_work_requested())) {
				continue;
			}

//			for (DiskStreamList::iterator i = diskstreams.begin(); !transport_work_requested() && butler_should_run && i != diskstreams.end(); ++i) {
//				cerr << "AFTER " << (*i)->name() << ": pb = " << (*i)->playback_buffer_load() << " cp = " << (*i)->capture_buffer_load() << endl;
//			}

			pthread_cond_signal (&butler_paused);
			pthread_cond_wait (&butler_request, butler_request_lock.mutex());
		}
	}
}


void
Session::request_overwrite_buffer (DiskStream* stream)
{
	Event *ev = new Event (Event::Overwrite, Event::Add, Event::Immediate, 0, 0, 0.0);
	ev->set_ptr (stream);
	queue_event (ev);
}

void
Session::overwrite_some_buffers (DiskStream* ds)
{
	/* executed by the audio thread */

	if (actively_recording()) {
		return;
	}

	if (ds) {

		ds->set_pending_overwrite (true);

	} else {

		LockMonitor dm (diskstream_lock, __LINE__, __FILE__);
		for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
			(*i)->set_pending_overwrite (true);
		}
	}

	post_transport_work = PostTransportWork (post_transport_work | PostTransportOverWrite);
	schedule_butler_transport_work ();
}

float
Session::read_data_rate () const
{
	/* disk i/o in excess of 10000MB/sec indicate the buffer cache
	   in action. ignore it.
	*/
	return _read_data_rate > 10485760000.0f ? 0.0f : _read_data_rate;
}

float
Session::write_data_rate () const
{
	/* disk i/o in excess of 10000MB/sec indicate the buffer cache
	   in action. ignore it.
	*/
	return _write_data_rate > 10485760000.0f ? 0.0f : _write_data_rate;
}

float
Session::playback_load () 
{
	LockMonitor lm (diskstream_lock, __LINE__, __FILE__);
	float worst = 1.0f;

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		if (!(*i)->hidden()) {
			worst = min (worst, (*i)->playback_buffer_load());
		}
	}

	return worst;
}

float
Session::capture_load () 
{
	LockMonitor lm (diskstream_lock, __LINE__, __FILE__);
	float worst = 1.0f;

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		if (!(*i)->hidden()) {
			worst = min (worst, (*i)->capture_buffer_load());
		}
	}

	return worst;
}
