/*
    Copyright (C) 1999-2003 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: session_transport.cc,v 1.51 2004/02/29 23:33:56 pauld Exp $
*/

#include <cmath>
#include <cerrno>
#include <unistd.h>

#include <pbd/error.h>
#include <pbd/lockmonitor.h>
#include <pbd/pthread_utils.h>

#include <midi++/mmc.h>
#include <midi++/port.h>

#include <ardour/ardour.h>
#include <ardour/audioengine.h>
#include <ardour/session.h>
#include <ardour/diskstream.h>
#include <ardour/auditioner.h>
#include <ardour/slave.h>
#include <ardour/location.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;

void
Session::request_input_change_handling ()
{
	Event* ev = new Event (Event::InputConfigurationChange, Event::Add, Event::Immediate, 0, 0.0);
	queue_event (ev);
}

void
Session::request_slave_source (SlaveSource src, jack_nframes_t pos)
{
	Event* ev = new Event (Event::SetSlaveSource, Event::Add, Event::Immediate, pos, 0.0);
	ev->slave = src;
	queue_event (ev);
}

void
Session::request_transport_speed (float speed)
{
	Event* ev = new Event (Event::SetTransportSpeed, Event::Add, Event::Immediate, 0, speed);
	queue_event (ev);
}

void
Session::request_diskstream_speed (DiskStream& ds, float speed)
{
	Event* ev = new Event (Event::SetDiskstreamSpeed, Event::Add, Event::Immediate, 0, speed);
	ev->set_ptr (&ds);
	queue_event (ev);
}

void
Session::request_roll ()
{
	if (_play_range) {
		if (!current_audio_range.empty()) {
			request_locate (current_audio_range.front().start, true);
		} else if (!current_music_range.empty()) {
			warning << _("can't handle auto play of musical ranges yet") << endmsg;
		}
		return;
	}
	else if (auto_loop) {
		Location * loc = _locations.auto_loop_location();
		if (loc && (_transport_frame < loc->start() || _transport_frame > loc->end())) {
			// relocate to beginning of loop
			request_locate (loc->start(), true);
			return;
		}
	}
	
	Event* ev = new Event (Event::SetTransportSpeed, Event::Add, Event::Immediate, 0, 1.0);
	queue_event (ev);
}

void
Session::request_stop (bool abort)
{
	Event* ev = new Event (Event::SetTransportSpeed, Event::Add, Event::Immediate, 0, 0.0, abort);
	queue_event (ev);
}

void
Session::request_locate (jack_nframes_t target_frame, bool with_roll)
{
	Event *ev = new Event (with_roll ? Event::LocateRoll : Event::Locate, Event::Add, Event::Immediate, target_frame, 0, true);
	queue_event (ev);
}

void
Session::request_auto_loop (bool yn)
{
	Event* ev;	

	Location *location = _locations.auto_loop_location();
	if (yn && location && transport_rolling()
	    && (_transport_frame < location->start() || _transport_frame > location->end()))
	{
		ev = new Event (Event::LocateRoll, Event::Add, Event::Immediate, location->start(), 0, true);
		queue_event (ev);
	}
	
	ev = new Event (Event::SetLoop, Event::Add, Event::Immediate, 0, 0.0, yn);
	queue_event (ev);
}

void
Session::realtime_stop (bool abort)
{
	/* assume that when we start, we'll be moving forwards */

	if (_transport_speed < 0.0f) {
		post_transport_work = PostTransportWork (post_transport_work | PostTransportStop | PostTransportReverse);
	} else {
		post_transport_work = PostTransportWork (post_transport_work | PostTransportStop);
	}

	if (actively_recording()) {

		post_transport_work = PostTransportWork (post_transport_work | PostTransportDidRecord);

		/* move the transport position back to where the
		   request for a stop was noticed. we rolled
		   past that point to pick up delayed input.
		*/

		decrement_transport_position (_worst_output_latency);

		if (_transport_frame > current_end) {
			current_end = _transport_frame;
			post_transport_work = PostTransportWork (post_transport_work | PostTransportDuration);
		}
	}

	if (abort) {
		post_transport_work = PostTransportWork (post_transport_work | PostTransportAbort);
	}

	clear_event_type (Event::StopOnce);
	if (_transport_speed != 0.0f) {
		_last_transport_speed = _transport_speed;
	}
	_transport_speed = 0;
	transport_sub_state = (auto_return ? AutoReturning : 0);
}

void
Session::butler_transport_work ()
{
	if (post_transport_work & PostTransportCurveRealloc) {
		for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {
			(*i)->curve_reallocate();
		}
	}

	if (post_transport_work & PostTransportInputChange) {
		for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
			(*i)->non_realtime_input_configuration_change ();
		}
	}

	if (post_transport_work & PostTransportSpeed) {
		non_realtime_set_speed ();
	}

	if (post_transport_work & PostTransportReverse) {

		clear_clicks();
		cumulative_rf_motion = 0;
		reset_rf_scale (0);

		for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
			if (!(*i)->hidden()) {
				if ((*i)->speed() != 1.0f || (*i)->speed() != -1.0f) {
					(*i)->seek ((jack_nframes_t) (_transport_frame * (double) (*i)->speed()));
				}
				else {
					(*i)->seek (_transport_frame);
				}
			}
		}
	}

	if (post_transport_work & PostTransportScrub) {
		non_realtime_start_scrub ();
	}

	if (post_transport_work & (PostTransportStop|PostTransportLocate)) {
		non_realtime_stop (post_transport_work & PostTransportAbort);
	}

	if (post_transport_work & PostTransportOverWrite) {
		non_realtime_overwrite ();
	}

	if (post_transport_work & PostTransportAudition) {
		non_realtime_set_audition ();
	}

	atomic_dec (&butler_should_do_transport_work);
}

void
Session::non_realtime_start_scrub ()
{
	LockMonitor lm (diskstream_lock, __LINE__, __FILE__);

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		if (!(*i)->hidden()) {
			(*i)->start_scrub (pending_locate_frame);
		}
	}

	 ScrubReady (); /* EMIT SIGNAL */
}

void
Session::non_realtime_set_speed ()
{
	LockMonitor lm (diskstream_lock, __LINE__, __FILE__);

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		(*i)->non_realtime_set_speed ();
	}
}

void
Session::non_realtime_overwrite ()
{
	LockMonitor lm (diskstream_lock, __LINE__, __FILE__);

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		if ((*i)->pending_overwrite) {
			(*i)->overwrite_existing_buffers ();
		}
	}
}

void
Session::non_realtime_stop (bool abort)
{
	struct tm* now;
	time_t     xnow;
	bool       did_record = (atomic_read (&_record_status) != Disabled);

	/* stop and locate are merged here because they share a lot of common stuff */
	
	time (&xnow);
	now = localtime (&xnow);

	if (auditioner) {
		auditioner->cancel_audition ();
	}

	clear_clicks();
	cumulative_rf_motion = 0;
	reset_rf_scale (0);
	
	if (did_record) {
		_have_captured = true;
		begin_reversible_command ("capture");
	}

	{ 
		LockMonitor lm (diskstream_lock, __LINE__, __FILE__);
		for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
			(*i)->transport_stopped (*now, xnow, abort);
		}
	}

	if (did_record) {
		commit_reversible_command ();
	}

	
	{ 
		LockMonitor lm (route_lock, __LINE__, __FILE__);
		_worst_track_latency = 0;

		for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {
			(*i)->transport_stopped (abort, (post_transport_work & PostTransportLocate), 
						 (!(post_transport_work & PostTransportLocate) || pending_locate_flush));

			jack_nframes_t track_latency = (*i)->update_total_latency ();

			if (!(*i)->hidden() && ((*i)->active())) {
				_worst_track_latency = max (_worst_track_latency, track_latency);
			}
		}

		for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {
			(*i)->set_latency_delay (_worst_track_latency);
		}
	}

	if (auto_return || (post_transport_work & PostTransportLocate)) {

		if (pending_locate_flush) {
			flush_all_redirects ();
		}

		if (auto_return && !(post_transport_work & PostTransportLocate)) {
			_transport_frame = last_stop_frame;
		} 

		for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
			if (!(*i)->hidden()) {
				if ((*i)->speed() != 1.0f || (*i)->speed() != -1.0f) {
					(*i)->seek ((jack_nframes_t) (_transport_frame * (double) (*i)->speed()));
				}
				else {
					(*i)->seek (_transport_frame);
				}
			}
		}

		deliver_mmc (MIDI::MachineControl::cmdLocate, _transport_frame);
		_engine.transport_locate (_transport_frame);
	}

	last_stop_frame = _transport_frame;

	send_full_time_code ();
	deliver_mmc (MIDI::MachineControl::cmdStop, 0);
	_engine.transport_stop ();

	if (did_record) {
		atomic_set (&_record_status, Disabled);
		 RecordDisabled (); /* EMIT SIGNAL */
	}
	
	/* save the current state of things if appropriate */
	
	if (post_transport_work & PostTransportDidRecord) {
		save_state (_current_snapshot_name);
	}

	if (post_transport_work & PostTransportDuration) {
		 DurationChanged (); /* EMIT SIGNAL */
	}

	 PositionChanged (_transport_frame); /* EMIT SIGNAL */
	 TransportStateChange (); /* EMIT SIGNAL */
}

void
Session::check_declick_out ()
{
	bool locate_required = transport_sub_state & PendingLocate;

	/* this is called after a process() iteration. if PendingDeclickOut was set,
	   it means that we were waiting to declick the output (which has just been
	   done) before doing something else. this is where we do that "something else".
	   
	   note: called from the audio thread.
	*/

	if (transport_sub_state & PendingDeclickOut) {

		if (locate_required) {
			locate (pending_locate_frame, pending_locate_roll, pending_locate_flush);
			transport_sub_state &= ~(PendingDeclickOut|PendingLocate);
		} else {
			transport_sub_state &= ~(PendingDeclickOut|PendingLocate);
			stop_transport (pending_abort);
		}
	}
}

void
Session::set_auto_loop (bool yn)
{
	if (auto_loop == yn || actively_recording() || _locations.auto_loop_location() == 0) {
		return;
	}
	
	set_dirty();

	if ((auto_loop = yn)) {

		Location *loc;
		
		if ((loc = _locations.auto_loop_location()) != 0) {

			Event* event = new Event (Event::AutoLoop, Event::Replace, loc->end(), loc->start(), 0.0f);
			merge_event (event);
		}

	} else {
		clear_event_type (Event::AutoLoop);
	}

	 ControlChanged (AutoLoop); /* EMIT SIGNAL */
}

void
Session::flush_all_redirects ()
{
	for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {
		(*i)->flush_redirects ();
	}
}

void
Session::locate (jack_nframes_t target_frame, bool with_roll, bool with_flush)
{
	if (actively_recording()) {
		return;
	}

	if (_transport_frame == target_frame) {
		if (with_roll) {
			set_transport_speed (1.0, false);
		}
		return;
	}

	_transport_frame = target_frame;

	if (_transport_speed) {

		/* schedule a declick. we'll be called again when its done */

		if (!(transport_sub_state & PendingDeclickOut)) {
			transport_sub_state |= (PendingDeclickOut|PendingLocate);
			pending_locate_frame = target_frame;
			pending_locate_roll = with_roll;
			pending_locate_flush = with_flush;
			return;
		} 
	}

	if (transport_rolling() && !auto_play && !with_roll) {
		realtime_stop (false);
	} 

	post_transport_work = PostTransportWork (post_transport_work | PostTransportLocate);

	if (with_roll) {
		post_transport_work = PostTransportWork (post_transport_work | PostTransportRoll);
	}
	
	schedule_butler_transport_work ();
}

void
Session::set_transport_speed (float speed, bool abort)
{
	if (speed > 0) {
		speed = min (8.0f, speed);
	} else if (speed < 0) {
		speed = max (-8.0f, speed);
	}

	if (transport_rolling() && speed == 0.0) {
		stop_transport (abort);
	} else if (transport_stopped() && speed == 1.0) {
		start_transport ();
	} else {

		if (actively_recording()) {
			return;
		}

		if (speed > 0.0f && _transport_frame == current_end) {
			return;
		}

		if (speed < 0.0f && _transport_frame == 0) {
			return;
		}
		
		clear_clicks ();

		/* if we are reversing relative to the current speed, or relative to the speed
		   before the last stop, then we have to do extra work.
		*/

		if ((_transport_speed && speed * _transport_speed < 0.0f) || (_last_transport_speed * speed < 0.0f)) {
			post_transport_work = PostTransportWork (post_transport_work | PostTransportReverse);
		}

		_transport_speed = speed;

		for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
			if ((*i)->realtime_set_speed ((*i)->speed())) {
				post_transport_work = PostTransportWork (post_transport_work | PostTransportSpeed);
			}
		}
		
		if (post_transport_work & (PostTransportSpeed|PostTransportReverse)) {
			schedule_butler_transport_work ();
		}
	}
}

void
Session::stop_transport (bool abort)
{
	if (_transport_speed == 0.0f) {
		return;
	}

	if (actively_recording() && !(transport_sub_state & StopPendingCapture) && 
	    _worst_output_latency > current_block_size) 
	{
		
		/* we need to capture the audio that has still not yet been received by the system
		   at the time the stop is requested, so we have to roll past that time.

		   we want to declick before stopping, so schedule the autostop for one
		   block before the actual end. we'll declick in the subsequent block,
		   and then we'll really be stopped.
		*/
		
		Event *ev = new Event (Event::StopOnce, Event::Replace, 
				       _transport_frame + _worst_output_latency - current_block_size,
				       0, 0, abort);
		
		merge_event (ev);
		transport_sub_state |= StopPendingCapture;
		pending_abort = abort;
		return;
	} 

	if (transport_sub_state & PendingDeclickOut == 0) {
		transport_sub_state |= PendingDeclickOut;
		/* we'll be called again after the declick */
		return;
	}

	realtime_stop (abort);
	schedule_butler_transport_work ();
}

void
Session::start_transport ()
{
	_last_roll_location = _transport_frame;

	/* if record status is Enabled, move it to Recording. if its
	   already Recording, move it to Disabled. 
	*/

	switch (record_status()) {
	case Enabled:
		if (!punch_in) {
			enable_record ();
		}
		break;

	case Recording:
		disable_record ();
		break;

	default:
		break;
	}

	transport_sub_state |= PendingDeclickIn;
	_transport_speed = 1.0;

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		(*i)->realtime_set_speed ((*i)->speed());
	}

	_engine.transport_start ();

	 TransportStateChange (); /* EMIT SIGNAL */
}

void
Session::post_transport ()
{
	if (post_transport_work & PostTransportScrub) {
		process_function = &Session::process_scrub;
	}

	if (post_transport_work & PostTransportAudition) {
		if (auditioner && auditioner->active()) {
			process_function = &Session::process_audition;
		} else {
			process_function = &Session::process_with_events;
		}
	}

	if (post_transport_work & PostTransportStop) {

		transport_sub_state = 0;
	}

	if (post_transport_work & PostTransportLocate) {

		if (auto_play || (post_transport_work & PostTransportRoll)) {
			
			start_transport ();
			
		} else {

			transport_sub_state = 0;
		}
	}

	set_next_event ();

	post_transport_work = PostTransportWork (0);
}


void
Session::set_rf_speed (float speed)
{
	rf_speed = speed;
	cumulative_rf_motion = 0;
	reset_rf_scale (0);
}

void
Session::reset_rf_scale (jack_nframes_t motion)
{
	cumulative_rf_motion += motion;

	if (cumulative_rf_motion < 4 * _current_frame_rate) {
		rf_scale = 1;
	} else if (cumulative_rf_motion < 8 * _current_frame_rate) {
		rf_scale = 4;
	} else if (cumulative_rf_motion < 16 * _current_frame_rate) {
		rf_scale = 10;
	} else {
		rf_scale = 100;
	}

	if (motion != 0) {
		set_dirty();
	}
}

int
Session::set_slave_source (SlaveSource src, jack_nframes_t frame)
{
	bool reverse = false;
	bool non_rt_required = false;

	if (_transport_speed) {
		error << _("please stop the transport before adjusting slave settings") << endmsg;
		/* help out non-MVC friendly UI's by telling them the slave type changed */
		 ControlChanged (SlaveType); /* EMIT SIGNAL */
		return 0;
	}

	if (src == _slave_type) {
		return 0;
	} 
	
	if (_slave) {
		delete _slave;
		_slave = 0;
		_slave_type = None;
	}

	if (_transport_speed < 0.0) {
		reverse = true;
	}

	switch (src) {
	case None:
		stop_transport ();
		break;

	case MTC:
		if (_mtc_port) {
			try {
				_slave = new MTC_Slave (*this, *_mtc_port);
			}
			catch (failed_constructor& err) {
				return -1;
			}
		} else {
			error << _("No MTC port defined: MTC slaving is impossible.") << endmsg;
			return -1;
		}
		_desired_transport_speed = _transport_speed;
		break;
		
	case ADAT:
		_slave = new ADAT_Slave ();
		_desired_transport_speed = _transport_speed;
		break;
		
	case JACK:
		_slave = new JACK_Slave ();
		_desired_transport_speed = _transport_speed;
		break;

	case Scrub:
		_slave = new Scrub_Slave ();
		_desired_transport_speed = 0.0;
		_transport_frame = frame;
		non_rt_required = true;
		process_function = &Session::process_scrub;
		break;
	};
	
	_slave_type = src;

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		if (!(*i)->hidden()) {
			if ((*i)->realtime_set_speed ((*i)->speed())) {
				non_rt_required = true;
			}
			(*i)->set_slaved (_slave);
		}
	}

	if (reverse) {
		reverse_diskstream_buffers ();
	}

	if (non_rt_required) {
		if (src == Scrub) {
			pending_locate_frame = frame;
			post_transport_work = PostTransportWork (post_transport_work | PostTransportScrub);
		} else {
			post_transport_work = PostTransportWork (post_transport_work | PostTransportSpeed);
		}
		schedule_butler_transport_work ();
	}

	set_dirty();
	 ControlChanged (SlaveType); /* EMIT SIGNAL */
	
	return 0;
}

jack_nframes_t
Session::scrub_buffer_size () const
{
	return 2 * frame_rate();
}

void
Session::start_scrub (jack_nframes_t start)
{
	request_slave_source (Scrub, start);
}

void
Session::stop_scrub ()
{
	/* if we're not moving, then move to wherever we stopped. otherwise,
	   move to the last heard frame, as accurately as we can.
	*/

	if (_transport_speed == 0.0) {
		locate (_transport_frame, false, true);
	} else {
		locate (audible_frame(), false, true);
	}

	set_slave_source (None, 0);

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		if (!(*i)->hidden()) {
			(*i)->end_scrub ();
		}
	}

	stop_transport (false);
}

void
Session::set_scrub_speed (float sp)
{
	_desired_transport_speed = sp;
}

void
Session::reverse_diskstream_buffers ()
{
	post_transport_work = PostTransportWork (post_transport_work | PostTransportReverse);
	schedule_butler_transport_work ();
}

void
Session::set_diskstream_speed (DiskStream* stream, float speed)
{
	if (stream->realtime_set_speed (speed)) {
		post_transport_work = PostTransportWork (post_transport_work | PostTransportSpeed);
		schedule_butler_transport_work ();
		set_dirty ();
	}
}

void
Session::set_audio_range (list<AudioRange>& range)
{
	Event *ev = new Event (Event::SetAudioRange, Event::Add, Event::Immediate, 0, 0.0f);
	ev->audio_range = range;
	queue_event (ev);
}

void
Session::set_music_range (list<MusicRange>& range)
{
	Event *ev = new Event (Event::SetMusicRange, Event::Add, Event::Immediate, 0, 0.0f);
	ev->music_range = range;
	queue_event (ev);
}

void
Session::request_play_range (bool yn)
{
	Event* ev = new Event (Event::SetPlayRange, Event::Add, Event::Immediate, 0, 0.0f, yn);
	queue_event (ev);
}

void
Session::set_play_range (bool yn)
{
	if (_play_range != yn) {
		_play_range = yn;
		setup_auto_play ();
		 ControlChanged (PlayRange); /* EMIT SIGNAL */
	}
}

void
Session::setup_auto_play ()
{
	Event* ev;

	clear_event_type (Event::RangeStop);
	clear_event_type (Event::RangeLocate);

	if (_play_range) {
		
		list<AudioRange>::size_type sz = current_audio_range.size();

		if (sz > 1) {

			list<AudioRange>::iterator i = current_audio_range.begin(); 
			list<AudioRange>::iterator next;

			while (i != current_audio_range.end()) {

				next = i;
				++next;

				/* locating/stopping is subject to delays for declicking.
				 */

				jack_nframes_t requested_frame = (*i).end;
				
				if (requested_frame > current_block_size) {
					requested_frame -= current_block_size;
				} else {
					requested_frame = 0;
				}

				if (next == current_audio_range.end()) {
					ev = new Event (Event::RangeStop, Event::Add, requested_frame, 0, 0.0f);
				} else {
					ev = new Event (Event::RangeLocate, Event::Add, requested_frame, (*next).start, 0.0f);
				}

				merge_event (ev);

				i = next;
			}

		} else if (sz == 1) {

			ev = new Event (Event::RangeStop, Event::Add, current_audio_range.front().end, 0, 0.0f);
			merge_event (ev);

		} else if (!current_music_range.empty()) {
			warning << _("can't handle music ranges yet") << endmsg;
		}

	}
}

void
Session::request_bounded_roll (jack_nframes_t start, jack_nframes_t end)
{
	request_stop ();
	Event *ev = new Event (Event::StopOnce, Event::Replace, Event::Immediate, end, 0.0);
	queue_event (ev);
	request_locate (start, true);
}

void
Session::engine_halted ()
{
	/* there will be no more calls to process(), so
	   we'd better clean up for ourselves, right now.

	   but first, make sure the butler is out of 
	   the picture.
	*/

	atomic_set (&butler_should_do_transport_work, 0);
	post_transport_work = PostTransportWork (0);
	stop_butler ();
	
	realtime_stop (false);
	non_realtime_stop (false);
	transport_sub_state = 0;
}


void
Session::xrun_recovery ()
{
	if (Config->get_stop_recording_on_xrun() && actively_recording()) {

		 HaltOnXrun (); /* EMIT SIGNAL */

		/* it didn't actually halt, but we need
		   to handle things in the same way.
		*/

		engine_halted();
	} 
}

int
Session::jack_slave_sync (jack_nframes_t nframes)
{
	jack_transport_info_t info;

	// _engine.get_transport_info (info);

	if (info.frame != _transport_frame + nframes) {
		/* compute the difference then a speed */
	}
	
	return 0;
}
