/*
    Copyright (C) 2003 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: crossfade.cc,v 1.18 2004/12/02 19:27:00 pauld Exp $
*/

#include <sigc++/bind.h>

#include <ardour/crossfade.h>
#include <ardour/crossfade_compare.h>
#include <ardour/audioregion.h>
#include <ardour/playlist.h>
#include <ardour/utils.h>

#include "i18n.h"
#include <locale.h>

using namespace std;
using namespace ARDOUR;
using namespace SigC;

/* XXX if and when we ever implement parallel processing of the process()
   callback, these will need to be handled on a per-thread basis.
*/

Sample* Crossfade::crossfade_buffer_left = 0;
Sample* Crossfade::crossfade_buffer_right = 0;

void
Crossfade::set_buffer_size (jack_nframes_t sz)
{
	if (crossfade_buffer_left) {
		delete [] crossfade_buffer_left;
		crossfade_buffer_left = 0;
	}

	if (crossfade_buffer_right) {
		delete [] crossfade_buffer_right;
		crossfade_buffer_right = 0;
	}

	if (sz) {
		crossfade_buffer_left = new Sample[sz];
		crossfade_buffer_right = new Sample[sz];
	}
}

bool
Crossfade::operator== (const Crossfade& other)
{
	return (_right == other._right) && (_left == other._left);
}

Crossfade::Crossfade (ARDOUR::AudioRegion& r, ARDOUR::AudioRegion& l, bool act)
	: _right (&r), 
	  _left (&l), 
	  _active (act),
	  _fade_in (0.0, 2.0, 1.0), // linear (gain coefficient) => -inf..+6dB
	  _fade_out (0.0, 2.0, 1.0) // linear (gain coefficient) => -inf..+6dB
{
	overlap_type = _right->coverage (_left->first_frame(), _left->last_frame());
	last_length = length();
	_current_state_id = 0;
	_in_update = false;

	fill ();

	_fade_in.StateChanged.connect (slot (*this, &Crossfade::fade_in_changed));
	_fade_out.StateChanged.connect (slot (*this, &Crossfade::fade_out_changed));

	_right->StateChanged.connect (slot (*this, &Crossfade::member_changed));
	_left->StateChanged.connect (slot (*this, &Crossfade::member_changed));

	save_state ("initial");
}

Crossfade::Crossfade (const Playlist& playlist, XMLNode& node)
	:  _fade_in (0.0, 2.0, 1.0), // linear (gain coefficient) => -inf..+6dB
	   _fade_out (0.0, 2.0, 1.0) // linear (gain coefficient) => -inf..+6dB
{
	Region* r;
	XMLProperty* prop;
	id_t id;
	LocaleGuard (X_("POSIX"));

	if ((prop = node.property ("right")) == 0) {
		error << _("Crossfade: no \"right\" region in state") << endmsg;
		throw failed_constructor();
	}
	
	sscanf (prop->value().c_str(), "%llu", &id);

	if ((r = playlist.find_region (id)) == 0) {
		error << compose (_("Crossfade: no right region %1 found in playlist %2"), id, playlist.name())
		      << endmsg;
		throw failed_constructor();
	}
	
	if ((_right = dynamic_cast<AudioRegion*> (r)) == 0) {
		throw failed_constructor();
	}

	if ((prop = node.property ("left")) == 0) {
		error << _("Crossfade: no \"left\" region in state") << endmsg;
		throw failed_constructor();
	}

	sscanf (prop->value().c_str(), "%llu", &id);

	if ((r = playlist.find_region (id)) == 0) {
		error << compose (_("Crossfade: no left region %1 found in playlist %2"), id, playlist.name())
		      << endmsg;
		throw failed_constructor();
	}

	if ((_left = dynamic_cast<AudioRegion*> (r)) == 0) {
		throw failed_constructor();
	}

	overlap_type = _right->coverage (_left->first_frame(), _left->last_frame());
	_active = true;
	last_length = length();
	_current_state_id = 0;
	_in_update = false;

	set_state (node);

	_fade_in.StateChanged.connect (slot (*this, &Crossfade::fade_in_changed));
	_fade_out.StateChanged.connect (slot (*this, &Crossfade::fade_out_changed));

	_right->StateChanged.connect (slot (*this, &Crossfade::member_changed));
	_left->StateChanged.connect (slot (*this, &Crossfade::member_changed));

	save_state ("initial");
}

Crossfade::~Crossfade ()
{
	for (StateMap::iterator i = states.begin(); i != states.end(); ++i) {
		delete *i;
	}
}

void
Crossfade::fill ()
{
	double len = length();

	_fade_out.freeze ();
	_fade_out.clear ();
	_fade_out.add (0.0, 1.0);
	_fade_out.add ((len * 0.1), 0.99);
	_fade_out.add ((len * 0.2), 0.97);
	_fade_out.add ((len * 0.35), 0.8);
	_fade_out.add ((len * 0.5), 0.5);
	_fade_out.add ((len * 0.6), 0.3);
	_fade_out.add ((len * 0.8), 0.03);
	_fade_out.add ((len * 0.9), 0.01);
	_fade_out.add (len, 0.0);
	_fade_out.thaw ();
	
	_fade_in.freeze ();
	_fade_in.clear ();
	_fade_in.add (0.0, 0.0);
	_fade_in.add ((len * 0.1),  0.01);
	_fade_in.add ((len * 0.2),  0.03);
	_fade_in.add ((len * 0.4),  0.3);
	_fade_in.add ((len * 0.5),  0.5);
	_fade_in.add ((len * 0.65), 0.8);
	_fade_in.add ((len * 0.8),  0.97);
	_fade_in.add ((len * 0.9),  0.99);
	_fade_in.add (len, 1.0);
	_fade_in.thaw ();
}

void
Crossfade::refill ()
{
	double factor = length () / (double) last_length;

	_fade_out.x_scale (factor);
	_fade_in.x_scale (factor);
}

jack_nframes_t
Crossfade::length () const
{
	return _left->last_frame() - _right->first_frame();
}

void
Crossfade::update ()
{
	jack_nframes_t newlen = length();
	_in_update = true;

	if (newlen != last_length) {

		refill ();

		last_length = newlen;
		save_state ("updated");

		Changed(); /* EMIT SIGNAL */
	}

	_in_update = false;
}

jack_nframes_t 
Crossfade::read_at (Sample *buf, Sample *mixdown_buffer, 
		    float *gain_buffer, jack_nframes_t position, jack_nframes_t cnt, uint32_t chan_n,
		    jack_nframes_t read_frames, jack_nframes_t skip_frames)
{
	jack_nframes_t offset;
	jack_nframes_t to_write;
	jack_nframes_t first_frame;

	if (!_active) {
		return 0;
	}

	first_frame = _right->position();

	if (position < first_frame) {

		/* handle an initial section of the read area that we do not
		   cover.
		*/
		

		offset = first_frame - position;

		if (offset < cnt) {
			cnt -= offset;
		} else {
			return 0;
		}

		position = first_frame;
		buf += offset;
	}

	_left->read_at (crossfade_buffer_left, mixdown_buffer, gain_buffer, position, cnt, chan_n, read_frames, skip_frames);
	_right->read_at (crossfade_buffer_right, mixdown_buffer, gain_buffer, position, cnt, chan_n, read_frames, skip_frames);
	
	offset = position - first_frame;
	to_write = min (last_length - offset, cnt);

	float* fiv = new float[to_write];
	float* fov = new float[to_write];

	_fade_in.get_vector (offset, offset+to_write, fiv, to_write);
	_fade_out.get_vector (offset, offset+to_write, fov, to_write);

	/* note: although we have not explicitly taken into account the return values
	   from _left->read_at() or _right->read_at(), the length() function does this
	   implicitly. why? because it computes a value based on the left+right regions'
	   position and length, and so we know precisely how much data they could return. 
	*/
	
	for (jack_nframes_t n = 0; n < to_write; ++n) {
		buf[n] = (crossfade_buffer_left[n] * fov[n]) + (crossfade_buffer_right[n] * fiv[n]);
	}

	delete [] fov;
	delete [] fiv;

	return to_write;
}	

OverlapType 
Crossfade::coverage (jack_nframes_t start, jack_nframes_t end) const
{
	jack_nframes_t rstart = _right->position();
	jack_nframes_t rend = _left->last_frame();
	
	if ((start >= rstart) && (end <= rend)) {
		return OverlapInternal;
	}
	if ((end >= rstart) && (end <= rend)) {
		return OverlapStart;
	}
	if ((start >= rstart) && (start <= rend)) {
		return OverlapEnd;
	}
	if ((rstart >= start) && (rstart <= end) && (rend <= end)) {
		return OverlapExternal;
	}
	return OverlapNone;
}

void
Crossfade::refresh ()
{
	OverlapType ot;
	
	ot = _right->coverage (_left->first_frame(), _left->last_frame());

	switch (ot) {
	case OverlapNone:
	case OverlapInternal:
	case OverlapExternal:
		Invalidated (this);
		return;

	default:
		break;
	}

	/* a bounds change may have caused us to switch to a new kind of overlap */
	
	if (ot != overlap_type) {
		Invalidated (this);
	} else {
		update ();
	}
}

void
Crossfade::set_active (bool yn)
{
	if (_active != yn) {
		_active = yn;
		save_state (_("active changed"));
		ActiveChanged ();
	}
}

void
Crossfade::member_changed (Change what_changed)
{
	Change what_we_care_about = Change (Region::LayerChanged|ARDOUR::BoundsChanged);

	if (what_changed & what_we_care_about) {
		refresh ();
	}
}

void
Crossfade::fade_in_changed (Change ignored)
{
	if (!_in_update) {
		update ();
	}
}

void
Crossfade::fade_out_changed (Change ignored)
{
	if (!_in_update) {
		update ();
	}
}

Change
Crossfade::restore_state (StateManager::State& state)
{
	CrossfadeState* xfstate = dynamic_cast<CrossfadeState*> (&state);

	_active = xfstate->active;

	xfstate->fade_in_memento ();
	xfstate->fade_out_memento ();

	return Change (0);
}

StateManager::State*
Crossfade::state_factory (std::string why) const
{
	CrossfadeState* state = new CrossfadeState (why);
	
	state->fade_in_memento = _fade_in.get_memento ();
	state->fade_out_memento = _fade_out.get_memento ();
	state->active = _active;

	return state;
}
	
UndoAction
Crossfade::get_memento() const
{
	return bind (slot (*(const_cast<Crossfade *> (this)), &StateManager::use_state), _current_state_id);
}

XMLNode&
Crossfade::get_state () 
{
	XMLNode* node = new XMLNode (X_("Crossfade"));
	XMLNode* child;
	char buf[64];
	LocaleGuard (X_("POSIX"));

	snprintf (buf, sizeof(buf), "%" PRIu64, _left->id());
	node->add_property ("left", buf);
	snprintf (buf, sizeof(buf), "%" PRIu64, _right->id());
	node->add_property ("right", buf);
	node->add_property ("active", (_active ? "yes" : "no"));

	child = node->add_child ("FadeIn");

	for (AutomationList::iterator ii = _fade_in.begin(); ii != _fade_in.end(); ++ii) {
		XMLNode* pnode;

		pnode = new XMLNode ("point");

		snprintf (buf, sizeof (buf), "%" PRIu32, (jack_nframes_t) floor ((*ii)->when));
		pnode->add_property ("x", buf);
		snprintf (buf, sizeof (buf), "%f", (*ii)->value);
		pnode->add_property ("y", buf);
		child->add_child_nocopy (*pnode);
	}

	child = node->add_child ("FadeOut");

	for (AutomationList::iterator ii = _fade_out.begin(); ii != _fade_out.end(); ++ii) {
		XMLNode* pnode;

		pnode = new XMLNode ("point");

		snprintf (buf, sizeof (buf), "%" PRIu32, (jack_nframes_t) floor ((*ii)->when));
		pnode->add_property ("x", buf);
		snprintf (buf, sizeof (buf), "%f", (*ii)->value);
		pnode->add_property ("y", buf);
		child->add_child_nocopy (*pnode);
	}

	return *node;
}

int
Crossfade::set_state (const XMLNode& node)
{
	XMLNodeConstIterator i;
	XMLNodeList children;
	XMLNode* fi;
	XMLNode* fo;
	const XMLProperty* prop;
	LocaleGuard (X_("POSIX"));

	if ((fi = find_named_node (node, "FadeIn")) == 0) {
		return -1;
	}
	
	if ((fo = find_named_node (node, "FadeOut")) == 0) {
		return -1;
	}

	if ((prop = node.property ("active")) == 0) {
		_active = false;
	} else {
		set_active (prop->value() == "yes");
	}
	
	/* fade in */
	
	_fade_in.clear ();
	
	children = fi->children();
	
	for (i = children.begin(); i != children.end(); ++i) {
		if ((*i)->name() == "point") {
			jack_nframes_t x;
			float y;
			
			prop = (*i)->property ("x");
			sscanf (prop->value().c_str(), "%lu", &x);
			
			prop = (*i)->property ("y");
			sscanf (prop->value().c_str(), "%f", &y);

			_fade_in.add (x, y);
		}
	}
	
        /* fade out */
	
	_fade_out.clear ();

	children = fo->children();
	
	for (i = children.begin(); i != children.end(); ++i) {
		if ((*i)->name() == "point") {
			jack_nframes_t x;
			float y;
			XMLProperty* prop;

			prop = (*i)->property ("x");
			sscanf (prop->value().c_str(), "%lu", &x);

			prop = (*i)->property ("y");
			sscanf (prop->value().c_str(), "%f", &y);
			
			_fade_out.add (x, y);
		}
	}

	return 0;
}

