/*
 * Copyright (c) 2019 ARM Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include <algorithm>

#include "arm_gemm.hpp"

#include <cstdint>
#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void a64_smallK_hybrid_s8s32_dot_4x6(const int8_t *A, int lda, const int8_t *B, int32_t *C, int ldc, int M, int N, int K, const int32_t *, Activation, bool) {
    const long loops_count = iceildiv(N, (int)4) - 1;
    const long ldab = lda * sizeof(int8_t);
    const long ldcb = ldc * sizeof(int32_t);
    const long odds_count = K % 4;
    K = (K + 3) / 4;

    for (int y0=0; y0<M; y0+=6) {
        long loops = loops_count;
        long oob_rows = std::max(6 - (M-y0), 0);
        long odds = odds_count;
        const int8_t *b_ptr0 = B;
        const int8_t *a_ptr0 = A + (y0 * lda);

        int32_t *c_ptr0 = C + (y0 * ldc);

        switch(K) {
            case 9:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "c_ptr1 .req X5\n"
                    "c_ptr2 .req X6\n"
                    "c_ptr3 .req X7\n"
                    "c_ptr4 .req X8\n"
                    "c_ptr5 .req X9\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q3, [a_ptr1], #0x10\n"
                    "ldr q6, [a_ptr2], #0x10\n"
                    "ldr q9, [a_ptr3], #0x10\n"
                    "ldr q12, [a_ptr4], #0x10\n"
                    "ldr q15, [a_ptr5], #0x10\n"
                    "ldr q1, [%[a_ptr0]], #0x10\n"
                    "ldr q4, [a_ptr1], #0x10\n"
                    "ldr q7, [a_ptr2], #0x10\n"
                    "ldr q10, [a_ptr3], #0x10\n"
                    "ldr q13, [a_ptr4], #0x10\n"
                    "ldr q16, [a_ptr5], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr s2, [%[a_ptr0]]\n"
                    "ldr s5, [a_ptr1]\n"
                    "ldr s8, [a_ptr2]\n"
                    "ldr s11, [a_ptr3]\n"
                    "ldr s14, [a_ptr4]\n"
                    "ldr s17, [a_ptr5]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ldr b2, [%[a_ptr0]]\n"
                    "ldr b5, [a_ptr1]\n"
                    "ldr b8, [a_ptr2]\n"
                    "ldr b11, [a_ptr3]\n"
                    "ldr b14, [a_ptr4]\n"
                    "ldr b17, [a_ptr5]\n"
                    "b 3f\n"
                    "4:\n"
                    "ldr h2, [%[a_ptr0]], #0x2\n"
                    "ldr h5, [a_ptr1], #0x2\n"
                    "ldr h8, [a_ptr2], #0x2\n"
                    "ldr h11, [a_ptr3], #0x2\n"
                    "ldr h14, [a_ptr4], #0x2\n"
                    "ldr h17, [a_ptr5], #0x2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v2.b}[2], [%[a_ptr0]]\n"
                    "ld1 {v5.b}[2], [a_ptr1]\n"
                    "ld1 {v8.b}[2], [a_ptr2]\n"
                    "ld1 {v11.b}[2], [a_ptr3]\n"
                    "ld1 {v14.b}[2], [a_ptr4]\n"
                    "ld1 {v17.b}[2], [a_ptr5]\n"
                    "3:\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "movi v28.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "movi v29.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "movi v30.4s, #0\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    "movi v31.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x40]\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x80]\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0xc0]\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x100]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x140]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x180]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "movi v26.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "str q27, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "movi v26.4s, #0\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v27.4s, #0\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    "6:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q28, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q29, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q30, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q31, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 10:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "c_ptr1 .req X5\n"
                    "c_ptr2 .req X6\n"
                    "c_ptr3 .req X7\n"
                    "c_ptr4 .req X8\n"
                    "c_ptr5 .req X9\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q3, [a_ptr1], #0x10\n"
                    "ldr q6, [a_ptr2], #0x10\n"
                    "ldr q9, [a_ptr3], #0x10\n"
                    "ldr q12, [a_ptr4], #0x10\n"
                    "ldr q15, [a_ptr5], #0x10\n"
                    "ldr q1, [%[a_ptr0]], #0x10\n"
                    "ldr q4, [a_ptr1], #0x10\n"
                    "ldr q7, [a_ptr2], #0x10\n"
                    "ldr q10, [a_ptr3], #0x10\n"
                    "ldr q13, [a_ptr4], #0x10\n"
                    "ldr q16, [a_ptr5], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr d2, [%[a_ptr0]]\n"
                    "ldr d5, [a_ptr1]\n"
                    "ldr d8, [a_ptr2]\n"
                    "ldr d11, [a_ptr3]\n"
                    "ldr d14, [a_ptr4]\n"
                    "ldr d17, [a_ptr5]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr s2, [%[a_ptr0]], #0x4\n"
                    "ldr s5, [a_ptr1], #0x4\n"
                    "ldr s8, [a_ptr2], #0x4\n"
                    "ldr s11, [a_ptr3], #0x4\n"
                    "ldr s14, [a_ptr4], #0x4\n"
                    "ldr s17, [a_ptr5], #0x4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v2.b}[4], [%[a_ptr0]]\n"
                    "ld1 {v5.b}[4], [a_ptr1]\n"
                    "ld1 {v8.b}[4], [a_ptr2]\n"
                    "ld1 {v11.b}[4], [a_ptr3]\n"
                    "ld1 {v14.b}[4], [a_ptr4]\n"
                    "ld1 {v17.b}[4], [a_ptr5]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v2.h}[2], [%[a_ptr0]], #2\n"
                    "ld1 {v5.h}[2], [a_ptr1], #2\n"
                    "ld1 {v8.h}[2], [a_ptr2], #2\n"
                    "ld1 {v11.h}[2], [a_ptr3], #2\n"
                    "ld1 {v14.h}[2], [a_ptr4], #2\n"
                    "ld1 {v17.h}[2], [a_ptr5], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v2.b}[6], [%[a_ptr0]]\n"
                    "ld1 {v5.b}[6], [a_ptr1]\n"
                    "ld1 {v8.b}[6], [a_ptr2]\n"
                    "ld1 {v11.b}[6], [a_ptr3]\n"
                    "ld1 {v14.b}[6], [a_ptr4]\n"
                    "ld1 {v17.b}[6], [a_ptr5]\n"
                    "3:\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "movi v28.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "movi v29.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "movi v30.4s, #0\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    "movi v31.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x40]\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x80]\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0xc0]\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x100]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x140]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x180]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "movi v26.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "str q27, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "movi v26.4s, #0\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v27.4s, #0\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    "6:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q28, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q29, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q30, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q31, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 11:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "c_ptr1 .req X5\n"
                    "c_ptr2 .req X6\n"
                    "c_ptr3 .req X7\n"
                    "c_ptr4 .req X8\n"
                    "c_ptr5 .req X9\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q3, [a_ptr1], #0x10\n"
                    "ldr q6, [a_ptr2], #0x10\n"
                    "ldr q9, [a_ptr3], #0x10\n"
                    "ldr q12, [a_ptr4], #0x10\n"
                    "ldr q15, [a_ptr5], #0x10\n"
                    "ldr q1, [%[a_ptr0]], #0x10\n"
                    "ldr q4, [a_ptr1], #0x10\n"
                    "ldr q7, [a_ptr2], #0x10\n"
                    "ldr q10, [a_ptr3], #0x10\n"
                    "ldr d2, [%[a_ptr0]], #0x8\n"
                    "ldr q13, [a_ptr4], #0x10\n"
                    "ldr d5, [a_ptr1], #0x8\n"
                    "ldr q16, [a_ptr5], #0x10\n"
                    "ldr d8, [a_ptr2], #0x8\n"
                    "ldr d11, [a_ptr3], #0x8\n"
                    "ldr d14, [a_ptr4], #0x8\n"
                    "ldr d17, [a_ptr5], #0x8\n"
                    "cbnz %[odds], 2f\n"
                    "ld1 {v2.s}[2], [%[a_ptr0]]\n"
                    "ld1 {v5.s}[2], [a_ptr1]\n"
                    "ld1 {v8.s}[2], [a_ptr2]\n"
                    "ld1 {v11.s}[2], [a_ptr3]\n"
                    "ld1 {v14.s}[2], [a_ptr4]\n"
                    "ld1 {v17.s}[2], [a_ptr5]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v2.b}[8], [%[a_ptr0]]\n"
                    "ld1 {v5.b}[8], [a_ptr1]\n"
                    "ld1 {v8.b}[8], [a_ptr2]\n"
                    "ld1 {v11.b}[8], [a_ptr3]\n"
                    "ld1 {v14.b}[8], [a_ptr4]\n"
                    "ld1 {v17.b}[8], [a_ptr5]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v2.h}[4], [%[a_ptr0]], #2\n"
                    "ld1 {v5.h}[4], [a_ptr1], #2\n"
                    "ld1 {v8.h}[4], [a_ptr2], #2\n"
                    "ld1 {v11.h}[4], [a_ptr3], #2\n"
                    "ld1 {v14.h}[4], [a_ptr4], #2\n"
                    "ld1 {v17.h}[4], [a_ptr5], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v2.b}[10], [%[a_ptr0]]\n"
                    "ld1 {v5.b}[10], [a_ptr1]\n"
                    "ld1 {v8.b}[10], [a_ptr2]\n"
                    "ld1 {v11.b}[10], [a_ptr3]\n"
                    "ld1 {v14.b}[10], [a_ptr4]\n"
                    "ld1 {v17.b}[10], [a_ptr5]\n"
                    "3:\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "movi v28.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "movi v29.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "movi v30.4s, #0\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    "movi v31.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x40]\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x80]\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0xc0]\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x100]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x140]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x180]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    ".inst 0x4f82ea9a // sdot v26.4s, v20.16b, v2.4b[2]\n"
                    ".inst 0x4f85ea9b // sdot v27.4s, v20.16b, v5.4b[2]\n"
                    ".inst 0x4f88ea9c // sdot v28.4s, v20.16b, v8.4b[2]\n"
                    ".inst 0x4f8bea9d // sdot v29.4s, v20.16b, v11.4b[2]\n"
                    ".inst 0x4f8eea9e // sdot v30.4s, v20.16b, v14.4b[2]\n"
                    ".inst 0x4f91ea9f // sdot v31.4s, v20.16b, v17.4b[2]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "movi v26.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "str q27, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82ea9a // sdot v26.4s, v20.16b, v2.4b[2]\n"
                    ".inst 0x4f85ea9b // sdot v27.4s, v20.16b, v5.4b[2]\n"
                    ".inst 0x4f88ea9c // sdot v28.4s, v20.16b, v8.4b[2]\n"
                    ".inst 0x4f8bea9d // sdot v29.4s, v20.16b, v11.4b[2]\n"
                    ".inst 0x4f8eea9e // sdot v30.4s, v20.16b, v14.4b[2]\n"
                    ".inst 0x4f91ea9f // sdot v31.4s, v20.16b, v17.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "movi v26.4s, #0\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v27.4s, #0\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    ".inst 0x4f82ea9a // sdot v26.4s, v20.16b, v2.4b[2]\n"
                    ".inst 0x4f85ea9b // sdot v27.4s, v20.16b, v5.4b[2]\n"
                    ".inst 0x4f88ea9c // sdot v28.4s, v20.16b, v8.4b[2]\n"
                    ".inst 0x4f8bea9d // sdot v29.4s, v20.16b, v11.4b[2]\n"
                    ".inst 0x4f8eea9e // sdot v30.4s, v20.16b, v14.4b[2]\n"
                    ".inst 0x4f91ea9f // sdot v31.4s, v20.16b, v17.4b[2]\n"
                    "6:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q28, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q29, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q30, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q31, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 12:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "c_ptr1 .req X5\n"
                    "c_ptr2 .req X6\n"
                    "c_ptr3 .req X7\n"
                    "c_ptr4 .req X8\n"
                    "c_ptr5 .req X9\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q3, [a_ptr1], #0x10\n"
                    "ldr q6, [a_ptr2], #0x10\n"
                    "ldr q9, [a_ptr3], #0x10\n"
                    "ldr q12, [a_ptr4], #0x10\n"
                    "ldr q15, [a_ptr5], #0x10\n"
                    "ldr q1, [%[a_ptr0]], #0x10\n"
                    "ldr q4, [a_ptr1], #0x10\n"
                    "ldr q7, [a_ptr2], #0x10\n"
                    "ldr q10, [a_ptr3], #0x10\n"
                    "ldr q13, [a_ptr4], #0x10\n"
                    "ldr q16, [a_ptr5], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr q2, [%[a_ptr0]]\n"
                    "ldr q5, [a_ptr1]\n"
                    "ldr q8, [a_ptr2]\n"
                    "ldr q11, [a_ptr3]\n"
                    "ldr q14, [a_ptr4]\n"
                    "ldr q17, [a_ptr5]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr d2, [%[a_ptr0]], #0x8\n"
                    "ldr d5, [a_ptr1], #0x8\n"
                    "ldr d8, [a_ptr2], #0x8\n"
                    "ldr d11, [a_ptr3], #0x8\n"
                    "ldr d14, [a_ptr4], #0x8\n"
                    "ldr d17, [a_ptr5], #0x8\n"
                    "ld1 {v2.s}[2], [%[a_ptr0]], #4\n"
                    "ld1 {v5.s}[2], [a_ptr1], #4\n"
                    "ld1 {v8.s}[2], [a_ptr2], #4\n"
                    "ld1 {v11.s}[2], [a_ptr3], #4\n"
                    "ld1 {v14.s}[2], [a_ptr4], #4\n"
                    "ld1 {v17.s}[2], [a_ptr5], #4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v2.b}[12], [%[a_ptr0]]\n"
                    "ld1 {v5.b}[12], [a_ptr1]\n"
                    "ld1 {v8.b}[12], [a_ptr2]\n"
                    "ld1 {v11.b}[12], [a_ptr3]\n"
                    "ld1 {v14.b}[12], [a_ptr4]\n"
                    "ld1 {v17.b}[12], [a_ptr5]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v2.h}[6], [%[a_ptr0]], #2\n"
                    "ld1 {v5.h}[6], [a_ptr1], #2\n"
                    "ld1 {v8.h}[6], [a_ptr2], #2\n"
                    "ld1 {v11.h}[6], [a_ptr3], #2\n"
                    "ld1 {v14.h}[6], [a_ptr4], #2\n"
                    "ld1 {v17.h}[6], [a_ptr5], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v2.b}[14], [%[a_ptr0]]\n"
                    "ld1 {v5.b}[14], [a_ptr1]\n"
                    "ld1 {v8.b}[14], [a_ptr2]\n"
                    "ld1 {v11.b}[14], [a_ptr3]\n"
                    "ld1 {v14.b}[14], [a_ptr4]\n"
                    "ld1 {v17.b}[14], [a_ptr5]\n"
                    "3:\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "movi v28.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "movi v29.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "movi v30.4s, #0\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    "movi v31.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x40]\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x80]\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0xc0]\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x100]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x140]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x180]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    ".inst 0x4f82ea9a // sdot v26.4s, v20.16b, v2.4b[2]\n"
                    ".inst 0x4f85ea9b // sdot v27.4s, v20.16b, v5.4b[2]\n"
                    ".inst 0x4f88ea9c // sdot v28.4s, v20.16b, v8.4b[2]\n"
                    ".inst 0x4f8bea9d // sdot v29.4s, v20.16b, v11.4b[2]\n"
                    ".inst 0x4f8eea9e // sdot v30.4s, v20.16b, v14.4b[2]\n"
                    ".inst 0x4f91ea9f // sdot v31.4s, v20.16b, v17.4b[2]\n"
                    ".inst 0x4fa2eaba // sdot v26.4s, v21.16b, v2.4b[3]\n"
                    ".inst 0x4fa5eabb // sdot v27.4s, v21.16b, v5.4b[3]\n"
                    ".inst 0x4fa8eabc // sdot v28.4s, v21.16b, v8.4b[3]\n"
                    ".inst 0x4fabeabd // sdot v29.4s, v21.16b, v11.4b[3]\n"
                    ".inst 0x4faeeabe // sdot v30.4s, v21.16b, v14.4b[3]\n"
                    ".inst 0x4fb1eabf // sdot v31.4s, v21.16b, v17.4b[3]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "movi v26.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "str q27, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    "ldr q22, [%[b_ptr0], #0x40]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    "ldr q23, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    "ldr q24, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82ea9a // sdot v26.4s, v20.16b, v2.4b[2]\n"
                    ".inst 0x4f85ea9b // sdot v27.4s, v20.16b, v5.4b[2]\n"
                    ".inst 0x4f88ea9c // sdot v28.4s, v20.16b, v8.4b[2]\n"
                    ".inst 0x4f8bea9d // sdot v29.4s, v20.16b, v11.4b[2]\n"
                    ".inst 0x4f8eea9e // sdot v30.4s, v20.16b, v14.4b[2]\n"
                    ".inst 0x4f91ea9f // sdot v31.4s, v20.16b, v17.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa2eaba // sdot v26.4s, v21.16b, v2.4b[3]\n"
                    ".inst 0x4fa5eabb // sdot v27.4s, v21.16b, v5.4b[3]\n"
                    ".inst 0x4fa8eabc // sdot v28.4s, v21.16b, v8.4b[3]\n"
                    ".inst 0x4fabeabd // sdot v29.4s, v21.16b, v11.4b[3]\n"
                    ".inst 0x4faeeabe // sdot v30.4s, v21.16b, v14.4b[3]\n"
                    ".inst 0x4fb1eabf // sdot v31.4s, v21.16b, v17.4b[3]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "movi v26.4s, #0\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v27.4s, #0\n"
                    ".inst 0x4f80e25a // sdot v26.4s, v18.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f83e25b // sdot v27.4s, v18.16b, v3.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f86e25c // sdot v28.4s, v18.16b, v6.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f89e25d // sdot v29.4s, v18.16b, v9.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8ce25e // sdot v30.4s, v18.16b, v12.4b[0]\n"
                    ".inst 0x4fa0e27a // sdot v26.4s, v19.16b, v0.4b[1]\n"
                    ".inst 0x4f8fe25f // sdot v31.4s, v18.16b, v15.4b[0]\n"
                    "ldr q18, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e27b // sdot v27.4s, v19.16b, v3.4b[1]\n"
                    ".inst 0x4fa6e27c // sdot v28.4s, v19.16b, v6.4b[1]\n"
                    ".inst 0x4fa9e27d // sdot v29.4s, v19.16b, v9.4b[1]\n"
                    ".inst 0x4face27e // sdot v30.4s, v19.16b, v12.4b[1]\n"
                    ".inst 0x4fafe27f // sdot v31.4s, v19.16b, v15.4b[1]\n"
                    "ldr q19, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80ea9a // sdot v26.4s, v20.16b, v0.4b[2]\n"
                    ".inst 0x4f83ea9b // sdot v27.4s, v20.16b, v3.4b[2]\n"
                    ".inst 0x4f86ea9c // sdot v28.4s, v20.16b, v6.4b[2]\n"
                    ".inst 0x4f89ea9d // sdot v29.4s, v20.16b, v9.4b[2]\n"
                    ".inst 0x4f8cea9e // sdot v30.4s, v20.16b, v12.4b[2]\n"
                    ".inst 0x4f8fea9f // sdot v31.4s, v20.16b, v15.4b[2]\n"
                    "ldr q20, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4fa0eaba // sdot v26.4s, v21.16b, v0.4b[3]\n"
                    ".inst 0x4fa3eabb // sdot v27.4s, v21.16b, v3.4b[3]\n"
                    ".inst 0x4fa6eabc // sdot v28.4s, v21.16b, v6.4b[3]\n"
                    ".inst 0x4fa9eabd // sdot v29.4s, v21.16b, v9.4b[3]\n"
                    ".inst 0x4faceabe // sdot v30.4s, v21.16b, v12.4b[3]\n"
                    ".inst 0x4fafeabf // sdot v31.4s, v21.16b, v15.4b[3]\n"
                    "ldr q21, [%[b_ptr0], #0x30]\n"
                    ".inst 0x4f81e2da // sdot v26.4s, v22.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    ".inst 0x4f84e2db // sdot v27.4s, v22.16b, v4.4b[0]\n"
                    ".inst 0x4f87e2dc // sdot v28.4s, v22.16b, v7.4b[0]\n"
                    ".inst 0x4f8ae2dd // sdot v29.4s, v22.16b, v10.4b[0]\n"
                    ".inst 0x4f8de2de // sdot v30.4s, v22.16b, v13.4b[0]\n"
                    ".inst 0x4f90e2df // sdot v31.4s, v22.16b, v16.4b[0]\n"
                    ".inst 0x4fa1e2fa // sdot v26.4s, v23.16b, v1.4b[1]\n"
                    ".inst 0x4fa4e2fb // sdot v27.4s, v23.16b, v4.4b[1]\n"
                    ".inst 0x4fa7e2fc // sdot v28.4s, v23.16b, v7.4b[1]\n"
                    ".inst 0x4faae2fd // sdot v29.4s, v23.16b, v10.4b[1]\n"
                    ".inst 0x4fade2fe // sdot v30.4s, v23.16b, v13.4b[1]\n"
                    ".inst 0x4fb0e2ff // sdot v31.4s, v23.16b, v16.4b[1]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f87eb1c // sdot v28.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8aeb1d // sdot v29.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8deb1e // sdot v30.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f90eb1f // sdot v31.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa7eb3c // sdot v28.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4faaeb3d // sdot v29.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4fadeb3e // sdot v30.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb0eb3f // sdot v31.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4f82e25a // sdot v26.4s, v18.16b, v2.4b[0]\n"
                    ".inst 0x4f85e25b // sdot v27.4s, v18.16b, v5.4b[0]\n"
                    ".inst 0x4f88e25c // sdot v28.4s, v18.16b, v8.4b[0]\n"
                    ".inst 0x4f8be25d // sdot v29.4s, v18.16b, v11.4b[0]\n"
                    ".inst 0x4f8ee25e // sdot v30.4s, v18.16b, v14.4b[0]\n"
                    ".inst 0x4f91e25f // sdot v31.4s, v18.16b, v17.4b[0]\n"
                    ".inst 0x4fa2e27a // sdot v26.4s, v19.16b, v2.4b[1]\n"
                    ".inst 0x4fa5e27b // sdot v27.4s, v19.16b, v5.4b[1]\n"
                    ".inst 0x4fa8e27c // sdot v28.4s, v19.16b, v8.4b[1]\n"
                    ".inst 0x4fabe27d // sdot v29.4s, v19.16b, v11.4b[1]\n"
                    ".inst 0x4faee27e // sdot v30.4s, v19.16b, v14.4b[1]\n"
                    ".inst 0x4fb1e27f // sdot v31.4s, v19.16b, v17.4b[1]\n"
                    ".inst 0x4f82ea9a // sdot v26.4s, v20.16b, v2.4b[2]\n"
                    ".inst 0x4f85ea9b // sdot v27.4s, v20.16b, v5.4b[2]\n"
                    ".inst 0x4f88ea9c // sdot v28.4s, v20.16b, v8.4b[2]\n"
                    ".inst 0x4f8bea9d // sdot v29.4s, v20.16b, v11.4b[2]\n"
                    ".inst 0x4f8eea9e // sdot v30.4s, v20.16b, v14.4b[2]\n"
                    ".inst 0x4f91ea9f // sdot v31.4s, v20.16b, v17.4b[2]\n"
                    ".inst 0x4fa2eaba // sdot v26.4s, v21.16b, v2.4b[3]\n"
                    ".inst 0x4fa5eabb // sdot v27.4s, v21.16b, v5.4b[3]\n"
                    ".inst 0x4fa8eabc // sdot v28.4s, v21.16b, v8.4b[3]\n"
                    ".inst 0x4fabeabd // sdot v29.4s, v21.16b, v11.4b[3]\n"
                    ".inst 0x4faeeabe // sdot v30.4s, v21.16b, v14.4b[3]\n"
                    ".inst 0x4fb1eabf // sdot v31.4s, v21.16b, v17.4b[3]\n"
                    "6:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q28, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q29, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q30, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q31, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 13:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "c_ptr1 .req X5\n"
                    "c_ptr2 .req X6\n"
                    "c_ptr3 .req X7\n"
                    "c_ptr4 .req X8\n"
                    "c_ptr5 .req X9\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q4, [a_ptr1], #0x10\n"
                    "ldr q8, [a_ptr2], #0x10\n"
                    "ldr q12, [a_ptr3], #0x10\n"
                    "ldr q16, [a_ptr4], #0x10\n"
                    "ldr q20, [a_ptr5], #0x10\n"
                    "ldr q1, [%[a_ptr0]], #0x10\n"
                    "ldr q5, [a_ptr1], #0x10\n"
                    "ldr q9, [a_ptr2], #0x10\n"
                    "ldr q13, [a_ptr3], #0x10\n"
                    "ldr q17, [a_ptr4], #0x10\n"
                    "ldr q21, [a_ptr5], #0x10\n"
                    "ldr q2, [%[a_ptr0]], #0x10\n"
                    "ldr q6, [a_ptr1], #0x10\n"
                    "ldr q10, [a_ptr2], #0x10\n"
                    "ldr q14, [a_ptr3], #0x10\n"
                    "ldr q18, [a_ptr4], #0x10\n"
                    "ldr q22, [a_ptr5], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr s3, [%[a_ptr0]]\n"
                    "ldr s7, [a_ptr1]\n"
                    "ldr s11, [a_ptr2]\n"
                    "ldr s15, [a_ptr3]\n"
                    "ldr s19, [a_ptr4]\n"
                    "ldr s23, [a_ptr5]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ldr b3, [%[a_ptr0]]\n"
                    "ldr b7, [a_ptr1]\n"
                    "ldr b11, [a_ptr2]\n"
                    "ldr b15, [a_ptr3]\n"
                    "ldr b19, [a_ptr4]\n"
                    "ldr b23, [a_ptr5]\n"
                    "b 3f\n"
                    "4:\n"
                    "ldr h3, [%[a_ptr0]], #0x2\n"
                    "ldr h7, [a_ptr1], #0x2\n"
                    "ldr h11, [a_ptr2], #0x2\n"
                    "ldr h15, [a_ptr3], #0x2\n"
                    "ldr h19, [a_ptr4], #0x2\n"
                    "ldr h23, [a_ptr5], #0x2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v3.b}[2], [%[a_ptr0]]\n"
                    "ld1 {v7.b}[2], [a_ptr1]\n"
                    "ld1 {v11.b}[2], [a_ptr2]\n"
                    "ld1 {v15.b}[2], [a_ptr3]\n"
                    "ld1 {v19.b}[2], [a_ptr4]\n"
                    "ld1 {v23.b}[2], [a_ptr5]\n"
                    "3:\n"
                    "movi v26.4s, #0\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "movi v27.4s, #0\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "movi v28.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x40]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x80]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0xc0]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x100]\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x140]\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x180]\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "movi v26.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "str q27, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "movi v26.4s, #0\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v27.4s, #0\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "6:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q28, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q29, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q30, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q31, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 14:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "c_ptr1 .req X5\n"
                    "c_ptr2 .req X6\n"
                    "c_ptr3 .req X7\n"
                    "c_ptr4 .req X8\n"
                    "c_ptr5 .req X9\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q4, [a_ptr1], #0x10\n"
                    "ldr q8, [a_ptr2], #0x10\n"
                    "ldr q12, [a_ptr3], #0x10\n"
                    "ldr q16, [a_ptr4], #0x10\n"
                    "ldr q20, [a_ptr5], #0x10\n"
                    "ldr q1, [%[a_ptr0]], #0x10\n"
                    "ldr q5, [a_ptr1], #0x10\n"
                    "ldr q9, [a_ptr2], #0x10\n"
                    "ldr q13, [a_ptr3], #0x10\n"
                    "ldr q17, [a_ptr4], #0x10\n"
                    "ldr q21, [a_ptr5], #0x10\n"
                    "ldr q2, [%[a_ptr0]], #0x10\n"
                    "ldr q6, [a_ptr1], #0x10\n"
                    "ldr q10, [a_ptr2], #0x10\n"
                    "ldr q14, [a_ptr3], #0x10\n"
                    "ldr q18, [a_ptr4], #0x10\n"
                    "ldr q22, [a_ptr5], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr d3, [%[a_ptr0]]\n"
                    "ldr d7, [a_ptr1]\n"
                    "ldr d11, [a_ptr2]\n"
                    "ldr d15, [a_ptr3]\n"
                    "ldr d19, [a_ptr4]\n"
                    "ldr d23, [a_ptr5]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr s3, [%[a_ptr0]], #0x4\n"
                    "ldr s7, [a_ptr1], #0x4\n"
                    "ldr s11, [a_ptr2], #0x4\n"
                    "ldr s15, [a_ptr3], #0x4\n"
                    "ldr s19, [a_ptr4], #0x4\n"
                    "ldr s23, [a_ptr5], #0x4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v3.b}[4], [%[a_ptr0]]\n"
                    "ld1 {v7.b}[4], [a_ptr1]\n"
                    "ld1 {v11.b}[4], [a_ptr2]\n"
                    "ld1 {v15.b}[4], [a_ptr3]\n"
                    "ld1 {v19.b}[4], [a_ptr4]\n"
                    "ld1 {v23.b}[4], [a_ptr5]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v3.h}[2], [%[a_ptr0]], #2\n"
                    "ld1 {v7.h}[2], [a_ptr1], #2\n"
                    "ld1 {v11.h}[2], [a_ptr2], #2\n"
                    "ld1 {v15.h}[2], [a_ptr3], #2\n"
                    "ld1 {v19.h}[2], [a_ptr4], #2\n"
                    "ld1 {v23.h}[2], [a_ptr5], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v3.b}[6], [%[a_ptr0]]\n"
                    "ld1 {v7.b}[6], [a_ptr1]\n"
                    "ld1 {v11.b}[6], [a_ptr2]\n"
                    "ld1 {v15.b}[6], [a_ptr3]\n"
                    "ld1 {v19.b}[6], [a_ptr4]\n"
                    "ld1 {v23.b}[6], [a_ptr5]\n"
                    "3:\n"
                    "movi v26.4s, #0\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "movi v27.4s, #0\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "movi v28.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x40]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x80]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0xc0]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x100]\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x140]\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x180]\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v26.4s, #0\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "str q27, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v26.4s, #0\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v27.4s, #0\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    "6:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q28, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q29, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q30, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q31, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 15:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "c_ptr1 .req X5\n"
                    "c_ptr2 .req X6\n"
                    "c_ptr3 .req X7\n"
                    "c_ptr4 .req X8\n"
                    "c_ptr5 .req X9\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q4, [a_ptr1], #0x10\n"
                    "ldr q8, [a_ptr2], #0x10\n"
                    "ldr q12, [a_ptr3], #0x10\n"
                    "ldr q16, [a_ptr4], #0x10\n"
                    "ldr q20, [a_ptr5], #0x10\n"
                    "ldr q1, [%[a_ptr0]], #0x10\n"
                    "ldr q5, [a_ptr1], #0x10\n"
                    "ldr q9, [a_ptr2], #0x10\n"
                    "ldr q13, [a_ptr3], #0x10\n"
                    "ldr q17, [a_ptr4], #0x10\n"
                    "ldr q21, [a_ptr5], #0x10\n"
                    "ldr q2, [%[a_ptr0]], #0x10\n"
                    "ldr q6, [a_ptr1], #0x10\n"
                    "ldr q10, [a_ptr2], #0x10\n"
                    "ldr q14, [a_ptr3], #0x10\n"
                    "ldr d3, [%[a_ptr0]], #0x8\n"
                    "ldr q18, [a_ptr4], #0x10\n"
                    "ldr d7, [a_ptr1], #0x8\n"
                    "ldr q22, [a_ptr5], #0x10\n"
                    "ldr d11, [a_ptr2], #0x8\n"
                    "ldr d15, [a_ptr3], #0x8\n"
                    "ldr d19, [a_ptr4], #0x8\n"
                    "ldr d23, [a_ptr5], #0x8\n"
                    "cbnz %[odds], 2f\n"
                    "ld1 {v3.s}[2], [%[a_ptr0]]\n"
                    "ld1 {v7.s}[2], [a_ptr1]\n"
                    "ld1 {v11.s}[2], [a_ptr2]\n"
                    "ld1 {v15.s}[2], [a_ptr3]\n"
                    "ld1 {v19.s}[2], [a_ptr4]\n"
                    "ld1 {v23.s}[2], [a_ptr5]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v3.b}[8], [%[a_ptr0]]\n"
                    "ld1 {v7.b}[8], [a_ptr1]\n"
                    "ld1 {v11.b}[8], [a_ptr2]\n"
                    "ld1 {v15.b}[8], [a_ptr3]\n"
                    "ld1 {v19.b}[8], [a_ptr4]\n"
                    "ld1 {v23.b}[8], [a_ptr5]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v3.h}[4], [%[a_ptr0]], #2\n"
                    "ld1 {v7.h}[4], [a_ptr1], #2\n"
                    "ld1 {v11.h}[4], [a_ptr2], #2\n"
                    "ld1 {v15.h}[4], [a_ptr3], #2\n"
                    "ld1 {v19.h}[4], [a_ptr4], #2\n"
                    "ld1 {v23.h}[4], [a_ptr5], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v3.b}[10], [%[a_ptr0]]\n"
                    "ld1 {v7.b}[10], [a_ptr1]\n"
                    "ld1 {v11.b}[10], [a_ptr2]\n"
                    "ld1 {v15.b}[10], [a_ptr3]\n"
                    "ld1 {v19.b}[10], [a_ptr4]\n"
                    "ld1 {v23.b}[10], [a_ptr5]\n"
                    "3:\n"
                    "movi v26.4s, #0\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "movi v27.4s, #0\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "movi v28.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x40]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x80]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0xc0]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x100]\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x140]\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x180]\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    ".inst 0x4f83eb1a // sdot v26.4s, v24.16b, v3.4b[2]\n"
                    ".inst 0x4f87eb1b // sdot v27.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8beb1c // sdot v28.4s, v24.16b, v11.4b[2]\n"
                    ".inst 0x4f8feb1d // sdot v29.4s, v24.16b, v15.4b[2]\n"
                    ".inst 0x4f93eb1e // sdot v30.4s, v24.16b, v19.4b[2]\n"
                    ".inst 0x4f97eb1f // sdot v31.4s, v24.16b, v23.4b[2]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "movi v26.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "str q27, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83eb1a // sdot v26.4s, v24.16b, v3.4b[2]\n"
                    ".inst 0x4f87eb1b // sdot v27.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8beb1c // sdot v28.4s, v24.16b, v11.4b[2]\n"
                    ".inst 0x4f8feb1d // sdot v29.4s, v24.16b, v15.4b[2]\n"
                    ".inst 0x4f93eb1e // sdot v30.4s, v24.16b, v19.4b[2]\n"
                    ".inst 0x4f97eb1f // sdot v31.4s, v24.16b, v23.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "movi v26.4s, #0\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v27.4s, #0\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    ".inst 0x4f83eb1a // sdot v26.4s, v24.16b, v3.4b[2]\n"
                    ".inst 0x4f87eb1b // sdot v27.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8beb1c // sdot v28.4s, v24.16b, v11.4b[2]\n"
                    ".inst 0x4f8feb1d // sdot v29.4s, v24.16b, v15.4b[2]\n"
                    ".inst 0x4f93eb1e // sdot v30.4s, v24.16b, v19.4b[2]\n"
                    ".inst 0x4f97eb1f // sdot v31.4s, v24.16b, v23.4b[2]\n"
                    "6:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q28, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q29, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q30, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q31, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            default:
            case 16:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "c_ptr1 .req X5\n"
                    "c_ptr2 .req X6\n"
                    "c_ptr3 .req X7\n"
                    "c_ptr4 .req X8\n"
                    "c_ptr5 .req X9\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q4, [a_ptr1], #0x10\n"
                    "ldr q8, [a_ptr2], #0x10\n"
                    "ldr q12, [a_ptr3], #0x10\n"
                    "ldr q16, [a_ptr4], #0x10\n"
                    "ldr q20, [a_ptr5], #0x10\n"
                    "ldr q1, [%[a_ptr0]], #0x10\n"
                    "ldr q5, [a_ptr1], #0x10\n"
                    "ldr q9, [a_ptr2], #0x10\n"
                    "ldr q13, [a_ptr3], #0x10\n"
                    "ldr q17, [a_ptr4], #0x10\n"
                    "ldr q21, [a_ptr5], #0x10\n"
                    "ldr q2, [%[a_ptr0]], #0x10\n"
                    "ldr q6, [a_ptr1], #0x10\n"
                    "ldr q10, [a_ptr2], #0x10\n"
                    "ldr q14, [a_ptr3], #0x10\n"
                    "ldr q18, [a_ptr4], #0x10\n"
                    "ldr q22, [a_ptr5], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr q3, [%[a_ptr0]]\n"
                    "ldr q7, [a_ptr1]\n"
                    "ldr q11, [a_ptr2]\n"
                    "ldr q15, [a_ptr3]\n"
                    "ldr q19, [a_ptr4]\n"
                    "ldr q23, [a_ptr5]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr d3, [%[a_ptr0]], #0x8\n"
                    "ldr d7, [a_ptr1], #0x8\n"
                    "ldr d11, [a_ptr2], #0x8\n"
                    "ldr d15, [a_ptr3], #0x8\n"
                    "ldr d19, [a_ptr4], #0x8\n"
                    "ldr d23, [a_ptr5], #0x8\n"
                    "ld1 {v3.s}[2], [%[a_ptr0]], #4\n"
                    "ld1 {v7.s}[2], [a_ptr1], #4\n"
                    "ld1 {v11.s}[2], [a_ptr2], #4\n"
                    "ld1 {v15.s}[2], [a_ptr3], #4\n"
                    "ld1 {v19.s}[2], [a_ptr4], #4\n"
                    "ld1 {v23.s}[2], [a_ptr5], #4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v3.b}[12], [%[a_ptr0]]\n"
                    "ld1 {v7.b}[12], [a_ptr1]\n"
                    "ld1 {v11.b}[12], [a_ptr2]\n"
                    "ld1 {v15.b}[12], [a_ptr3]\n"
                    "ld1 {v19.b}[12], [a_ptr4]\n"
                    "ld1 {v23.b}[12], [a_ptr5]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v3.h}[6], [%[a_ptr0]], #2\n"
                    "ld1 {v7.h}[6], [a_ptr1], #2\n"
                    "ld1 {v11.h}[6], [a_ptr2], #2\n"
                    "ld1 {v15.h}[6], [a_ptr3], #2\n"
                    "ld1 {v19.h}[6], [a_ptr4], #2\n"
                    "ld1 {v23.h}[6], [a_ptr5], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v3.b}[14], [%[a_ptr0]]\n"
                    "ld1 {v7.b}[14], [a_ptr1]\n"
                    "ld1 {v11.b}[14], [a_ptr2]\n"
                    "ld1 {v15.b}[14], [a_ptr3]\n"
                    "ld1 {v19.b}[14], [a_ptr4]\n"
                    "ld1 {v23.b}[14], [a_ptr5]\n"
                    "3:\n"
                    "movi v26.4s, #0\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "movi v27.4s, #0\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "movi v28.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x40]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x80]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0xc0]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x100]\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x140]\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr5, #0x180]\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83eb1a // sdot v26.4s, v24.16b, v3.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87eb1b // sdot v27.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8beb1c // sdot v28.4s, v24.16b, v11.4b[2]\n"
                    ".inst 0x4f8feb1d // sdot v29.4s, v24.16b, v15.4b[2]\n"
                    ".inst 0x4f93eb1e // sdot v30.4s, v24.16b, v19.4b[2]\n"
                    ".inst 0x4f97eb1f // sdot v31.4s, v24.16b, v23.4b[2]\n"
                    ".inst 0x4fa3eb3a // sdot v26.4s, v25.16b, v3.4b[3]\n"
                    ".inst 0x4fa7eb3b // sdot v27.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4fabeb3c // sdot v28.4s, v25.16b, v11.4b[3]\n"
                    ".inst 0x4fafeb3d // sdot v29.4s, v25.16b, v15.4b[3]\n"
                    ".inst 0x4fb3eb3e // sdot v30.4s, v25.16b, v19.4b[3]\n"
                    ".inst 0x4fb7eb3f // sdot v31.4s, v25.16b, v23.4b[3]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v26.4s, #0\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "str q27, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83eb1a // sdot v26.4s, v24.16b, v3.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87eb1b // sdot v27.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8beb1c // sdot v28.4s, v24.16b, v11.4b[2]\n"
                    ".inst 0x4f8feb1d // sdot v29.4s, v24.16b, v15.4b[2]\n"
                    ".inst 0x4f93eb1e // sdot v30.4s, v24.16b, v19.4b[2]\n"
                    ".inst 0x4f97eb1f // sdot v31.4s, v24.16b, v23.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa3eb3a // sdot v26.4s, v25.16b, v3.4b[3]\n"
                    ".inst 0x4fa7eb3b // sdot v27.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4fabeb3c // sdot v28.4s, v25.16b, v11.4b[3]\n"
                    ".inst 0x4fafeb3d // sdot v29.4s, v25.16b, v15.4b[3]\n"
                    ".inst 0x4fb3eb3e // sdot v30.4s, v25.16b, v19.4b[3]\n"
                    ".inst 0x4fb7eb3f // sdot v31.4s, v25.16b, v23.4b[3]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v26.4s, #0\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v27.4s, #0\n"
                    ".inst 0x4f80e31a // sdot v26.4s, v24.16b, v0.4b[0]\n"
                    "str q28, [c_ptr2]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f84e31b // sdot v27.4s, v24.16b, v4.4b[0]\n"
                    "str q29, [c_ptr3]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f88e31c // sdot v28.4s, v24.16b, v8.4b[0]\n"
                    "str q30, [c_ptr4]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f8ce31d // sdot v29.4s, v24.16b, v12.4b[0]\n"
                    "str q31, [c_ptr5]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f90e31e // sdot v30.4s, v24.16b, v16.4b[0]\n"
                    ".inst 0x4fa0e33a // sdot v26.4s, v25.16b, v0.4b[1]\n"
                    ".inst 0x4f94e31f // sdot v31.4s, v24.16b, v20.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa4e33b // sdot v27.4s, v25.16b, v4.4b[1]\n"
                    ".inst 0x4fa8e33c // sdot v28.4s, v25.16b, v8.4b[1]\n"
                    ".inst 0x4face33d // sdot v29.4s, v25.16b, v12.4b[1]\n"
                    ".inst 0x4fb0e33e // sdot v30.4s, v25.16b, v16.4b[1]\n"
                    ".inst 0x4fb4e33f // sdot v31.4s, v25.16b, v20.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f80eb1a // sdot v26.4s, v24.16b, v0.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f84eb1b // sdot v27.4s, v24.16b, v4.4b[2]\n"
                    ".inst 0x4f88eb1c // sdot v28.4s, v24.16b, v8.4b[2]\n"
                    ".inst 0x4f8ceb1d // sdot v29.4s, v24.16b, v12.4b[2]\n"
                    ".inst 0x4f90eb1e // sdot v30.4s, v24.16b, v16.4b[2]\n"
                    ".inst 0x4f94eb1f // sdot v31.4s, v24.16b, v20.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa0eb3a // sdot v26.4s, v25.16b, v0.4b[3]\n"
                    ".inst 0x4fa4eb3b // sdot v27.4s, v25.16b, v4.4b[3]\n"
                    ".inst 0x4fa8eb3c // sdot v28.4s, v25.16b, v8.4b[3]\n"
                    ".inst 0x4faceb3d // sdot v29.4s, v25.16b, v12.4b[3]\n"
                    ".inst 0x4fb0eb3e // sdot v30.4s, v25.16b, v16.4b[3]\n"
                    ".inst 0x4fb4eb3f // sdot v31.4s, v25.16b, v20.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81e31a // sdot v26.4s, v24.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85e31b // sdot v27.4s, v24.16b, v5.4b[0]\n"
                    ".inst 0x4f89e31c // sdot v28.4s, v24.16b, v9.4b[0]\n"
                    ".inst 0x4f8de31d // sdot v29.4s, v24.16b, v13.4b[0]\n"
                    ".inst 0x4f91e31e // sdot v30.4s, v24.16b, v17.4b[0]\n"
                    ".inst 0x4f95e31f // sdot v31.4s, v24.16b, v21.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1e33a // sdot v26.4s, v25.16b, v1.4b[1]\n"
                    ".inst 0x4fa5e33b // sdot v27.4s, v25.16b, v5.4b[1]\n"
                    ".inst 0x4fa9e33c // sdot v28.4s, v25.16b, v9.4b[1]\n"
                    ".inst 0x4fade33d // sdot v29.4s, v25.16b, v13.4b[1]\n"
                    ".inst 0x4fb1e33e // sdot v30.4s, v25.16b, v17.4b[1]\n"
                    ".inst 0x4fb5e33f // sdot v31.4s, v25.16b, v21.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f81eb1a // sdot v26.4s, v24.16b, v1.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f85eb1b // sdot v27.4s, v24.16b, v5.4b[2]\n"
                    ".inst 0x4f89eb1c // sdot v28.4s, v24.16b, v9.4b[2]\n"
                    ".inst 0x4f8deb1d // sdot v29.4s, v24.16b, v13.4b[2]\n"
                    ".inst 0x4f91eb1e // sdot v30.4s, v24.16b, v17.4b[2]\n"
                    ".inst 0x4f95eb1f // sdot v31.4s, v24.16b, v21.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa1eb3a // sdot v26.4s, v25.16b, v1.4b[3]\n"
                    ".inst 0x4fa5eb3b // sdot v27.4s, v25.16b, v5.4b[3]\n"
                    ".inst 0x4fa9eb3c // sdot v28.4s, v25.16b, v9.4b[3]\n"
                    ".inst 0x4fadeb3d // sdot v29.4s, v25.16b, v13.4b[3]\n"
                    ".inst 0x4fb1eb3e // sdot v30.4s, v25.16b, v17.4b[3]\n"
                    ".inst 0x4fb5eb3f // sdot v31.4s, v25.16b, v21.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82e31a // sdot v26.4s, v24.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86e31b // sdot v27.4s, v24.16b, v6.4b[0]\n"
                    ".inst 0x4f8ae31c // sdot v28.4s, v24.16b, v10.4b[0]\n"
                    ".inst 0x4f8ee31d // sdot v29.4s, v24.16b, v14.4b[0]\n"
                    ".inst 0x4f92e31e // sdot v30.4s, v24.16b, v18.4b[0]\n"
                    ".inst 0x4f96e31f // sdot v31.4s, v24.16b, v22.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2e33a // sdot v26.4s, v25.16b, v2.4b[1]\n"
                    ".inst 0x4fa6e33b // sdot v27.4s, v25.16b, v6.4b[1]\n"
                    ".inst 0x4faae33c // sdot v28.4s, v25.16b, v10.4b[1]\n"
                    ".inst 0x4faee33d // sdot v29.4s, v25.16b, v14.4b[1]\n"
                    ".inst 0x4fb2e33e // sdot v30.4s, v25.16b, v18.4b[1]\n"
                    ".inst 0x4fb6e33f // sdot v31.4s, v25.16b, v22.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f82eb1a // sdot v26.4s, v24.16b, v2.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f86eb1b // sdot v27.4s, v24.16b, v6.4b[2]\n"
                    ".inst 0x4f8aeb1c // sdot v28.4s, v24.16b, v10.4b[2]\n"
                    ".inst 0x4f8eeb1d // sdot v29.4s, v24.16b, v14.4b[2]\n"
                    ".inst 0x4f92eb1e // sdot v30.4s, v24.16b, v18.4b[2]\n"
                    ".inst 0x4f96eb1f // sdot v31.4s, v24.16b, v22.4b[2]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa2eb3a // sdot v26.4s, v25.16b, v2.4b[3]\n"
                    ".inst 0x4fa6eb3b // sdot v27.4s, v25.16b, v6.4b[3]\n"
                    ".inst 0x4faaeb3c // sdot v28.4s, v25.16b, v10.4b[3]\n"
                    ".inst 0x4faeeb3d // sdot v29.4s, v25.16b, v14.4b[3]\n"
                    ".inst 0x4fb2eb3e // sdot v30.4s, v25.16b, v18.4b[3]\n"
                    ".inst 0x4fb6eb3f // sdot v31.4s, v25.16b, v22.4b[3]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83e31a // sdot v26.4s, v24.16b, v3.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87e31b // sdot v27.4s, v24.16b, v7.4b[0]\n"
                    ".inst 0x4f8be31c // sdot v28.4s, v24.16b, v11.4b[0]\n"
                    ".inst 0x4f8fe31d // sdot v29.4s, v24.16b, v15.4b[0]\n"
                    ".inst 0x4f93e31e // sdot v30.4s, v24.16b, v19.4b[0]\n"
                    ".inst 0x4f97e31f // sdot v31.4s, v24.16b, v23.4b[0]\n"
                    "ldr q24, [%[b_ptr0]]\n"
                    ".inst 0x4fa3e33a // sdot v26.4s, v25.16b, v3.4b[1]\n"
                    ".inst 0x4fa7e33b // sdot v27.4s, v25.16b, v7.4b[1]\n"
                    ".inst 0x4fabe33c // sdot v28.4s, v25.16b, v11.4b[1]\n"
                    ".inst 0x4fafe33d // sdot v29.4s, v25.16b, v15.4b[1]\n"
                    ".inst 0x4fb3e33e // sdot v30.4s, v25.16b, v19.4b[1]\n"
                    ".inst 0x4fb7e33f // sdot v31.4s, v25.16b, v23.4b[1]\n"
                    "ldr q25, [%[b_ptr0], #0x10]\n"
                    ".inst 0x4f83eb1a // sdot v26.4s, v24.16b, v3.4b[2]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f87eb1b // sdot v27.4s, v24.16b, v7.4b[2]\n"
                    ".inst 0x4f8beb1c // sdot v28.4s, v24.16b, v11.4b[2]\n"
                    ".inst 0x4f8feb1d // sdot v29.4s, v24.16b, v15.4b[2]\n"
                    ".inst 0x4f93eb1e // sdot v30.4s, v24.16b, v19.4b[2]\n"
                    ".inst 0x4f97eb1f // sdot v31.4s, v24.16b, v23.4b[2]\n"
                    ".inst 0x4fa3eb3a // sdot v26.4s, v25.16b, v3.4b[3]\n"
                    ".inst 0x4fa7eb3b // sdot v27.4s, v25.16b, v7.4b[3]\n"
                    ".inst 0x4fabeb3c // sdot v28.4s, v25.16b, v11.4b[3]\n"
                    ".inst 0x4fafeb3d // sdot v29.4s, v25.16b, v15.4b[3]\n"
                    ".inst 0x4fb3eb3e // sdot v30.4s, v25.16b, v19.4b[3]\n"
                    ".inst 0x4fb7eb3f // sdot v31.4s, v25.16b, v23.4b[3]\n"
                    "6:\n"
                    "str q26, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q27, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q28, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q29, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q30, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q31, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
        }
    }
}

} // namespace arm_gemm

#endif // __aarch64__
