//
// Copyright © 2017 Arm Ltd. All rights reserved.
// SPDX-License-Identifier: MIT
//
#pragma once

#include "Layer.hpp"

namespace armnn
{

/// This layer dequantizes the input tensor.
class MergeLayer : public Layer
{
public:
    /// Makes a workload for the Merge type.
    /// @param [in] graph The graph where this layer can be found.
    /// @param [in] factory The workload factory which will create the workload.
    /// @return A pointer to the created workload, or nullptr if not created.
    virtual std::unique_ptr<IWorkload> CreateWorkload(const Graph& graph,
                                                      const IWorkloadFactory& factory) const override;

    /// Creates a dynamically-allocated copy of this layer.
    /// @param [in] graph The graph into which this layer is being cloned.
    MergeLayer* Clone(Graph& graph) const override;

    /// Check if the input tensor shape(s)
    /// will lead to a valid configuration of @ref MergeLayer.
    void ValidateTensorShapesFromInputs() override;

    /// Infers the output shapes from given input shapes.
    /// @param [in] inputShapes The input shapes layer has.
    /// @return A vector to the inferred output shape.
    std::vector<TensorShape> InferOutputShapes(const std::vector<TensorShape>& inputShapes) const override;

    void Accept(ILayerVisitor& visitor) const override;

protected:
    /// Constructor to create a MergeLayer.
    /// @param [in] name Optional name for the layer.
    MergeLayer(const char* name);

    /// Default destructor
    ~MergeLayer() = default;
};

} // namespace armnn
