#!/usr/bin/env python
# -*- coding: utf8 -*-
# Copyright: 2013-2014, Maximiliano Curia <maxy@debian.org>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51
# Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
'''
Arriero is a tool for simplifying maintaining many debian packages.
It allows to quickly update, build, push and upload many packages at the same
time.
'''

# System imports
import ConfigParser
import argparse
import collections
import git
import glob
import inspect
import logging
import os
import shutil
import subprocess
import sys

# Own imports
import actions
import moo
import util


class Arriero(object):
    '''Main class to interface with the user.

    This class handles the CLI, the config values.  The individual actions are
    delegated to the actions module.
    '''

    defaults = {
        'config_files': ['/etc/arriero.conf', '~/.config/arriero.conf'],
        'upstream-branch': 'upstream',
        'debian-branch': 'master',
        'pristine-tar': 'False',
        'pristine-tar-branch': 'pristine-tar',
        'is_merged': 'False',
        'filter-orig': 'False',
        'basedir': '~',
        'path': '%(basedir)s/%(package)s',
        'depends': '',
        'upload-command': 'dput local %(changes_file)s',
        'vcs-git': '',
    }

    def __init__(self):

        # Some hardcoded commands that should go away.
        self.commands = {
            'set-debian-push': (self.set_debian_push, ''),
            'checkout-debian': (self.checkout_debian, ''),
        }

        # TODO: dynamically read other files and their actions
        self.commands.update(actions.AVAILABLE_ACTIONS)
        self.config = None
        self._cache = {}
        self.packages = {}
        self._architecture = None
        self._build_images = collections.defaultdict(bool)

    def list_commands(self):
        return self.commands.keys()

    def show_commands_help(self):
        result = ['\nAvailable commands:']
        commands = sorted(actions.AVAILABLE_ACTIONS.items())
        for command, (classname, helptext) in commands:
            result.append('  %-22s%s.' % (command, helptext))
        return '\n'.join(result)

    @property
    def architecture(self):
        if not self._architecture:
            self._architecture = subprocess.check_output(
                ['dpkg-architecture', '-qDEB_BUILD_ARCH']).rstrip('\n')
        return self._architecture

    def update_config(self, config_files=None):
        '''Read the configuration file, update package sets.'''
        if config_files:
            self.config_files = util.split(config_files)
        else:
            self.config_files = self.defaults['config_files']
        self.config_files = map(os.path.expanduser, self.config_files)
        self.config = ConfigParser.SafeConfigParser()
        self.config.read(self.config_files)

        self._cache['_modules'] = set()
        self._cache['_packages'] = set()
        self._cache['_parents'] = collections.defaultdict(set)

        for section in self.config.sections():
            if self.config.has_option(section, 'packages'):
                self._cache['_modules'].add(section)
                section_packages = util.split(
                    self.config.get(section, 'packages'))
                self._cache['_packages'].update(section_packages)
                for package in section_packages:
                    self._cache['_parents'][package].add(section)
            else:
                self._cache['_packages'].add(section)

    def get_config_option(self, section, option, raw=False, inherit=True):
        '''Obtain the requested config option for section.'''
        if not self.config:
            self.update_config()
        if self.config.has_section(section) and \
                self.config.has_option(section, option):
            return self.config.get(section, option, raw=raw)
        elif section == 'DEFAULT' and self.config.has_option(section, option):
            return self.config.get(section, option, raw=raw)

        if inherit:
            # Inherit configuration options from parent modules
            done = set([section])
            queue = collections.deque(self._cache['_parents'][section])
            while queue:
                parent = queue.popleft()
                if parent in done:
                    continue
                done.add(parent)

                if self.config.has_option(parent, option):
                    return self.config.get(parent, option, raw=raw)
                queue.extend(self._cache['_parents'][parent])

        return self.defaults.get(option, None)

    def write_config(self):
        '''Writes any changes to the config file to disk.'''
        config_file = self.config_files[-1]
        if os.path.exists(config_file):
            shutil.copyfile(config_file, config_file + '.bak')

        config_open = open(config_file, 'w')
        self.config.write(config_open)
        config_open.close()

    def add_new_package(self, package_name, git_url, path, debian_branch,
                        upstream_branch, pristine_tar):
        '''Adds a new package to the configuration.'''

        if package_name in self.list_all():
            logging.error(
                'Package %s definition already in the config file. '
                'Not adding it.', package_name)
            return False

        basedir = self.get_config_option('DEFAULT', 'basedir')
        basedir = os.path.expanduser(basedir)

        if path.startswith(basedir):
            path = '%(basedir)s' + path[len(basedir):]

        self.config.add_section(package_name)
        self.config.set(package_name, 'vcs-git', git_url)
        self.config.set(package_name, 'path', path)
        self.config.set(package_name, 'debian-branch', debian_branch)
        self.config.set(package_name, 'pristine-tar', str(pristine_tar))

        if (upstream_branch):
            self.config.set(package_name, 'upstream-branch', upstream_branch)

        self.write_config()
        self.update_config()

        return True

    def list_modules(self):
        if not self.config:
            self.update_config()

        return self._cache['_modules']

    def list_packages(self):
        if not self.config:
            self.update_config()

        return self._cache['_packages']

    def list_all(self):
        return self.list_modules() | self.list_packages()

    def call(self, cmd, argv):
        '''Execute the command the user requested.'''
        exit_code = 0
        action = self.commands[cmd][0]
        if inspect.isclass(action):
            try:
                instance = action(self, argv)
                exit_code = instance.run()
                instance.print_status()
                return exit_code
            except actions.ActionError as e:
                logging.critical(e.message)
                return -1

        # Default argument parser.
        parser = argparse.ArgumentParser()
        # Empty list means all modules:
        names_choices = set(['']) | self.list_all()
        parser.add_argument('names', nargs='*', choices=names_choices,
                            default='')
        options = parser.parse_args(argv)
        names = options.names
        if not names:
            names = self.list_modules()
        for name in names:
            res = self.commands[cmd][0](name)
            if res:
                exit_code = res
        return exit_code

    def _read_packages(self, name):
        '''Returns the set of packages in a particular module.'''
        if name in self._cache:
            return self._cache[name].packages

        config_packages = self.get_config_option(name, 'packages',
                                                 inherit=False)
        if config_packages:
            packages = set(util.split(config_packages))
        else:
            packages = set((name,))

        return packages

    def _read_depends(self, name):
        '''Returns the dependencies for a particular module.'''
        if name in self._cache:
            return self._cache[name].depends

        raw_depends = self.get_config_option(name, 'depends')
        not_expanded = set()
        if raw_depends:
            not_expanded.update(util.split(raw_depends))

        if name in not_expanded:
            not_expanded.remove(name)

        depends = set()
        for dependency in not_expanded:
            # Do not expand parents
            if name in self._cache['_parents'] and \
                    dependency in self._cache['_parents'][name]:
                # Add it as a package
                if dependency in self._cache['_packages']:
                    depends.add(dependency)
                continue
            depends.update(self._read_packages(dependency))

        if name in depends:
            depends.remove(name)

        return depends

    def get_module(self, name):
        '''Returns a Module object.'''
        if name in self._cache:
            return self._cache[name]

        packages = self._read_packages(name)
        basedir = self.get_config_option(name, 'basedir')
        depends = self._read_depends(name)

        self._cache[name] = moo.Module(name, packages, basedir, depends)
        return self._cache[name]

    def get_package(self, name):
        '''Returns a Package object.'''
        if name not in self.packages:
            self.packages[name] = moo.Package(self, self.get_module(name))
        return self.packages[name]

    def get_orig(self, directory, package, version):
        file_glob = '%s_%s.orig.*' % (package, version)
        ls = glob.glob(os.path.join(directory, file_glob))
        return ls[0]

    def build_depends_graph(self, pending, ignored):
        # TODO: this should be in moo.py
        # TODO: this should also be a class of its own
        # build the graph
        graph = collections.defaultdict(lambda: moo.Node(set(), set()))

        visited = set()
        to_visit = set(pending)
        ready = collections.deque()

        while to_visit:
            name = to_visit.pop()
            if name in visited:
                continue
            visited.add(name)

            package = self.get_package(name)
            depends = package.depends

            ignored |= depends - pending
            depends &= pending

            if not depends:
                ready.append(name)
                continue

            graph[name].input.update(depends)
            for input in depends:
                graph[input].output.add(name)
            to_visit.update(depends)

        return graph, ready

    def update_depends_graph(self, graph, package_name, ready):
        ''' Remove the built package from the needed dependencies of depending
            packages.
        '''
        for child in graph[package_name].output:
            graph[child].input.remove(package_name)
            # if there are no more dependencies, the package can be built
            if not graph[child].input:
                ready.append(child)

    def sort_buildable(self, packages, done=None, ignored=None, error=None):
        if done is None:
            done = set()
        if ignored is None:
            ignored = set()
        if error is None:
            error = set()

        pending = set(packages)
        graph, ready = self.build_depends_graph(pending, ignored)

        # ready is a set that contains the packages ready to be built
        while ready:
            name = ready.popleft()
            yield name
            if name in error:
                continue
            done.add(name)
            self.update_depends_graph(graph, name, ready)

        if len(done) != len(packages) and not error:
            not_ready = set(name for name in graph if graph[name].input)
            error.update(not_ready)
            raise moo.GraphError('Not a DAG?')
        elif error:
            not_ready = set(name for name in graph if graph[name].input)
            error.update(not_ready)

    def set_debian_push(self, name):
        module = self.get_module(name)

        # TODO: fix, ugly ugly
        for package_name in module.packages:
            package = self.get_package(package_name)

            try:
                remote = package.git.config('branch.%s.remote' % (
                    package.debian_branch,))
            except git.exc.GitCommandError:
                remote = 'origin'

            try:
                ref = package.git.config('branch.%s.merge' % (
                    package.debian_branch,))
            except git.exc.GitCommandError:
                ref = 'refs/heads/%s' % (package.debian_branch,)

            try:
                package.git.config('--get', 'remote.%s.push' % (remote,),
                                   ref)
            except git.exc.GitCommandError:
                package.git.config('--add', 'remote.%s.push' % (remote,),
                                   ref)

    def checkout_debian(self, name):
        module = self.get_module(name)

        for package_name in module.packages:
            package = self.get_package(package_name)
            if not package.switch_branches(package.debian_branch):
                logging.error('Failure while switching branches for: %s',
                              package_name)

    def ensure_builder_image(self, distribution, architecture):
        '''Once per run, checks that the build image is up to date.'''

        distribution = distribution.lower()
        if self._build_images[(distribution, architecture)]:
            return

        env = {'ARCH': architecture, 'DIST': distribution}

        # First tries to update, if it fails, tries to create.
        try:
            cmd = ['git-pbuilder', 'update']
            logging.info('Updating build image for %s-%s', architecture,
                         distribution)
            subprocess.check_call(cmd, env=dict(os.environ, **env))
        except subprocess.CalledProcessError:
            # Distribution unreleased is usually intended to be unstable.
            # If the user wants something different, they can create the file or
            # symlink it manually to something else
            if distribution == 'unreleased':
                env['GIT_PBUILDER_OPTIONS'] = '--distribution=unstable'

            logging.warning('Build image for %s-%s not found. Creating...',
                            architecture, distribution)
            cmd = ['git-pbuilder', 'create']
            subprocess.check_call(cmd, env=dict(os.environ, **env))

        self._build_images[(distribution, architecture)] = True


def main():

    arriero = Arriero()
    cmds = arriero.list_commands()

    parser = argparse.ArgumentParser(
        description=__doc__, epilog=arriero.show_commands_help(),
        formatter_class=argparse.RawTextHelpFormatter)
    parser.add_argument('-c', '--config', help='Specify a config file.',
                        metavar='FILE')
    parser.add_argument('-v', '--verbose', action='store_true',
                        help='Show more information.')
    parser.add_argument('-q', '--quiet', action='store_true',
                        help='Show only critical errors.')
    parser.add_argument('command', choices=cmds, metavar='COMMAND',
                        help='Command to execute. See options below.')
    options, argv = parser.parse_known_args()

    if options.config:
        arriero.update_config(options.config)

    # Initialize logging
    logging.basicConfig(format='[%(levelname)s] %(message)s',
                        level=logging.WARNING)
    logger = logging.getLogger()
    if options.verbose:
        logger.setLevel(logging.DEBUG)
    elif options.quiet:
        logger.setLevel(logging.CRITICAL)

    try:
        exit_code = arriero.call(options.command, argv)
    except argparse.ArgumentError as error:
        logging.error('Error while parsing arguments: %s', error)

    return exit_code

if __name__ == '__main__':
    exit_code = main()
    sys.exit(exit_code)

# vi:expandtab:softtabstop=4:shiftwidth=4:smarttab
