/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	KDE mp3 burner frontend. This file contains
 *	main(), the application class, utility class
 *	implementations, and utility functions.
 *
 *	by Tony Sideris	(05:28AM Jul 31, 2001)
 *================================================*/
#include "arson.h"

#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#include <qfileinfo.h>
#include <qstring.h>
#include <qxml.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qheader.h>

#include <kio/netaccess.h>
#include <kcmdlineargs.h>
#include <kmessagebox.h>
#include <kaboutdata.h>
#include <dcopclient.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kstddirs.h>
#include <kglobal.h>
#include <knotifyclient.h>

#ifdef ARSON_KDE3
#include <kstdguiitem.h>
#endif

#include "progressdlg.h"
#include "listwnd.h"
#include "mainwnd.h"
#include "konfig.h"
#include "cdwriter.h"
#include "cdcopy.h"
#include "isofs.h"
#include "docwidget.h"
#include "logwnd.h"
#include "process.h"
#include "infprogbar.h"
#include "tempfile.h"
#include "tools.h"

#ifdef ARSONDBG
arson_instance_counter g_INSTANCE_COUNTER;
#endif	//	ARSONDBG

KCmdLineOptions g_cmdline_options[] = {
	{ "w", 0, 0 },
	{ "write <file>", I18N_NOOP("Burn image file (iso, cue) or directory, then quit"), 0 },
	{ "p", 0, 0 },
	{ "write-prompt", I18N_NOOP("Prompt for an image file, then burn it"), 0 },
	{ "c", 0, 0 },
	{ "copy", I18N_NOOP("CD-to-CD Copy, then quit"), 0 },
	{ "+[arg]", I18N_NOOP(".arson files..."), 0 },
	{ 0, 0, 0 },
};

/*========================================================*/
/*	Main
 *========================================================*/

int main (int argc, char **argv)
{
	KAboutData about(
		PACKAGE,
		I18N_NOOP(DISPLAYNAME),
		VERSION,
		I18N_NOOP(DESCRIPTION),
		KAboutData::License_GPL,
		"(c) 2001,2002 Tony Sideris",
		I18N_NOOP("A KDE frontend for CD burning, and CD ripping."),
		I18N_NOOP(HOMEPAGE),
		I18N_NOOP(BUGEMAIL)
		);

	about.addAuthor(MYNAME, 0, BUGEMAIL, HOMEPAGE);
	about.addAuthor("Markus Triska",
		I18N_NOOP("Temporary folder option, decoded audio file caching."),
		"triska@gmx.at", "http://triskam.virtualave.net");

	KCmdLineArgs::init(argc, argv, &about);
	KCmdLineArgs::addCmdLineOptions(g_cmdline_options);
	KApplication::addCmdLineOptions();

	return ArsonApp(argc, argv).main();
}

/*========================================================*/

void ArsonError::report (void) const
{
	const QString &desc = description();

	if (desc != QString::null)
		arsonErrorMsg(desc);
}

void ArsonStartupError::report (void) const
{
	if (arsonWarning(description() + i18n("\nContinue?"), m_cfgKey))
		throw 0xF00;
}

/*========================================================*/
/*	Utility functions
 *========================================================*/

#if defined(ARSONDBG) || defined(ARSONTRACE)
#include <stdarg.h>

void Ts_Trace (const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);

	vfprintf(stderr, fmt, ap);
	va_end(ap);
}
#endif

/*========================================================*/

QString arsonDisplayTime (int secs)
{
	QString temp;

	temp.sprintf("%u:%02d",
		secs / 60,
		secs % 60);

	return temp;
}

/*========================================================*/

#define InBetween(i,n,m)	((i)>=(n)&&(i)<=(m))
#define KILOBYTE		(1024)
#define MEGABYTE		(KILOBYTE*KILOBYTE)
#define GIGABYTE		(MEGABYTE*KILOBYTE)
#define BYTEARG(n)		arg((n), 0, 'f', 2)

QString arsonByteSize (uint nbytes)
{
	if (nbytes < KILOBYTE)
		return i18n("%1").arg(nbytes);

	else if (InBetween(nbytes, KILOBYTE, MEGABYTE))
		return i18n("%1 KB").BYTEARG(double(nbytes) / double(KILOBYTE));

	else if (InBetween(nbytes, MEGABYTE, GIGABYTE))
		return i18n("%1 MB").BYTEARG(double(nbytes) / double(MEGABYTE));

	return i18n("%1 GB").BYTEARG(double(nbytes) / double(GIGABYTE));
}

/*========================================================*/

bool arsonIsExecutable (const char *path)
{
	QFileInfo fi (QFile::encodeName(path));

	if (!fi.isDir() && fi.isExecutable())
		return true;

	return false;
}

/*========================================================*/

QCString arsonDataFile (const char *filename, const char *res, bool local)
{
	QCString result;

	if (!res)
		res = "appdata";

	if (local)
		result = locateLocal(res, filename);
	else
	{
		/*	In debug builds we want the data files in the
		 *	current working dir, these are the devel files
		 *	not the installed copies...
		 */
#ifndef ARSONDBG
		KStandardDirs *dirs = KGlobal::dirs();

		result = dirs->findResource(res, filename);
#endif	//	ARSONDBG
	}

	if (result.isEmpty())
		result = filename;

	return QFile::encodeName(result);
}

/*========================================================*/

void arsonErrorMsg (const QString &msg)
{
	KMessageBox::error(
		ArsonFrame::getFrame(),
		msg, i18n("Error"));
}

bool arsonWarning (const QString &msg, const QString &save_at)
{
	if (save_at != QString::null)
		return (KMessageBox::warningContinueCancel(ArsonFrame::getFrame(),
					msg, i18n("Error"),
#ifdef ARSON_KDE3
					KStdGuiItem::guiItem(KStdGuiItem::Continue),
#else
					i18n("Continue"),
#endif	//	ARSON_KDE3
					save_at,
					true) == KMessageBox::Continue);
	else
		return (KMessageBox::warningContinueCancel(ArsonFrame::getFrame(),
					msg, i18n("Error"),
#ifdef ARSON_KDE3
					KStdGuiItem::guiItem(KStdGuiItem::Continue)
#else
					i18n("Continue")
#endif	//	ARSON_KDE3
					) == KMessageBox::Continue);
}

void arsonNotify (const QString &msg)
{
	KNotifyClient::event(KNotifyClient::notification, msg);
}

/*========================================================*/

void arsonSaveLogWindow (QListBox *pl, const QString &fn)
{
	QString path (fn);

	if (fn == QString::null)
		path = KFileDialog::getSaveFileName();

	if (path != QString::null)
	{
		QFile file (QFile::encodeName(path));

		//	Open the selected file for output
		if (!file.open(IO_WriteOnly | IO_Truncate))
			arsonErrorMsg(
				i18n("Error opening %1 for writing").arg(path));
		else
		{
			const int count = pl->count();

			//	Write each line of the output box to the file
			for (int index = 0; index < count; ++index)
			{
				const QString text = pl->text(index) + "\n";

				file.writeBlock(text, text.length());
			}

			file.close();
		}
	}
}

/*========================================================*/
/*	Open an XML document
 *========================================================*/

bool arsonOpenDomDocument (const KURL &url, QDomDocument &dom)
{
	ArsonNetFile nf (url);

	if (arsonOpenDomDocument(nf.path(), dom))
		return true;

	return false;
}

bool arsonOpenDomDocument (const QString &fn, QDomDocument &dom)
{
	QFile file (QFile::encodeName(fn));

	if (file.open(IO_ReadOnly))
	{
		dom.setContent(&file);
		return true;
	}

	Trace("Failed to open DOM Document: %s\n",
		(const char *) QFile::encodeName(fn));
	return false;
}

/*========================================================*/
/*	Open an arson document (.arson)
 *========================================================*/

class arsonDocHandler : public QXmlDefaultHandler
{
public:
	arsonDocHandler (void)
		: QXmlDefaultHandler(),
		m_pLI(NULL),
		m_pDoc(NULL)
	{}

	~arsonDocHandler (void) { delete m_pLI; }

	ArsonDocWidget *doc (void) { return m_pDoc; }

	virtual bool startElement (const QString &ns, const QString &local,
		const QString &name, const QXmlAttributes &attr)
	{
		if (!m_pDoc)
		{
			ArsonFrame *pFrm = ArsonFrame::getFrame();

			if (pFrm && (m_pDoc = pFrm->findDocument(name)))
			{
				ArsonLvPos pos;

				m_pDoc->newDocument();
				m_pDoc->setSizeCanChange(false);

				m_pDoc->defaultLvPos(pos);
				m_pLI = new ArsonListInserter(pos.parent, pos.after);

				m_pDoc->onStartElement(name, attr, *m_pLI);
				return true;
			}

			return false;
		}
		else
			return m_pDoc->onStartElement(name, attr, *m_pLI);
	}

	virtual bool endElement (const QString &ns,
		const QString &local, const QString &name)
	{
		if (m_pDoc)
			return m_pDoc->onEndElement(name, *m_pLI);

		return false;
	}

	virtual bool endDocument (void)
	{
		if (m_pDoc)
			m_pDoc->setSizeCanChange(true);
		
		return true;
	}

private:
	ArsonListInserter *m_pLI;
	ArsonDocWidget *m_pDoc;
};

void arsonOpenDoc (const KURL &url)
{
	arsonDocHandler adh;

	if (arsonParseXml(url, &adh))
	{
		KRecentFilesAction *pa = (KRecentFilesAction *)
			ArsonFrame::getFrame()->actionCollection()
			->action("file_open_recent");

		if (pa)
			pa->addURL(url);
		
		adh.doc()->setDocumentName(url);
		adh.doc()->setModified(false);
	}
}

/*========================================================*/
/*	XML parser utilities
 *========================================================*/

bool arsonParseXml (QTextStream *ps, QXmlContentHandler *ph)
{
	QXmlInputSource input (*ps);
	QXmlSimpleReader rdr;

	rdr.setContentHandler(ph);

	if (!rdr.parse(input))
	{
		Trace("XML reader failed! reason:\n%s\n",
			ph->errorString().latin1());

		return false;
	}

	return true;
}

bool arsonParseXml (const KURL &url, QXmlContentHandler *ph)
{
	ArsonNetFile nf (url);
	QFile file (nf.path());

	if (file.open(IO_ReadOnly))
	{
		QTextStream ts (&file);
		return arsonParseXml(&ts, ph);
	}

	return false;
}

/*========================================================*/
/*	Arson application class
 *========================================================*/

ArsonApp::ArsonApp (int argc, char **argv)
	: KApplication()
{
	srand(time(NULL));
	srand(rand());
	
	dcopClient()->registerAs(name(), false);
}

/*========================================================*/
/*	Handle all arson-specific command line parameters
 *========================================================*/

KCmdLineArgs *ArsonApp::handleCmdLine (void)
{
	KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
	QCString imgfile = args->getOption("write");

#define TRACEARG(arg)	if (!(arg).isNull()) Trace("%s: %s\n", #arg, (const char *) arg);

	TRACEARG(imgfile);

	if (!imgfile.isNull() || args->isSet("write-prompt"))
		arsonWriteImg(imgfile);
	else if (args->isSet("copy"))
		arsonCdCopy();
	else
		return args;

	return NULL;
}

/*========================================================*/
/*	Main program.
 *========================================================*/

int ArsonApp::main (void)
{
	int result = 1;
	ArsonFrame *pWnd;
	
	if(isRestored())
		RESTORE(ArsonFrame)
	else
	{
		//	Load the user's configuration
		try {
			ACONFIG.load();
		}
		catch (ArsonStartupError &err)
		{
			try { err.report(); }
			catch (...) {
			}
		}

		/*	Check command line args... will return NULL
		 *	is a single operation was specified on the
		 *	command line, otherwise, operate normally.
		 */
		if (KCmdLineArgs *args = handleCmdLine())
		{
			pWnd = new ArsonFrame;

			pWnd->setGeometry(ACONFIG.geometry());
			setMainWidget(pWnd);
			pWnd->show();

			/*	Open any .arson or .md5 file
			 *	specified on the command line
			 */
			if (args->count() > 0)
			{
				const QCString arg (args->arg(0));

				if (arg.right(3).lower() == "md5")
				{
					ArsonFileListDoc *pd = (ArsonFileListDoc *)
						pWnd->setCurrentDocument(DOCUMENT_AUDIO);

					pd->openMd5File(QString("file:") + arg.data());
				}
				else
					arsonOpenDoc(QString("file:") + arg.data());
			}

			//	Run program, run...
			result = exec();

			ArsonLogWindow::finalize();
			ARSON_INSTANCE_DUMP_ALL();
		}

		//	Save user's config
		ACONFIG.save();
	}

	ArsonTempFile::deleteTempFiles();
	return result;
}

/*========================================================*/
/*	A net-aware filename
 *========================================================*/

ArsonNetFile::ArsonNetFile (const KURL &url)
	: m_local(QString::null)
{
//	Trace("ArsonNetFile: %s\n", url.path().latin1());

	if ((m_bLocal = url.isLocalFile()))
		m_local = url.path();

	else if (!KIO::NetAccess::download(url, m_local))
		throw ArsonError(
			i18n("Failed to download %1")
			.arg(url.path()));
}

ArsonNetFile::~ArsonNetFile (void)
{
	if (!m_bLocal && m_local != QString::null)
		QFile::remove(QFile::encodeName(m_local));
}

/*========================================================*/
/*	A file filter builder
 *========================================================*/

ArsonFileFilter::ArsonFileFilter (const QString &init, uint flags)
	: m_flags(flags)
{
	fromString(init);
}

ArsonFileFilter::ArsonFileFilter (const QString *arr, size_t cnt, uint flags)
	: m_flags(flags)
{
	for (int index = 0; index < cnt; ++index)
		addFilter(arr[index]);
}

ArsonFileFilter::ArsonFileFilter (ArsonFileListDoc *pd, uint flags)
	: m_flags(flags)
{
	pd->buildFileFilter(*this);
}

/*========================================================*/

void ArsonFileFilter::Filter::fromString (const QString &str)
{
	const int at = str.find('|');

	if (at != -1)
	{
		m_filter = str.left(at);
		m_desc = str.right(str.length() - at - 1);
	}
}

void ArsonFileFilter::fromString (const QString &str)
{
	const QStringList sl = QStringList::split(QString("\n"), str);

	for (QStringList::ConstIterator it = sl.begin(), end = sl.end();
		 it != end; ++it)
		addFilter((*it));
}

/*========================================================*/

QString ArsonFileFilter::toString (void) const
{
	QString result, all;

	for (FILTERS::ConstIterator it = m_filters.begin(), end = m_filters.end();
		 it != end; ++it)
	{
		if (!result.isEmpty())
			result.append("\n");

		if (m_flags & AllTypes &&
			(*it).filter() != "*" && (*it).filter() != "*.*")
		{
			if (!all.isEmpty())
				all.append(" ");

			all.append((*it).filter());
		}

		result.append((*it).toString());
	}

	if (m_flags & AllFiles)
		result.append(QString("\n") + Filter(i18n("All Files"), "*").toString());

	if (m_flags & AllTypes)
		return Filter(i18n("All Supported Types"), all).toString() + "\n" + result;

	return result;
}

QString ArsonFileFilter::Filter::toString (void) const
{
	QString result (m_filter + "|" + m_desc);

	if (m_filter.length() <= 16)
		result.append(QString(" (") + m_filter + ")");

	return result;
}

/*========================================================*/
/*	Only the filters, no descriptions (like for QDir)
 *========================================================*/

QString ArsonFileFilter::filters (void) const
{
	QString result;

	for (FILTERS::ConstIterator it = m_filters.begin(), end = m_filters.end();
		 it != end; ++it)
	{
		if ((*it).filter() == "*" || (*it).filter() == "*.*")
			continue;

		if (!result.isEmpty())
			result.append(" ");

		result.append((*it).filter());
	}

	return result;
}

/*========================================================*/

void ArsonFileFilter::addFilter (const QString &desc, const QString &filter)
{
	m_filters.append(Filter(desc, filter));
}

void ArsonFileFilter::addFilter (const QString &str)
{
	m_filters.append(Filter(str));
}

/*========================================================*/
/*	A radio button dialog, so the user can choose an item.
 *========================================================*/

ArsonRadioDlg::ArsonRadioDlg (QWidget *parent)
	: ArsonListviewDlgBase(parent ? parent : kapp->mainWidget(), NULL, true),
	m_group(NULL), m_last(NULL)
{
	setCaption(i18n("Select Disk"));

	teh_list->addColumn("");
	teh_list->header()->hide();

	m_group = new QCheckListItem(teh_list,
		i18n("Disk Sets:"), QCheckListItem::Controller);

	m_group->setPixmap(0, QPixmap());
	m_group->setOpen(true);
}

void ArsonRadioDlg::addItem (const QString &item)
{
	const bool empty = (m_group->firstChild() == NULL);
	QCheckListItem *pi = new QCheckListItem(m_group,
		item, QCheckListItem::RadioButton);

	teh_list->moveItem(pi, m_group, m_last);
	(m_last = pi)->setOn(empty);
}

void ArsonRadioDlg::accept (void)
{
	for (QCheckListItem *pi = (QCheckListItem *) m_group->firstChild();
		 pi; (pi = (QCheckListItem *) pi->nextSibling()))
		if (pi->isOn())
		{
			m_sel = pi->text(0);
			break;
		}

	ArsonListviewDlgBase::accept();
}

/*========================================================*/
