/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Mp3 (actually audio, not just mp3) doc/view
 *	class implementations
 *
 *	by Tony Sideris	(04:32AM Aug 02, 2001)
 *================================================*/
#include "arson.h"

#include <qtextstream.h>
#include <qcheckbox.h>
#include <qfile.h>
#include <qlabel.h>

#include <klocale.h>

#include "audiowriter.h"
#include "tempfile.h"
#include "audiofile.h"
#include "audiodoc.h"
#include "konfig.h"
#include "listwnd.h"

/*========================================================*/
/*	Document class implementation
 *========================================================*/

ArsonMp3Doc::ArsonMp3Doc (QWidget *parent, const char *name)
	: ArsonFileListDoc(parent, name)
{
	//	Nothing...
}

/*========================================================*/

QString ArsonMp3Doc::propDocType (void) const
{
	return "audio";
}

/*========================================================*/

void ArsonMp3Doc::delItem (QListViewItem *ptr)
{
	ArsonFileListFileItem *pi = (ArsonFileListFileItem *) item(ptr);

	if (pi && !pi->isDir())
		ArsonTempFile::deleteTemporaryFor(pi->local());

	ArsonFileListDoc::delItem(ptr);
}

/*========================================================*/

ArsonFileListItem *ArsonMp3Doc::createFileItem (const KURL &url) const
{
	try {
		ArsonMp3ListItem *ptr = new ArsonMp3ListItem(url);
		return ptr;
	}
	catch (ArsonError &err) {
		err.report();
		return NULL;
	}
}

/*========================================================*/

ArsonProcessMgr *ArsonMp3Doc::createCdWriter (ArsonProcessUI *pUI)
{
	ArsonAudioWriter *ptr;

	if (ACONFIG.seq().seq(PROGGRP_AUDIOCD) == AUDIOSEQ_CDRECORD)
		ptr = new ArsonTaoWriter(pUI);
	else
		ptr = new ArsonDaoWriter(pUI);

	for (ItemIterator it (this); it; ++it)
		if (ArsonMp3ListItem *pi = (ArsonMp3ListItem *) it.item())
			ptr->addTrack(pi);

	return ptr;
}

/*========================================================*/

void ArsonMp3Doc::buildFileFilter (ArsonFileFilter &filter)
{
	ArsonAudioFile::audioFilter(filter);
}

/*========================================================*/
/*	Custom progress dialog (with Normalize option)
 *========================================================*/

class audioProgressDlg : public ArsonCdWriterProgressDlg
{
public:
	audioProgressDlg (QWidget *parent, ArsonMp3Doc *pDoc)
		: ArsonCdWriterProgressDlg(parent, "audio"), m_pDoc(pDoc)
	{
		QLabel *pl;
		const QString modes[] = {
			i18n("None"),
			i18n("Batch Mode"),
			i18n("Mix Mode"),
		};

		pl = new QLabel(i18n("&Normalization:"), ctrlParent(), "normlabel");
		m_pNormalize = new ArsonProgressCombobox(
			ctrlParent(), "normmeth");

		for (int index = 0; index < ARRSIZE(modes); ++index)
			m_pNormalize->insertItem(modes[index]);

		m_pNormalize->setCurrentItem(0);
		pl->setSizePolicy(
			QSizePolicy(
				QSizePolicy::Maximum,
				QSizePolicy::Minimum));
		pl->setBuddy(m_pNormalize);

		m_pOnTheFly = addCheckbox(
			i18n("&On the fly (requires cdrecord as Audio Burner)"), "ontehfly");
		m_pByteSwap = addCheckbox(
			i18n("&Byte swap"), "byteswap");

		QObject::connect(m_pOnTheFly, SIGNAL(toggled(bool)),
			m_pNormalize, SLOT(setDisabled(bool)));
		QObject::connect(m_pOnTheFly, SIGNAL(toggled(bool)),
			pl, SLOT(setDisabled(bool)));

		layoutRow() << pl << m_pNormalize;
	}

protected:
	inline bool usingCdrecord (void) const {
		return (
			ACONFIG.seq().seq(PROGGRP_AUDIOCD) == AUDIOSEQ_CDRECORD);
	}

	inline bool onTheFly (void) const {
		return (usingCdrecord() && m_pOnTheFly->isChecked());
	}

	virtual void processOpts (ArsonProcessOpts &opts)
	{
		ArsonCdWriterProgressDlg::processOpts(opts);
		opts.addLong("normalize", m_pNormalize->currentItem());
		opts.addBool("byteswap", m_pByteSwap->isChecked());
		opts.addBool("onthefly", onTheFly());
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return m_pDoc->createCdWriter(ui());
	}

	virtual void reconfigure (void)
	{
		const bool onthefly = onTheFly();
		const bool normalize = (
			ACONFIG.program(ArsonConfig::PROGRAM_NORMALIZE)->valid() &&
			(!usingCdrecord() || !onthefly));

		ArsonCdWriterProgressDlg::reconfigure();

		m_pNormalize->setEnabled(normalize);
		((QLabel *) ctrlParent()->child("normlabel"))
			->setEnabled(normalize);

		m_pOnTheFly->setEnabled(usingCdrecord());
	}

	QComboBox *m_pNormalize;
	QCheckBox *m_pOnTheFly;
	QCheckBox *m_pByteSwap;
	ArsonMp3Doc *m_pDoc;
};

ArsonProgressDlg *ArsonMp3Doc::createProgressDlg (void)
{
	return new audioProgressDlg(kapp->mainWidget(), this);
}

/*========================================================*/
/*	List item type
 *========================================================*/

ArsonMp3ListItem::ArsonMp3ListItem (const KURL &url)
	: ArsonFileListFileItem(url)
{
	ArsonAudioFile af (local());

	if (!af.initialize())
		throw ArsonError(
			i18n("Not a valid audio file (%1)")
			.arg(url.filename()));

	m_length = af.length();
	m_title = af.title();
}

/*========================================================*/

QString ArsonMp3ListItem::displayTime (void) const
{
	/*	Total byte length of decoded track:
	 *	SAMPPERSEC * CHANS * (BITSPERSAMP / BITSPERBYTE) * LENGTHINSEC
	 */
//	const uint bytes = (44100 * (2 * (16 / 8))) * m_length;

	return arsonDisplayTime(m_length);/*
 +
		QString(" (%1)").arg(arsonByteSize(bytes));*/
}

/*========================================================*/

void ArsonMp3ListItem::refresh (QListViewItem *pi, ArsonDocWidget *pd)
{
	static QPixmap pm;

	if (pm.isNull())
		pm = loadIcon("sound");

	pi->setPixmap(0, pm);
	pi->setText(0, display());
	pi->setText(1, title());
	pi->setText(2, displayTime());
}

/*========================================================*/

QString ArsonMp3Doc::progressString (int length) const
{
	return i18n("Total time: %1 / %2 minutes")
		.arg(arsonDisplayTime(length))
		.arg(getMaxProgress() / 60);
}

int ArsonMp3Doc::getMaxProgress (void) const
{
	return (ACONFIG.cdlenMin() * 60);
}

/*========================================================*/

ArsonListWnd *ArsonMp3Doc::createListWnd (void)
{
	return new ArsonMp3List(this);
}

/*========================================================*/
/*	TOC file class implementation
 *========================================================*/

ArsonTocFile::ArsonTocFile (void)
	: m_bTemp(false)
{
	//	Nothing...
}

ArsonTocFile::~ArsonTocFile (void)
{
	//	Cleanup temporary TOC file
	if (m_bTemp)
		QFile::remove(QFile::encodeName(m_filename));
}

/*========================================================*/

bool ArsonTocFile::addFile (const QString &fname)
{
	m_files.append(fname);
	return true;
}

/*========================================================*/

bool ArsonTocFile::writeFile (const char *filename)
{
	int index;
	QFile *pf = NULL;
	ArsonTempFile *pt = NULL;
	QTextStream *ps = NULL;

	//	Open a text stream
	if (!filename)
	{
		pt = new ArsonTempFile("arson", "toc");
		ps = pt->textStream();

		m_filename = pt->name();
		m_bTemp = true;
	}
	else
	{
		pf = new QFile(QFile::encodeName(filename));
		m_filename = filename;

		if (!pf->open(IO_WriteOnly))
		{
			m_strError = i18n("Failed to open output file %1")
				.arg(filename);
			return false;
		}

		ps = new QTextStream(pf);
	}

	//	Write the TOC file
	for ((*ps) << "CD_DA\n", index = 0; index < size(); index++)
	{
		(*ps) << "TRACK AUDIO\n"
			  << "FILE \"" << m_files[index] << "\" 0\n";
	}

	//	Clean up
	if (pt)
		delete pt;
	else
	{
		pf->close();
		delete ps;
		delete pf;
	}

	return true;
}

/*========================================================*/
