/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	A uniform wrapper around various types of
 *	audio files. Any information that depends on
 *	the type (format) of audio file, should come
 *	from here.
 *
 *	by Tony Sideris	(06:23PM Sep 12, 2001)
 *================================================*/
#include "arson.h"

#include <qfileinfo.h>
#include <qfile.h>

#include <kio/netaccess.h>
#include <klocale.h>
#include <kstddirs.h>

#include "audiofile.h"
#include "tempfile.h"
#include "mp3info.h"
#include "wavinfo.h"
#include "process.h"

#include <fstream>
#include <utility>

#ifdef FLAC
/*
extern "C" {
#	include <FLAC/stream_decoder.h>
#	include <FLAC/file_decoder.h>
}
*/
#	include <FLAC/metadata.h>
#endif	//	FLAC

#ifdef OGG
#	include <vorbis/vorbisfile.h>
#endif	//	OGG

/*========================================================*/
/*	Base audio file class
 *========================================================*/

ArsonAudioBase::ArsonAudioBase (const QString &path)
	: m_filename(path)
{}

/*========================================================*/
/*	A base for non-wav files (files that require decoding)
 *========================================================*/

class encodedBase : public ArsonAudioBase
{
public:
	encodedBase (const QString &path)
		: ArsonAudioBase(path)
	{}

	virtual QString decodedFile (void)
	{
		return ArsonTempFile::temporaryFor(filename());
	}

protected:
	bool wantDecoder (void) const
	{
		return !QFileInfo(ArsonTempFile::temporaryFor(filename())).exists();
	}
};


/*========================================================*/
/*	An mp3 audiofile class
 *========================================================*/

class file_mp3 : public encodedBase
{
public:
	file_mp3 (const QString &path)
		: encodedBase(path),
		m_info(NULL)
	{}

	~file_mp3 (void)
	{ delete m_info; }

	virtual ArsonAudioDecoderProcess *decoder(
		ArsonProcessMgr *pMgr, const char *outfile)
	{
		if (wantDecoder())
			return new ArsonMpg123Process(pMgr, filename(), outfile);

		return NULL;
	}

	virtual QString title (void)
	{
		if (initialize())
		{
			const char *sz;
			const mp3::Tag *pt = m_info->tag();
			const mp3::Id3v2 *pi = m_info->id3v2();

			if (pi && (sz = pi->value(mp3::Id3v2::TITLE)))
				return sz;

			if (pt)
				return pt->title();
		}

		return QString();
	}
	
	virtual ulong length (void)
	{
		if (initialize())
			return m_info->playtime();

		return 0;
	}

	virtual bool initialize (void)
	{
		if (!m_info)
		{
			m_info = new mp3::Info;

			if (!m_info->load(filename()))
			{
				delete m_info;
				m_info = NULL;
			}
		}

		return (m_info != NULL);
	}

private:
	mp3::Info *m_info;
};

/*========================================================*/
/*	A class for ogg vorbis files
 *========================================================*/
#ifdef OGG
class file_ogg : public encodedBase
{
public:
	file_ogg (const QString &path)
		: encodedBase(path),
		m_file(NULL)
	{}

	~file_ogg (void)
	{
		if (m_file)
			ov_clear(&m_ov);
	}

	virtual ArsonAudioDecoderProcess *decoder(
		ArsonProcessMgr *pMgr, const char *outfile)
	{
		if (wantDecoder())
			return new ArsonOgg123Process(pMgr, filename(), outfile);

		return NULL;
	}

	virtual QString title (void)
	{
		if (vorbis_comment *vc = ov_comment(&m_ov, -1))
		{
			Trace("Vendor: %s\n", vc->vendor);
			
			for (int index = 0; index < vc->comments; ++index)
			{
				Trace("OVComment: %s\n", vc->user_comments[index]);

				if (!qstrncmp(vc->user_comments[index], "title=", 6))
					return QString(vc->user_comments[index] + 6);
			}
		}

		return QString();
	}

	virtual ulong length (void)
	{
		if (initialize())
			return (ulong) ov_time_total(&m_ov, -1);

		return 0;
	}

	virtual bool initialize (void)
	{
		if (!m_file)
		{
			if (!(m_file = fopen(filename(), "rb")))
				return false;

			if (ov_open(m_file, &m_ov, NULL, 0))
			{
				fclose(m_file);
				m_file = NULL;
				return false;
			}
		}

		return true;
	}

private:
	OggVorbis_File m_ov;
	FILE *m_file;
};
#endif	//	OGG
/*========================================================*/
/*	The FLAC class
 *========================================================*/
#ifdef FLAC
class file_flac : public encodedBase
{
public:
	file_flac (const QString &path)
		: encodedBase(path),
		m_samples(0), m_secs(0) { }
	
	virtual ArsonAudioDecoderProcess *decoder(
		ArsonProcessMgr *pMgr, const char *outfile)
	{
		if (wantDecoder())
			return new ArsonFlacDecoderProcess(pMgr, filename(), outfile);

		return NULL;
	}

	virtual QString title (void)
	{
		return QString();	//	FIXME
	}

	virtual ulong length (void)
	{
		if (initialize())
			return m_secs;

		return 0;
	}

	virtual bool initialize (void)
	{
		if (!m_secs || !m_samples)
		{
			FLAC__StreamMetadata sd;

			if (!FLAC__metadata_get_streaminfo(filename().latin1(), &sd))
				return false;

			m_samples = sd.data.stream_info.total_samples;
			m_secs = m_samples / sd.data.stream_info.sample_rate;
/*
			FLAC__FileDecoder *pd;
			
			if (!(pd = FLAC__file_decoder_new()))
				return false;

			FLAC__file_decoder_set_filename(pd, filename().latin1());
			FLAC__file_decoder_set_md5_checking(pd, false);
			FLAC__file_decoder_set_metadata_callback(pd, on_metadata);
			FLAC__file_decoder_set_error_callback(pd, on_error);
			FLAC__file_decoder_set_write_callback(pd, on_write);
			FLAC__file_decoder_set_client_data(pd, this);

			if (FLAC__file_decoder_init(pd) == FLAC__FILE_DECODER_OK)
			{
				FLAC__file_decoder_process_metadata(pd);
				FLAC__file_decoder_finish(pd);
			}

			FLAC__file_decoder_delete(pd);
*/
		}

		return (m_secs || m_samples);
	}

private:	//	FLAC callbacks
/*
	static void on_metadata (const FLAC__FileDecoder *pd,
		const FLAC__StreamMetaData *pm, void *pu)
	{
		if (pm->type == FLAC__METADATA_TYPE_STREAMINFO)
		{
			const FLAC__StreamMetaData_StreamInfo *pi = &(pm->data.stream_info);
			file_flac *pf = (file_flac *) pu;

			pf->m_secs = pi->total_samples / pi->sample_rate;
			pf->m_samples = pi->total_samples;
		}
	}

	static FLAC__StreamDecoderWriteStatus on_write (const FLAC__FileDecoder *pd,
		const FLAC__Frame *pf, const FLAC__int32 *const buffer[], void *pc)
	{
		return FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE;
	}

	static void on_error (const FLAC__FileDecoder *pd,
		FLAC__StreamDecoderErrorStatus stat, void *pc)
	{
		Trace("received FLAC error: %d\n", stat);
	}
*/
	ulong m_samples;
	ulong m_secs;
};
#endif	//	FLAC
/*========================================================*/
/*	A wav file class
 *========================================================*/

class file_wav : public ArsonAudioBase
{
public:
	file_wav (const QString &path)
		: ArsonAudioBase(path),
		m_hdr(NULL)
	{}

	~file_wav (void)
	{ delete m_hdr; }

	virtual QString decodedFile (void)
	{
		QString tmpname (ArsonTempFile::tempFileName("wav", "wav"));
		const QFileInfo fi (tmpname);
		const QString url ("file:");

		//	Backup WAV files so that we never modify originals
		if (//(!fi.exists() || !fi.size()) &&
			!KIO::NetAccess::upload(filename(), url + tmpname))
		{
			arsonErrorMsg(
				i18n("Failed to backup WAV file %1 to %2:\n%3")
				.arg(filename())
				.arg(tmpname)
				.arg(KIO::NetAccess::lastErrorString()));

			tmpname = QString::null;
		}

		return tmpname;
	}

	virtual ulong length (void)
	{
		if (initialize())
			return wav::seconds(
				m_hdr->format(),
				m_hdr->dataLen());
		
		return 0;
	}

	virtual bool initialize (void)
	{
		if (!m_hdr)
		{
			std::ifstream fin (filename());

			if (fin)
			{
				m_hdr = new wav::stdhdr (fin);

				if (!m_hdr->valid())
				{
					delete m_hdr;
					m_hdr = NULL;
				}
			}
		}

		return (m_hdr != NULL);
	}
	
private:
	wav::stdhdr *m_hdr;
};

/*========================================================*/
/*	A SHN file type
 *========================================================*/

class file_shn : public encodedBase
{
public:
	file_shn (const QString &path)
		: encodedBase(path),
		m_length(0)
	{}

	virtual ArsonAudioDecoderProcess *decoder(
		ArsonProcessMgr *pMgr, const char *outfile)
	{
		if (wantDecoder())
			return new ArsonShortenProcess(pMgr, filename(), outfile);

		return NULL;
	}
	
	virtual ulong length (void)
	{
		if (initialize())
			return m_length;

		return 0;
	}

	virtual bool initialize (void)
	{
		QFileInfo fi (filename());

		if (!fi.exists() || !fi.size())
			return false;
		
		if (m_length == 0)
		{
			ArsonShnlenProcess proc (ACONFIG, filename());

			if (proc.execute())
			{
				m_length = proc.seconds();
				return true;
			}

			return false;
		}

		return true;
	}

private:
	int m_length;
};

/*========================================================*/
/*	Interface class.
 *========================================================*/

#define INIT(x)			if(ext == #x) { m_pFile=new file_##x(QFile::encodeName(path)); m_type=type_##x; return; }

ArsonAudioFile::ArsonAudioFile (const QString &path)
	: ArsonAudioBase(path),
	m_type(type_unknown),
	m_pFile(NULL)
{
	QFileInfo fi (QFile::encodeName(path));
	const QString ext = fi.extension(false).lower();

	INIT(mp3);

#ifdef OGG
	INIT(ogg);
#endif	//	OGG

	INIT(wav);
	INIT(shn);

#ifdef FLAC
	INIT(flac);
#endif

	throw ArsonError(i18n("Unsupported or invalid file: %1").arg(path));
}

ArsonAudioFile::~ArsonAudioFile (void)
{
	delete m_pFile;
}

/*========================================================*/
/*	Forward to implementation of proper type.
 *========================================================*/

QString ArsonAudioFile::decodedFile (void)
{
	return m_pFile->decodedFile();
}

ArsonAudioDecoderProcess *ArsonAudioFile::decoder(
	ArsonProcessMgr *pMgr, const char *outfile)
{
	return m_pFile->decoder(pMgr, outfile);
}

QString ArsonAudioFile::title (void)
{
	return m_pFile->title();
}

ulong ArsonAudioFile::length (void)
{
	return m_pFile->length();
}

bool ArsonAudioFile::initialize (void)
{
	return m_pFile->initialize();
}

/*========================================================*/

void ArsonAudioFile::audioFilter (ArsonFileFilter &filter)
{
	const QString arr[] = {
		i18n("*.mp3|Mp3 Files"),
		i18n("*.wav|Wave Files"),
#ifdef OGG
		i18n("*.ogg|Ogg Vorbis Files"),
#endif
		i18n("*.shn|SHN Files"),
#ifdef FLAC
		i18n("*.flac|FLAC Files"),
#endif
	};

	filter = ArsonFileFilter(arr, ARRSIZE(arr));
}

/*========================================================*/
