/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Implements the CD copy dialog
 *
 *	by Tony Sideris	(07:49PM Nov 12, 2001)
 *================================================*/
#include <qradiobutton.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qlabel.h>

#include <kfiledialog.h>
#include <kapp.h>
#include <klocale.h>

#include "tempfile.h"
#include "progressdlg.h"
#include "konfig.h"
#include "cdcopy.h"
#include "mainwnd.h"
#include "process.h"

/*========================================================*/
/*	CD Copy specific progress dialog class
 *========================================================*/

class cdCopyProgressDlg : public ArsonCdWriterProgressDlg
{
public:
	cdCopyProgressDlg (const ArsonCopyOpts &opts)
		: ArsonCdWriterProgressDlg(ArsonFrame::getFrame(), "cdcopy"),
		m_opts(opts)
	{
		message->setText(
			i18n("Please insert the CD to be copied into the source drive, and a blank CDR into the burner, press Start to begin."));

		setCaption(
			i18n("CD Copy"));
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		try
		{
			ArsonProcessMgr *pm;

			if (ACONFIG.seq().seq(PROGGRP_CDCOPY) == COPYSEQ_CDRDAO)
				pm = new ArsonCdrdaoCopyMgr(ui(), m_opts);
			else
			{
				if (!arsonWarning(
						i18n("readcd/cdrecord can only be used to copy data CDs. If you are trying to copy an audio CD, please use cdrdao. Continue?"),
						"audio-warning"))
					return NULL;

				if (m_opts.bDirect)
					pm = new ArsonDirectCopyMgr(ui(), m_opts);
				else
					pm = new ArsonLongCopyMgr(ui(), m_opts);
			}

			return pm;
		}
		catch (ArsonError &err) {
			err.report();
		}

		return NULL;
	}

protected:
	ArsonCopyOpts m_opts;
};

/*========================================================*/
/*	The CD Copy dialog class implementation
 *========================================================*/

ArsonCdCopyDlg::ArsonCdCopyDlg (void)
	: ArsonCdCopyDlgBase(ArsonFrame::getFrame(), NULL, true),
	m_devs(ACONFIG.devices().uiList(true))
{
	ArsonCopyOpts opts;
	opts.load();

	tmp->setText(ArsonTempFile::tempFileName("copy", "iso"));
	m_devs.fillList(dropdev, opts.srcdev);
	direct->setChecked(opts.bDirect);
	del->setChecked(opts.bDelete);
}

/*========================================================*/

void ArsonCdCopyDlg::browse_clicked (void)
{
	const QString file = KFileDialog::getSaveFileName();

	if (file != QString::null)
		tmp->setText(file);
}

/*========================================================*/

bool ArsonCdCopyDlg::directCopy (void) const
{
	return direct->isChecked();
}

/*========================================================*/
/*	OK was pressed, create the progress dialog and the
 *	appropriate process manager.
 *========================================================*/

void ArsonCdCopyDlg::accept (void)
{
	ArsonCopyOpts opts;

	opts.srcdev = m_devs.fromList(dropdev);
	opts.temppath = tmp->text();
	opts.bDelete = del->isChecked();
	opts.bDirect = directCopy();

	if (opts.srcdev != QString::null)
	{
		cdCopyProgressDlg dlg (opts);

		ArsonCdCopyDlgBase::accept();
		opts.save();
		dlg.exec();
	}
}

/*========================================================*/
/*	The various cd copy process managers
 *========================================================*/

ArsonCdCopyMgr::ArsonCdCopyMgr (ArsonProcessUI *pUI, const ArsonCopyOpts &opts)
	: ArsonProcessMgr(pUI), m_opts(opts)
{
	//	Nothing...
}

/*========================================================*/
/*	Reads image from CD, the burns to CDR, in two
 *	seperate steps.
 *========================================================*/

ArsonLongCopyMgr::ArsonLongCopyMgr (ArsonProcessUI *pUI, const ArsonCopyOpts &opts)
	: ArsonCdCopyMgr(pUI, opts), m_stage(stageBegin)
{
	if (m_opts.temppath == QString::null)
		throw ArsonError(
			i18n("No temporary file specified!"));
}

/*========================================================*/

void ArsonLongCopyMgr::removeImg (void)
{
	ArsonProcessUI *pUI = ui();

	IFPTR(pUI)->setText(i18n("Removing temporary image file..."));

//	QFile::remove(QFile::encodeName(m_opts.temppath));
	ArsonFileCleanup() << m_opts.temppath;

	IFPTR(pUI)->setText(i18n("Done..."));
}

/*========================================================*/

void ArsonLongCopyMgr::taskComplete (ArsonProcess *ptr)
{
	switch (m_stage)
	{
	case stageRead:
		try {
			setProcess(
				new ArsonCdrecordDataProcess(
					this, m_opts.temppath));

			m_stage = stageWrite;
		}
		catch (ArsonError &err) {
			err.report();
		}
		break;

	case stageWrite:
		if (m_opts.bDelete)
			removeImg();
		
		setProcess(NULL);
		break;
	}
}

/*========================================================*/

void ArsonLongCopyMgr::abort (void)
{
	ArsonCdCopyMgr::abort();
	removeImg();
}

/*========================================================*/

void ArsonLongCopyMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonCdCopyMgr::begin(opts);

	try {
		setProcess(
			new ArsonReadCdProcess(
				this, m_opts.temppath, m_opts.srcdev));

		m_stage = stageRead;
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	Reads CD into CDR process in one shot
 *========================================================*/

ArsonDirectCopyMgr::ArsonDirectCopyMgr (ArsonProcessUI *pUI, const ArsonCopyOpts &opts)
	: ArsonCdCopyMgr(pUI, opts)
{
	//	Nothing...
}

/*========================================================*/

void ArsonDirectCopyMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonCdCopyMgr::begin(opts);

	try {
		setProcess(
			new ArsonCdrecordIsosizeProcess(
				this, m_opts.srcdev));
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	CDRDAO copy manager
 *========================================================*/

ArsonCdrdaoCopyMgr::ArsonCdrdaoCopyMgr (ArsonProcessUI *pUI, const ArsonCopyOpts &opts)
	: ArsonCdCopyMgr(pUI, opts)
{
	//	Nothing...
}

/*========================================================*/

void ArsonCdrdaoCopyMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonCdCopyMgr::begin(opts);

	try {
		setProcess(
			new ArsonCdrdaoCopyProcess(this,
				m_opts.bDelete,
				m_opts.bDirect,
				m_opts.srcdev));
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	Copy options impl
 *========================================================*/

void ArsonCopyOpts::load (void)
{
	if (KConfig *pk = kapp->config())
	{
		pk->setGroup("cdcopy");
		bDirect = pk->readBoolEntry("direct", false);
		bDelete = pk->readBoolEntry("delete", true);
		srcdev = pk->readEntry("dev");
	}
}

/*========================================================*/

void ArsonCopyOpts::save (void) const
{
	if (KConfig *pk = kapp->config())
	{
		pk->setGroup("cdcopy");

		pk->writeEntry("direct", bDirect);
		pk->writeEntry("delete", bDelete);
		pk->writeEntry("dev", srcdev);
	}
}

/*========================================================*/
