/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	A progress bar that can zoom back and forth
 *	knight-rider style to indicate a process of
 *	unknown length.
 *
 *	by Tony Sideris	(05:49PM Mar 18, 2002)
 *================================================*/
#include "arson.h"

#include <qpalette.h>
#include <qpixmap.h>
#include <qtimer.h>
#include <qpainter.h>
#include <qstyle.h>

#include <kpixmapeffect.h>
#include <kpixmap.h>

#include "infprogbar.h"


#define TS_TIMEOUT		75
#define TS_STEP			8

#define TS_EDGE			1
#define TS_EDGE2		2

#define IMGWIDTH(w)		((w) / 2)

/*========================================================*/

ArsonInfiniteProgressBar::ArsonInfiniteProgressBar (QWidget *parent, const char *name)
	: QProgressBar(parent, name),
	m_nMode(arsonProgressBlank),
	m_bDirLeft(false),
	m_nAniPos(0),
	m_pTimer(NULL)
{
	const QRect inner = getRect();
	m_nAniPos = -(IMGWIDTH(inner.width()));
}

/*========================================================*/

void ArsonInfiniteProgressBar::slotTimeout (void)
{
	const QRect inner = getRect();
	const int step = (int) ((double(inner.width()) * 0.75) / double(1000 / TS_TIMEOUT));

	if (!m_bDirLeft)
	{
		if (m_nAniPos <= inner.width())
			m_nAniPos += step;
		else
			m_bDirLeft = true;
	}
	else
	{
		if (m_nAniPos >= -(IMGWIDTH(inner.width())))
			m_nAniPos -= step;
		else
			m_bDirLeft = false;
	}

	repaint();
}

/*========================================================*/

void ArsonInfiniteProgressBar::createPixmaps (const QSize &size)
{
	if (infinite())
	{
		const QColor left = colorGroup().text();
		const QColor rght = colorGroup().background();

		m_pml.resize(IMGWIDTH(size.width() - TS_EDGE2),
			size.height() - TS_EDGE2);
		m_pmr.resize(IMGWIDTH(size.width() - TS_EDGE2),
			size.height() - TS_EDGE2);

		KPixmapEffect::gradient(m_pml, left, rght,
			KPixmapEffect::HorizontalGradient);
		KPixmapEffect::gradient(m_pmr, rght, left,
			KPixmapEffect::HorizontalGradient);
	}
	else
	{
		m_pml.resize(0,0);
		m_pmr.resize(0,0);
	}
}

/*========================================================*/

QRect ArsonInfiniteProgressBar::getRect (bool edge) const
{
	QRect rc = contentsRect();

	if (!edge)
		return QRect(rc.left() + TS_EDGE, rc.y() + TS_EDGE,
			rc.width() - TS_EDGE2, rc.height() - TS_EDGE2);

	return rc;
}

/*========================================================*/

void ArsonInfiniteProgressBar::setMode (int mode)
{
	if ((m_nMode = mode) == arsonProgressInfinite)
	{
		m_pTimer = new QTimer(this);

		QObject::connect(m_pTimer, SIGNAL(timeout()),
			this, SLOT(slotTimeout()));

		m_pTimer->start(TS_TIMEOUT, false);
	}
	else
	{
		delete m_pTimer;
		m_pTimer = NULL;
	}

	createPixmaps(size());
	repaint();
}

/*========================================================*/

void ArsonInfiniteProgressBar::drawContents (QPainter *ptr)
{
	if (m_nMode != arsonProgressNormal)
	{
		const QRect rc = getRect(true);
		const QRect inner = getRect();

#ifndef ARSON_KDE3
		style().drawPanel(ptr, rc.x(), rc.y(),
			rc.width(), rc.height(), colorGroup(), true, 1,
			&(colorGroup().brush(QColorGroup::Background)));
#else
		style().drawPrimitive(QStyle::PE_Panel,
			ptr, rc, colorGroup(),
			QStyle::Style_Default | QStyle::Style_Sunken,
			QStyleOption(1,0));
#endif	//	ARSON_KDE3

		if (m_nMode == arsonProgressInfinite)
		{
			int srcw = IMGWIDTH(inner.width());

			if (m_nAniPos < 0)	srcw += m_nAniPos;
			else if (m_nAniPos > srcw)
				srcw = inner.width() - m_nAniPos;

			ptr->drawPixmap(
				inner.x() + QMAX(m_nAniPos, 0),
				inner.y(),
				m_bDirLeft ? m_pml : m_pmr,
				(m_nAniPos < 0) ? (m_nAniPos * -1) : 0,
				0,
				srcw,
				inner.height());
		}
	}
	else	//	arsonProgressNormal: Default handling
		QProgressBar::drawContents(ptr);
}

/*========================================================*/

void ArsonInfiniteProgressBar::resizeEvent (QResizeEvent *ptr)
{
	createPixmaps(ptr->size());
	QProgressBar::resizeEvent(ptr);
}

/*========================================================*/
