/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Read header and tag info from Mp3 files.
 *
 *	by Tony Sideris	(05:55AM Aug 03, 2001)
 *================================================*/
#include "mp3info.h"

#ifdef HAVE_CONFIG_H
#	include "../config.h"
#	if STDC_HEADERS
#		include <string.h>
#	else
#		if !HAVE_STRCHR
#			define strrchr rindex
#			define strchr index
#		endif
		char *strchr(), *strrchr();
#		if !HAVE_MEMCPY
#			define memmove(d, s, n) bcopy ((s), (d), (n))
#			define memcpy(d, s, n) bcopy ((s), (d), (n))
#		endif
#	endif
#endif	//	HAVE_CONFIG_H

#ifdef TESTMP3
//#if 0
#	define Mp3out	printf
#else
#	define Mp3out	(true)?0:printf
#endif

namespace mp3 {
	inline ushort ntohs (ushort val)
	{
		byte *buf = (byte *) &val;

		return (
			(buf[0] << 8) |
			(buf[1]));
	}

	inline ulong ntohl (ulong val)
	{
		byte *buf = (byte *) &val;
	
		return (
			(buf[0] << 24) |
			(buf[1] << 16) |
			(buf[2] << 8) |
			(buf[3]));
	}
};

/*========================================================*/
/*	MP3 Info class implementation
 *========================================================*/

mp3::Info::Info (void)
	: m_pVBR(NULL),
	m_pTag(NULL),
	m_pId3(NULL),
	m_filelen(0)
{
	//	Nothing...
}

mp3::Info::~Info (void)
{
	delete m_pId3;
	delete m_pVBR;
	delete m_pTag;
}

/*========================================================*/

#define HEADER_BUFSIZE		(400 * 1024)

bool mp3::Info::load (const char *filename)
{
	FILE *file;

	if ((file = fopen(filename, "rb")))
	{
		ulong nread, pos = 0;
		bool bres = false;
		byte *buf = new byte[HEADER_BUFSIZE];

		//	Get the filesize
		fseek(file, 0, SEEK_END);
		m_filelen = ftell(file);
		rewind (file);

		/*	Read in the first 4kb of the file,
		 *	if no valid headers exist in that
		 *	much data then its prolly not a
		 *	valid file.
		 */
		nread = fread (buf, sizeof(byte),
			HEADER_BUFSIZE, file);

		//	Find the header in the buffer
		if (nread > sizeof(long) && (bres = m_header.find (buf, nread, &pos)))
		{
			//	Don't count the leading pad (Id3V2)
			m_filelen -= (pos - 4);

			//	Skip to the VBR header
			if (m_header.version() == Header::V1)
				pos += (m_header.channelmode() == Header::SINGCHAN) ? 17 : 32;
			else
				pos += (m_header.channelmode() == Header::SINGCHAN) ? 9 : 17;

			m_pVBR = new Vbr;

			//	If there's no VBR then free the object
			if (!m_pVBR->find (buf + pos, nread - pos, &pos))
			{
				delete m_pVBR;
				m_pVBR = NULL;
			}

			if (m_filelen > 128)
			{
				fseek (file, -128L, SEEK_END);

				//	Read in the possible V1 tag
				if (fread (buf, 1, 128, file) == 128)
				{
					m_pTag = new Tag;

					//	Free non-existant V1 tag
					if (!m_pTag->find (buf))
					{
						delete m_pTag;
						m_pTag = NULL;
					}
				}
			}

			fseek(file, 0L, SEEK_SET);
			m_pId3 = new Id3v2;

			if (!m_pId3->read(file))
			{
				delete m_pId3;
				m_pId3 = NULL;
			}
		}

		//	Clean up
		fclose (file);
		delete[] buf;

		return bres;
	}

	return false;
}

/*========================================================*/
/*	FIXME: This func is still fux0red methinks (unless
 *	winamp is wrong), but the playtime() calculation
 *	is right in all tested cases, so fuck it for now...
 *	(only fux0red in non-VBR files)
 *========================================================*/

mp3::ulong mp3::Info::frames (void) const
{
	if (!m_pVBR)
	{
		const double btr = (1000.00 * (double) m_header.bitrate());
		const double ave = (
			(m_header.layer() == Header::L1) ? 12 : 144) *
			btr / (double) m_header.samplerate();

		return (ulong) ((double) m_filelen / ave);
	}

	return m_pVBR->frames();
}

/*========================================================*/

mp3::ulong mp3::Info::bitrate (void) const
{
	if (m_pVBR)
	{
		double ave = ((double) m_filelen / (double) frames());

		return (ulong) (
			(ave * (double) m_header.samplerate()) /
			(1000.00 * ((m_header.layer() == Header::L1) ? 12.00 : 144.00)));
	}

	return m_header.bitrate();
}

/*========================================================*/
/*	MP3 Header class implementation
 *========================================================*/

mp3::tbl_bitrate mp3::Header::bitRateTbl[5] = {
	{ 0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0 },
	{ 0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, 0 },
	{ 0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 0 },
	{ 0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0 },
	{ 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0 },
};

mp3::tbl_samplerate mp3::Header::sampleRateTbl[4] = {
	{ 11025, 12000, 8000, 0 },
	{ 0, 0, 0, 0 },
	{ 22050, 24000, 16000, 0 },
	{ 44100, 48000, 32000, 0 },
};

/*========================================================*/

void mp3::Header::load (byte *buf)
{
	byte *ptr = (byte *) &m_data;

	ptr[0] = buf[3];
	ptr[1] = buf[2];
	ptr[2] = buf[1];
	ptr[3] = buf[0];
}

bool mp3::Header::valid (void) const
{
	return (
		version() != V0 &&
		layer() != L0 &&
		bitrate() != 0 &&
		samplerate() != 0
		);
}

/*========================================================*/

mp3::ulong mp3::Header::bitrate (void) const
{
	return (biVersion.getValue(m_data) == V1)
		? bitRateTbl[biVersion.getValue(m_data) ^ biLayer.getValue(m_data)][biBitrateIndex.getValue(m_data)]
		: bitRateTbl[(biLayer.getValue(m_data) == L1) ? 3 : 4][biBitrateIndex.getValue(m_data)];
}

/*========================================================*/

bool mp3::Header::find (byte *buf, ulong count, ulong *ptr)
{
	ulong start;
	byte last, ch;

	//	Find the start of the frame header in the input buffer
	for (last = ch = 0, start = 0;
		 start < (count - sizeof(m_data));
		 ++start, last = ch)
	{
		ch = buf[start];

		if (last == 255 && (ch >> 5) == 7)
		{
			//	Set the header
			load (buf + start - 1);

			//	The above test is not nessecarily enough...
			if (valid())
			{
				/*	Save position after this LONG
				 *	and the position of the header
				 */
				m_hdrpos = (start - 1);
				*ptr = (start + 3);

				return true;
			}

			reset();
		}
	}

	return false;
}

/*========================================================*/
/*	Vbr header class implementation
 *========================================================*/

mp3::Vbr::Vbr (void)
	: m_frames(0)
{
}

/*========================================================*/

bool mp3::Vbr::find (byte *buf, ulong pos, ulong *ptr)
{
	if (!memcmp (buf, "Xing", 4))
	{
		const ulong flags = fromBuf(buf, 4);

		if (flags & FRAMES)
		{
			m_frames = fromBuf(buf, 8);
			return true;
		}
	}

	return false;
}

/*========================================================*/

mp3::ulong mp3::Vbr::fromBuf (byte *buf, int index)
{
	ulong *ptr = (ulong *) (buf + index);

	return ntohl(*ptr);
}

/*========================================================*/
/*	MP3 tag class implementation
 *========================================================*/

mp3::Tag::Tag (void)
	: m_genre(0),
	m_year(0)
{
}

/*========================================================*/

bool mp3::Tag::find (byte *buf)
{
	if (!memcmp(buf, "TAG", 3))
	{
		char yearbuf[5] = {0};

		m_title = getString(buf, 3);
		m_artist = getString(buf, 33);
		m_album = getString(buf, 63);
		m_comment = getString(buf, 97);

		memcpy (yearbuf, buf + 93, 4);

		m_year = strtoul(yearbuf, NULL, 0);
		m_genre = buf[127];
		return true;
	}

	return false;
}

std::string mp3::Tag::getString (byte *ptr, int pos)
{
	char buf[31] = {0};

	memcpy(buf, ptr + pos, 30);
	return std::string(buf);
}

/*========================================================*/
/*	Id3v2 class
 *========================================================*/

mp3::Id3v2::Id3v2 (void)
	: m_verHi(0), m_verLo(0),
	m_flags(0), m_size(0)
{
	//	Nothing...
}

/*========================================================*/

#define CHECK_READ(r)	if ((r) == -1) return false

bool mp3::Id3v2::read (FILE *file)
{
	char buf[3];
	int nread = 10;

	CHECK_READ(fread(buf, 1, 3, file));

	if (!memcmp("ID3", buf, 3))
	{
		CHECK_READ(fread(&m_verHi, 1, 1, file));
		CHECK_READ(fread(&m_verLo, 1, 1, file));
		CHECK_READ(fread(&m_flags, 1, 1, file));

		CHECK_READ(fread(&m_size, 4, 1, file));
		m_size = ntohl(m_size);

		Mp3out("Flags: %08x\n", m_flags);

		if (m_flags & EXTHDR)
		{
			ulong xsize;

			CHECK_READ(fread(&xsize, 4, 1, file));
			xsize = ntohl(xsize);

			Mp3out("Found extended header\n");

			fseek(file, xsize - 4, SEEK_CUR);
			nread += xsize;
		}

		if (m_flags & FOOTER)
		{
			Mp3out("Contains Id3v2 footer\n");
			m_size -= 10;
		}

#ifdef TESTMP3
		if (m_flags & UNSYNC) Mp3out("UNSYNC\n");
		if (m_flags & EXPERM) Mp3out("EXPERM\n");
#endif
		
		Mp3out("About to find frames...\n");

		while (nread < m_size)
		{
			frame fm;

			if (!fm.read(file))
				break;

			nread += (fm.size() + 10);

			m_frames.insert(
				FRAMES::value_type(
					fm.frameType(), fm.data()));

			Mp3out("Read %u bytes, added %s/%s\n",
				nread, fm.frameType().c_str(),
				fm.data().c_str());
		}

		return true;
	}

	return false;
}

/*========================================================*/

const char *mp3::Id3v2::value (const char *key) const
{
	FRAMES::const_iterator it = m_frames.find(key);

	if (it != m_frames.end())
		return it->second.c_str();

	return NULL;
}

const char *mp3::Id3v2::value (int key) const
{
	const char *keys[] = {
		"TPE1",
		"TIT2",		//	hehe... (.)(.)
		"TALB",
		"TYER",
		"TRCK",
	};

	return value(keys[key]);
}		

/*========================================================*/
/*	Id3v2 Frame class
 *========================================================*/

mp3::Id3v2::frame::frame (void)
	: m_size(0), m_flags(0)
{
	memset(m_type, 0, sizeof(m_type));
}

bool mp3::Id3v2::frame::read (FILE *file)
{
	if (fread(m_type, 1, 4, file) == 4 &&	//	Read frame type
		m_type[0] &&						//	'\0' means start of padding
		fread(&m_size, 4, 1, file) == 1 &&	//	Read size
		fread(&m_flags, 2, 1, file) == 1)	//	Read flags
	{
		byte *ptr = NULL;

		m_flags = ntohs(m_flags);

		/*	This isn't defined (not that i saw at
		 *	least), but it happens... and it's BAD.
		 */
		if (!(m_size = ntohl(m_size)))
			return false;

		ptr = new byte[m_size];
		fread(ptr, 1, m_size, file);

		//	We're not currently supporting UNICODE frames
		if (*ptr == 0 || *ptr == 3)
			m_data = std::string((const char *) (ptr + 1), m_size - 1);
		
		delete[] ptr;
		return true;
	}

	return false;
}

/*========================================================*/
/*	A testing main
 *========================================================*/

#ifdef TESTMP3
#include <string.h>
#include <stdlib.h>

int error (const char *sz) { fprintf (stderr, "%s\n", sz); exit(1); }

void show_mp3 (const char *fn)
{
	mp3::Info info;

	if (!info.load(fn))
		error ("no header found");

	printf ("\n%s (file %d bytes, header at %d)\n",
		fn, info.fileLength(),
		info.header().pos());

	printf ("Version %s Layer %s\n",
		info.header().versionString(),
		info.header().layerString());

	printf ("\t%dkbit/%dhz %s\n",
		info.bitrate(),
		info.header().samplerate(),
		info.header().channelmodeString());

	printf ("\tMissing CRC: %s\n\tCopyright: %s\n\tOriginal: %s\n\tEmphasis: %s\n",
		mp3::biNoCRC.boolString(info.header().binary()),
		mp3::biCopyright.boolString(info.header().binary()),
		mp3::biOriginal.boolString(info.header().binary()),
		info.header().emphasisString());

	printf ("\tFrames: %d\n", info.frames());

	const int seconds = info.playtime();

	printf ("\t%d:%02d (%d seconds)\n",
		seconds / 60,
		seconds % 60,
		seconds);

	const mp3::Vbr *pVBR = info.vbr();

	printf ("This %s a VBR file (%d frames)...\n",
		pVBR ? "IS" : "IS NOT");

	const mp3::Tag *pTag = info.tag();

	if (pTag)
	{
		printf ("MP3 Id3V1 Tag:\n");
		printf ("\tTitle: %s\n", pTag->title());
		printf ("\tArtist: %s\n", pTag->artist());
		printf ("\tAlbum: %s\n", pTag->album());
		printf ("\tComment: %s\n", pTag->comment());
		printf ("\tYear/Genre: %d/%d\n",
			pTag->year(), pTag->genre());
	}

	const mp3::Id3v2 *pId3 = info.id3v2();

	if (pId3)
	{
		printf ("MP3 Id3v2 Tag:\n");

		for (int index = 0; index < mp3::Id3v2::COUNT; ++index)
			if (const char *val = pId3->value(index))
				printf(" %s\n", val);
	}
}

int main (int argc, char **argv)
{
	for (int index = 1; index < argc; ++index)
		show_mp3(argv[index]);

	return 0;
}
#endif	//	TESTMP3
/*========================================================*/
