/*
 * Copyright (c) 2002 Markus Triska   triska@gmx.at
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Arson temporary file, class implementation
 *
 *	by Markus Triska (11:15PM Mar 14, 2002)
 *================================================*/
#include "arson.h"

#include <sys/types.h>
#include <sys/stat.h>

#ifdef ARSONDBG
#include <errno.h>
#endif

#include <qapplication.h>
#include <qfileinfo.h>
#include <qfile.h>
#include <qcursor.h>

#include <kstddirs.h>

#include "konfig.h"
#include "tempfile.h"

/*========================================================*/

ArsonTempFile::TEMPMAP ArsonTempFile::m_temporaryFiles;

/*========================================================*/

namespace arson {
	QString getPrefix (bool append = true)
	{
		const QString sarson (append ? "arson-" : "");
		QString dir = ACONFIG.temporaryDirectory();

		if (dir.isEmpty())	//	Checking for QString::null is not enough
			return locateLocal("tmp", sarson);

		if (dir[dir.length() - 1] != '/')
			dir.append("/");

		return (dir + sarson);
	}
};

/*========================================================*/

ArsonTempFile::ArsonTempFile (const QString &pref, const QString &ext, int mode)
	: KTempFile(arson::getPrefix() + pref, QString(".") + ext, mode)
{
	// If users have specifed their own temporary directory, use
	// it instead of the default KDE "tmp"-locale.
}

/*========================================================*/

#define ME_PARANOID		31337

QString ArsonTempFile::tempFileName (const char *prefix, const char *ext)
{
	const QString tmpl = arson::getPrefix() + prefix;

	for (int index = 0; index < ME_PARANOID; ++index)
	{
		const QString fn = tmpl + QString("%1").arg(rand(), 6, 16) + "." + ext;

		if (!QFileInfo(fn).exists())
			return QFile::encodeName(fn);
	}

	//	This should never happen
	return QFile::encodeName(tmpl + "d0h." + ext);
}

/*========================================================*/

QString ArsonTempFile::temporaryFor (const QString &name, const QString &suffix)
{
	TEMPMAP::ConstIterator it = m_temporaryFiles.find(name);
	
	if (it == m_temporaryFiles.end()) {
		QString basename = QFileInfo(name).baseName(
#ifdef ARSON_KDE3
			true
#endif
			);
		
		QString tfilename = arson::getPrefix(false) + basename + "." + suffix;
		int try_ext = 2;
		while (QFileInfo(tfilename).exists()) {
			QString s_ext = QString("_%1.%2").arg(try_ext).arg(suffix);
			tfilename = arson::getPrefix(false) + basename + s_ext;
			try_ext++;
		}
		m_temporaryFiles[name] = tfilename;

		Trace("New temporary file: %s\n", (const char *) tfilename);
		return tfilename;
	}

	Trace("Re-using temporary file: %s\n", it.data().latin1());
	return it.data();
}

/*========================================================*/

void ArsonTempFile::deleteTempFiles()
{
	ArsonFileCleanup ac;
	TEMPMAP::ConstIterator it;

	for (it = m_temporaryFiles.begin(); it != m_temporaryFiles.end(); it++)
		ac << it.data();

	m_temporaryFiles.clear();
}

/*========================================================*/

void ArsonTempFile::deleteTemporaryFor(const QString &name)
{
	TEMPMAP::Iterator it = m_temporaryFiles.find(name);
	if (it != m_temporaryFiles.end()) {
		Trace("Removing temporary file %s\n", name.latin1());
		ArsonFileCleanup() << it.data();
		m_temporaryFiles.remove(it);
	}
}

/*========================================================*/
/*	Cleanup multiple files, display wait cursor...
 *========================================================*/

ArsonFileCleanup &ArsonFileCleanup::operator<< (const QStringList &sl)
{
	m_files += sl;
	return *this;
}

ArsonFileCleanup &ArsonFileCleanup::operator<< (const QString &fn)
{
	m_files.append(fn);
	return *this;
}

/*========================================================*/

#ifdef ARSON_KDE3
#	define CURSOR	QCursor::WaitCursor
#else
#	define CURSOR	WaitCursor
#endif	//	ARSON_KDE3

ArsonFileCleanup::~ArsonFileCleanup (void)
{
	QStringList::ConstIterator it, end;

	QApplication::setOverrideCursor(QCursor(CURSOR));
	Trace("Cleanup removing:\n");

	for (it = m_files.begin(), end = m_files.end(); it != end; ++it)
		if (QFileInfo(QFile::encodeName(*it)).exists())
		{
			QFile::remove(QFile::encodeName(*it));
			Trace("  %s\n", (*it).latin1());
		}
	
	QApplication::restoreOverrideCursor();
}

/*========================================================*/

void ArsonFileCleanup::reset (void)
{
	m_files.clear();
}

/*========================================================*/
/*	A named pipe class to wrap mkfifo()
 *========================================================*/

ArsonFifo::ArsonFifo (const QString &fn, int mode)
	: m_filename(fn), m_bAutoDel(true)
{
	ARSON_INSTANCE_INCR("FIFO");
	
	create(mode);
}

ArsonFifo::ArsonFifo (const QString &prefix, const QString &ext, int mode)
	: m_bAutoDel(true)
{
	ARSON_INSTANCE_INCR("FIFO");
	
	m_filename = ArsonTempFile::tempFileName(prefix, ext);
	create(mode);
}

/*========================================================*/

ArsonFifo::~ArsonFifo (void)
{
	ARSON_INSTANCE_DECR("FIFO");
	
	if (autoDelete())
		unlink();

	ARSON_INSTANCE_DUMP("FIFO");
}

/*========================================================*/

bool ArsonFifo::create (int mode)
{
	if (mkfifo(QFile::encodeName(m_filename), mode))
	{
		Trace("Failed to create FIFO, errno=%d\n", errno);

		m_filename = QString::null;
		return false;
	}

	return true;
}

/*========================================================*/

void ArsonFifo::unlink (void)
{
	if (valid())
		QFile::remove(m_filename);
}

/*========================================================*/
