/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Implements the CD copy dialog
 *
 *	by Tony Sideris	(07:49PM Nov 12, 2001)
 *================================================*/
#include "arson.h"

#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qlabel.h>

#include <kfiledialog.h>
#include <kapp.h>
#include <klocale.h>

#include "tempfile.h"
#include "progressdlg.h"
#include "konfig.h"
#include "cdcopy.h"
#include "process.h"
#include "device.h"
#include "wizard.h"

/*========================================================*/
/*	CD Copy specific progress dialog class
 *========================================================*/

class cdCopyProgress : public ArsonCdWriterProgress
{
public:
	cdCopyProgress (QWidget *parent)
		: ArsonCdWriterProgress(parent, "cdcopyprog")
	{
		message->setText(
			i18n("Please insert the CD to be copied into the source drive, and a blank CDR into the burner, press Start to begin."));

		setCaption(
			i18n("CD Copy"));

		m_pCdrdao = new ArsonCdrdaoCtrlGrp(ctrlParent());
		widgetMgr() << m_pCdrdao;
	}

	bool usingCdrdao (void) const
	{ return (ACONFIG.programPref(PROGGRP_CDCOPY) == "cdrdao"); }

	void setOpts (const ArsonCopyOpts &opts)
	{
		m_opts = opts;

		Trace("set copy opts:\n %s\n %s\n %d/%d\n",
			m_opts.temppath.latin1(), m_opts.srcdev.latin1(),
			m_opts.bDirect, m_opts.bDelete);
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		try
		{
			ArsonProcessMgr *pm;

			if (usingCdrdao())
				pm = new ArsonCdrdaoCopyMgr(ui(), m_opts);
			else
			{
				if (!arsonWarning(
						i18n("readcd/cdrecord can only be used to copy data CDs. If you are trying to copy an audio CD, please use cdrdao. Continue?"),
						"audio-warning"))
					return NULL;

				if (m_opts.bDirect)
					pm = new ArsonDirectCopyMgr(ui(), m_opts);
				else
					pm = new ArsonLongCopyMgr(ui(), m_opts);
			}

			return pm;
		}
		catch (ArsonError &err) {
			err.report();
		}

		return NULL;
	}

	virtual void processOpts (ArsonProcessOpts &opts)
	{
		if (usingCdrdao())
			m_pCdrdao->applyTo(opts);

		ArsonCdWriterProgress::processOpts(opts);
	}

	virtual void reconfigure (void)
	{
		m_pCdrdao->setEnabled(usingCdrdao());
		ArsonCdWriterProgress::reconfigure();
	}

protected:
	ArsonCdrdaoCtrlGrp *m_pCdrdao;
	ArsonCopyOpts m_opts;
};

/*========================================================*/
/*	CD Copy wizard code
 *========================================================*/

class cdcopyWizardFactory : public ArsonWizardFactory
{
public:
	cdcopyWizardFactory (void)
		: ArsonWizardFactory(
			ARSON_WIZARD_CDCOPY,
			I18N_NOOP("CD Copy"),
			SecondaryAction)
	{ }
	
	virtual void addPages (ArsonWizard *pw)
	{
		ArsonCdCopyDlg *pCopy = new ArsonCdCopyDlg(pw, "cdcopydlg");
		cdCopyProgress *pProgress = new cdCopyProgress(pw);

		QObject::connect(pw, SIGNAL(pageChange(QWidget*,QWidget*)),
			pCopy, SLOT(slotPageChange(QWidget*,QWidget*)));
/*
		QObject::connect(pw->finishButton(), SIGNAL(clicked()),
			pProgress, SLOT(start_clicked()));
*/
		
		pw->addPage(pCopy, i18n("CD Copy Setup"));
		pw->addPage(pProgress, i18n("CD Copy Progress"));
	}
};

//	Global wizard factory instance.
cdcopyWizardFactory cdcopyFactory;

/*========================================================*/
/*	The CD Copy dialog class implementation
 *========================================================*/

ArsonCdCopyDlg::ArsonCdCopyDlg (QWidget *parent, const char *name)
	: ArsonCdCopyDlgBase(parent, name),
	m_devs(NULL)
{
	ArsonCopyOpts opts;
	opts.load();

	m_devs = new ArsonDeviceUiList(ACONFIG.devices().uiList(true));
	m_devs->fillList(dropdev, opts.srcdev);

	tmp->setText(ArsonTempFile::tempFileName("copy", "iso"));
	direct->setChecked(opts.bDirect);
	del->setChecked(opts.bDelete);
}

ArsonCdCopyDlg::~ArsonCdCopyDlg (void)
{
	delete m_devs;
}

/*========================================================*/

void ArsonCdCopyDlg::browse_clicked (void)
{
	const QString file = KFileDialog::getSaveFileName();

	if (file != QString::null)
		tmp->setText(file);
}

/*========================================================*/

bool ArsonCdCopyDlg::directCopy (void) const
{
	return direct->isChecked();
}

/*========================================================*/

void ArsonCdCopyDlg::slotPageChange (QWidget *was, QWidget *now)
{
	Trace("slotPageChange\n");
	
	if (was == this && now)
	{
 		ArsonCopyOpts opts;
		cdCopyProgress *pd = (cdCopyProgress *) now;

		opts.srcdev = m_devs->fromList(dropdev);
		opts.temppath = tmp->text();
		opts.bDelete = del->isChecked();
		opts.bDirect = directCopy();

		if (opts.srcdev != QString::null)
		{
			pd->setOpts(opts);
			opts.save();
		}
	}
}

/*========================================================*/
/*	The various cd copy process managers
 *========================================================*/

ArsonCdCopyMgr::ArsonCdCopyMgr (ArsonProcessUI *pUI, const ArsonCopyOpts &opts)
	: ArsonProcessMgr(pUI), m_opts(opts)
{
	//	Nothing...
}

/*========================================================*/
/*	Reads image from CD, the burns to CDR, in two
 *	seperate steps.
 *========================================================*/

ArsonLongCopyMgr::ArsonLongCopyMgr (ArsonProcessUI *pUI, const ArsonCopyOpts &opts)
	: ArsonCdCopyMgr(pUI, opts), m_stage(stageBegin)
{
	if (m_opts.temppath == QString::null)
		throw ArsonError(
			i18n("No temporary file specified!"));
}

/*========================================================*/

void ArsonLongCopyMgr::removeImg (void)
{
	ArsonProcessUI *pUI = ui();

	IFPTR(pUI)->setText(i18n("Removing temporary image file..."));

	ArsonFileCleanup() << m_opts.temppath;

	IFPTR(pUI)->setText(i18n("Done..."));
}

/*========================================================*/

void ArsonLongCopyMgr::taskComplete (ArsonProcess *ptr)
{
	switch (m_stage)
	{
	case stageRead:
		try {
			setProcess(
				new ArsonCdrecordDataProcess(
					this, m_opts.temppath));

			m_stage = stageWrite;
		}
		catch (ArsonError &err) {
			err.report();
		}
		break;

	case stageWrite:
		if (m_opts.bDelete)
			removeImg();
		
		setProcess(NULL);
		break;
	}
}

/*========================================================*/

void ArsonLongCopyMgr::abort (void)
{
	ArsonCdCopyMgr::abort();
	removeImg();
}

/*========================================================*/

void ArsonLongCopyMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonCdCopyMgr::begin(opts);

	try {
		setProcess(
			new ArsonReadCdProcess(
				this, m_opts.temppath, m_opts.srcdev));

		m_stage = stageRead;
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	Reads CD into CDR process in one shot
 *========================================================*/

ArsonDirectCopyMgr::ArsonDirectCopyMgr (ArsonProcessUI *pUI, const ArsonCopyOpts &opts)
	: ArsonCdCopyMgr(pUI, opts)
{
	//	Nothing...
}

/*========================================================*/

void ArsonDirectCopyMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonCdCopyMgr::begin(opts);

	try {
		setProcess(
			new ArsonCdrecordIsosizeProcess(
				this, m_opts.srcdev));
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	CDRDAO copy manager
 *========================================================*/

ArsonCdrdaoCopyMgr::ArsonCdrdaoCopyMgr (ArsonProcessUI *pUI, const ArsonCopyOpts &opts)
	: ArsonCdCopyMgr(pUI, opts)
{
	//	Nothing...
}

/*========================================================*/

void ArsonCdrdaoCopyMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonCdCopyMgr::begin(opts);

	try {
		setProcess(
			new ArsonCdrdaoCopyProcess(this,
				m_opts.bDelete,
				m_opts.bDirect,
				m_opts.srcdev));
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	Copy options impl
 *========================================================*/

void ArsonCopyOpts::load (void)
{
	if (KConfig *pk = kapp->config())
	{
		pk->setGroup("cdcopy");
		bDirect = pk->readBoolEntry("direct", false);
		bDelete = pk->readBoolEntry("delete", true);
		srcdev = pk->readEntry("dev");
	}
}

/*========================================================*/

void ArsonCopyOpts::save (void) const
{
	if (KConfig *pk = kapp->config())
	{
		pk->setGroup("cdcopy");

		pk->writeEntry("direct", bDirect);
		pk->writeEntry("delete", bDelete);
		pk->writeEntry("dev", srcdev);
	}
}

/*========================================================*/
