// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CClassInfo_h__
#define __CClassInfo_h__

// class/struct info 
// knows its base and derived classes

#include "Puma/CRecord.h"
#include "Puma/CTypeInfo.h"
#include "Puma/CBaseClassInfo.h"
#include "Puma/CTokens.h"
#include "Puma/CProtection.h"

namespace Puma {

/** \file 
 *  Representation of a C/C++ 'class' or 'struct'.
 *  \see Puma::CClassInfo */

/** \class CClassInfo CClassInfo.h Puma/infos/CClassInfo.h
 *  A CClassInfo object represents a C++ 'class' or 'struct'. Most of 
 *  the functionality of CClassInfo inherited by Puma::CRecord. It
 *  mainly manages the list of base classes and the list of derived
 *  classes. Note that a 'union', although syntactically very similar,
 *  is not represented by a CClassInfo object. */
class CClassInfo : public CRecord {
  Array<CBaseClassInfo*> _BaseClasses;
  Array<CClassInfo*> _Derived;
  // true if class has dependent base classes
  bool _DepBaseClass;

protected:
  /** This constructor is to be used by classes inherited from
   *  CClassInfo. It makes it possible to set another object id
   *  than \c CLASS_INFO for abstractions not representing a 
   *  standard C++ class (e.g. CClassInstance).
   *  \param id The object id to be used instead of \c CLASS_INFO. */
  CClassInfo (ObjectId id);

public: 
  /** Construct a new CClassInfo object. */
  CClassInfo ();
  /** Destroy the CClassInfo object. If the object id is 
   *  \c CLASS_INFO, Puma::CObjectInfo::CleanUp() is called. */
  ~CClassInfo ();

  /** Get the base info object of the info object. <b>The 
   *  corresponding information is not yet set!</b>
   *  \return This method always returns \c NULL.
   *  \warning Do not use this method. */
  CClassInfo *BaseObject () const;
  /** Get the info object of the class definition. 
   *  \return The info or \e this if the class is not defined. */
  CClassInfo *DefObject () const;
  /** Get the type information object for the class. */
  CTypeClass *TypeInfo () const;

  /** Get the number of the base classes of the class. */
  unsigned BaseClasses () const;
  /** Get the number of classes derived from the class. */
  unsigned DerivedClasses() const;
  /** Get the n-th base class. 
   *  \param n The number of the base class to return. 
   *  \return The base class info or \c NULL if \e n is not valid. */
  CBaseClassInfo *BaseClass (unsigned n) const;
  /** Get the base class with the given name. 
   *  \param name The name of the base class to return.
   *  \return The base class info or \c NULL if there is no base 
   *          class with the given name. */
  CBaseClassInfo *BaseClass (const char *name) const;
  /** Get the n-th derived class. 
   *  \param n The number of the derived class to return. 
   *  \return The class info of the derived class or \c NULL 
   *          if \e n is not valid. */
  CClassInfo *DerivedClass (unsigned n) const;
  /** Get the derived class with the given name. 
   *  \param name The name of the derived class to return.
   *  \return The class info of the derived class or \c NULL 
   *          if there is no derived class with the given name. */
  CClassInfo *DerivedClass (const char *name) const;
  
  /** Return \e true if the given class is a base class of the class.
   *  \param ci The class info of the possible base class.
   *  \param recursive If \e false it is only checked whether \e ci
   *         is a direct base class. */
  bool isBaseClass (const CClassInfo *ci, bool recursive = false) const;
  /** Return \e true if the given class is a derived class of the class.
   *  \param ci The class info of the possible base class.
   *  \param recursive If \e false it is only checked whether \e ci
   *         is directly derived from the class. */
  bool isDerivedClass (const CClassInfo *ci, bool recursive = false) const;
  /** Mark the class to have a base class that depends on template 
   *  parameters. 
   *  \param depends \e true if there is a dependent base class. */
  void hasDepBaseClass (bool depends);
  /** Return \e true if the class has a base class that depends on 
   *  template parameters. */
  bool hasDepBaseClass () const;

  /** Add a base class to the class.
   *  \param bc The base class info object. */
  void addBaseClass (CBaseClassInfo *bc);
  /** Add a derived class to the class.
   *  \param dc The class info object of the derived class. */
  void addDerivedClass (CClassInfo *dc);
  /** Remove the given base class from the class.
   *  \param bc The base class info object to remove. */
  void removeBaseClass (const CBaseClassInfo *bc);
  /** Remove the given base class from the class.
   *  \param bc The class info object of the base class to remove. */
  void removeBaseClass (const CClassInfo *bc);
  /** Remove the given derived class from the class.
   *  \param dc The class info object of the derived class to remove. */
  void removeDerivedClass (const CClassInfo *dc);

  /** Create a new base class info object.
   *  \param bc The class info of the base class or \c NULL to create
   *         an empty base class info object. 
   *  \return The new base class info object. */
  CBaseClassInfo *newBaseClass (CClassInfo *bc = 0);

  /** Return \e true if the class is declared using the keyword 'struct'. */
  bool isStruct () const;
  
  /** Returns the accessibility of a particular member or base class member.
   *  For example, if the accessibility is "private", the object can be
   *  accessed by members of this class, but not by other classes.
   *  \param oi The object info of the checked function/attribute/...
   *  \return The accessibility, i.e. private, public, protected, or none */
  CProtection::Type Accessibility (CObjectInfo *oi) const;
  
  /** Returns true if the argument function overrides any virtual function
   *  defined in this class or any of its base classes.
   *  \param fi function info object of the checked function. */
   bool overridesVirtual (const CFunctionInfo *fi) const;
   
  /** Returns true if this class is an aggregate according to 8.5.1-1. */
   bool isAggregate () const;

};

inline CClassInfo::CClassInfo (ObjectId id) :
  CRecord (id),
  _DepBaseClass (false)
 {}
inline CClassInfo::CClassInfo () :
  CRecord (CObjectInfo::CLASS_INFO),
  _DepBaseClass (false)
 {}
 
inline CClassInfo *CClassInfo::BaseObject () const
 { return (CClassInfo*)CObjectInfo::BaseObject (); }
inline CTypeClass *CClassInfo::TypeInfo () const
 { return (CTypeClass*)CObjectInfo::TypeInfo (); }

inline unsigned CClassInfo::BaseClasses () const
 { return _BaseClasses.length (); }
inline unsigned CClassInfo::DerivedClasses () const
 { return _Derived.length (); }
inline CBaseClassInfo *CClassInfo::BaseClass (unsigned n) const
 { return _BaseClasses.lookup (n); }
inline CClassInfo *CClassInfo::DerivedClass (unsigned n) const
 { return _Derived.lookup (n); }

inline bool CClassInfo::hasDepBaseClass () const
 { return _DepBaseClass; }
inline void CClassInfo::hasDepBaseClass (bool v)
 { _DepBaseClass = v; }


} // namespace Puma

#endif /* __CClassInfo_h__ */
