// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __InstantiationCandidate__
#define __InstantiationCandidate__

#include "Puma/Array.h"
#include "Puma/ErrorSink.h"
#include "Puma/DeducedArgument.h"

namespace Puma {


class CTree;
class CUnit;
class CTypeInfo;
class CConstant;
class CStructure;
class CT_ExprList;
class CObjectInfo;
class CT_SimpleName;
class CTemplateInfo;
class CTemplateParamInfo;

class InstantiationCandidate {
  typedef Array<CTree*> ArgumentList;
  typedef Array<DeducedArgument*> DeducedArgumentList;
  
  ArgumentList         arguments;
  DeducedArgumentList  darguments;
  CObjectInfo         *obj_info;
  CTemplateInfo       *tpl_info;
  CTree               *poi;
  ErrorSink           *err;
  
public:
  InstantiationCandidate (); 
  ~InstantiationCandidate ();
  
  void reset ();
  void initialize (ErrorSink *e);
  void initialize (CTree *p, CObjectInfo *o);
  void initialize (CTree *p, CObjectInfo *o, CTemplateInfo *t, ErrorSink *e = 0);
  void addArgument (CTree *);
  void forgetDeducedArgs ();

  bool deduceArguments (bool real_inst);
  bool match ();
  int compare (InstantiationCandidate &);

  CObjectInfo *ObjectInfo () const;
  CTemplateInfo *TemplateInfo () const;
  CTree *PointOfInstantiation () const;

  unsigned Arguments () const;
  unsigned DeducedArgs () const;
  CTree *Argument (unsigned) const;
  DeducedArgument *DeducedArg (unsigned) const;

  int getPosition (CTemplateParamInfo *) const;

private:
  bool deduceArguments (unsigned skip);
  bool deduceArguments (CTypeInfo *, CTypeInfo *, DeducedArgumentList &);
  bool matchArgument (CTree *pexpr, CTree *aexpr);
  bool matchArgument (CTypeInfo *ptype, CTypeInfo *atype, CTree *aexpr);
  bool sameValue (int pos, CConstant *value);
  bool sameType (int pos, CTypeInfo *type);
};

inline CObjectInfo *InstantiationCandidate::ObjectInfo () const
 { return obj_info; }
inline CTemplateInfo *InstantiationCandidate::TemplateInfo () const
 { return tpl_info; }
inline CTree *InstantiationCandidate::PointOfInstantiation () const
 { return poi; }

inline unsigned InstantiationCandidate::Arguments () const 
 { return arguments.length (); }
inline unsigned InstantiationCandidate::DeducedArgs () const
 { return darguments.length (); }
inline CTree *InstantiationCandidate::Argument (unsigned i) const
 { return arguments.lookup (i); }
inline DeducedArgument *InstantiationCandidate::DeducedArg (unsigned i) const
 { return darguments.lookup (i); }

inline void InstantiationCandidate::addArgument (CTree *a)
 { arguments.append (a); }
inline void InstantiationCandidate::forgetDeducedArgs () 
 { darguments.reset (); }


} // namespace Puma

#endif /* __InstantiationCandidate__ */
