// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CScopeInfo_h__
#define __CScopeInfo_h__

// scope info

#include "Puma/CObjectInfo.h"
#include "Puma/Array.h"

namespace Puma {


class CLocalScope;
class CFunctionInfo;
class CNamespaceInfo;
class CClassInfo;
class CUnionInfo;
class CTemplateInfo;

class CScopeInfo : public CObjectInfo {
  Array<CScopeInfo*> _Children; // child scopes
  CScopeInfo *_Parent; // parent scope
 
protected:
  CScopeInfo (ObjectId);

public:
  ~CScopeInfo ();

  // Get ... 
  unsigned Children () const;           // children of this scope
  CScopeInfo *Child (unsigned) const;
  CScopeInfo *Parent () const;          // parent scope
  bool isFile () const;                 // is file?
  bool isNamespace () const;            // is namespace?
  bool isRecord () const;               // is class or union?
  bool isClass () const;                // is class?
  bool isUnion () const;                // is union?
  bool isFunction () const;             // is function?
  bool isMethod () const;               // is class method?
  bool isClassTemplate () const;        // is class template?
  bool isFctTemplate () const;          // is function template?
  bool insideTemplate () const;         // is inside class or function template?
  bool insideRecord () const;           // is inside class or union?
  bool insideClass () const;            // is inside class?
  bool insideUnion () const;            // is inside union?
  bool insideFunction () const;         // is inside function?
  bool insideMethod () const;           // is inside class method?
  bool isLocalScope () const;           // is local (block) scope?
  bool isLocalRecord () const;          // is local (block scope) class or union?
  bool isLocalClass () const;           // is local (block scope) class?
  bool isLocalUnion () const;           // is local (block scope) union?
  
  bool GlobalScope () const;            // global (file) scope

  // Set ...
  void Parent (const CScopeInfo *);
  void addChild (CScopeInfo *);
  void removeChild (const CScopeInfo *); // WILL NOT BE DESTROYED!
  
  // Create/Delete ...
  CLocalScope *newLocalScope ();
  CFunctionInfo *newFunction (bool = false);
  CClassInfo *newClass (bool = false);
  CUnionInfo *newUnion (bool = false);
  CNamespaceInfo *newNamespace ();
  CTemplateInfo *newTemplate ();
  void deleteLocalScope (const CLocalScope *);
  void deleteFunction (const CFunctionInfo *);
  void deleteClass (const CClassInfo *);
  void deleteUnion (const CUnionInfo *);
  void deleteNamespace (const CNamespaceInfo *);
  void deleteTemplate (const CTemplateInfo *);
};


inline CScopeInfo::CScopeInfo (CObjectInfo::ObjectId id) :
  CObjectInfo (id),
  _Children (20, 20),
  _Parent ((CScopeInfo*)this)
 {}
 
inline unsigned CScopeInfo::Children () const
 { return _Children.length (); }
inline CScopeInfo *CScopeInfo::Child (unsigned n) const
 { return _Children.lookup (n); }
inline CScopeInfo *CScopeInfo::Parent () const
 { return _Parent; }

inline bool CScopeInfo::isLocalScope () const
 { return LocalScope (); }
inline bool CScopeInfo::isFile () const 
 { return FileInfo (); }
inline bool CScopeInfo::isNamespace () const 
 { return NamespaceInfo (); }
inline bool CScopeInfo::isRecord () const 
 { return Record (); }
inline bool CScopeInfo::isClass () const 
 { return ClassInfo (); }
inline bool CScopeInfo::isUnion () const 
 { return UnionInfo (); }
inline bool CScopeInfo::isFunction () const 
 { return FunctionInfo (); }
inline bool CScopeInfo::GlobalScope () const 
 { return FileInfo (); }

inline bool CScopeInfo::insideFunction () const 
 { return isFunction () || (Parent () && Parent () != this && 
                            Parent ()->insideFunction ()); }
inline bool CScopeInfo::insideMethod () const 
 { return isMethod () || (Parent () && Parent () != this && 
                          Parent ()->insideMethod ()); }
inline bool CScopeInfo::insideRecord () const 
 { return isRecord () || (Parent () && Parent () != this && 
                          Parent ()->insideRecord ()); }
inline bool CScopeInfo::insideTemplate () const 
 { return isTemplate () || (Parent () && Parent () != this && 
                            Parent ()->insideTemplate ()); }
inline bool CScopeInfo::insideClass () const 
 { return isClass () || (Parent () && Parent () != this && 
                         Parent ()->insideClass ()); }
inline bool CScopeInfo::insideUnion () const 
 { return isUnion () || (Parent () && Parent () != this && 
                         Parent ()->insideUnion ()); }

inline bool CScopeInfo::isLocalClass () const 
 { return isClass () && insideFunction (); }
inline bool CScopeInfo::isLocalUnion () const 
 { return isUnion () && insideFunction (); }
inline bool CScopeInfo::isLocalRecord () const 
 { return isRecord () && insideFunction (); }


} // namespace Puma

#endif /* __CScopeInfo_h__ */
