// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "ModelBuilder.h"
#include "TrackerDog.h"
#include "IntroductionUnit.h"
#include "PointCutContext.h"
#include "PointCutEvaluator.h"
#include "PointCutExpr.h"
#include "ACConfig.h"

#include "Puma/CFileInfo.h"
#include "Puma/ACAspectInfo.h"
#include "Puma/CClassDatabase.h"
#include "Puma/VerboseMgr.h"
#include "Puma/ErrorStream.h"
#include "Puma/SysCall.h"

// start phase 1
void ModelBuilder::setup_phase1 (CTranslationUnit& tunit) {
  assert(_phase == 0);
  _phase = 1;
  build (tunit);
  _phase = 2;
}

// start phase 2
void ModelBuilder::setup_phase2 (CTranslationUnit& tunit, list<CTree*> &ah_trees) {
  assert(_phase == 2);
  build (tunit);

  // run the tracker dog
  TrackerDog tracker (tunit, *this);
  tracker.run ();

  // reset the phase => we are done
  _phase = 0;
}

void ModelBuilder::build (CStructure &scope, JPL_Name *jpl) {

  // iterate through all namespace in this scope
  for (unsigned int n = 0; n < scope.Namespaces (); n++) {
    CNamespaceInfo *ni = scope.Namespace (n);

    // check if this namespace belongs to our project
    FileUnit *funit = source_unit (ni->Tree ());
    if (!funit || !_db->Project ()->isBelow (funit))
      continue;

    JPL_Namespace *jpl_namespace = register_namespace (ni, jpl);
    build (*ni, jpl_namespace);
  }
  
  // insert all classes and handle their contents
  for (unsigned int c = 0; c < scope.Types (); c++) {
    CClassInfo *ci = scope.Type (c)->ClassInfo ();
    if (!ci) continue;

    // check if this class belongs to our project
    FileUnit *funit = source_unit (ci->Tree ());
    if (!funit || !_db->Project ()->isBelow (funit))
      continue;
    
    // check if this class is a class slice
    ACSliceInfo *acsi = _db->SliceInfo (ci);
    if (acsi) {
      register_class_slice (acsi,
        !ci->CObjectInfo::QualifiedScope () && ci->Scope () == &scope ? jpl : 0);
    }
    else {
      // it is an ordinary class (or aspect)
      ACAspectInfo *ai = _db->AspectInfo (ci->DefObject ());
      JPL_Class *jpl_class = 0;
      if (ai)
        jpl_class = register_aspect (ai, !ci->CObjectInfo::QualifiedScope () &&
          ci->Scope () == &scope ? jpl : 0);
      else
        jpl_class = register_class (ci, !ci->CObjectInfo::QualifiedScope () &&
          ci->Scope () == &scope ? jpl : 0);
      if (!jpl_class) continue;
            
      if (ci->isDefined ()) {
   
        // setup the base class relations
        for (unsigned b = 0; b < ci->BaseClasses (); b++) {
          CClassInfo *base = ci->BaseClass (b)->Class ();
          JPL_Class *base_loc = 0;
          ACAspectInfo *base_aspect = _db->AspectInfo (base->DefObject ());
          if (base_aspect)
            base_loc = register_aspect (base_aspect);
          else
            base_loc = register_class (base);
          // ignore base classes that are not part of the model
          if (base_loc && (base_loc->type () == JoinPointLoc::Class ||
              base_loc->type () == JoinPointLoc::Aspect))
            jpl_class->base ((JPL_Class*)base_loc);
  //        else
  //          cout << "*** base not found: " << ci->QualName () << " " << base->QualName () << endl;
        }
        
        // recursively go down one step
        build (*ci, jpl_class);
  
        // setup the built-in member functions
        if (_phase == 2 && !ci->isStruct () &&
            (_db->Project ()->isBelow (ci->SourceInfo ()->SrcUnit ()) ||
             IntroductionUnit::cast (ci->SourceInfo ()->SrcUnit ()))) {
          for (unsigned f = 0; f < ci->Functions (); f++) {
            CFunctionInfo *fi = ci->Function (f)->DefObject ();
            if (!fi->isBuiltin ())
              continue;
            if (fi->isConstructor ())
              register_construction (register_function (fi, jpl_class));
            else if (fi->isDestructor ())
              register_destruction (register_function (fi, jpl_class));
          }
        }
        
        // is it a definition?
        if (jpl_class->type () == JoinPointLoc::Aspect) {
          JPL_Aspect *jpl_aspect = (JPL_Aspect*)jpl_class;
          ACAspectInfo *ai = _db->AspectInfo (ci->DefObject ());
          _vm << (ai->is_abstract () ? "Abstract" : "Concrete")
              << " aspect " << jpl_aspect->signature () << endvm;
          _vm++;
          advice_infos (jpl_aspect, ai);
          _vm--;
        }
      }
    }
  }
  
  // insert all functions
  for (unsigned f = 0; f < scope.Functions (); f++) {
    CFunctionInfo *fi = scope.Function (f);
    if (fi->isBuiltin ())
      continue;

    // check if this function belongs to our project
    FileUnit *funit = source_unit (fi->Tree ());
    if (!funit || !_db->Project ()->isBelow (funit))
      continue;

    // only functions that belong to the project are registered here
    // other functions might be register later if they are referenced
    register_function (fi,
      !fi->CObjectInfo::QualifiedScope () && fi->Scope () == &scope ? jpl : 0);
  }
}

void ModelBuilder::advice_infos (JPL_Aspect *jpl_aspect, ACAspectInfo *acai) {
  // collect all introduction advices
  if (_phase == 1) {
    for (int i = 0; i < acai->IntroNodes (); i++) {
      ACIntroductionInfo *acii = acai->IntroNode (i);
      TU_Introduction *intro = new TU_Introduction;
      intro->parent (jpl_aspect);
      intro->expr (acii->def_node ()->Pointcut ());
      intro->intro_info (acii);
      register_elem (intro);
      add_source_loc (intro, acii->def_node ());
      check (jpl_aspect, acii, intro);
      _vm << "intro: " << intro->introduced ()->type_str () << " "
          << intro->introduced ()->signature () << endvm;
    }
    // collect the order advice of this aspect
    for (int i = 0; i < acai->OrderNodes (); i++) {
      CT_AdviceDecl *ad = acai->OrderNode (i);
      TU_Order *order = new TU_Order;
      order->parent (jpl_aspect);
      order->expr (ad->Pointcut ());
      CT_OrderList *lst = ((CT_OrderDecl*)ad->Decl ())->OrderList ();
      for (int n = 0; n < lst->Entries (); n++) {
        order->add_pce (lst->Entry (n));
      }
      order->tree (ad);
      register_elem (order);
      add_source_loc (order, ad);
    }
  }
  // collect the advice nodes
  for (int i = 0; i < acai->AdviceNodes (); i++) {
    CT_AdviceDecl *ad = acai->AdviceNode (i);
    string sig  = TI_AdviceCode::name (ad);
    string name = ((CT_FctDef*)ad->Decl ())->Object ()->Name ().c_str ();
    TU_AdviceCode *new_elem = (TU_AdviceCode*)new_advice_code (jpl_aspect, name, sig);
    new_elem->expr (ad->Pointcut ());
    new_elem->function_type(JPL_Function::MEMBER);
    new_elem->tree (ad);
    new_elem->phase (_phase);
    add_source_loc (new_elem, ad);
  }
}

void ModelBuilder::check (JPL_Aspect *jpl_aspect, ACIntroductionInfo *acii,
  JPL_Introduction *intro) {

  // get the node type of the introduction declaration
  CTree *decl_node = acii->def_node ()->Decl ();
  const char *decl_node_name = decl_node->NodeName ();

  JPL_ClassSlice::class_slice_type _type = JPL_ClassSlice::CS_ERROR;
  Unit *_slice_unit = 0;
  CObjectInfo *obj = 0;
  CProtection::Type prot = acii->prot ();
  string name;
  
  if (decl_node_name == CT_Intro::NodeId ()) {
    // Copy all tokens of the intro pattern => intro source code can be removed
    // The pattern contains no whitespace or any kind of formatting!
    // Token positions are identical with the entry number.
    CT_Intro *intro = (CT_Intro*)decl_node;
    // set the name for #line directives
    Unit _pattern;
    _slice_unit = (Unit*)intro->token ()->belonging_to ();
    _pattern.name (_slice_unit->name ());
    bool name_truncated = false;
    for (int e = 0; e < intro->Entries (); e++) {
      Token *tok = intro->Entry (e)->token ();
      _pattern.append (*tok->duplicate ());
      if (name.length () < 40) {
        name += tok->text ();
        name += " ";
      }
      else
        name_truncated = true;
    }
    if (name_truncated)
      name += "...";
    _type = JPL_ClassSlice::CS_OLD_OTHER;
  }
  else if (decl_node_name == CT_SliceRef::NodeId ()) {
    _type = JPL_ClassSlice::CS_NORMAL;
    obj = ((CT_SliceRef*)decl_node)->name ()->Object ();
//    analyze_slice (obj);
    CT_ClassSliceDecl *csd = (CT_ClassSliceDecl*)obj->Tree ();
    _slice_unit = (Unit*)csd->token ()->belonging_to ();
    
  }
  else if (decl_node_name == CT_ClassSliceDecl::NodeId ()) {
    _type = JPL_ClassSlice::CS_NORMAL;
    obj = ((CT_ClassSliceDecl*)decl_node)->Object ();
//    analyze_slice (obj);
    CT_ClassSliceDecl *csd = (CT_ClassSliceDecl*)obj->Tree ();
    _slice_unit = (Unit*)csd->token ()->belonging_to ();
  }
  else if (decl_node_name == CT_ObjDecl::NodeId ()) {
    CT_DeclaratorList *dl = ((CT_ObjDecl*)decl_node)->Declarators();
    if (dl->Entries () > 0) {
      obj = ((CT_InitDeclarator*)dl->Entry (0))->Object ();
      
      // check for baseclass introductions
      if (obj->FunctionInfo () && strcmp (obj->Name (), "baseclass") == 0) {
        _type = JPL_ClassSlice::CS_OLD_BASE;
        CClassInfo *ci = obj->FunctionInfo ()->Argument (0u)->TypeInfo ()->ClassInfo ();
        _slice_unit = (Unit*)ci->Tree ()->token ()->belonging_to ();
        name = ci->QualName ();
      }
      else
        obj = 0;
    }
  }

  // to be found or created:
  TU_ClassSlice *class_slice = 0;

  // check whether the slice is defined
  Location loc = decl_node->token ()->location ();
  if (_type == JPL_ClassSlice::CS_NORMAL) {
    ACSliceInfo *acsi = obj->ClassDB ()->SliceInfo (obj);
    assert (acsi);
    if (!acsi->definition ()) {
      _err << sev_error << loc
          << "undefined slice '" << obj->QualName ()
          << "' used in advice" << endMessage;
    }
    if (acsi->type () == ACSliceInfo::SL_UNION) {
      _err << sev_error << loc
           << "union slices are not supported by ac++, yet" << endMessage;
    }
    if (obj->ClassInfo ()) {
      class_slice = register_class_slice (acsi);
    }
    if (!class_slice) {
      _err << sev_fatal << loc
           << "class slice '" << obj->QualName () << "' for intro lost" << endMessage;
    }
  }
  else {
    if (_conf.warn_deprecated ())
      _err << sev_warning << loc
          << "deprecated introduction syntax, "
          << "use slice instead." << endMessage;
    // create an anonymous slice object in the model for this old-style introduction
    class_slice = (TU_ClassSlice*)new_class_slice (jpl_aspect, name);
    class_slice->slice_type (_type);
    class_slice->prot (prot);
    class_slice->obj_info (obj);
    class_slice->slice_unit (_slice_unit);
    add_source_loc (class_slice, decl_node);
  }
  
  if (class_slice) {
    if (decl_node_name == CT_Intro::NodeId ()) {
      // Copy all tokens of the intro pattern => intro source code can be removed
      // The pattern contains no whitespace or any kind of formatting!
      // Token positions are identical with the entry number.
      CT_Intro *intro = (CT_Intro*)decl_node;
      // set the name for #line directives
      Unit &pattern = class_slice->pattern ();
      pattern.name (_slice_unit->name ());
      for (int e = 0; e < intro->Entries (); e++) {
        Token *tok = intro->Entry (e)->token ();
        pattern.append (*tok->duplicate ());
      }
    }

    intro->introduced (class_slice);
  }
}

string ModelBuilder::model_filename (FileUnit *unit) {
  const char *fname = unit->absolutePath ();
  ACProject &prj = _conf.project ();
  // if the file does not belong to the project return the absolute path
  if (!unit->belongsTo (prj))
    return fname;
  // iterate over all project paths
  for (long p = 0; p < prj.numPaths (); p++) {
    Filename dir_abs;
    if (!SysCall::canonical (prj.src (p), dir_abs)) {
      assert (false);
      return fname;
    }
    int dir_len = strlen (dir_abs.name ());
    if (strncmp (dir_abs.name (), fname, dir_len) == 0) {
      return fname + dir_len + 1;
    }
  }
  // the file has to be below any of the directories => fatal error here
  // assert (false); acgen.c does not belong to the project with this test
  return fname;
}

void ModelBuilder::build (CTranslationUnit& tunit) {
  _db = &tunit.db ();
  
  // TODO: the number of lines is wrong
  if (_phase == 1) {
    _tunit_file = register_file (model_filename ((FileUnit*)tunit.unit ()), -1, true);
    _file_map.insert (FileMapPair (tunit.unit (), _tunit_file));
  }
  
  CFileInfo *fi = _db->FileInfo (0);
  JPL_Namespace *jpl_namespace = register_namespace (fi);
  build (*fi, jpl_namespace);
}

bool ModelBuilder::inside_template (CScopeInfo *scope) const {
  if (scope->isTemplate () ||
      (scope->isTemplateInstance () &&
       scope->TemplateInstance ()->isPseudoInstance ()))
    return true;
  if (scope->QualifiedScope ())
    return inside_template (scope->QualifiedScope ());
  if (scope->Parent () != scope)
    return inside_template (scope->Parent ());
  return false;
}

bool ModelBuilder::is_valid_model_class (CClassInfo *ci) const {
  // Don't consider
  // * anonymous classes like template instances(?)
  // * the generated class JoinPoint
  // * classes defined in the special namespace AC
  if (strncmp (ci->Name (), "%", 1) == 0 ||
      strcmp (ci->QualName (), "JoinPoint") == 0 ||
      strncmp (ci->QualName (), "AC::", 4) == 0)
    return false;

  // Templates and classes nested in template class are not considered for
  // matching, only instances
  if (inside_template (ci))
    return false;

  return true;
}

bool ModelBuilder::is_valid_model_function (CFunctionInfo* fi) const {
  // Don't consider
  // * member functions of an invalid class
  // * ac++ generated functions
  // * pointcuts

  CRecord *cls = fi->ClassScope ();
  if (cls && !is_valid_model_class ((CClassInfo*)cls->DefObject ()))
    return false;
  
  CFunctionInfo *def = fi->DefObject ();
  if (inside_template (def))
    return false;
    
  if (strncmp (fi->Name (), "%a", 2) == 0 ||
      strncmp (fi->Name (), "__a", 3) == 0 ||
      strcmp (fi->Name (), "aspectof") == 0 ||
      strcmp (fi->Name (), "aspectOf") == 0)
    return false;
  
  if (_db->PointcutInfo (def))
    return false;

  if (!fi->isBuiltin () &&
      strcmp (def->SourceInfo ()->FileName (), "<anonymous unit>") == 0)
    return false;
  
  return true;
}

bool ModelBuilder::is_valid_model_namespace (CNamespaceInfo *ni) const {
  // no template instance namespace, but anonymous namespaces!
  if (strstr (ni->Name (), "<") == 0 &&
      strcmp (ni->QualName (), "AC") != 0 &&
      strncmp (ni->Name (), "__puma", 6) != 0)
    return true;
  return false;
}

TU_Type *ModelBuilder::register_type (CTypeInfo *ti) {
  TU_Type *new_elem = (TU_Type*)new_type (TI_Type::name (ti));
  new_elem->type_info (ti);
  return new_elem;
}

TU_Function *ModelBuilder::register_function (CFunctionInfo *fi,
  JPL_Name *parent) {
  if (!is_valid_model_function (fi))
    return 0;

  CFunctionInfo *def = fi->DefObject ();

  // find the parent model element
  if (!parent && !(parent = register_scope (def)))
    return 0;

  // build the name and signature of the function
  string sig  = TI_Function::name (def);
  string name = fi->Name ().c_str ();
  
  // register the element
  TU_Function * elem = (TU_Function*)new_function (parent, name, sig);

  // set function attributes
  if (elem->phase () == 0) {
    JPL_Function::FunctionType ft = JPL_Function::NON_MEMBER;
    if (fi->isConstructor ())
      ft = JPL_Function::CONSTRUCTOR;
    else if (fi->isDestructor ())
      ft = JPL_Function::DESTRUCTOR;
    else if (fi->isMethod ()) {
      if (fi->isStaticMethod ())
        ft = JPL_Function::STATIC_MEMBER;
      else if (fi->isVirtual () || fi->overridesVirtual ())
        ft = JPL_Function::VIRTUAL_MEMBER;
      else
        ft = JPL_Function::MEMBER;
    }
    elem->function_type (ft);
    elem->is_built_in (fi->isBuiltin ());
  }

  if (elem->phase () != _phase) {
    elem->func_info (def);
    elem->phase (_phase);
    // TODO: too often?
    add_source_loc (elem, fi, fi->isFctDef () ? SLK_DEF : SLK_DECL);

    elem->reset_types ();
    CTypeInfo *rtype = (fi->isConversion ()) ?
      fi->ConversionType() : fi->TypeInfo ()->BaseType ();
    elem->result_type (register_type (rtype));
    // argument types are the types of the function declaration
    CTypeList *arg_types = fi->TypeInfo ()->ArgTypes ();
    for (unsigned a = 0; a < arg_types->Entries (); a++)
      elem->add_arg_type (register_type (arg_types->Entry (a)));

    if (_phase == 2 && !fi->isBuiltin () && !fi->isPureVirtual () &&
      (fi->ClassDB ()->Project ()->isBelow (fi->SourceInfo ()->SrcUnit ()) ||
       IntroductionUnit::cast (fi->SourceInfo ()->SrcUnit ()))) {
      if (fi->isConstructor ())
        register_construction (elem);
      else if (fi->isDestructor ())
        register_destruction (elem);
      else
        register_execution (elem);

      // constructors and and destructors cannot be called
      if (!(fi->isConstructor () || fi->isDestructor ()))
        // register a 'pseudo call join point'
        register_call (def, 0, 0, 0);
    }
  }

  return elem;
}

TU_ClassSlice *ModelBuilder::register_class_slice (ACSliceInfo *acsi,
  JPL_Name *parent) {
  
  CClassInfo *ci = acsi->object ()->ClassInfo ();

  // find the parent model element
  if (!parent && !(parent = register_scope (scope_obj (ci))))
    return 0;

  // build the class slice name
  string name = TI_ClassSlice::name (acsi);

  // register the slice as a child of its parent
  TU_ClassSlice *new_elem = (TU_ClassSlice*)new_class_slice (parent, name);

  // set class slice attributes
  Unit *unit = (Unit*)ci->Tree ()->token ()->belonging_to ();
  new_elem->obj_info (acsi->definition ()->object ());
  new_elem->slice_unit (unit);
  CProtection::Type prot = (acsi->type () == ACSliceInfo::SL_STRUCT) ?
    CProtection::PROT_PUBLIC : CProtection::PROT_PRIVATE;
  new_elem->prot (prot);
  new_elem->slice_type (JPL_ClassSlice::CS_NORMAL);
  add_source_loc (new_elem, ci);
  
  return new_elem;
}

TU_Class *ModelBuilder::register_class (CClassInfo *ci, JPL_Name *parent) {
  // only classes are relevant
  if (!is_valid_model_class (ci))
    return 0;

  // find the parent model element
  if (!parent && !(parent = register_scope (ci)))
    return 0;

  // deterine the name
  string name = TI_Class::name (ci);
  
  // register an object with that name
  TU_Class *elem = (TU_Class*)new_class (parent, name);
  // set the class attributes
  elem->class_info (ci);
  if (elem->phase () == 0) // only once ...
    elem->intro_target (is_intro_target (ci->DefObject ()));
  elem->phase (_phase);
  add_source_loc (elem, ci, ci->isDefined () ? SLK_DEF : SLK_DECL);
  return elem;
}

TU_Aspect *ModelBuilder::register_aspect (ACAspectInfo *ai, JPL_Name *parent) {
  // only classes are relevant
  CClassInfo *ci = ai->ClassInfo ();
  if (!is_valid_model_class (ci))
    return 0;

  // find the parent model element
  if (!parent && !(parent = register_scope (ci)))
    return 0;

  // determine the name
  string name = TI_Class::name (ci);
  
  // register an object with that name
  TU_Aspect *elem = (TU_Aspect*)new_aspect (parent, name);
  // set the aspect attributes
  elem->aspect_info (ai);
  if (elem->phase () == 0) // only once ...
    elem->intro_target (is_intro_target (ci->DefObject ()));
  elem->phase (_phase);
  add_source_loc (elem, ci, ci->isDefined () ? SLK_DEF : SLK_DECL);
  return elem;
}

bool ModelBuilder::is_intro_target (CClassInfo *def) const {
  return !(!_db->Project ()->isBelow (def->SourceInfo ()->SrcUnit ()) ||
           def->isTemplateInstance () ||
           !def->isDefined ());
}

TU_Namespace *ModelBuilder::register_namespace (CNamespaceInfo *n,
  JPL_Name *parent) {
  // not all Puma namespaces should be registered
  if (!is_valid_model_namespace (n))
    return 0;

  // find the parent model element
  if (!parent && !n->GlobalScope () && !(parent = register_scope (n)))
    return 0;

  // build the namespace name
  string name = TI_Namespace::name (n);

  // register this namespace as a child of its parent in the model
  TU_Namespace *new_elem = (TU_Namespace*)new_namespace (parent, name);

  // set namespace attributes
  new_elem->namespace_info (n);
  if (!new_elem->is_root ())
    add_source_loc (new_elem, n);

  return new_elem;
}


// create a new call join point in the join point model
TU_MethodCall *ModelBuilder::register_call (CFunctionInfo *called, CT_Call *call_node,
    CObjectInfo *caller, int local_id) {

  // find the called function in the join point model
  JPL_Function *called_func = register_function (called);
  if (!called_func) {
//  if a called function is, for instance, a member of a local class, it is
//  perfectly valid that we don't find it in the model -> ignore call join-point
//    _err << sev_error << "called function \'" << called->QualName ()
//         << "\' not found in join point model" << endMessage;
    return 0;
  }

  // what is the lexical scope of this call?
  JPL_Name *lexical = 0;
  if (!caller) {
    lexical = 0; // a pseudo call join point
  }
  else if (caller->FunctionInfo ()) {
    // TODO: better cache the JPL object of the current function
    JoinPointLoc *loc = register_function (caller->FunctionInfo (), 0);
    if (!loc) {
      // TODO: calls in advice code are silently ignored here at the moment
//      _err << sev_error << call_node->token ()->location ()
//           << "location of function call invalid" << endMessage;
      return 0;
    }
    lexical = (JPL_Name*)loc;
  }
  else {
    CScopeInfo *scope = caller->QualifiedScope ();
    if (!scope) scope = caller->Scope ();
    lexical = register_scope (scope);
    if (!lexical) {
      _err << sev_error << call_node->token ()->location ()
           << "location of function call invalid" << endMessage;
      return 0;
    }
  }
  
  TU_MethodCall *new_elem = (TU_MethodCall*)new_call (called_func, local_id);
  new_elem->called (called);
  new_elem->caller (caller);
  new_elem->tree (call_node);
  if (call_node)
    add_source_loc (new_elem, call_node);
  // set the parent in the join point model structure
  if (lexical) { // pseudo-calls are invisible
    new_elem->parent (lexical);
  }
  
  // set the result type
  CTypeInfo *rtype = TI_Type::of (called_func->result_type ())->type_info ();
  new_elem->result_type (register_type (rtype));
  
  // argument types are the types from the target function declaration
  CTypeList *formal_arg_types = called->TypeInfo ()->ArgTypes ();
  int args = (int)formal_arg_types->Entries ();
  // For functions with default arguments, not more than the number of args
  // in the call expression is used.
  bool no_operator = (call_node &&
                      call_node->NodeName () == CT_CallExpr::NodeId () &&
                      ((CT_CallExpr *)call_node)->Arguments ());
  if (no_operator) {
    CT_ExprList *current_arg_list = ((CT_CallExpr *)call_node)->Arguments ();
    if (current_arg_list->Entries () < args) {
      args = current_arg_list->Entries ();
    }
  }
  // take the arguments types from the formal argument list
  bool have_ellipsis = false;
  int a = 0;
  for (; a < args; a++) {
    CTypeInfo *arg_type = formal_arg_types->Entry (a);
    if (arg_type->is_ellipsis ()) {
      have_ellipsis = true;
      break;
    }
    else {
      new_elem->add_arg_type (register_type (arg_type));
    }
  }
  
  // For functions with variable argument lists, the types that match '...'
  // in the declaration are taken from the call expression
  if (have_ellipsis && no_operator) {
    CT_ExprList *current_arg_list = ((CT_CallExpr *)call_node)->Arguments ();
    for (; a < current_arg_list->Entries (); a++) {
      CTypeInfo *arg_type =
        ((CT_Expression*)current_arg_list->Entry (a))->Type ();
      new_elem->add_arg_type (register_type (arg_type));
    }
  }

  register_elem (new_elem);
  return new_elem;
}

// create a new execution join point
TU_Method *ModelBuilder::register_execution (JPL_Function *ef) {
  CFunctionInfo *func = ((TI_Function*)ef->transform_info ())->func_info ();
  TU_Method *new_elem = (TU_Method*)new_execution (ef);
  new_elem->func_info (func);
  register_elem (new_elem);
  return new_elem;
}
  
// create a new construction join point
TU_Construction *ModelBuilder::register_construction (JPL_Function *cf) {
  assert (cf);
  CFunctionInfo *func = ((TI_Function*)cf->transform_info ())->func_info ();
  TU_Construction *new_elem = (TU_Construction*)new_construction (cf);
  new_elem->func_info (func);
  register_elem (new_elem);
  return new_elem;
}

// create a new construction join point
TU_Destruction *ModelBuilder::register_destruction (JPL_Function *df) {
  assert (df);
  CFunctionInfo *func = ((TI_Function*)df->transform_info ())->func_info ();
  TU_Destruction *new_elem = (TU_Destruction*)new_destruction (df);
  new_elem->func_info (func);
  register_elem (new_elem);
  return new_elem;
}

// TODO: temporary solution for dac++
void ModelBuilder::register_attr_access (CAttributeInfo *attr, CTree *node) {
  _access_infos.push_back (AccessInfo (attr, node));
}

JPL_Name *ModelBuilder::register_scope (CObjectInfo *obj) {
  JPL_Name *result = 0;
  CScopeInfo *scope = scope_obj (obj);
  if (scope) {
    if (scope->NamespaceInfo ())
      result = register_namespace (scope->NamespaceInfo ());
    else if (scope->FunctionInfo ())
      result = register_function (scope->FunctionInfo ());
    else if (scope->ClassInfo ()) {
      ACAspectInfo *ai = _db->AspectInfo (scope->ClassInfo ()->DefObject ());
      if (ai)
        result = register_aspect (ai);
      else
        result = register_class (scope->ClassInfo ());
    }
  }
  
//  in some cases, e.g. join-points within local classes it can happen that
//  the scope of a join-point is not known in the model -> no error!
//  if (!result) {
//    _err << sev_error << "parent '" << scope_name (obj).c_str ()
//       << "' of model element " << obj->QualName () << " not found"
//       << endMessage;
//  }
  
  return result;
}

CScopeInfo *ModelBuilder::scope_obj (CObjectInfo *oi) {
  CScopeInfo *scope = 0;
  if (oi->TemplateInstance ()) {
    scope = oi->TemplateInstance ()->Template ()->CObjectInfo::QualifiedScope ();
    if (!scope) scope = oi->TemplateInstance ()->Template ()->Scope ();
  }
  else {
    scope = oi->QualifiedScope ();
    if (!scope) scope = oi->Scope ();
  }
  // if this is a template instance scope, go to the parent
  while (scope->isAnonymous () && strstr (scope->Name (), "<")) {
    scope = scope->Parent (); 
  }
  return scope;
}

string ModelBuilder::scope_name (CObjectInfo *oi) {
  CScopeInfo *scope = scope_obj (oi);
  assert (scope);
  // is it the globale scope
  if (scope->GlobalScope ())
    return "::";
  ostringstream scope_name;
  if (scope->TypeInfo () && !scope->TypeInfo ()->isUndefined ())
    scope_name << *scope->TypeInfo ();
  else if (scope->isAnonymous ())
    scope_name << "<noname>";
  else
    scope_name << scope->QualName ();
  return scope_name.str ();
}

// add the source location to a model element by using the syntax tree node
void ModelBuilder::add_source_loc (JoinPointLoc *name, CTree *tree, SourceLocKind kind) {
    
  // check if this file belong to our project
  FileUnit *funit = source_unit (tree);
  if (!funit)
    return;
  
  if (!_db->Project ()->isBelow (funit)) {
    if (name->type () & JoinPointLoc::Name) {
      JPL_Name *jpl_name = (JPL_Name*)name;
      jpl_name->tunits ().insert (_tunit_file->id ());
    }
    return;
  }
  FileMap::iterator i = _file_map.find (funit);
  File *file = 0;
  if (i != _file_map.end ())
    file = i->second;
  else {
    // TODO: temporary hack
    int len = ((Token*)funit->last ())->location ().line ();
    file = register_file (model_filename (funit), len, false);
    file->included_by (_tunit_file);
    _file_map.insert (FileMapPair (funit, file));
  }
  int line = tree->token ()->location ().line ();
  int len = tree->end_token ()->location ().line () - line + 1;
  SourceLoc source_loc (file->id (), line, len, kind);
  name->source_loc (source_loc);
}

FileUnit *ModelBuilder::source_unit (CTree *tree) {
  if (!tree || !tree->token ())
    return 0;
    
  Unit *unit   = (Unit*)tree->token ()->belonging_to ();
  while (unit && unit->isMacroExp ()) {
    unit = ((MacroUnit*)unit)->CallingUnit ();
  }
  // if the 'insert unit' is an 'introduction unit', find the intro target unit
  IntroductionUnit *intro_unit = 0;
  while ((intro_unit = IntroductionUnit::cast (unit)) != 0)
    unit = intro_unit->target_unit ();
  
  // at this point we must have reached a file unit
  if (!unit->isFile ())
    return 0;

  return (FileUnit*)unit;
}
