// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __transform_info_h__
#define __transform_info_h__

#include "JoinPointLoc.h"
#include "ThisJoinPoint.h"

#include "Puma/CClassDatabase.h"
#include "Puma/CNamespaceInfo.h"
#include "Puma/DeducedArgument.h"
#include "Puma/CTemplateInstance.h"
#include "Puma/MacroUnit.h"

class TransformInfo : public ModelTransformInfo {
public:
  virtual JoinPointLoc &jpl () = 0;
  virtual CTree *tree () const = 0;
  virtual CObjectInfo *obj_info () const = 0;
  // associated unique parser object (e.g. for name mangling)
  virtual CObjectInfo *assoc_obj () const = 0;
  int line () const {
    return tree ()->token ()->location ().line ();
  }
  int lines () const {
    return tree ()->end_token ()->location ().line () - line () + 1;
  }
  // find the unit (non-macro) in which a join point is located
  Unit *unit () const {
    CTree *node = tree ();
    if (!node)
      return 0;
    Token *token = node->token ();
    Unit *unit   = (Unit*)token->belonging_to ();
    while (unit && unit->isMacroExp ()) {
      unit = ((MacroUnit*)unit)->CallingUnit ();
    }
    return unit;
  }

  static inline const TransformInfo *of (const JoinPointLoc &loc);
  static inline CTree *tree (const JoinPointLoc &loc);
  static inline int line (const JoinPointLoc &loc);
  static inline int lines (const JoinPointLoc &loc);
  static inline Unit *unit (const JoinPointLoc &loc);
  static inline Location location (const JoinPointLoc &loc);
};


inline const TransformInfo *TransformInfo::of (const JoinPointLoc &loc) {
  return loc.transform_info () ? (TransformInfo*)loc.transform_info () : 0;
}
inline CTree *TransformInfo::tree (const JoinPointLoc &loc) {
  return loc.transform_info () ?
    ((TransformInfo*)loc.transform_info ())->tree () : 0;
}
inline int TransformInfo::line (const JoinPointLoc &loc) {
  return loc.transform_info () ?
    ((TransformInfo*)loc.transform_info ())->line () : -1;
}
inline int TransformInfo::lines (const JoinPointLoc &loc) {
  return loc.transform_info () ?
    ((TransformInfo*)loc.transform_info ())->lines () : -1;
}
inline Unit *TransformInfo::unit (const JoinPointLoc &loc) {
  return loc.transform_info () ?
    ((TransformInfo*)loc.transform_info ())->unit () : 0;
}
inline Location TransformInfo::location (const JoinPointLoc &loc) {
  return loc.transform_info () ?
    tree (loc)->token ()->location () : Location ();
}

class TI_Namespace : public TransformInfo {
  // pointer to the Puma namespace object (for transformation)
  CNamespaceInfo *_namespace_obj;
public:
  TI_Namespace () : _namespace_obj (0) {}
    
  void namespace_info (CNamespaceInfo *n) { _namespace_obj = n; }
  CNamespaceInfo *namespace_info () const { return _namespace_obj; }
  virtual CObjectInfo *obj_info () const { return _namespace_obj; }
  virtual CObjectInfo *assoc_obj () const { return _namespace_obj; }
  virtual CTree *tree () const { return _namespace_obj->Tree (); }

  static string name (CNamespaceInfo *n) {
    if (n->GlobalScope ())
      return "::";
    else
      return n->QualName ();
  } 
};

class TI_Class : public TransformInfo {
  CClassInfo *_class_obj;
  int _phase;
public:
  TI_Class () : _class_obj (0), _phase (0) {}

  void phase (int p) { _phase = p; }
  int phase () const { return _phase; }
  void class_info (CClassInfo *c) { _class_obj = c->DefObject (); }
  CClassInfo *class_info () const { return _class_obj; }

  virtual CObjectInfo *obj_info () const { return _class_obj; }
  virtual CObjectInfo *assoc_obj () const { return _class_obj; }
  virtual CTree *tree () const { return _class_obj->Tree (); }
  
  static string name (CClassInfo *c) {
    ostringstream class_name;
    class_name << *c->TypeInfo ();
    return class_name.str ();
  }

  static const TI_Class *of (const JPL_Class &loc) {
    return (TI_Class*)loc.transform_info ();
  }
};

class TI_Aspect : public TI_Class {
  ACAspectInfo *_aspect_obj;
public:
  TI_Aspect () : _aspect_obj (0) {}
    
  void aspect_info (ACAspectInfo *ai) {
    _aspect_obj = ai;
    class_info (ai->ClassInfo ());
  }
  ACAspectInfo *aspect_info () const { return _aspect_obj; }
  CFunctionInfo *aspectof () const {
    // make this better!
    CFunctionInfo *aof = ClassInfo ()->Function ("aspectof");
    if (!aof)
      aof = ClassInfo ()->Function ("aspectOf");
    return aof ? aof->DefObject () : (CFunctionInfo*)0;
  }
  const char *name () const { return _aspect_obj->name (); }
  ACAspectInfo *acnode () const { return _aspect_obj; }
  CClassInfo *ClassInfo () const { return _aspect_obj->ClassInfo (); }
  
  static string name (ACAspectInfo *ai) {
    ostringstream class_name;
    class_name << *ai->ClassInfo ()->TypeInfo ();
    return class_name.str ();
  }

  static const TI_Aspect *of (const JPL_Aspect &loc) {
    return (TI_Aspect*)loc.transform_info ();
  }

};

class TI_Function : public TransformInfo {
  CFunctionInfo *_func_obj;
  int _phase;
public:
  TI_Function () : _func_obj (0), _phase (0) {}
    
  void func_info (CFunctionInfo *fi) { _func_obj = fi->DefObject (); }
  CFunctionInfo *func_info () const { return _func_obj; }
  void phase (int p) { _phase = p; }
  int phase () const { return _phase; }
  virtual CObjectInfo *obj_info () const { return _func_obj; }
  virtual CObjectInfo *assoc_obj () const { return _func_obj; }
  virtual CTree *tree () const { return _func_obj->Tree (); }

  static string name (CFunctionInfo *func_info) {
    ostringstream sig, name;
    if (func_info->isMethod () && !func_info->isStaticMethod ())
      name << func_info->Name ();
    else
      name << func_info->QualName ();
      
    // add the template arguments if it is a function template instance
    CTemplateInstance *instance = func_info->TemplateInstance ();
    if (instance) {
      name << "< ";
      for (unsigned a = 0; a < instance->DeducedArgs (); a++) {
        if (a > 0) name << ",";
        DeducedArgument *arg = instance->DeducedArg (a);
        if (arg->Type ())
          name << *arg->Type ();
        else if (arg->Value ()) {
          if (arg->Value ()->isSigned ())
            name << arg->Value ()->convert_to_int ();
          else if (arg->Value ()->isUnsigned ())
            name << arg->Value ()->convert_to_uint ();
          else if (arg->Value ()->isFloat ())
            name << arg->Value ()->convert_to_float ();
        }
        else
          name << "*invalid template arg*";
      }
      name << " >";
    }
    func_info->TypeInfo ()->TypeText (sig, name.str ().c_str ());
    return sig.str ();
  }

  static const TI_Function *of (const JPL_Function &loc) {
    return (TI_Function*)loc.transform_info ();
  }
};

class TI_Type : public TransformInfo {
  CTypeInfo *_type_info;
public:
  TI_Type () : _type_info (0) {}
  
  void type_info (CTypeInfo *ti) { _type_info = ti; }
  CTypeInfo *type_info () const { return _type_info; }

  virtual CObjectInfo *obj_info () const { return 0; }
  virtual CObjectInfo *assoc_obj () const { return 0; }
  virtual CTree *tree () const { return 0; }

  static string name (CTypeInfo *type_info) {
    ostringstream out;
    out << *type_info;
    return out.str ();
  }
  
  static const TI_Type *of (const JPL_Type &loc) {
    return (TI_Type*)loc.transform_info ();
  }
};

class TI_Code : public TransformInfo {
public:
  virtual CFunctionInfo *tjp_type() const = 0;
  virtual CFunctionInfo *tjp_target() const = 0;
  virtual CFunctionInfo *tjp_that() const = 0;

  // that types (for the JoinPoint-API)
  virtual CTypeInfo *that_type () const = 0;

  // target type (for the JoinPoint-API)
  virtual CTypeInfo *target_type () const = 0;
  
  // interface needed to generate proper proceed code
  virtual bool proceed_needs_that () const { return false; }
  virtual bool proceed_needs_target () const { return false; }
  virtual bool proceed_needs_fptr () const { return false; }
  
  // helper functions for derived classes
  static CTypeInfo *get_that_type (CObjectInfo *obj) {
    CFunctionInfo *func = obj->FunctionInfo ();
    if (func) {
      CObjectInfo *this_obj = func->Attribute ("this");
      if (this_obj) {
        CTypeInfo *type = this_obj->TypeInfo ();
        // find the type which is referenced by 'this'
        while (type && !type->isRecord ())
          type = type->BaseType ();
        assert (type);
        return type;
      }
    }
    if (obj->ClassScope ())
      return obj->ClassScope ()->TypeInfo ();
    return &CTYPE_VOID;
  }

  // helper function: check if a function is a method (needs a 'this' pointer)
  static bool needs_this (CFunctionInfo *func) {
    if (func->isMethod () && !func->isStaticMethod ()) {
      return !(func->isOperator () &&
        (strcmp (func->Name (), "operator new") == 0 ||
         strcmp (func->Name (), "operator new[]") == 0 ||
         strcmp (func->Name (), "operator delete") == 0 ||
         strcmp (func->Name (), "operator delete[]") == 0));
    }
    return false;
  }
};

class TI_Method : public TI_Code {
  CFunctionInfo *_func_info;
public:
  TI_Method () : _func_info (0) {}

  void func_info (CFunctionInfo *f) { _func_info = f; }
  virtual CObjectInfo *obj_info () const { return _func_info; }
  virtual CObjectInfo *assoc_obj () const { return _func_info; }
  virtual CTree *tree () const { return _func_info->Tree (); }

  virtual CFunctionInfo *tjp_type() const {return _func_info;}
  virtual CFunctionInfo *tjp_target() const {return _func_info;}
  virtual CFunctionInfo *tjp_that() const {return _func_info;}

  // that type (for the JoinPoint-API)
  virtual CTypeInfo *that_type () const { return get_that_type (_func_info); }

  // interface needed to generate proper proceed code
  virtual bool proceed_needs_that () const { return needs_this (_func_info); }
  
  // target type (for the JoinPoint-API)
  virtual CTypeInfo *target_type () const {
    CRecord *scope = _func_info->ClassScope ();
    return scope ? scope->TypeInfo () : &CTYPE_VOID;
  }
};

class TI_MethodCall : public TI_Code {
  CFunctionInfo *called_func;
  CObjectInfo *caller_obj;
  CT_Call *node;
public:
  TI_MethodCall () : called_func (0), caller_obj (0), node (0) {}

  void called (CFunctionInfo *c) { called_func = c; }
  CFunctionInfo *called () const { return called_func; }
  void caller (CObjectInfo *c) { caller_obj = c; }
  CFunctionInfo *caller () { 
    return caller_obj ? caller_obj->FunctionInfo () : 0;
  }
  void tree (CT_Call *n) { node = n; }
  virtual CTree *tree () const { return node; }
  virtual CObjectInfo *obj_info () const { return called_func; }
  virtual CObjectInfo *assoc_obj () const { return caller_obj; }
  
  virtual CFunctionInfo *tjp_type() const { return called_func;}
  virtual CFunctionInfo *tjp_target() const { return called_func;}
  virtual CFunctionInfo *tjp_that() const { return caller_obj->FunctionInfo ();}

  // interface needed to generate proper proceed code
  virtual bool proceed_needs_target () const { return needs_this (called_func); }
  virtual bool proceed_needs_fptr () const { return needs_rights (); }

  // that type (for the JoinPoint-API)
  virtual CTypeInfo *that_type () const { return get_that_type (caller_obj); }

  // target type (for the JoinPoint-API)
  virtual CTypeInfo *target_type () const {
    bool is_ptr;
    CTree *expr = target_expr (is_ptr);
    if (expr) {
      CTypeInfo *type = expr->Type ();
      // if this is a pointer or reference, take the base type
      while (type && (type->TypeAddress () || !type->isRecord ()))
        type = type->BaseType ();
      assert (type);
      return type;
    }
    else if (called_func->isMethod ()) {
      if (called_func->isStaticMethod ()) {
        return called_func->ClassScope ()->TypeInfo ();
      }
      else {
        if (caller_obj->FunctionInfo ()) {
          assert (caller_obj->ClassScope ());
          return caller_obj->ClassScope ()->TypeInfo ();
        }
        else
          return &CTYPE_VOID;
      }
    }
    else
      return &CTYPE_VOID;
  }

  // the target object of the call or NULL
  CT_Expression *target_expr (bool &is_ptr) const {
  
    // check if this call has a target object
    if (!(called_func->isMethod () && !called_func->isStaticMethod ()))
      return 0;
      
    CTree *result = 0;
  
    // what kind of node was used for the call?
    const char *calltype = node->NodeName ();
    
    // in most case we have no pointer
    is_ptr = false;
  
    if (calltype == CT_CallExpr::NodeId ()) {
      CTree *expr = ((CT_CallExpr*)node)->Expr ();
      const char *fctcalltype = expr->NodeName ();
      if (fctcalltype == CT_MembRefExpr::NodeId ()) {
        // <target>.method()
        result = expr->Son (0);
      } else if (fctcalltype == CT_MembPtrExpr::NodeId ()) {
        // <target-ptr>->method()
        is_ptr = true;
        result = expr->Son (0);
      } else {
        // method()
        is_ptr = true; // here 'this' is passed implicitly
      }
    }
    else if (calltype == CT_BinaryExpr::NodeId ()) {
      // <target> <op> <arg>
      result = node->Son (0);
    }
    else if (calltype == CT_UnaryExpr::NodeId ()) {
      // <op> <target>
      result = node->Son (1);
    }
    else if (calltype == CT_PostfixExpr::NodeId ()) {
      // <target> <op>
      result = node->Son (0);
    }
    else if (calltype == CT_IndexExpr::NodeId ()) {
      // <target> [ <index> ]
      result = node->Son (0);
    }
    else if (calltype == CT_ImplicitCall::NodeId ()) {
      // <target>
      result = node->Son (0);
    }
    else {
      cout << "unexpected node type " << node->NodeName () << " in "
           << "JPL_MethodCall::target_expr()" << endl;
    }
    return (CT_Expression*)result;
  }
      
  // the call expression node in the syntax tree
  CT_Call *CallNode() const { return node;}

  // checks if the original call uses a qualified target function name
  bool is_qualified () const {
    if (node->NodeName () != CT_CallExpr::NodeId ())
      return false;
    CTree *expr = ((CT_CallExpr*)node)->Expr ();
    const char *nodename = expr->NodeName ();
    if (nodename == CT_MembPtrExpr::NodeId () || 
        nodename == CT_MembRefExpr::NodeId ()) {
      expr = expr->Son (2); // some access function in PUMA missing!
      nodename = expr->NodeName ();
    }
    return nodename == CT_QualName::NodeId () || 
           nodename == CT_RootQualName::NodeId ();
  }
  
  // returns true if the call needs special access rights
  bool needs_rights () const {
    // get the target object type
    CTypeInfo *type = target_type ()->UnqualType ();
    
    // no member function => no accessibility problem
    if (type->isVoid ())
      return false;
      
    // static member => no problem only if public
    if (called_func->isStaticMethod ())
      return (called_func->Protection () != CProtection::PROT_PUBLIC);
      
    // normal member function => look up the accessibility
    if (type->ClassInfo () &&
      type->ClassInfo ()->Accessibility (called_func) == CProtection::PROT_PUBLIC)
      return false;
    
    return true;
  }
};

class TI_Construction : public TI_Code {
  CFunctionInfo *_func_info;
public:
  TI_Construction () : _func_info (0) {}

  void func_info (CFunctionInfo *f) { _func_info = f; }
  virtual CObjectInfo *obj_info () const { return _func_info; }
  virtual CObjectInfo *assoc_obj () const { return _func_info; }
  virtual CTree *tree () const { return _func_info->ClassScope ()->Tree (); }
  
  virtual CFunctionInfo *tjp_type() const {return _func_info;}
  virtual CFunctionInfo *tjp_target() const {return _func_info;}
  virtual CFunctionInfo *tjp_that() const {return _func_info;}

  // that type (for the JoinPoint-API)
  virtual CTypeInfo *that_type () const { return get_that_type (_func_info); }

  // target type (for the JoinPoint-API)
  virtual CTypeInfo *target_type () const {
    CRecord *scope = _func_info->ClassScope ();
    return scope ? scope->TypeInfo () : &CTYPE_VOID;
  }

  // interface needed to generate proper proceed code
  virtual bool proceed_needs_that () const { return true; }
};

class TI_Destruction : public TI_Code {
  CFunctionInfo *_func_info;
public:
  TI_Destruction () : _func_info (0) {}

  void func_info (CFunctionInfo *f) { _func_info = f; }
  virtual CObjectInfo *obj_info () const { return _func_info; }
  virtual CObjectInfo *assoc_obj () const { return _func_info; }
  virtual CTree *tree () const { return _func_info->ClassScope ()->Tree (); }

  virtual CFunctionInfo *tjp_type() const {return _func_info;}
  virtual CFunctionInfo *tjp_target() const {return _func_info;}
  virtual CFunctionInfo *tjp_that() const {return _func_info;}

  // that type (for the JoinPoint-API)
  virtual CTypeInfo *that_type () const { return get_that_type (_func_info); }

  // target type (for the JoinPoint-API)
  virtual CTypeInfo *target_type () const {
    CRecord *scope = _func_info->ClassScope ();
    return scope ? scope->TypeInfo () : &CTYPE_VOID;
  }

  // interface needed to generate proper proceed code
  virtual bool proceed_needs_that () const { return true; }
};

class TI_AdviceCode : public TransformInfo {
  CT_AdviceDecl *_tree;
  ThisJoinPoint _this_join_point;
  int _phase;
  
public:
  TI_AdviceCode () : _tree (0), _phase (0) {}
  
  void phase (int p) { _phase = p; }
  int phase () const { return _phase; }
  void tree (CT_AdviceDecl *ad) { _tree = ad; }
  virtual CTree *tree () const { return _tree; }
  
  virtual CObjectInfo *obj_info () const {
    return ((CT_FctDef*)_tree->Decl ())->Object ();
  }
  virtual CObjectInfo *assoc_obj () const { return obj_info (); }

  CScopeInfo *Scope () const {
    return ((CT_FctDef*)_tree->Decl ())->Object ()->QualifiedScope ();
  }
  const char *name () const {
    return ((CT_FctDef*)_tree->Decl ())->Object ()->Name ();
  }
  CFunctionInfo *function() const {
    stringstream fname;
    fname << "__" << (name () + 1);
    return Scope ()->ClassInfo ()->Function (fname.str ().c_str ());
  }

  ThisJoinPoint &this_join_point () { return _this_join_point; }
  const ThisJoinPoint &this_join_point () const { return _this_join_point; }
  
  static string name (CT_AdviceDecl *ad) {
    const char *name = ((CT_FctDef*)ad->Decl ())->Object ()->Name () + 1;
    CScopeInfo *scope = ((CT_FctDef*)ad->Decl ())->Object ()->QualifiedScope ();
    
    stringstream fname;
    fname << scope->QualName () << "::__" << name;
    return fname.str ();
  }
  
  static TI_AdviceCode *of (const JPL_AdviceCode &loc) {
    return (TI_AdviceCode*)loc.transform_info ();
  }
};

class TI_Introduction : public TransformInfo {
  ACIntroductionInfo *_acii;
  // the following elements are only needed for the *old* repository code
  Location _loc;
  int _lines;
  Unit *_unit;
  
public:
  TI_Introduction () : _acii (0) {}
  
  void intro_info (ACIntroductionInfo *acii) {
    _acii = acii;
    _loc = tree ()->token ()->location ();
    _lines = tree ()->end_token ()->location ().line () - line () + 1;
    _unit = (Unit*)tree ()->token ()->belonging_to ();
  }
      
  virtual CObjectInfo *obj_info () const { return 0; }
  virtual CObjectInfo *assoc_obj () const { return obj_info (); }
  virtual CT_AdviceDecl *tree () const { return _acii->def_node (); }

  int line () const { return _loc.line (); }
  int lines () const { return _lines; }
  Unit *unit () const { return _unit; }

  static TI_Introduction *of (const JPL_Introduction &loc) {
    return (TI_Introduction*)loc.transform_info ();
  }
};

class TI_Order : public TransformInfo {
  CT_AdviceDecl *_tree;
public:
  TI_Order () : _tree (0) {}
  void tree (CT_AdviceDecl *ad) { _tree = ad; }
  virtual CObjectInfo *obj_info () const {
    return ((CT_FctDef*)_tree->Decl ())->Object ();
  }
  virtual CObjectInfo *assoc_obj () const { return obj_info (); }
  virtual CTree *tree () const { return _tree; }

  static TI_Order *of (const JPL_Order &loc) {
    return (TI_Order*)loc.transform_info ();
  }
};

class TI_ClassSlice : public TransformInfo {
  CObjectInfo *_obj;
  const Unit *_slice_unit;

public:
  TI_ClassSlice () : _obj (0), _slice_unit (0) {}

  void obj_info (CObjectInfo *obj) { _obj = obj; }
  virtual CObjectInfo *obj_info () const { return _obj; }
  void slice_unit (const Unit *su) { _slice_unit = su; }
  const Unit &slice_unit () const { return *_slice_unit; }

  virtual CObjectInfo *assoc_obj () const { return obj_info (); }
//  virtual CTree *tree () const { return _acsi->def_node (); }
  virtual CTree *tree () const { return _obj->Tree (); }


  static TI_ClassSlice *of (const JPL_ClassSlice &loc) {
    return (TI_ClassSlice*)loc.transform_info ();
  }
  
  static string name (ACSliceInfo *acsi) {
    return acsi->object ()->QualName ();
  }
};

#endif // __transform_info_h__
