// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __pre_syntax_tree_nodes__
#define __pre_syntax_tree_nodes__

// Contains all node classes for the preprocessor syntax tree.

#include "Puma/Unit.h"
#include "Puma/PreVisitor.h"
#include "Puma/PreTreeToken.h"
#include "Puma/PreTreeComposite.h"

namespace Puma {


// The root node of the preprocessor syntax tree.
class PreProgram : public PreTreeComposite {
public:
  PreProgram (PreTree* a) : PreTreeComposite (1, 0) 
    { add_son (a); };

  void accept (PreVisitor &v) {
    v.visitPreProgram_Pre (this);
    v.iterateNodes (this);
    v.visitPreProgram_Post (this);
  }            
};      


// Assemble groups of directive groups to a `super' group => the program.
class PreDirectiveGroups : public PreTreeComposite {
public:
  PreDirectiveGroups () : PreTreeComposite (-1, 0) {};

  void accept (PreVisitor& v) {
    v.visitPreDirectiveGroups_Pre (this);
    v.iterateNodes (this);
    v.visitPreDirectiveGroups_Post (this);
  }            
};      


// Assemble conditional directives to a logical group.
// Like: #if ... #elif ... #else ... #endif.
class PreConditionalGroup : public PreTreeComposite {
public:
  PreConditionalGroup (PreTree* a, PreTree* b, PreTree* c) : 
    PreTreeComposite (3, 0) // 3 sons, no daughters.
    { add_son (a); add_son (b); add_son (c); };
        
  PreConditionalGroup (PreTree* a, PreTree* b, PreTree* c, PreTree* d) : 
    PreTreeComposite (4, 0) // 4 sons, no daughters.
    { add_son (a); add_son (b); add_son (c); add_son (d); };

  void accept (PreVisitor& v) {
    v.visitPreConditionalGroup_Pre (this);
    v.iterateNodes (this);
    v.visitPreConditionalGroup_Post (this);
  }
};      


// Assemble conditional directives within the else part of a 
// conditional group to a logical group.
// Like: ... #elif ... #elif ... #elif ... #else.
class PreElsePart : public PreTreeComposite {
public:
  PreElsePart (PreTree* a, PreTree* b) : 
    PreTreeComposite (2, 0) // 2 sons, no daughters.
    { add_son (a); add_son (b); };
                                        
  PreElsePart (PreTree* a, PreTree* b, PreTree* c) : 
    PreTreeComposite (3, 0) // 3 sons, no daughters.
    { add_son (a); add_son (b); add_son (c); };

  void accept (PreVisitor& v) {
    v.visitPreElsePart_Pre (this);
    v.iterateNodes (this);
    v.visitPreElsePart_Post (this);
  }
};      


// Assemble conditional directives in the elif part of a conditional 
// group to a logical group.
// Like: ... #elif ... #elif ... #elif.
class PreElifPart : public PreTreeComposite
 {
    public:
                                // Dynamic sons, no daughters.
        PreElifPart () : PreTreeComposite (-1, 0) {};
        
        // Add two sons.
        void addSons (PreTree* a, PreTree* b)
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreElifPart_Pre (this);
            v.iterateNodes (this);
            v.visitPreElifPart_Post (this);
        }
 };      


// The #if directive.
class PreIfDirective : public PreTreeComposite
 {
    public:
                                        
        PreIfDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 1) // 2 sons, 1 daughter.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreIfDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreIfDirective_Post (this);
        }
 };      


// The #ifdef directive.
class PreIfdefDirective : public PreTreeComposite
 {
    public:
                                        
        PreIfdefDirective (PreTree* a, PreTree* b, PreTree* c) 
                        : PreTreeComposite (3, 1) // 3 sons, 1 daughter.
            { add_son (a); add_son (b); add_son (c); };
        PreIfdefDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughter.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreIfdefDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreIfdefDirective_Post (this);
        }
 };      


// The #ifndef directive.
class PreIfndefDirective : public PreTreeComposite
 {
    public:
                                        
        PreIfndefDirective (PreTree* a, PreTree* b, PreTree* c) 
                        : PreTreeComposite (3, 1) // 3 sons, 1 daughter.
            { add_son (a); add_son (b); add_son (c); };
        PreIfndefDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughter.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreIfndefDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreIfndefDirective_Post (this);
        }
 };      


// The #elif directive.
class PreElifDirective : public PreTreeComposite
 {
    public:
                                        
        PreElifDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 1) // 2 sons, 1 daughter.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreElifDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreElifDirective_Post (this);
        }
 };      


// The #else directive.
class PreElseDirective : public PreTreeComposite
 {
    public:
                                        
        PreElseDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreElseDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreElseDirective_Post (this);
        }
 };      


// The #endif directive.
class PreEndifDirective : public PreTreeComposite
 {
    public:
                                        
        PreEndifDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreEndifDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreEndifDirective_Post (this);
        }
 };      


// The #include and #include_next directive.
class PreIncludeDirective : public PreTreeComposite
 {
        int _depth;

    public:
                                        
        PreIncludeDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 1) // 2 sons, 1 daughter.
            { add_son (a); add_son (b); _depth = -1; };

        void accept (PreVisitor& v)
        {
            v.visitPreIncludeDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreIncludeDirective_Post (this);
        }
        
        int depth () const 
        {
            return _depth;
        }

        void depth (int d)
        {
            _depth = d;
        }
        
        bool is_forced () const {
          return !((Unit*)startToken ()->belonging_to ())->isFile ();
        }
 };      


// The #assert directive.
class PreAssertDirective : public PreTreeComposite
 {
    public:
                                        
        PreAssertDirective (PreTree* a, PreTree* b, PreTree* c) 
                        : PreTreeComposite (3, 0) // 3 sons, no daughters.
            { add_son (a); add_son (b); add_son (c); };
        PreAssertDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreAssertDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreAssertDirective_Post (this);
        }
 };      


// The #unassert directive.
class PreUnassertDirective : public PreTreeComposite
 {
    public:
                                        
        PreUnassertDirective (PreTree* a, PreTree* b, PreTree* c) 
                        : PreTreeComposite (3, 0) // 3 sons, no daughters.
            { add_son (a); add_son (b); add_son (c); };
        PreUnassertDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreUnassertDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreUnassertDirective_Post (this);
        }
 };      


// The #define directive that defines function-like macros.
// Like: #define FUNCTION (id,...,id) ... ... ...
class PreDefineFunctionDirective : public PreTreeComposite
 {
    public:
                                        
        PreDefineFunctionDirective (PreTree* a, PreTree* b, 
                                    PreTree* c, PreTree* d, 
                                    PreTree* e, PreTree* f, 
                                    PreTree* g, PreTree* h) 
                        : PreTreeComposite (8, 0) // 8 sons, no daughters.
            { add_son (a); add_son (b); add_son (c); add_son (d);
              add_son (e); add_son (f); add_son (g); add_son (h); }

        PreDefineFunctionDirective (PreTree* a, PreTree* b, 
                                    PreTree* c, PreTree* d, 
                                    PreTree* e, PreTree* f, 
                                    PreTree* g) 
                        : PreTreeComposite (7, 0) // 7 sons, no daughters.
            { add_son (a); add_son (b); add_son (c); add_son (d);
              add_son (e); add_son (f); add_son (g); }

        PreDefineFunctionDirective (PreTree* a, PreTree* b, 
                                    PreTree* c, PreTree* d, 
                                    PreTree* e, PreTree* f) 
                        : PreTreeComposite (6, 0) // 6 sons, no daughters.
            { add_son (a); add_son (b); add_son (c); add_son (d);
              add_son (e); add_son (f); }
        
        PreDefineFunctionDirective (PreTree* a, PreTree* b, PreTree* c, 
                                   PreTree* d, PreTree* e) 
                        : PreTreeComposite (5, 0) // 5 sons, no daughters.
            { add_son (a); add_son (b); add_son (c); add_son (d); add_son (e); }

        void accept (PreVisitor& v)
        {
            v.visitPreDefineFunctionDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreDefineFunctionDirective_Post (this);
        }
 };              


// The #define directive that defines symbolic constants macros.
// Like: #define CONSTANT ... ... ...
class PreDefineConstantDirective : public PreTreeComposite
 {
    public:
                                        
        PreDefineConstantDirective (PreTree* a, PreTree* b, PreTree* c) 
                        : PreTreeComposite (3, 0) // 3 sons, no daughters.
            { add_son (a); add_son (b); add_son (c); };
        PreDefineConstantDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreDefineConstantDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreDefineConstantDirective_Post (this);
        }
 };      


// The #undef directive.
class PreUndefDirective : public PreTreeComposite
 {
    public:
                                        
        PreUndefDirective (PreTree* a, PreTree* b, PreTree* c) 
                        : PreTreeComposite (3, 0) // 3 sons, no daughters.
            { add_son (a); add_son (b); add_son (c); };
        PreUndefDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreUndefDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreUndefDirective_Post (this);
        }
 };      


// The #warning directive.
class PreWarningDirective : public PreTreeComposite
 {
    public:
                                        
        PreWarningDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreWarningDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreWarningDirective_Post (this);
        }
 };      


// The #error directive.
class PreErrorDirective : public PreTreeComposite
 {
    public:
                                        
        PreErrorDirective (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreErrorDirective_Pre (this);
            v.iterateNodes (this);
            v.visitPreErrorDirective_Post (this);
        }
 };      


// Collects comma separated argument identifiers for a function-like macro.
// Like: id, id, ..., id.
class PreIdentifierList : public PreTreeComposite
 {
    public:
                                        // Dynamic sons, no daughters. 
        PreIdentifierList (PreTree* a) : PreTreeComposite (-1, 0)
            { add_son (a); };
        
        // Add two sons.
        void addSons (PreTree* a, PreTree* b)
            { add_son (a); add_son (b); };

        void accept (PreVisitor& v)
        {
            v.visitPreIdentifierList_Pre (this);
            v.iterateNodes (this);
            v.visitPreIdentifierList_Post (this);
        }
 };      


// Collects every kind of token  (except comments) for the macro body.
class PreTokenList : public PreTreeComposite
 {
    public:
                                        
        PreTokenList (PreTree* a, PreTree* b) 
                        : PreTreeComposite (2, 0) // 2 sons, no daughters.
            { add_son (a); add_son (b); };
        PreTokenList (PreTree* a) 
                        : PreTreeComposite (1, 0) // 1 son, no daughters.
            { add_son (a); };
        PreTokenList () : PreTreeComposite (0, 0) // no sons, no daughters.
            { };
        
        void accept (PreVisitor& v)
        {
            v.visitPreTokenList_Pre (this);
            v.iterateNodes (this);
            v.visitPreTokenList_Post (this);
        }
 };      


// Contains every kind of token  (except comments) for the macro body.
class PreTokenListPart : public PreTreeComposite
 {
    public:
                                        // Dynamic sons, no daughters.
        PreTokenListPart (PreTree* a) : PreTreeComposite (-1, 0)
            { add_son (a); };
        
        void accept (PreVisitor& v)
        {
            v.visitPreTokenListPart_Pre (this);
            v.iterateNodes (this);
            v.visitPreTokenListPart_Post (this);
        }
 };      


// Semantic node for conditionals.
class PreCondSemNode : public PreTree
 {
        // Boolean value of the condition of a conditional.
        bool _value;

    public:

        PreCondSemNode (bool value) : _value (value) {};
        
        void accept (PreVisitor& v)
        {
            v.visitPreCondSemNode (this);
        }
        
        // Get the value of the conditional.
        bool value () const { return _value; }
 };      


// Semantic node for the #include directive.
class PreInclSemNode : public PreTree
 {
        // Pointer of the included file unit.
        Unit* _unit;
        
        // true if the inclusion was not done, because of an active include
        // guard
        bool _guarded;

    public:

        PreInclSemNode (Unit* unit, bool guarded) :
          _unit (unit), _guarded (guarded) {};
        
        void accept (PreVisitor& v)
        {
            v.visitPreInclSemNode (this);
        }
        
        // Get the included file unit.
        Unit* unit () const { return _unit; }
        
        // Check if the inclusion was not done, because of an include guard
        bool guarded () const { return _guarded; }
 };      


// Special parse error node to show the location of an error in the
// preprocessor syntax tree.
class PreError : public PreTree
 {
    public:

        PreError () {};
        
        void accept (PreVisitor& v)
        {
            v.visitPreError (this);
        }
 };      


} // namespace Puma

#endif /* __pre_syntax_tree_nodes__ */
