// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CEnumInfo_h__
#define __CEnumInfo_h__

// enumeration info 
// knows its enumerators (enumeration constants)

#include "Puma/CEnumeratorInfo.h"
#include "Puma/CTypeInfo.h"
#include "Puma/Array.h"
#include "Puma/CScopeRequest.h"

namespace Puma {


/** \file 
 *  Representation of a C/C++ 'enum'.
 *  \see Puma::CEnumInfo */

/** \class CEnumInfo CEnumInfo.h Puma/infos/CEnumInfo.h
 *  A CEnumInfo object represents a C++ 'enum'. */
class CEnumInfo : public CScopeRequest {
  Array<CEnumeratorInfo*> _Enumerators;
  CTypeInfo *_UnderlyingType;

public: 
  /** Construct a new enumeration info object. */
  CEnumInfo ();
  /** Destroy the enum info object. If the object id is 
   *  \c ENUM_INFO, Puma::CObjectInfo::CleanUp() is called. */
  ~CEnumInfo ();

  /** Get the number of enumerators in this enumeration. */
  unsigned Enumerators () const;
  /** Get the n-th enumerator info.
   *  \param n The number of the enumerator.
   *  \return The enumerator info or \c NULL if \e n is invalid. */
  CEnumeratorInfo *Enumerator (unsigned n) const;
  /** Get the info of the enumerator with the given name. 
   *  \param name The name of the enumerator.
   *  \return The enumerator info or \c NULL if no enumerator with that name. */
  CEnumeratorInfo *Enumerator (const char *name) const;

  /** Get the base info object of the info object. <b>The 
   *  corresponding information is not yet set!</b>
   *  \return This method always returns \c NULL.
   *  \warning Do not use this method. */
  CEnumInfo *BaseObject () const;
  /** Get the info object of the enumeration definition. 
   *  \return The info or \e this if the enumeration is not defined. */
  CEnumInfo *DefObject () const;
  /** Get the type information object for the enumeration. */
  CTypeEnum *TypeInfo () const;

  /** Add a new enumerator to the enumeration.
   *  \param ei The enumerator info to add. */
  void addEnumerator (CEnumeratorInfo *ei);
  /** Remove the given enumerator from the enumeration.
   *  \param ei The enumerator info to remove. */
  void removeEnumerator (const CEnumeratorInfo *ei);

  /** Create a new enumerator info. The enumerator info is
   *  added to the enumeration. */
  CEnumeratorInfo *newEnumerator ();
  /** Destroy the given enumerator info. The enumerator info is
   *  removed from the enumeration.
   *  \param ei The enumerator info to destroy. */
  void deleteEnumerator (const CEnumeratorInfo *ei);
  
  /** Return \e true if the enumeration is defined. */
  bool isDefined () const;
  /** Return \e true if the enumeration is complete (defined)
   *  at the given position.
   *  \param pos The current source code position. */
  bool isComplete (unsigned long pos = 0) const;

  /** Return the underlying data type. */
  CTypeInfo *UnderlyingType () const;
  /** Set the underlying data type.
   *  \param type The underlying type. */
  void UnderlyingType (CTypeInfo *type);
};

inline CEnumInfo::CEnumInfo () :
  CScopeRequest (CObjectInfo::ENUM_INFO),
  _UnderlyingType (&CTYPE_INT)
 {}
 
inline unsigned CEnumInfo::Enumerators () const
 { return _Enumerators.length (); }
inline CEnumeratorInfo *CEnumInfo::Enumerator (unsigned n) const
 { return _Enumerators.lookup (n); }

inline CEnumInfo *CEnumInfo::BaseObject () const
 { return (CEnumInfo*)CObjectInfo::BaseObject (); }
inline CTypeEnum *CEnumInfo::TypeInfo () const
 { return (CTypeEnum*)CObjectInfo::TypeInfo (); }
//inline CT_EnumSpec *CEnumInfo::Tree () const
// { return (CT_EnumSpec*)CObjectInfo::Tree (); }

inline CTypeInfo *CEnumInfo::UnderlyingType () const
 { return (CTypeInfo*)_UnderlyingType; }
inline void CEnumInfo::UnderlyingType (CTypeInfo *type)
 { _UnderlyingType = type; }


} // namespace Puma

#endif /* __CEnumInfo_h__ */
