# include "automaton.h"
# include "charconst.h"
# include <stdio.h>

ListEntry::ListEntry (ListData *New)
 { Data = New;
 }

void ListEntry::Insert (ListEntry* Behind, ListEntry* Before)
 { if (Behind)
      Behind->Next = this;
   Prev = Behind;
   if (Before)
      Before->Prev = this;
   Next = Before;
 }

void ListEntry::Remove (void)
 { if (Next)
      Next->Prev = Prev;
   if (Prev)
      Prev->Next = Next;
 }

ListEntry* ListEntry::NextEntry (void)
 { return Next;
 }

ListEntry* ListEntry::PrevEntry (void)
 { return Prev;
 }

ListData* ListEntry::GetData (void)
 { return Data;
 }

List::List (void)
 { First = Last = NULL;
   Entries = 0;
 }

List::~List (void)
 { ListEntry* NextEntry;
   ListEntry* CurrEntry;

   CurrEntry = First;
   while (CurrEntry != NULL)
    { NextEntry = CurrEntry->NextEntry ();
      delete CurrEntry;
      CurrEntry = NextEntry;
    }
 }

int List::Empty (void)
 { return (Entries == 0);
 }

long List::Length (void)
 { return Entries;
 }

void List::Append (ListData* New)
 { ListEntry* NewEntry;

   NewEntry = new ListEntry (New);
   NewEntry->Insert (Last, NULL);
   if (!First)
      First = NewEntry;
   Last = NewEntry;
   Entries++;
 }

void List::Remove (ListData* ToRemove)
 { ListEntry* CurrEntry;

   CurrEntry = First;
   while (CurrEntry->GetData () != ToRemove)
      CurrEntry = CurrEntry->NextEntry ();
   if (First == CurrEntry)
      First = CurrEntry->NextEntry ();
   if (Last == CurrEntry)
      Last = CurrEntry->PrevEntry ();
   CurrEntry->Remove ();
   delete CurrEntry;
   Entries--;
 }

void List::StartScanning (ListEntry*& Context)
 { Context = First;
 }

ListData* List::GetNext (ListEntry*& Context)
 { ListEntry* CurrEntry;

   CurrEntry = Context;
   if (Context)
    { Context = Context->NextEntry ();
      return CurrEntry->GetData ();
    }
   else
      return NULL;
 }

void List::ShiftEntries (List& List)
 { 
   if (List.Entries == 0)
      return;
   List.First->Insert (Last, List.First->NextEntry ());
   if (!First)
      First = List.First;
   Last = List.Last;
   Entries += List.Entries;
   List.First = List.Last = NULL;
   List.Entries = 0;
 }

void List::CopyEntries (List& ToCopy)
 { ListEntry* Context;
   ListData* Data;

   ToCopy.StartScanning (Context);
   while ((Data = ToCopy.GetNext (Context)) != NULL)
      Append (Data);
 }

int List::IsIn (ListData* ToFind)
 { ListEntry* Context;
   ListData* CurrData;
   
   StartScanning (Context);
   while ((CurrData = GetNext (Context)) != NULL)
    { if (CurrData == ToFind)
	 return 1;
    }
   return 0;
 }

int List::IsEqual (List& List)
 { ListEntry* Context;
   ListData* CurrData;

   if (Entries != List.Entries)
      return 0;
   StartScanning (Context);
   while ((CurrData = GetNext (Context)) != NULL)
      if (!List.IsIn (CurrData))
	 return 0;
   return 1;
 }

void List::Print (void)
 { ListEntry* Context;
   ListData* CurrEntry;

   printf ("\nList-Start\n");
   StartScanning (Context);
   while ((CurrEntry = GetNext (Context)) != NULL)
      CurrEntry->Print ();
   printf ("\nList-End\n");
 }

void Stack::PushList (List& List)
 { CopyEntries (List);
 }

void Stack::Push (ListData* New)
 { Append (New);
 }

ListData* Stack::Pop (void)
 { ListData *Top;
   ListEntry* LastEntry;

   LastEntry = Last;
   Top = LastEntry->GetData ();
   if (First == Last)
      First = NULL;
   Last = Last->PrevEntry ();
   LastEntry->Remove ();
   delete LastEntry;
   Entries--;
   return Top;
 }

Transition::Transition (TransitionType NewType, State* NewState)
 { Type = NewType;
   TransitionState = NewState;
 }

Transition::Transition (TransitionType NewType, char NewChar, State* NewState)
 { Type = NewType;
   TransitionState = NewState;
   Temp.Char = NewChar;
 }

Transition::Transition (TransitionType NewType, char* NewName, State* NewState)
 { Type = NewType;
   TransitionState = NewState;
   Temp.Name = NewName;
 }

Transition::Transition (TransitionType NewType, ClassId NewId, State* NewState)
 { Type = NewType;
   TransitionState = NewState;
   Id = NewId;
 }

void Transition::Print (void)
 { unsigned char CharBuff[5];
   unsigned char CurrChar;

   printf ("--");
   switch (Type)
    { case EPSILON_TRANS:
	 printf ("Epsilon");
	 break;
      case CHAR_CLASS_TRANS:
	 Id.Print ();
	 break;
      case CLASS_NAME_TRANS:
	 printf ("Class %s", Temp.Name);
	 break;
      case CHAR_TRANS:
         CurrChar = (unsigned char)Temp.Char;
	 CharToString (CurrChar, CharBuff);
	 printf ("\'%s\'", CharBuff);
	 break;
    }
   printf ("--> [%p]\n", TransitionState);
 }


State::State (void)
 { Type = NORMAL_STATE;
   LookAhead  = 0;
   Expression = 0;
 }

State::~State (void)
 { ListEntry* Context;
   Transition* CurrTransition;
   
   EpsilonTransitions.StartScanning (Context);
   while ((CurrTransition = (Transition *)
	   EpsilonTransitions.GetNext (Context)) != NULL)
      delete CurrTransition;
   NonEpsilonTransitions.StartScanning (Context);
   while ((CurrTransition = (Transition *)
	   NonEpsilonTransitions.GetNext (Context)) != NULL)
      delete CurrTransition;
 }

int State::Expr () 
 { return Expression; }

void State::Expr (int E)
 { Expression = E; }

void State::AddTransition (TransitionType Type, State* State)
 { 
   if (Type == EPSILON_TRANS)
      EpsilonTransitions.Append (new Transition (Type, State));
   else
      NonEpsilonTransitions.Append (new Transition (Type, State));
 }

void State::AddTransition (TransitionType Type, char Char, State* State)
 { 
   if (Type == EPSILON_TRANS)
      EpsilonTransitions.Append (new Transition (Type, Char, State));
   else
      NonEpsilonTransitions.Append (new Transition (Type, Char, State));
 }

void State::AddTransition (TransitionType Type, char* Name, State* State)
 { 
   if (Type == EPSILON_TRANS)
      EpsilonTransitions.Append (new Transition (Type, Name, State));
   else
      NonEpsilonTransitions.Append (new Transition (Type, Name, State));
 }

void State::AddTransition (TransitionType Type, ClassId Id, State* State)
 { 
   if (Type == EPSILON_TRANS)
      EpsilonTransitions.Append (new Transition (Type, Id, State));
   else
      NonEpsilonTransitions.Append (new Transition (Type, Id, State));
 }

void State::ShiftTransitionList (State* State)
 { EpsilonTransitions.ShiftEntries (State->EpsilonTransitions);
   NonEpsilonTransitions.ShiftEntries (State->NonEpsilonTransitions);
  
 } 

void State::Print (void)
 { 
   printf ("[%p] - ", this);
   if (LookAhead)
      printf ("LOOK_AHEAD, ");
   switch (Type)
    { case NORMAL_STATE:
	 printf ("NORMAL");
	 break;
      case END_STATE:
	 printf ("END (Expression %d)", Expr ());
	 break;
      case LOOK_AHEAD_END_STATE:
	 printf ("LOOK_AHEAD_END (Expression %d)", Expr ());
	 break;
    }
   printf ("\nGroup %d\n", Group);
   printf ("Transitions:\n");
   NonEpsilonTransitions.Print ();
   EpsilonTransitions.Print ();
   printf ("\n");
 }

Automaton::Automaton (void)
 { 
   Start = End = NULL;
 }

Automaton::~Automaton (void)
 { ListEntry* Context;
   State* CurrState;

   States.StartScanning (Context);
   while ((CurrState = (State*)(States.GetNext (Context))) != NULL)
      delete CurrState;
 }

void Automaton::ShiftStates (Automaton& Automaton)
 { States.ShiftEntries (Automaton.States);
 }

State* Automaton::AddState (void)
 { State* NewState;

   NewState = new State;
   States.Append (NewState);
   return NewState;
 }

void Automaton::RemoveState (State* State)
 { States.Remove (State);
   delete State;
 }

void Automaton::ShiftState (Automaton& From, State* State)
 { From.States.Remove (State);
   States.Append (State);
 }

void Automaton::ScanStates (ListEntry*& Context)
 {
   States.StartScanning (Context);
 }

State* Automaton::NextState (ListEntry*& Context)
 { 
   return (State*)(States.GetNext (Context));
 }

void Automaton::Print (void)
 { ListEntry* Context;
   State* CurrState;

   printf ("Start: [%p]\n\n", Start);
   ScanStates (Context);
   while ((CurrState = NextState (Context)) != NULL)
      CurrState->Print ();
   printf ("\n");
 }

void DFAStateDescriptor::Print (void)
 { printf ("DFA-State-Descriptor\n");
   printf ("--------------------\n");
   printf ("DFA-State:\n");
   DFAState->Print ();
   printf ("NFA-States:\n");
   NFAStates.Print ();
 }



