/*
 * Chan_Misdn -- Channel Driver for Asterisk
 *
 * Interface to Asterisk
 *
 * Copyright (C) 2004, Christian Richter
 *
 * Christian Richter <crich@beronet.com>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License
 */


#include <stdio.h>
#include <pthread.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <sys/ioctl.h>


#include <asterisk/channel.h>
#include <asterisk/config.h>
#include <asterisk/logger.h>
#include <asterisk/module.h>
#include <asterisk/pbx.h>
#include <asterisk/options.h>
#include <asterisk/io.h>
#include <asterisk/frame.h>
#include <asterisk/translate.h>
#include <asterisk/cli.h>
#include <asterisk/musiconhold.h>
#include <asterisk/dsp.h>
#include <asterisk/translate.h>
#include <asterisk/config.h>
#include <asterisk/file.h>
#include <asterisk/callerid.h>
#include <asterisk/indications.h>
#include <asterisk/app.h>
#include <asterisk/features.h>
#include "chan_misdn_if.h"
#include "te_lib.h"
#include "crypt.h"

#ifdef ASTERISK_STABLE
#include <asterisk/channel_pvt.h>
#endif
//#define MISDN_DEBUG 1

#ifdef ASTERISK_STABLE
#define MISDN_ASTERISK_TECH_PVT(ast) ast->pvt->pvt
#define MISDN_ASTERISK_PVT(ast) ast->pvt
#else
#define MISDN_ASTERISK_TECH_PVT(ast) ast->tech_pvt
#define MISDN_ASTERISK_PVT(ast) 1
#endif

static  char *desc = "Channel driver for mISDN Support (Bri/Pri)";
static  char *type = "mISDN";
#ifdef ASTERISK_STABLE
static  char *tdesc = "This driver enables the asterisk to use hardware which is supported by the new  isdn4linux System mISDN";
#endif


static int usecnt =0;

static char default_context[255]="misdn";
static char default_language[255]="en";
static char default_msns[1024]="";
static int default_dialplan=0;
static char default_natprefix[128]="0";
static char default_internatprefix[128]="00";
static int default_volume=0;
static int default_te_choose_channel=0;

static char tracefile[512]="";
static int tracing =0 ;

static int stop_tone_after_first_digit=1;
static int append_digits2exten=1;

char **misdn_key_vector=NULL;
int misdn_key_vector_size=0;

static int misdn_dynamic_crypting=0;
static char *misdn_crypt_prefix=NULL;
static char *misdn_crypt_keys=NULL; 



/* Only alaw and mulaw is allowed for now */
static int prefformat =  AST_FORMAT_ALAW; // AST_FORMAT_SLINEAR ;  //AST_FORMAT_ULAW |


static ast_mutex_t usecnt_lock; 

extern int misdn_crypt_debug; 
int misdn_debug=0;

chan_list_t dummy_cl;

chan_list_t *cl_te=NULL;
pthread_mutex_t cl_te_lock;



typedef struct msn_list_s msn_list_t; 
struct msn_list_s {
  char *msn;
  msn_list_t *next;
};

static msn_list_t *msn_list=NULL;


typedef struct config_list_s config_list_t;
pthread_mutex_t config_mutex; 

struct config_list_s {
  int port;
  int ptp;
  int volume;
  int te_choose_channel;
  
  char *group;
  char *context;
  char *language;
  char *callerid;
  int dialplan; 
  char *natprefix;
  char *internatprefix;
  

#ifndef ASTERISK_STABLE
  ast_group_t callgroup;		/* Call group */
  ast_group_t pickupgroup;		/* Pickup group */
#endif
  
  
  int immediate;
  int hold_allowed;
  msn_list_t *msn_list;
  config_list_t *next;
};

static config_list_t *config_list=NULL;


manager_te_t *mgr_te;
enum te_event_response_e
cb_te_events(manager_te_t *mgr, enum te_event_e event, bchannel_te_t *bc, void *user_data);


void cl_queue_chan(chan_list_t **list, chan_list_t *chan);
void cl_dequeue_chan(chan_list_t **list, chan_list_t *chan);
chan_list_t *cl_find_chan_by_bc_te(chan_list_t *list, bchannel_te_t *bc);
void * audio_thread( void * data);
void chan_misdn_log(char *tmpl, ...);
int check_ret (manager_te_t *mgr,bchannel_te_t * bc, enum te_event_e event, int ret);
static int start_bc_tones(chan_list_t *cl);
static int stop_bc_tones(chan_list_t *cl);
static void release_chan(bchannel_te_t *bc);
void misdn_reload_config ();

static int misdn_set_opt_exec(struct ast_channel *chan, void *data);


/*************** Helpers *****************/

static chan_list_t * get_chan_by_ast(struct ast_channel *ast)
{
  chan_list_t *tmp;
  
  for (tmp=cl_te; tmp; tmp = tmp->next) {
    if ( tmp->ast == ast ) return tmp;
  }
  
  return NULL;
}


static chan_list_t * get_chan_by_ast_name(char *name)
{
  chan_list_t *tmp;
  
  for (tmp=cl_te; tmp; tmp = tmp->next) {
    if ( tmp->ast  && strcmp(tmp->ast->name,name) == 0) return tmp;
  }
  
  return NULL;
}


static int get_dialplan(bchannel_te_t *bc)
{
  config_list_t *config=config_list;
  
  for (;config; config=config->next) {
    if( (config->port == bc->stack->port) && config->dialplan >=0 ) return config->dialplan;
  }
  
  return default_dialplan;
}


static char *get_nat_prefix (bchannel_te_t *bc) {
  config_list_t *config=config_list;
  
  for (;config; config=config->next) {
    if( (config->port == bc->stack->port) && (strlen(config->natprefix) > 0 )) return config->natprefix;
  }
  
  return default_natprefix;
}


static char *get_internat_prefix (bchannel_te_t *bc) {
  config_list_t *config=config_list;
  
  for (;config; config=config->next) {
    if( (config->port == bc->stack->port) && (strlen(config->internatprefix) > 0 )) return config->internatprefix;
  }
  
  return default_internatprefix;
}




static char *get_context(bchannel_te_t *bc) {
  config_list_t *config=config_list;
  
  for (;config; config=config->next) {
    if( (config->port == bc->stack->port) && (strlen(config->context) > 0 )) return config->context;
  }
  
  return default_context;
}

static char *get_language(config_list_t *config) {
  if (config->language && (strlen(config->language) > 0))  return config->language;
  return default_language;
}

static int get_volume(config_list_t *config) {
  if (config->volume != 0)  return config->volume;
  return default_volume;
}

static int get_te_choose_channel(config_list_t *config) {
  if (config->te_choose_channel>=0) return config->te_choose_channel;
  return default_te_choose_channel; 
}


static char *get_callerid(config_list_t *config) {
  if (config->callerid && (strlen(config->callerid) > 0))  return config->callerid;
  return NULL;
}

static char* tone2str(bchannel_te_t *bc)
{
  static struct {
    char name[16];
    enum tone_e tone;
  } *tone, tone_buf[] = {
    {"NOTONE",TONE_NONE},
    {"DIAL",TONE_DIAL},
    {"BUSY",TONE_BUSY},
    {"ALERT",TONE_ALERTING},
    {"",TONE_NONE}
  };
  
  
  for (tone=&tone_buf[0]; tone->name[0]; tone++) {
    if (tone->tone == bc->tone) return tone->name;
  }
  return NULL;
}


static config_list_t *find_config_by_port(int port)
{
  config_list_t *config=config_list;
  
  for (;config; config=config->next) {
    if (config->port==port)
      return config;
  }
  return NULL;
}

static void print_bearer(bchannel_te_t *bc) 
{
  switch (bc->capability) {
  case INFO_CAPABILITY_SPEECH:
    if (misdn_debug>1) chan_misdn_log(" --> Bearer: Speech\n");
    break;
  case INFO_CAPABILITY_AUDIO:
    if (misdn_debug>1) chan_misdn_log(" --> Bearer: Audio\n");
    break;
  case INFO_CAPABILITY_DIGITAL_UNRESTRICTED:
    if (misdn_debug>1) chan_misdn_log(" --> Bearer: Unres Digital\n");
    break;
  case INFO_CAPABILITY_DIGITAL_RESTRICTED:
    if (misdn_debug>1) chan_misdn_log(" --> Bearer: Res Digital\n");
    break;
  default: 
    if (misdn_debug>1) chan_misdn_log(" --> Unknown Bearer\n");
    break;
  }
}
/*************** Helpers END *************/





void send_digit_to_chan(chan_list_t *cl, char digit )
{
  static const char* dtmf_tones[] = {
    "!941+1336/100,!0/100",	/* 0 */
    "!697+1209/100,!0/100",	/* 1 */
    "!697+1336/100,!0/100",	/* 2 */
    "!697+1477/100,!0/100",	/* 3 */
    "!770+1209/100,!0/100",	/* 4 */
    "!770+1336/100,!0/100",	/* 5 */
    "!770+1477/100,!0/100",	/* 6 */
    "!852+1209/100,!0/100",	/* 7 */
    "!852+1336/100,!0/100",	/* 8 */
    "!852+1477/100,!0/100",	/* 9 */
    "!697+1633/100,!0/100",	/* A */
    "!770+1633/100,!0/100",	/* B */
    "!852+1633/100,!0/100",	/* C */
    "!941+1633/100,!0/100",	/* D */
    "!941+1209/100,!0/100",	/* * */
    "!941+1477/100,!0/100" };	/* # */
  struct ast_channel *chan=cl->ast; 
  
  if (digit >= '0' && digit <='9')
    ast_playtones_start(chan,0,dtmf_tones[digit-'0'], 0);
  else if (digit >= 'A' && digit <= 'D')
    ast_playtones_start(chan,0,dtmf_tones[digit-'A'+10], 0);
  else if (digit == '*')
    ast_playtones_start(chan,0,dtmf_tones[14], 0);
  else if (digit == '#')
    ast_playtones_start(chan,0,dtmf_tones[15], 0);
  else {
    /* not handled */
    ast_log(LOG_DEBUG, "Unable to handle DTMF tone '%c' for '%s'\n", digit, chan->name);
    
    
  }
}
/*** CLI HANDLING ***/
static int misdn_set_debug(int fd, int argc, char *argv[])
{
  if (argc != 4 )return RESULT_SHOWUSAGE; 
  
  misdn_debug = atoi(argv[3]);
  ast_cli(fd,"changing debug level to %d\n",misdn_debug);

  return 0;
}


static int misdn_set_crypt_debug(int fd, int argc, char *argv[])
{
  if (argc != 5 )return RESULT_SHOWUSAGE; 
  
  misdn_crypt_debug = atoi(argv[4]);
  ast_cli(fd,"changing debug level to %d\n",misdn_crypt_debug);

  return 0;
}

static int misdn_flush_stack (int fd, int argc, char *argv[])
{
  manager_te_t *mgr = mgr_te;
  int port, i;
  stack_te_t *stack;
  
  if (argc != 4)
    return RESULT_SHOWUSAGE;
  
  port = atoi(argv[3]);

  for (stack=mgr->stack_list;
       stack;
       stack=stack->next ) {
    if (stack->port == port) {
      for (i=0; i< stack->b_num; i++) {
	bchannel_te_t *mybc=&stack->bc_te[i];
	mybc->in_use=0;
      }
    }
  }
  
  
  return 0;
}



static int misdn_restart_port (int fd, int argc, char *argv[])
{
  int port;
  
  if (argc != 4)
    return RESULT_SHOWUSAGE;
  
  port = atoi(argv[3]);

  manager_te_port_restart(mgr_te, port);
  
  return 0;
}


static int misdn_show_config (int fd, int argc, char *argv[])
{
  config_list_t *config=config_list;

  ast_cli(fd,"Misdn Config: \n"); 
  
  ast_cli(fd,"--> VERSION: " CHAN_MISDN_VERSION "\n");
  ast_cli(fd,"--> debug_level (current): %d\n",misdn_debug);
  ast_cli(fd,"--> append_digits2exten: %d\n",append_digits2exten);
  ast_cli(fd,"--> stop_tone_after_first_digit: %d\n",stop_tone_after_first_digit);
  ast_cli(fd,"--> dynamic_crypting: %d\n",misdn_dynamic_crypting);
  ast_cli(fd,"\n"); 
  
  for (;config; config=config->next) {
    msn_list_t *msn;
    
    ast_cli(fd,"* Port:%d Group:%s Context:%s Lanugage:%s hold_allowed:%d immediate:%d\n",config->port,config->group, config->context, config->language?config->language:"No Lang", config->hold_allowed, config->immediate);
    for (msn=config->msn_list; msn; msn=msn->next) 
      ast_cli(fd, " --> msn %s\n",msn->msn);
    ast_cli(fd," --> Callerid:%s\n", config->callerid);
#ifndef ASTERISK_STABLE
    {
      char buf[256];
      ast_cli(fd," --> Callgroup: %s\n",ast_print_group(buf,sizeof(buf),config->callgroup));
      ast_cli(fd," --> Pickupgroup: %s\n",ast_print_group(buf,sizeof(buf),config->pickupgroup));
    }
#endif
  }

  return 0;
}



struct state_struct {
  enum misdn_chan_state state;
  char txt[255] ;
} ;

struct state_struct state_array[] = {
  {MISDN_NOTHING,"NOTHING"}, //at beginning
  {MISDN_WAITING4DIGS,"WAITING4DIGS"}, // when waiting for infos
  {MISDN_EXTCANTMATCH,"EXTCANTMATCH"}, // when asterisk couldnt match our ext
  {MISDN_DIALING,"DIALING"}, // when pbx_start
  {MISDN_CALLING,"CALLING"}, // when misdn_call is called
  {MISDN_ALERTING,"ALERTING"}, // when Alerting
  {MISDN_BUSY,"BUSY"}, // when BUSY
  {MISDN_CONNECTED,"CONNECTED"}, // when connected
  {MISDN_BRIDGED,"BRIDGED"}, // when bridged
  {MISDN_CLEANING,"CLEANING"}, //when hangup from * but we were connected before
  {MISDN_HUNGUP_FROM_MISDN,"HUNGUP_FROM_MISDN"}, //when DISCONNECT/RELEASE/REL_COMP  cam from misdn
  {MISDN_HOLDED,"HOLDED"}, //when DISCONNECT/RELEASE/REL_COMP  cam from misdn
  {MISDN_HOLD_DISCONNECT,"HOLD_DISCONNECT"}, //when DISCONNECT/RELEASE/REL_COMP  cam from misdn
  { MISDN_HUNGUP_FROM_AST,"HUNGUP_FROM_AST"} //when DISCONNECT/RELEASE/REL_COMP came out of
  //misdn_hangup
};




char *misdn_get_ch_state(chan_list_t *p) 
{
  int i;
  if( !p) return NULL;
  
  for (i=0; i< sizeof(state_array)/sizeof(struct state_struct); i++) {
    if ( state_array[i].state == p->state) return state_array[i].txt; 
  }
  
  return NULL;
}

static int misdn_reload (int fd, int argc, char *argv[])
{
  ast_cli(fd, "Reloading mISDN Config\n");
  misdn_reload_config();
  
  return 0;
}

static void print_bc_info (int fd, chan_list_t* help, bchannel_te_t* bc)
{
  struct ast_channel *ast=help->ast;
  ast_cli(fd,
	  "* Pid:%d Prt:%d Ch:%d Mode:%s Org:%s dad:%s oad:%s ctx:%s state:%s\n",
	  bc->pid, bc->stack->port, bc->channel,
	  bc->stack->mode==NT_MODE?"NT":"TE",
	  help->orginator == ORG_AST?"*":"I",
	  ast?ast->exten:NULL,
	  ast?AST_CID_P(ast):NULL,
	  ast?ast->context:NULL,
	  misdn_get_ch_state(help)
	  );
  if (misdn_debug > 0)
    ast_cli(fd,
	    "  --> astname: %s\n"
	    "  --> ch_l3id: %x\n"
	    "  --> bc_addr: %x\n"
	    "  --> bc_l3id: %x\n"
	    "  --> display: %s\n"
#if 0
	    "  --> nodtmf: %d\n"
	    "  --> curprx: %d\n"
	    "  --> curptx: %d\n"
	    "  --> crypt: %d\n"
#endif
	    "  --> activated: %d\n"
	    "  --> echo_cancel: %d\n"
	    "  --> echo_tics: %d\n"
	    "  --> notone : rx %d tx:%d\n"
	    "  --> bc_hold: %d holded_bc :%d\n",
	    help->ast->name,
	    help->l3id,
	    bc->addr,
	    bc?bc->l3_id:-1,
	    bc->display,

#if 0
	    bc->nodtmf,
	    bc->curprx,
	    bc->curptx,
	    bc->crypt,
#endif
	    bc->active,
	    bc->ec_enable,
	    bc->ec_tics, 
	    help->norxtone,help->notxtone,
	    bc->holded, help->holded_bc?1:0
	    );
  
}


static int misdn_show_cls (int fd, int argc, char *argv[])
{
  chan_list_t *help=cl_te;
  
  ast_cli(fd,"Chan List: %p\n",cl_te); 
  
  for (;help; help=help->next) {
    bchannel_te_t *bc=help->bc_te;   
    struct ast_channel *ast=help->ast;
    if (misdn_debug > 2) ast_cli(fd, "Bc:%p Ast:%p\n", bc, ast);
    if (bc) {
      print_bc_info(fd, help, bc);
    } else if ( (bc=help->holded_bc) ) {
      chan_misdn_log("ITS A HOLDED BC:\n");
      print_bc_info(fd, help,  bc);
    } else {
      ast_cli(fd,"* Channel in unknown STATE !!! Exten:%s, Callerid:%s\n", ast->exten, AST_CID_P(ast));
    }
  }
  
  
  return 0;
}



static int misdn_show_cl (int fd, int argc, char *argv[])
{
  chan_list_t *help=cl_te;

  if (argc != 4)
    return RESULT_SHOWUSAGE;
  
  for (;help; help=help->next) {
    bchannel_te_t *bc=help->bc_te;   
    struct ast_channel *ast=help->ast;
    
    if (bc && ast) {
      if (!strcasecmp(ast->name,argv[3])) {
	print_bc_info(fd, help, bc);
	break; 
      }
    } 
  }
  
  
  return 0;
}

int MAXTICS=8;

static int misdn_set_tics (int fd, int argc, char *argv[])
{
  if (argc != 4)
    return RESULT_SHOWUSAGE;
  
  MAXTICS=atoi(argv[3]);
  
  return 0;
}


static int misdn_show_fullstacks (int fd, int argc, char *argv[])
{
  manager_te_t *mgr = mgr_te;
  
  stack_te_t *stack;
  ast_cli(fd, "BEGIN STACK_LIST:\n");
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next ) {
    int i;
    ast_cli(fd, "* Stack Addr: Uid %x Port %d Type %s Prot. %s Link %s\n",stack->upper_id, stack->upper_id & IF_CONTRMASK, stack->mode==NT_MODE?"NT":"TE", stack->ptp?"PTP":"PMP", stack->l2link?"UP":"DOWN");
    for (i=0; i< stack->b_num; i++) {
      bchannel_te_t *mybc=&stack->bc_te[i];
      ast_cli(fd," --> bchan: addr %x channel %d pid %d cr %x tone %s inuse %d\n", mybc->addr,mybc->channel, mybc?mybc->pid:-1,mybc?mybc->l3_id:-1 , tone2str(mybc), mybc->in_use);
    }
  }

  return 0;
}

static int misdn_show_stacks (int fd, int argc, char *argv[])
{
  manager_te_t *mgr = mgr_te;
  stack_te_t *stack;
    
  ast_cli(fd, "BEGIN STACK_LIST:\n");
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next ) {
    ast_cli(fd, "* Stack Addr: Port %d Type %s Prot. %s L2Link %s L1Link:%s\n", stack->upper_id & IF_CONTRMASK, stack->mode==NT_MODE?"NT":"TE", stack->ptp?"PTP":"PMP", stack->l2link?"UP":"DOWN", stack->l1link?"UP":"DOWN");
  }

  return 0;

}

static int misdn_show_port (int fd, int argc, char *argv[])
{
  manager_te_t *mgr = mgr_te;
  stack_te_t *stack;
  int i;
  int port;
  
  if (argc != 4)
    return RESULT_SHOWUSAGE;
  
  port = atoi(argv[3]);
  
  ast_cli(fd, "BEGIN STACK_LIST:\n");
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next ) {
    if (stack->port == port) {
      ast_cli(fd, "* Stack Addr: %x Port %d Type %s Prot. %s L2Link %s L1Link:%s\n",stack->upper_id, stack->upper_id & IF_CONTRMASK, stack->mode==NT_MODE?"NT":"TE", stack->ptp?"PTP":"PMP", stack->l2link?"UP":"DOWN", stack->l1link?"UP":"DOWN");
      for (i=0; i <stack->b_num; i++) {
	ast_cli(fd,"Idx: %d stack->chan: %d Chan %d InUse:%d\n",i,stack->channels[i], i+1, stack->bc_te[i].in_use);
      }
      //manager_te_get_port_info(mgr, port);
    }
  }

  return 0;
}

static int misdn_send_digit (int fd, int argc, char *argv[])
{
  char *channame; 
  char *msg; 
  
  if (argc != 5)
    return RESULT_SHOWUSAGE;
  
  channame = argv[3];
  msg = argv[4];

  ast_cli(fd, "Sending %s to %s\n",msg, channame);
  
  {
    chan_list_t *tmp=get_chan_by_ast_name(channame);
    
    if (!tmp) {
      ast_cli(fd, "Sending %s to %s failed Channel does not exist\n",msg, channame);
      return 0; 
    } else {
#if 1
      int i; 
      for (i=0; i<strlen(msg); i++) {
	ast_cli(fd, "Sending: %c\n",msg[i]);
	send_digit_to_chan(tmp, msg[i]);
	//res = ast_safe_sleep(tmp->ast, 250);
	usleep(250000);
	//res = ast_waitfor(tmp->ast,100);
      }
#else
      int res;
      res = ast_dtmf_stream(tmp->ast,NULL,msg,250);
#endif
    }
  }
  
  return 0; 
}

static int misdn_toggle_echocancel (int fd, int argc, char *argv[])
{
  char *channame; 

  if (argc != 4)
    return RESULT_SHOWUSAGE;
  
  channame = argv[3];
  
  ast_cli(fd, "Toggling EchoCancel on %s\n", channame);
  
  {
    chan_list_t *tmp=get_chan_by_ast_name(channame);
    
    if (!tmp) {
      ast_cli(fd, "Toggling EchoCancel %s failed Channel does not exist\n", channame);
      return 0; 
    } else {
      
      tmp->bc_te->ec_enable=tmp->bc_te->ec_enable?0:1;
      
    }
  }
  
  return 0; 
}



static int misdn_send_display (int fd, int argc, char *argv[])
{
  char *channame; 
  char *msg; 
  
  if (argc != 5)
    return RESULT_SHOWUSAGE;
  
  channame = argv[3];
  msg = argv[4];

  ast_cli(fd, "Sending %s to %s\n",msg, channame);
  {
    chan_list_t *tmp;
    tmp=get_chan_by_ast_name(channame);
    
    if (tmp && tmp->bc_te) {
      strcpy(tmp->bc_te->display, msg);
      manager_te_send_event(mgr_te, tmp->bc_te, EVENT_INFORMATION);
    } else {
      ast_cli(fd,"No such channel %s\n",channame);
      return RESULT_FAILURE;
    }
  }

  return RESULT_SUCCESS ;
}




static char *complete_ch_helper(char *line, char *word, int pos, int state, int rpos)
{
	struct ast_channel *c;
	int which=0;
	char *ret;
	if (pos != rpos)
		return NULL;
	c = ast_channel_walk_locked(NULL);
	while(c) {
		if (!strncasecmp(word, c->name, strlen(word))) {
			if (++which > state)
				break;
		}
		ast_mutex_unlock(&c->lock);
		c = ast_channel_walk_locked(c);
	}
	if (c) {
		ret = strdup(c->name);
		ast_mutex_unlock(&c->lock);
	} else
		ret = NULL;
	return ret;
}

static char *complete_ch(char *line, char *word, int pos, int state)
{
	return complete_ch_helper(line, word, pos, state, 3);
}

static struct ast_cli_entry cli_send_digit =
  { {"misdn","send","digit", NULL},
    misdn_send_digit,
    "Sends DTMF Digit to mISDN Channel", // Short description
    "Usage: misdn send digit <channel> \"<msg>\" \n"
    "       Send <digit> to <channel> as DTMF Tone\n"
    "       when channel is a mISDN channel\n",
    complete_ch
  };


static struct ast_cli_entry cli_toggle_echocancel =
  { {"misdn","toggle","echocancel", NULL},
    misdn_toggle_echocancel,
    "Toggles EchoCancel on mISDN Channel", // Short description
    "Usage: misdn toggle echocancel <channel>\n", 
    complete_ch
  };



static struct ast_cli_entry cli_send_display =
  { {"misdn","send","display", NULL},
    misdn_send_display,
    "Sends Text to mISDN Channel", // Short description
    "Usage: misdn send display <channel> \"<msg>\" \n"
    "       Send <msg> to <channel> as Display Message\n"
    "       when channel is a mISDN channel\n",
    complete_ch
  };


static struct ast_cli_entry cli_show_config =
  { {"misdn","show","config", NULL},
    misdn_show_config,
    "Shows internal mISDN config, read from cfg-file", // Short description
    "Usage: misdn show config\n"
  };
 

static struct ast_cli_entry cli_reload =
  { {"misdn","reload", NULL},
    misdn_reload,
    "Reloads internal mISDN config, read from cfg-file", // Short description
    "Usage: misdn reload\n"
  };

static struct ast_cli_entry cli_set_tics =
  { {"misdn","set","tics", NULL},
    misdn_set_tics,
    "", // Short description
    "\n"
  };


static struct ast_cli_entry cli_show_cls =
  { {"misdn","show","channels", NULL},
    misdn_show_cls,
    "Shows internal mISDN chan_list", // Short description
    "Usage: misdn show channels\n"
  };

static struct ast_cli_entry cli_show_cl =
  { {"misdn","show","channel", NULL},
    misdn_show_cl,
    "Shows internal mISDN chan_list", // Short description
    "Usage: misdn show channels\n",
    complete_ch
  };

static struct ast_cli_entry cli_show_fullstacks =
  { {"misdn","show","fullstacks", NULL},
    misdn_show_fullstacks,
    "Shows internal mISDN stack_list with bchannels", // Short description
    "Usage: misdn show fullstacks\n"
  };

static struct ast_cli_entry cli_flush_stack =
  { {"misdn","flush","stack", NULL},
    misdn_flush_stack,
    "Flushes the in_use flag", // Short description
    "Usage: misdn flush stack\n"
  };

static struct ast_cli_entry cli_restart_port =
  { {"misdn","restart","port", NULL},
    misdn_restart_port,
    "Restarts the given port", // Short description
    "Usage: misdn restart port\n"
  };


static struct ast_cli_entry cli_show_stacks =
  { {"misdn","show","stacks", NULL},
    misdn_show_stacks,
    "Shows internal mISDN stack_list", // Short description
    "Usage: misdn show stacks\n"
  };

static struct ast_cli_entry cli_show_port =
  { {"misdn","show","port", NULL},
    misdn_show_port,
    "Shows detailed information for given port", // Short description
    "Usage: misdn show port <port>\n"
  };



static struct ast_cli_entry cli_set_debug =
  { {"misdn","set","debug", NULL},
    misdn_set_debug,
    "Sets Debuglevel of chan_misdn, at the moment, level={1,2}", // Short description
    "Usage: misdn set debug <level>\n"
  };

static struct ast_cli_entry cli_set_crypt_debug =
  { {"misdn","set","crypt","debug", NULL},
    misdn_set_crypt_debug,
    "Sets CryptDebuglevel of chan_misdn, at the moment, level={1,2}", // Short description
    "Usage: misdn set crypt debug <level>\n"
  };
/*** CLI END ***/


/*****************************/
/*** AST Indications Start ***/
/*****************************/

static int misdn_call(struct ast_channel *ast, char *dest, int timeout)
{
  int port=1;
  char *prt; 
  char *ext;
  char dest_cp[128]="";
  char group[128]="";
  char *tokb;
  int r;
  bchannel_te_t *newbc=NULL;
  char *opts;
  
  if (!ast) {
    chan_misdn_log( "! IND : CALL Without Asterisk Channel !?!?!\n");
    if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
    ast->hangupcause=41;
    ast_setstate(ast, AST_STATE_DOWN);
    return -1;
  }
  
  if (!dest) {
    chan_misdn_log( "! IND : CALL dad:%s tech:%s WITHOUT PORT, check extension.conf\n",ext,ast->name, ast->context);
    if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
    ast->hangupcause=41;
    ast_setstate(ast, AST_STATE_DOWN);
    return -1;
  }
  
  strcpy(dest_cp, dest);
  
  if (misdn_debug>0) chan_misdn_log( "* CALL: %s\n",dest_cp);
    
  prt=strtok_r(dest_cp,"/", &tokb);
  
  if (prt) {
    if (prt[0]=='g' && prt[1]==':' ) {
      //We make a group call lets checkout which ports are in my group
      //
      prt +=2;
      strcpy(group,prt);
      if (misdn_debug >1) chan_misdn_log( " --> Group Call group: %s\n",group);
    } else {
      port=atoi(prt);
    }
    ext=strtok_r(NULL,"/", &tokb);
    if (!ext) {
      chan_misdn_log( " --> ! IND : CALL dad:%s tech:%s WITH WRONG ARGS, check extension.conf\n",ext,ast->name, ast->context);
      if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
      ast_setstate(ast, AST_STATE_DOWN);
      ast->hangupcause=41;
      return -1;
    } else {
      opts=strtok_r(NULL,"/", &tokb);
    }
  } else {
    chan_misdn_log( " --> ! IND : CALL dad:%s tech:%s WITHOUT PORT/Group, check extension.conf\n",ext,ast->name, ast->context);
    if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
      ast_setstate(ast, AST_STATE_DOWN);
      ast->hangupcause=41;
      return -1;
  }

  
  if ( strlen(group) >0 ) {
    config_list_t *config=config_list; 
    for (;config; config=config->next) {
      if (!strcasecmp(config->group, group)) {
	newbc=manager_te_get_free_bc(mgr_te, config->port );
#ifdef GOT_L1_INFO_FROM_MISDN_IN_TE
	if (newbc && newbc->stack->l1link == 1)
#else
	  if (newbc )
#endif
	    {
	      port=config->port;
	      break;
	    }
      }
    }
    if (!newbc) {
      if (misdn_debug > 0) chan_misdn_log(" --> ! No free channel chan %p even after Group Call\n",ast);
      if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
      ast->hangupcause=34;      
      ast_setstate(ast, AST_STATE_DOWN);
      return -1;
      
    }
  } else {
    newbc=manager_te_get_free_bc(mgr_te, port );
    if (!newbc) {
      if (misdn_debug > 0) chan_misdn_log(" --> ! No free channel chan %p\n",ast);
      if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
      ast->hangupcause=34;
      ast_setstate(ast, AST_STATE_DOWN);
      return -1;
      
    }
  }
  
  
  if (misdn_debug>0) chan_misdn_log( " --> * dad:%s tech:%s ctx:%s\n",ext,ast->name, ast->context);
  
  if ((ast->_state != AST_STATE_DOWN) && (ast->_state != AST_STATE_RESERVED)) {
    ast_log(LOG_WARNING, " --> ! misdn_call called on %s, neither down nor reserved\n", ast->name);
    ast->hangupcause=41;
    return -1;
  }


  {
    config_list_t *config=find_config_by_port(port);
    
    
    if(config) {
      strcpy(ast->context, config->context);
      if (misdn_debug>1) chan_misdn_log(" --> * Setting Context to %s\n",config->context);
      strcpy(ast->language, get_language(config) );

      newbc->volume=get_volume(config);
      newbc->stack->te_choose_channel=get_te_choose_channel(config);
      if (get_callerid(config)) {
	if (misdn_debug>0) chan_misdn_log(" --> * Setting Cid to %s\n", get_callerid(config));
	strcpy(newbc->oad,get_callerid(config));

	switch (newbc->numplan) {
	case NUMPLAN_INTERNATIONAL:
	case NUMPLAN_NATIONAL:
	case NUMPLAN_SUBSCRIBER:
	case NUMPLAN_UNKNOWN:
	  newbc->numplan = get_dialplan(newbc);
	  break;
	default:
	  chan_misdn_log(" --> !!!! Wrong dialplan setting, please see the misdn.conf sample file\n ");
	  break;
	}

	
      }
      
#ifndef ASTERISK_STABLE
      {
	char buf[256];
	if (misdn_debug>1) chan_misdn_log(" --> * CallGrp:%s PickupGrp:%s\n",ast_print_group(buf,sizeof(buf),config->callgroup),ast_print_group(buf,sizeof(buf),config->pickupgroup));
	ast->pickupgroup=config->pickupgroup;
	ast->callgroup=config->callgroup;
      }
#endif
      
    } 
  }
  
  
  
  if (misdn_debug > 2) chan_misdn_log(" --> * adding2newbc ext %s\n",ext);
  if (ext) strcpy(newbc->dad,ext);
  if (misdn_debug > 2) chan_misdn_log(" --> * adding2newbc callerid %s\n",AST_CID_P(ast));
  if (!strlen(newbc->oad))
    if (AST_CID_P(ast)) strcpy(newbc->oad,AST_CID_P(ast));

#ifndef ASTERISK_STABLE
    switch (ast->cid.cid_pres){
    case AST_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED:
      if (misdn_debug > 0) chan_misdn_log(" --> Number Not Screened\n");
      newbc->pres=1;
      break;
      
    case AST_PRES_ALLOWED_USER_NUMBER_PASSED_SCREEN:
      if (misdn_debug > 0) chan_misdn_log(" --> Number Screened\n");
      newbc->pres=0;
      break;
    default:
      if (misdn_debug > 0) chan_misdn_log(" --> Other Screened\n");
    }
#endif

    {
      chan_list_t *ch=malloc(sizeof(chan_list_t));
      if (!ch) { perror("malloc for chan_list"); exit(0);}
      
      memset(ch,0,sizeof(chan_list_t));
      ch->bc_te = newbc;
      ch->orginator=ORG_AST;
      ch->ast = ast;
      
      MISDN_ASTERISK_TECH_PVT(ast) = ch ;


      
      if (opts) misdn_set_opt_exec(ast,opts);
      
      if (ast_test_flag(ast, AST_FLAG_DIGITAL) || newbc->digital) {
	newbc->capability=INFO_CAPABILITY_DIGITAL_UNRESTRICTED;
	newbc->digital=1; 
	if (misdn_debug>1) chan_misdn_log(" --> * Call with flag Digital\n");
      }

      
      cl_queue_chan(&cl_te, ch) ;
      ch->state=MISDN_CALLING;
      
      r=manager_te_send_event(mgr_te, newbc, EVENT_SETUP );
      
      /** we should habe l3id after sending setup **/
      ch->l3id=newbc->l3_id;
      
    }

    //newbc->channel=1;
    //chan_misdn_log("Sending Setup Port:%d (stack->port:%d)\n",port,newbc->stack->port);
    
    
    //manager_bchannel_setup(newbc);
    
    if (check_ret(mgr_te, newbc, EVENT_SETUP, r) ) {
      chan_misdn_log(" --> * Theres no Channel at the moment .. !\n");
      if (misdn_debug>0) chan_misdn_log(" --> * SEND: State Down pid:%d\n",newbc?newbc->pid:-1);
      ast->hangupcause=34;
      ast_setstate(ast, AST_STATE_DOWN);
      return -1;
    }
    
    if (misdn_debug>0) chan_misdn_log(" --> * SEND: State Dialing pid:%d\n",newbc?newbc->pid:1);

    ast_setstate(ast, AST_STATE_DIALING);
    //ast_setstate(ast, AST_STATE_RING);
    ast->hangupcause=16;
    return 0; 
 
}


int misdn_answer(struct ast_channel *ast)
{
  chan_list_t *p;


  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
  //printf("Got Answer\n");
  if (misdn_debug>0) chan_misdn_log("* ANSWER:\n");
  
  if (!p) {
    chan_misdn_log(" --> Channel not connected ??\n");
    ast_queue_hangup(ast);
  }

  if (!p->bc_te) {
    if (misdn_debug>0) chan_misdn_log(" --> Got Answer, but theres no bc obj ??\n");

    ast_queue_hangup(ast);

  }

  {
    char *tmp_key = pbx_builtin_getvar_helper(p->ast, "CRYPT_KEY");
    
    if (tmp_key ) {
      if (misdn_debug>0) chan_misdn_log(" --> Connection will be BF crypted\n");
      strcpy(p->bc_te->crypt_key,tmp_key);
    } else {
      if (misdn_debug>2) chan_misdn_log(" --> Connection is without BF encryption\n");
    }
    
  }
  
  p->state = MISDN_CONNECTED;
  manager_te_send_event(mgr_te, p->bc_te, EVENT_CONNECT);
  start_bc_tones(p);
  
  
  return 0;
}

int misdn_digit(struct ast_channel *ast, char digit )
{
  chan_list_t *p;


  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;


  if (misdn_debug>0) chan_misdn_log("* IND : Digit %c\n",digit);

  if (!p->bc_te) {
    chan_misdn_log(" --> !! Got Digit Event withut having bchannel Object\n");
    return -1;
  }
  

  if (p->bc_te->stack->mode==TE_MODE) 
    misdn_crypt_event(p, CRYPT_EVENT_AST_DIGIT, (void*) &digit);
  
  switch (p->state ) {
  case MISDN_CALLING:
    {
      bchannel_te_t *bc=p->bc_te;
      char buf[8];
      buf[0]=digit;
      buf[1]=0;
      
      strcat(bc->infos_pending,buf);
    }
    break;
  case MISDN_CALLING_ACKNOWLEDGE:
    {
      bchannel_te_t *bc=p->bc_te;
      bc->info_dad[0]=digit;
      bc->info_dad[1]=0;
      
      strcat(bc->dad,bc->info_dad);
      strcpy(p->ast->exten, bc->dad);
      
      manager_te_send_event(mgr_te, bc, EVENT_INFORMATION);
    }
    break;

  default:
    /*if ( 1 )
      send_digit_to_chan(p,digit); */
    //chan_misdn_log("");
    break;
  }
  return 0;
}


int misdn_fixup(struct ast_channel *oldast, struct ast_channel *ast)
{
  chan_list_t *p, *p_old;

  
  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;

  if (misdn_debug>0) chan_misdn_log("* IND: Got Fixup State:%s Holded:%d L3id:%x\n", misdn_get_ch_state(p), p->holded, p->l3id);
  
  if (misdn_debug>0) chan_misdn_log(" --> OldAst State:%s Holded:%d\n", misdn_get_ch_state(p_old), p_old?p_old->holded:-1);
  
  //p_old->state=MISDN_HOLD_DISCONNECT; 
  
  p->ast = ast ;
  p->state=MISDN_CONNECTED;
  
  return 0;
}

int misdn_soption(struct ast_channel *ast, int opt, void *data, int len )
{
  chan_list_t *p;

  
  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
  if (misdn_debug>0) chan_misdn_log("* IND : Got option %d\n",opt);
  return 0;
}

int misdn_qoption(struct ast_channel *ast, int opt, void *data, int *len )
{
  chan_list_t *p;

  
  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
  if (misdn_debug>0) chan_misdn_log("* IND : Got Query option %d\n",opt);
  return 0;
}

int misdn_transfer (struct ast_channel *ast, char *dest)
{
  chan_list_t *p;

  
  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
  if (misdn_debug>0) chan_misdn_log("* IND : Got Transfer %s\n",dest);
  return 0;
}



int misdn_indication(struct ast_channel *ast, int cond)
{
  chan_list_t *p;

  
  if (!ast || ! MISDN_ASTERISK_PVT(ast)) {
    chan_misdn_log("Returnded -1 in misdn_indication\n");
    return -1;
  }
  p = MISDN_ASTERISK_TECH_PVT(ast) ;
  
  if (misdn_debug>0) chan_misdn_log("* IND : Indication from %s\n",ast->exten);
  
  switch (cond) {
  case AST_CONTROL_BUSY:
    if (misdn_debug>0) chan_misdn_log("* IND :\tbusy\n");
    
    //manager_send_tone();
    
    if (p) p->state=MISDN_BUSY;
    
    if (p && p->bc_te) {
      //manager_te_send_event(mgr_te, p->bc_te, EVENT_ALERTING);
      if (misdn_debug>0) chan_misdn_log(" --> * SEND: State Busy pid:%d\n",p->bc_te?p->bc_te->pid:-1);
      ast_setstate(ast,AST_STATE_BUSY);

      p->bc_te->cause=17;
      if (p->state != MISDN_CONNECTED) {
	manager_te_send_event(mgr_te, p->bc_te, EVENT_DISCONNECT);
	manager_send_tone(mgr_te, p->bc_te, TONE_BUSY);
      } else {
	chan_misdn_log(" --> !! Got Busy in Connected State !?! port:%d ast:%s\n",
		       p->bc_te->stack->port, ast->name);
      }
      
    }
    break;
  case AST_CONTROL_RING:
    if (p && p->bc_te) {
      if (misdn_debug>0)    chan_misdn_log(" --> * IND :\tring pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    }
    break;
  case AST_CONTROL_RINGING:

    if (p && p->state == MISDN_ALERTING) {
      if (misdn_debug>0) chan_misdn_log(" --> * IND :\tringing pid:%d but I ws Ringing before, so ignoreing it\n",p->bc_te?p->bc_te->pid:-1);
      break;
    }
    if (p) p->state=MISDN_ALERTING;
    
    if (p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> * IND :\tringing pid:%d\n",p->bc_te?p->bc_te->pid:-1);
      
      manager_te_send_event(mgr_te, p->bc_te, EVENT_ALERTING);
      manager_send_tone(mgr_te, p->bc_te, TONE_ALERTING);
      if (misdn_debug > 0 ) chan_misdn_log(" --> * SEND: State Ring pid:%d\n",p->bc_te?p->bc_te->pid:-1);
      ast_setstate(ast,AST_STATE_RINGING);
    }
    //ast_playtones_start(ast, 0, "1111", 0);
    //tone_zone_play_tone(ast, ZT_TONE_RINGTONE);
    break;
  case AST_CONTROL_ANSWER:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> * IND :\tanswer pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    }
    break;
  case AST_CONTROL_TAKEOFFHOOK:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> *\ttakeoffhook pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    }
    break;
  case AST_CONTROL_OFFHOOK:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> *\toffhook pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    }
    break; 
  case AST_CONTROL_FLASH:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> *\tflash pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    }
    break;
  case AST_CONTROL_PROGRESS:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> * IND :\tprogress pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    }
    if (p->bc_te && p->bc_te->stack->mode == NT_MODE) {
      //manager_te_send_event(mgr_te, p->bc_te, EVENT_PROCEEDING);
      //start_bc_tones(p->bc_te);
      //ast_setstate(ast,AST_STATE_RINGING);
    }
    
    break;
  case AST_CONTROL_CONGESTION:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> * IND :\tcongestion pid:%d\n",p->bc_te?p->bc_te->pid:-1);

      p->bc_te->cause=42;
      if (p->state != MISDN_CONNECTED) {
	manager_te_send_event(mgr_te, p->bc_te, EVENT_RELEASE_COMPLETE);
	//manager_send_tone(mgr_te, p->bc_te, TONE_BUSY);
      } else {
	chan_misdn_log(" --> !! Got Congestion in Connected State !?! port:%d ast:%s\n",
		       p->bc_te->stack->port, ast->name);
      }
      
    }
    break;
  case -1 :
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> * IND :\t-1! pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    } else {
      if (misdn_debug>0) chan_misdn_log(" --> * IND :\t-1!\n");
    }
    //misdn_answer(ast);
    break;
#ifndef ASTERISK_STABLE
  case AST_CONTROL_HOLD:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> *\tHOLD pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    }
    break;
  case AST_CONTROL_UNHOLD:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log(" --> *\tUNHOLD pid:%d\n",p->bc_te?p->bc_te->pid:-1);
    }
    break;
#endif
  default:
    if (p && p->bc_te) {
      chan_misdn_log(" --> * Unknown Indication:%d pid:%d\n",cond,p->bc_te?p->bc_te->pid:-1);
    } else {
      chan_misdn_log(" --> * Unknown Indication :%d\n",cond);
    }
  }
  
  return 0;
}

/*static void misdn_destroy(chan_list_t *ch)
  {
  }*/


int misdn_hangup(struct ast_channel *ast)
{
  chan_list_t *p;
  bchannel_te_t *bc_te=NULL;

  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;
  
  
  ast_log(LOG_DEBUG, "misdn_hangup(%s)\n", ast->name);
  
  if (!p) {
    if (misdn_debug>2) chan_misdn_log("misdn_hangup called, without chan_list obj.\n");
    return 0 ;
  }
  
  bc_te=p->bc_te;
  
  if (bc_te) {
    char *varcause=NULL;
    bc_te->cause=ast->hangupcause?ast->hangupcause:16;
    
    if ( (varcause=pbx_builtin_getvar_helper(ast, "HANGUPCAUSE")) ||
	 (varcause=pbx_builtin_getvar_helper(ast, "PRI_CAUSE"))) {
      int tmpcause=atoi(varcause);
      bc_te->cause=tmpcause?tmpcause:16;
    }
    
    if (misdn_debug > 0) chan_misdn_log("* IND : HANGUP\tpid:%d ctx:%s dad:%s oad:%s State:%s\n",p->bc_te?p->bc_te->pid:-1, ast->context, ast->exten, AST_CID_P(ast), misdn_get_ch_state(p));
    if (misdn_debug > 1) chan_misdn_log(" --> l3id:%x\n",p->l3id);
    if (misdn_debug > 0) chan_misdn_log(" --> cause:%d\n",bc_te->cause);
    
    //manager_send_tone(mgr_te, bc_te, TONE_BUSY);
    switch (p->state) {
    case MISDN_CALLING:
    case MISDN_HOLDED:
    case MISDN_DIALING:
      manager_send_tone(mgr_te, bc_te, TONE_BUSY);
      p->state=MISDN_CLEANING;

      //bc_te->cause=16;

      
      
      manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
      //if (bc_te->stack->mode == NT_MODE)
      //manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
      //else
      //manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
      //manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      
      break;
      
    case MISDN_ALERTING:
      if (misdn_debug > 1) chan_misdn_log(" --> * State Alerting\n");
      //bc_te->cause=16;
      if (p->orginator != ORG_AST) 
	manager_send_tone(mgr_te, bc_te, TONE_BUSY);
      
      p->state=MISDN_CLEANING;
      manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      //manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
      break;
    case MISDN_CONNECTED:
      // Alerting or Disconect
      if (misdn_debug > 1) chan_misdn_log(" --> * State Connected\n");
      
      //bc_te->cause=16;
      
      
      manager_send_tone(mgr_te, bc_te, TONE_BUSY);
      //stop_bc_tones(p);
      /*if (bc_te->stack->mode == NT_MODE) {
	manager_send_tone(mgr_te, bc_te, TONE_BUSY);
	}*/
      
      manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);

      
      p->state=MISDN_CLEANING; //MISDN_HUNGUP_FROM_AST;
      break;

    case MISDN_CLEANING:
      break;
      
    case MISDN_HOLD_DISCONNECT:
      //need to send release here
      if (misdn_debug > 1) chan_misdn_log(" --> state HOLD_DISC\n");
      stop_bc_tones(p);
      
      if (misdn_debug > 0) chan_misdn_log(" --> cause %d\n",bc_te->cause);
      //bc_te->cause=16;
      manager_te_send_event(mgr_te,bc_te,EVENT_RELEASE);
      //manager_te_send_event(mgr_te,bc_te,EVENT_RELEASE_COMPLETE);
      //manager_te_send_event(mgr_te,bc_te,EVENT_DISCONNECT);
      break;
    default:
      // Alerting or Disconect
      
      stop_bc_tones(p);
      //bc_te->cause=16;
      if (bc_te->stack->mode == NT_MODE)
	manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE);
      else
	manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      p->state=MISDN_CLEANING; //MISDN_HUNGUP_FROM_AST;
    }
    
    MISDN_ASTERISK_TECH_PVT(p->ast)=NULL;
    p->ast=NULL;
  }
  
  return 0;
}

struct ast_frame  *misdn_read(struct ast_channel *ast)
{
  static int i = 0 ;
  chan_list_t *p;

  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return NULL;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
  if (i==0) {
    printf("read caled\n");
    i=1;
  }
  
  return NULL;
}

int misdn_write(struct ast_channel *ast, struct ast_frame *frame)
{
  chan_list_t *p;
  int i  = 0;

  if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
  p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
  if (p->bc_te && p->bc_te->tone != TONE_NONE)
    manager_send_tone(mgr_te,p->bc_te,TONE_NONE);


  if (p->holded ) {
    if (misdn_debug>4)chan_misdn_log("misdn_write: Returning because holded\n");
    return 0;
  }
  
  if (p->notxtone) {
    if (misdn_debug>4)chan_misdn_log("misdn_write: Returning because notxone\n");
    return 0;
  }
  
  if ( frame->subclass != prefformat) {
    chan_misdn_log("Got Frame with Format:%d\n", frame->subclass);
  }

  
  //printf("tx2misdn\n");
  //if (ast->_state != AST_STATE_UP) return 0;
  //if (misdn_debug > 2) printf("Txing: Pid:%d\n",p->bc_te->pid);
#if MISDN_DEBUG
  {
    int i, max=5>frame->samples?frame->samples:5;
    
    printf("write2mISDN %p %d bytes: ", p, frame->samples);
    
    for (i=0; i<  max ; i++) printf("%2.2x ",((char*) frame->data)[i]);
    printf ("\n");
  }
#endif
  if (p->bc_te) {
    if ( !p->bc_te->digital ) 
      manager_flip_buf_bits(frame->data, frame->samples);
    
    /*if (misdn_debug > 1 && p->bc_te->digital)
      chan_misdn_log("tx2misdn [%s]\n",ast->exten);*/
    
    i= manager_send_frame(p->bc_te, frame->data, frame->samples);
  } else {
    chan_misdn_log ("NO bc_te !!\n");
  }

  return 0;
}

int misdn_bridge (struct ast_channel *c0, struct ast_channel *c1, int flags,
		  struct ast_frame **fo, struct ast_channel **rc)
{
  chan_list_t *ch1,*ch2;
  
  return 1;
  
  ch1=get_chan_by_ast(c0);
  ch2=get_chan_by_ast(c1);
  
  if (ch1 && ch2 ) ;
  else
    return -1;
  
  return 1;
}

/** AST INDICATIONS END **/

static int start_bc_tones(chan_list_t* cl)
{
  manager_bchannel_activate(cl->bc_te);
  manager_send_tone(mgr_te, cl->bc_te ,TONE_NONE);
  cl->notxtone=0;
  cl->norxtone=0;
  return 0;
}

static int stop_bc_tones(chan_list_t *cl)
{
  if (cl->bc_te) {
    //manager_send_tone(mgr_te, bc, TONE_NONE);
    manager_bchannel_deactivate(cl->bc_te);
  }
  cl->notxtone=1;
  cl->norxtone=1;
  
  return 0;
}

static int chan_avail(char *port)
{
  int prt=-1;
  char group[128]="";
  
  if (port) {
    if (port[0]=='g' && port[1]==':' ) {
      //We make a group call lets checkout which ports are in my group
      //
      port +=2;
      strcpy(group,port);
      if (misdn_debug >0) chan_misdn_log( "Checking Availbl. Chan in Group: %s\n",group);
    } else {
      prt=atoi(port);
    }


    
    if ( strlen(group) >0 ) {
      config_list_t *config=config_list; 
      for (;config; config=config->next) {
	if (!strcasecmp(config->group, group)) {
	  if (manager_te_free_bc_avail(mgr_te, config->port )) return 1;
	}
      }
      
    } else {
      if (manager_te_free_bc_avail(mgr_te, prt )) return 1;
    }
    
  }
  
  return 0;
}


static struct ast_channel *misdn_request(const char *type, int format, void *data, int *cause)
{
  //chan_list_t *p;
  struct ast_channel *tmp = NULL;
  char buf[128];
  char buf2[128], *oad, *port;
  char *tokb;
  
  sprintf(buf,"%s/%s",type,(char*)data);
  strcpy(buf2,data);
  port=strtok_r(buf2,"/", &tokb);
  if (port)
    oad=strtok_r(NULL,"/", &tokb);
  dummy_cl.bc_te=NULL;
  //tmp = misdn_new( &dummy_cl, AST_STATE_DOWN, buf, "","",oad);
  
  if (!chan_avail(port)) {
    chan_misdn_log("Sorry on group/port %s is no channel free\n", port);
    return NULL;
  }
  
  tmp = misdn_new( &dummy_cl, AST_STATE_RESERVED, buf, "","",oad);
  
  //chan_misdn_log("type:%s oad:%s data:%s\n", type, oad, data);
  /*if (!tmp)
    misdn_destroy(p);*/
  
  return tmp;
}


#ifndef ASTERISK_STABLE
struct ast_channel_tech misdn_tech = {
  .type="mISDN",
  .description="Channel driver for mISDN Support (Bri/Pri)",
  .capabilities=AST_FORMAT_ALAW,
  .requester=misdn_request,
  .send_digit=misdn_digit,
  .call=misdn_call,
  .hangup=misdn_hangup,
  .answer=misdn_answer,
  .read=misdn_read,
  .write=misdn_write,
  .indicate=misdn_indication,
  .fixup=misdn_fixup,
  .transfer=misdn_transfer
};
#endif


struct ast_channel *misdn_new(chan_list_t *chlist, int state, char * name, char * context, char *exten, char *callerid)
{
  struct ast_channel *tmp;
  
  tmp = ast_channel_alloc(1);
  
  if (tmp) {
    if (misdn_debug>1)chan_misdn_log(" --> * NEW CHANNEL dad:%s oad:%s ctx:%s\n",exten,callerid, context);
    
    snprintf(tmp->name, sizeof(tmp->name), name);
    tmp->type = type;
    
    tmp->nativeformats = prefformat;
#ifdef ASTERISK_STABLE
    tmp->pvt->pvt = chlist;
    tmp->pvt->rawreadformat = prefformat;
    tmp->pvt->rawwriteformat = prefformat;
    
    tmp->pvt->call = misdn_call;
    tmp->pvt->hangup = misdn_hangup;
    tmp->pvt->read = misdn_read;
    tmp->pvt->write = misdn_write;
    tmp->pvt->answer = misdn_answer;
    tmp->pvt->indicate = misdn_indication; 
    //tmp->pvt->bridge = misdn_bridge;
    tmp->pvt->fixup = misdn_fixup;
    tmp->pvt->setoption = misdn_soption;
    tmp->pvt->queryoption = misdn_qoption;
    tmp->pvt->transfer = misdn_transfer;
    tmp->pvt->send_digit = misdn_digit; 
#else
    tmp->tech_pvt = chlist;
    tmp->tech = &misdn_tech;
#endif
    
    tmp->writeformat = prefformat;
    tmp->readformat = prefformat;
    tmp->priority=1;
    
    
    strncpy(tmp->context, context, sizeof(tmp->context)-1);
    strncpy(tmp->exten, exten,  sizeof(tmp->exten) - 1);

    if (callerid) {
#ifndef ASTERISK_STABLE
      char *cid_name, *cid_num;
      
      ast_callerid_parse(callerid, &cid_name, &cid_num);
      if (cid_name)
	tmp->cid.cid_name=strdup(cid_name);
      if (cid_num)
	tmp->cid.cid_num=strdup(cid_num);
#else
      AST_CID_P(tmp)=strdup(callerid);
#endif
    }
    
    strcpy(tmp->language, default_language);
    //i->owner = tmp;

    if (chlist->bc_te) {
#ifndef ASTERISK_STABLE
      config_list_t * config = find_config_by_port(chlist->bc_te->stack->port);

      if (config) {
	if (misdn_debug>2) chan_misdn_log(" --> Setting Pickupgroup\n");
	tmp->callgroup=config->callgroup;
	tmp->pickupgroup=config->pickupgroup;
      } else {
	chan_misdn_log(" --> !! Config Not found in misdn_new port:%d\n",chlist->bc_te->stack->port);
      }
#endif
    } else {
      if (misdn_debug>2) chan_misdn_log(" --> Not Setting Pickupgroup, we have no bc yet\n");
    }
    
    /*ast_mutex_lock(&usecnt_lock);
      usecnt++;
      ast_mutex_unlock(&usecnt_lock);
      ast_update_use_count(); */

    ast_setstate(tmp, state);
    if (state == AST_STATE_RING)
      tmp->rings = 1;
    else
      tmp->rings = 0;
    
    
  } else
    ast_log(LOG_WARNING, "Unable to allocate channel structure\n");

  
  
  return tmp;
}



int chan_misdn_write_frm(chan_list_t * tmp, char * buf,  int len )
{
#ifdef MISDN_CHECK_DTMFS
  struct ast_frame frame, *f=NULL;
#else
  struct ast_frame frame;
#endif
  

  //If in hold state we drop frame ..
  if (tmp->holded || tmp->state == MISDN_CLEANING ) return 0; 
  //if (tmp->state == MISDN_WAITING4DIGS ) return 0;
  //if (tmp->bc_te && tmp->bc_te->tone != TONE_NONE) return 0 ;
  
  //return 0;
  //printf("tx2*\n");
  /*if (misdn_debug > 1 && tmp->bc_te->digital)
    chan_misdn_log("tx2* [%s]\n", tmp->bc_te->dad);*/
  
  if (tmp->norxtone) {
    if (misdn_debug>2)
      chan_misdn_log("chan_misdn_write_frm: Returning because norxtone\n");
    return 0;
  }
  
  frame.frametype  = AST_FRAME_VOICE;
  frame.subclass = prefformat;
  frame.datalen = len;
  frame.samples = len ;
  frame.mallocd =0 ;
  frame.offset= 0 ;
  frame.src = NULL;
  frame.data = buf ;
  
  
#ifdef MISDN_CHECK_DTMFS
  if (tmp->trans) 
    f=ast_translate(tmp->trans, &frame,0);
  else 
    chan_misdn_log("No T-Path found\n");

  //chan_misdn_log("bc->notdmf:%d\n",tmp->bc_te->nodtmf );
  if (tmp->dsp && f && !tmp->bc_te->nodtmf ) {
    char c=ast_dsp_digitdetect(tmp->dsp, f);
    if (c) {
      int digit = 0;
      char buf[8];
      digit=ast_dsp_getdigits(tmp->dsp, buf, 1);
      
      if(digit) {
	// sending INFOS as DTMF-Frames :)
	struct ast_frame fr;
	fr.frametype = AST_FRAME_DTMF;
	fr.subclass = buf[0] ;
	fr.src=NULL;
	fr.data = NULL ;
	fr.datalen = 0;
	fr.samples = 0 ;
	fr.mallocd =0 ;
	fr.offset= 0 ;

	if (tmp->state != MISDN_CONNECTED) {
	  char buf[8];
	  buf[0]=digit;
	  buf[1]=0;
	  
	  if (append_digits2exten) {
	    strcat(tmp->bc_te->dad,buf);
	    strcpy(tmp->ast->exten, tmp->bc_te->dad);
	  }
	}
    
	
	if (tmp->ast && MISDN_ASTERISK_PVT(tmp->ast) && MISDN_ASTERISK_TECH_PVT(tmp->ast)) { 
	  if ( misdn_debug > 1) chan_misdn_log("* SEND: DTMF :%c\n",buf[0]);

	  ast_queue_frame(tmp->ast, &fr);

	}
      }
      //return 0;
    }
    ast_frfree (f);
  } else {
    chan_misdn_log(" Frame Not Transed\n");
  }

#endif

  
  if (tmp && tmp->ast && MISDN_ASTERISK_PVT (tmp->ast) && MISDN_ASTERISK_TECH_PVT(tmp->ast) ) {
#if MISDN_DEBUG
    int i, max=5>len?len:5;
    
    printf("write2* %p %d bytes: ",tmp, len);
    
    for (i=0; i<  max ; i++) printf("%2.2x ",((char*) frame.data)[i]);
    printf ("\n");
#endif
    //if (tmp->ast->_state == AST_STATE_UP)
    //ast_mutex_lock(&tmp->ast->lock);
    ast_queue_frame(tmp->ast,&frame);
    //ast_mutex_unlock(&tmp->ast->lock);
    
  }  else {
    chan_misdn_log ("No ast || ast->pvt || ch\n");
  }

  return 0;
}

/** Channel Queue ***/

chan_list_t *cl_find_chan_by_l3id(chan_list_t *list, unsigned long l3id)
{
  chan_list_t *help=list;
  for (;help; help=help->next) {
    //if (help->bc_te == bc && !help->holded) return help;
    if (help->l3id == l3id) return help;
  }
  
  if (misdn_debug>3) chan_misdn_log("$$$ find_chan: No channel found with l3id:%x\n",l3id);
  
  return NULL;
}


chan_list_t *cl_find_chan_by_bc_te(chan_list_t *list, bchannel_te_t *bc)
{
  chan_list_t *help=list;
  for (;help; help=help->next) {
    //if (help->bc_te == bc && !help->holded) return help;
    if (help->bc_te == bc) return help;
  }
  
  if (misdn_debug>3) chan_misdn_log("$$$ find_chan: No channel found for oad:%s dad:%s\n",bc->oad,bc->dad);
  
  return NULL;
}

chan_list_t *cl_find_chan_by_bc_te_holded(chan_list_t *list, bchannel_te_t *bc)
{
  chan_list_t *help=list;
  for (;help; help=help->next)
    if (help->holded_bc == bc && help->state == MISDN_HOLDED) return help;
  
  if (misdn_debug>3) chan_misdn_log("$$$ find_chan_holded: No channel found for oad:%s dad:%s\n",
				    bc->oad,bc->dad);

  return NULL;
}

void cl_queue_chan(chan_list_t **list, chan_list_t *chan)
{
  if (misdn_debug>3) chan_misdn_log("* Queuing chan %p\n",chan);

#ifdef MISDN_CHECK_DTMFS
  chan->dsp = ast_dsp_new();
  if (chan->dsp) ast_dsp_set_features(chan->dsp, DSP_FEATURE_DTMF_DETECT);
  //if (chan->dsp) ast_dsp_set_features(chan->dsp, DSP_FEATURE_BUSY_DETECT);
  chan->trans=ast_translator_build_path(AST_FORMAT_SLINEAR, prefformat);
#endif
  
  pthread_mutex_lock(&cl_te_lock);
  if (!*list) {
    //if (misdn_debug>3) chan_misdn_log("* List is empty so new chan is Listroot\n");
    *list = chan;
  } else {
    chan_list_t *help=*list;
    for (;help->next; help=help->next); 
    help->next=chan;
    //if (misdn_debug> 3) chan_misdn_log("* just adding Chan to the End of List\n");
  }
  chan->next=NULL;
  pthread_mutex_unlock(&cl_te_lock);
}

void cl_dequeue_chan(chan_list_t **list, chan_list_t *chan) 
{
  
  //if (misdn_debug>3)chan_misdn_log("* Dequeuing chan %p from List %p\n",chan, list);
  
#ifdef MISDN_CHECK_DTMFS
  if (chan->dsp) 
    ast_dsp_free(chan->dsp);
  if (chan->trans)
    ast_translator_free_path(chan->trans);
#endif


  pthread_mutex_lock(&cl_te_lock);
  if (!*list) {
    //if (misdn_debug >2) chan_misdn_log("List is empty ?\n");
    pthread_mutex_unlock(&cl_te_lock);
    return;
  }
  
  if (*list == chan) {
    //if (misdn_debug>2) chan_misdn_log("* Its the first Chan\n");
    *list=(*list)->next;
    pthread_mutex_unlock(&cl_te_lock);
    return ;
  }
  
  {
    chan_list_t *help=*list;
    for (;help->next; help=help->next) {
      if (help->next == chan) {
	//if (misdn_debug>2)chan_misdn_log("* Its not the first Chan\n");
	help->next=help->next->next;
	pthread_mutex_unlock(&cl_te_lock);
	return;
      }
    }
  }
  //chan_misdn_log("Nothin dequed from chan list\n");
  pthread_mutex_unlock(&cl_te_lock);
}

/** Channel Queue End **/


/** Isdn asks us to release channel, pendant to misdn_hangup **/
static void release_chan(bchannel_te_t *bc) {
  chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
  if (!ch) ch=cl_find_chan_by_l3id(cl_te, bc->l3_id);
  
  if (ch) {
    
    if (ch->ast && MISDN_ASTERISK_PVT(ch->ast)) {
      if (misdn_debug>0) chan_misdn_log("* RELEASING CHANNEL pid:%d ctx:%s dad:%s oad:%s state: %s\n",bc?bc->pid:-1, ch->ast->context, ch->ast->exten,AST_CID_P(ch->ast),misdn_get_ch_state(ch));
      //ast_setstate(ch->ast, AST_STATE_BUSY);
      if (misdn_debug>2) chan_misdn_log(" --> * State Down\n");


      MISDN_ASTERISK_TECH_PVT(ch->ast)=NULL;
      
      
      if (ch->ast->_state != AST_STATE_RESERVED) {
	if (misdn_debug> 2) chan_misdn_log(" --> Setting AST State to down\n");
	ast_setstate(ch->ast, AST_STATE_DOWN);
      }
      
      switch(ch->state) {
      case MISDN_EXTCANTMATCH:
      case MISDN_WAITING4DIGS:
	{
	  if (misdn_debug > 2 ) chan_misdn_log(" --> * State Wait4dig | ExtCantMatch\n");
	  //ast_queue_hangup(ch->ast);
	  ast_hangup(ch->ast);
	  
	}
	break;
	
      case MISDN_CALLING:
      case MISDN_CALLING_ACKNOWLEDGE:
      case MISDN_DIALING:
	if (misdn_debug > 1 ) chan_misdn_log("* --> In State Calling|Dialing\n");
	if (misdn_debug > 1 ) chan_misdn_log("* --> Queue Hangup\n");
	

	ast_queue_control(ch->ast, AST_CONTROL_HANGUP);
	break;
      case MISDN_CLEANING:
	// this state comes out of ast so we mustnt call a ast function !
	if (misdn_debug > 1 ) chan_misdn_log("* --> In StateCleaning\n");
	//ast_hangup(ch->ast);
	
	break;
      case MISDN_HOLD_DISCONNECT:
	if (misdn_debug > 1 ) chan_misdn_log("* --> In HOLD_DISC\n");
	break;
      default:
	if (misdn_debug > 1 ) chan_misdn_log("* --> In State Default\n");
	if (misdn_debug > 1 ) chan_misdn_log("* --> Queue Hangup\n");
	
	
	if (ch->ast && MISDN_ASTERISK_PVT(ch->ast)) {
	  ast_queue_hangup(ch->ast);
	} else {
	  chan_misdn_log ("!! Not really queued!\n");
	}
      }
    }
    cl_dequeue_chan(&cl_te, ch);
    free(ch);
  } else {
    // chan is already cleaned, so exiting 
  }
  
}
/*** release end **/

void misdn_transfer_bc(chan_list_t *tmp_ch, chan_list_t *holded_chan)
{
  tmp_ch->state=MISDN_HOLD_DISCONNECT;
  
  ast_moh_stop(AST_BRIDGED_P(holded_chan->ast));

  holded_chan->state=MISDN_CONNECTED;
  holded_chan->holded=0;
  manager_te_transfer(mgr_te, holded_chan->bc_te?holded_chan->bc_te:holded_chan->holded_bc);
  
  ast_channel_masquerade(holded_chan->ast, AST_BRIDGED_P(tmp_ch->ast));
}


int is_msn_valid(bchannel_te_t *bc)
{
  msn_list_t *msn=msn_list;
  
  config_list_t *config; 

  //if (misdn_debug > 2) chan_misdn_log("Locking Config Mutex\n");
  
  pthread_mutex_lock(&config_mutex);
  config = find_config_by_port(bc->stack->port);
  //if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
  pthread_mutex_unlock(&config_mutex);
  
  if (!config) return 0;
  
  //if (misdn_debug > 2) chan_misdn_log("Locking Config Mutex\n");
  pthread_mutex_lock(&config_mutex);
  for(msn=config->msn_list;
      msn;
      msn=msn->next) {
    
    if (!bc->dad || ! msn->msn) continue;
    
    if (!strcmp(bc->dad,msn->msn)) {
      //if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
      pthread_mutex_unlock(&config_mutex);
      return 1;
    }
    if (msn->msn[0]=='*') {
      //if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
      pthread_mutex_unlock(&config_mutex);
      return 1;
    }
  }

  pthread_mutex_unlock(&config_mutex);
  //if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
  
  if (!strcmp(bc->dad,default_msns)) 
    return 1;
  
  if (default_msns[0]=='*') 
    return 1;
  
  return 0;
}

/** State Machine for sending ISDN Messages to mISDN **/

int check_ret_te (manager_te_t *mgr,bchannel_te_t * bc, enum te_event_e event, int ret)
{
  switch (event) {
  case EVENT_SETUP:
    switch(ret) {
    case -ENOCHAN:
      return 1;
    default:
      break;
    }
    break;
  default:
    break;
  }
  return 0;
}

int check_ret (manager_te_t *mgr,bchannel_te_t * bc, enum te_event_e event, int ret)
{
if (bc->stack->mode == TE_MODE) return check_ret_te(mgr, bc, event, ret);
  
 switch (event) {
 case EVENT_SETUP_ACKNOWLEDGE:

   switch(ret) {
   case -ENOCHAN:
     manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE);
     return 1;
     break;
   }
   break;
 case EVENT_SETUP:
   switch(ret) {
   case -ENOCHAN:
     return 1;
     break;
   }
   break;
 default:
   break;
 }
 
 return 0;
}

/** Send mISDN State Machine End **/

void do_immediate_setup(bchannel_te_t *bc,chan_list_t *ch , struct ast_channel *ast)
{
  char predial[256]="";
  char *p = predial;
  
  struct ast_frame fr;
  
  strncpy(predial, ast->exten, sizeof(predial) -1 );
  
  ch->state=MISDN_DIALING;

  //manager_bchannel_setup(bc);
  
  if (bc->stack->mode == NT_MODE) {
    int ret; 
    ret = manager_te_send_event(mgr_te, bc, EVENT_SETUP_ACKNOWLEDGE );
  } else {
    int ret;
    ret = manager_te_send_event(mgr_te, bc, EVENT_SETUP_ACKNOWLEDGE );
  }

  manager_send_tone(mgr_te, bc,TONE_DIAL);  
  
  if (misdn_debug >0)
    chan_misdn_log("* Starting Ast ctx:%s dad:%s oad:%s with 's' extension\n", ast->context, ast->exten, AST_CID_P(ast));
  
  strcpy(ast->exten,"s");
  
  if (ast_pbx_start(ast)<0) {
    ast=NULL;
    manager_send_tone(mgr_te,bc,TONE_BUSY);
    if (bc->stack->mode == NT_MODE)
      manager_te_send_event(mgr_te, bc, EVENT_RELEASE_COMPLETE );
    else
      manager_te_send_event(mgr_te, bc, EVENT_DISCONNECT );
  }
  
  
  while (strlen(p) ) {
    fr.frametype = AST_FRAME_DTMF;
    fr.subclass = *p ;
    fr.src=NULL;
    fr.data = NULL ;
    fr.datalen = 0;
    fr.samples = 0 ;
    fr.mallocd =0 ;
    fr.offset= 0 ;

    if (ch->ast && MISDN_ASTERISK_PVT(ch->ast) && MISDN_ASTERISK_TECH_PVT(ch->ast)) {
      ast_queue_frame(ch->ast, &fr);
    }
    p++;
  }
}

enum te_event_response_e
cb_te_events(manager_te_t *mgr, enum te_event_e event, bchannel_te_t *bc, void *user_data)
{
  
  if (event != EVENT_BCHAN_DATA) { // Debug Only Non-Bchan
    if (misdn_debug >0) chan_misdn_log("I IND :%s\tpid:%d\tmode:%s\taddr:%x\tport:%d\n",manager_isdn_get_info( event),bc?bc->pid:-1, bc->stack->mode==NT_MODE?"NT":"TE", bc->addr,bc->stack->port);
    
    if (misdn_debug >0) chan_misdn_log(" --> l3id:%x cause:%d dad:%s oad:%s channel:%d port:%d\n",bc->l3_id, bc->cause, bc->dad,bc->oad,bc->channel, bc->stack->port);
  }
  
  switch (event) {
  case EVENT_NEW_L3ID:
#if 1
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch) {
	ch->l3id=bc->l3_id;
      } else {
	chan_misdn_log("Chan not existing at the moment\n");
      }
    }
#endif 
    break;
    
  case EVENT_DTMF_TONE:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      // sending INFOS as DTMF-Frames :)
      struct ast_frame fr;
      memset(&fr, 0 , sizeof(fr));
      fr.frametype = AST_FRAME_DTMF;
      fr.subclass = bc->dtmf ;
      fr.src=NULL;
      fr.data = NULL ;
      fr.datalen = 0;
      fr.samples = 0 ;
      fr.mallocd =0 ;
      fr.offset= 0 ;

      if (!ch) {
	chan_misdn_log("!! Got DTMF Tone without channel on port:%d\n",bc->stack->port);
	return -1;
      }

      if (ch->ast && MISDN_ASTERISK_PVT(ch->ast) && MISDN_ASTERISK_TECH_PVT(ch->ast)) { 
	if ( misdn_debug > 1) chan_misdn_log(" --> * SEND: DTMF :%c\n",bc->dtmf);
	
#if 0
	//just testing .. 
	if (bc->dtmf == '*' )
	   ch->norxtone=1;
	 if (bc->dtmf == '1' )
	   ch->norxtone=0;
#endif
	ast_queue_frame(ch->ast, &fr);
	
      }

      if (misdn_dynamic_crypting)
	misdn_crypt_event(ch, CRYPT_EVENT_MISDN_DIGIT, (void*) & bc->dtmf);
    }
    
    break;
  case EVENT_STATUS:
    if (misdn_debug>0) chan_misdn_log(" --> cause:%d\n",bc->cause);
    break;
    
  case EVENT_INFORMATION:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);

      if (ch) {
	config_list_t *config = find_config_by_port(bc->stack->port);
	if ( config) {
	  if ( stop_tone_after_first_digit && bc->tone != TONE_NONE) {
	    //if (misdn_debug > 2) chan_misdn_log(" --> Stoping Tones after first digit\n");
	    manager_send_tone(mgr,bc,TONE_NONE);
	  }
	  
	}
      }
	
      if (misdn_debug >1) chan_misdn_log(" --> keypad: %s info_dad: %s\n",bc->keypad,bc->info_dad);
      if (ch->state == MISDN_WAITING4DIGS ) {
	// Ok, incomplete Setup, waiting till extension exists
	
	strcat(bc->dad,bc->info_dad);
	if (!ch->ast) { chan_misdn_log("!! ERROR: Infos without ast-obj??\n");return RESPONSE_ERR;}
	strcpy(ch->ast->exten, bc->dad);


	//chan_misdn_log("Can Match Extension: dad:%s oad:%s\n",bc->dad,bc->oad);
	
	if(!ast_canmatch_extension(ch->ast, get_context(bc), bc->dad, 1, bc->oad)) {
	  chan_misdn_log("Extension can never match, so disconnecting\n");
	  manager_send_tone(mgr,bc,TONE_BUSY);
	  ch->state=MISDN_EXTCANTMATCH;
	  if (bc->stack->mode == NT_MODE)
	    manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	  else
	    manager_te_send_event(mgr, bc, EVENT_DISCONNECT );
	  break;
	}
	
	if (ast_exists_extension(ch->ast, get_context(bc), bc->dad, 1, bc->oad)) {
	  ch->state=MISDN_DIALING;
	  
	  //manager_send_tone(mgr, bc,TONE_NONE);
	  //stop_bc_tones(ch);
	  manager_send_tone(mgr,bc,TONE_NONE);
	  if (misdn_debug >0) chan_misdn_log(" --> * Starting Ast ctx:%s dad:%s oad:%s\n", ch->ast->context, ch->ast->exten, AST_CID_P(ch->ast));
	  if (ast_pbx_start(ch->ast)<0) {
	    chan_misdn_log("ast_pbx_start returned <0 in INFO\n");
	    manager_send_tone(mgr,bc,TONE_BUSY);
	    if (bc->stack->mode == NT_MODE)
	      manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	    else
	      manager_te_send_event(mgr, bc, EVENT_DISCONNECT );
	  }
	}
	
      } else {
	// sending INFOS as DTMF-Frames :)
	struct ast_frame fr;
	fr.frametype = AST_FRAME_DTMF;
	fr.subclass = bc->info_dad[0] ;
	fr.src=NULL;
	fr.data = NULL ;
	fr.datalen = 0;
	fr.samples = 0 ;
	fr.mallocd =0 ;
	fr.offset= 0 ;

	if (ch->ast && MISDN_ASTERISK_PVT(ch->ast)) {
	  
	  if (ch->state != MISDN_CONNECTED && append_digits2exten) {
	    //if (misdn_debug>1) chan_misdn_log(" --> Appending Digits %s\n",bc->info_dad);
	    strcat(bc->dad,bc->info_dad);
	    strcpy(ch->ast->exten, bc->dad);
	    ast_cdr_update(ch->ast);
	  } else {
	    //if (misdn_debug>1) chan_misdn_log(" --> Not Appending Digits %s\n",bc->info_dad);
	  }
	  
	  ast_queue_frame(ch->ast, &fr);

	}
	else
	  chan_misdn_log("Ast Channel doesn't exist anymore @ ast_queue_frame??\n");
      }
    }
    break;
  case EVENT_SETUP:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      
      if (ch && ch->state != MISDN_NOTHING && ch->state != MISDN_HOLDED) {
	if (misdn_debug>0) chan_misdn_log(" --> Ignoring Call we have already one\n");
	return RESPONSE_IGNORE_SETUP_WITHOUT_CLOSE; // Ignore MSNs which are not in our List
      }
    }
    
    if (bc->stack->mode == TE_MODE && ! is_msn_valid(bc)) {
      if (misdn_debug>0) chan_misdn_log(" --> Ignoring Call, its not in our MSN List\n");
      return RESPONSE_IGNORE_SETUP; // Ignore MSNs which are not in our List
    }
    
    if (misdn_debug>0) chan_misdn_log(" --> Keypad: %s\n",bc->keypad);
    
    print_bearer(bc);
    
    {
      chan_list_t *ch=malloc(sizeof(chan_list_t));
      struct ast_channel *chan;
      char name[128];
      if (!ch) { perror("malloc for chan_list"); exit(0);}
      memset(ch,0,sizeof(chan_list_t));
      ch->bc_te = bc;
      ch->l3id=bc->l3_id;
      ch->orginator = ORG_MISDN;
      
      {
	char *prefix="";
	char tmp[255];
	if ( !strlen(bc->dad) && strlen(bc->keypad)) strcpy(bc->dad,bc->keypad);
	switch( bc->numplan ) {
	case NUMPLAN_INTERNATIONAL:
	  prefix=get_nat_prefix(bc);
	  if (misdn_debug>0) chan_misdn_log(" --> Numplan: Internat. prefix %s\n", prefix);
	  break;
	  
	case NUMPLAN_NATIONAL:
	  prefix=get_internat_prefix(bc);
	  if (misdn_debug>0) chan_misdn_log(" --> Numplan: Nat. prefix %s\n", prefix);
	  break;
	  
	  
	case NUMPLAN_SUBSCRIBER:
	  // dunno what to do here ?
	  if (misdn_debug>0) chan_misdn_log(" --> Numplan: Subscriber\n");
	  break;

	case NUMPLAN_UNKNOWN:
	  if (misdn_debug>0) chan_misdn_log(" --> Numplan: Unknown\n");
	  break;
	default:
	  if (misdn_debug>0) chan_misdn_log(" --> Numplan: No Numplan\n");
	  break;
	}
	
	strcpy(tmp,prefix);
	strcat(tmp,bc->oad);
	strcpy(bc->oad,tmp);
	
	if (strlen(bc->oad))
	  sprintf(name,"mISDN/%d/%s",bc->stack->port,bc->oad);
	else
	  sprintf(name,"mISDN/%d",bc->stack->port);
	
	chan=misdn_new(ch, AST_STATE_RING,name ,get_context(bc), bc->dad, bc->oad);
	
      }

#ifndef ASTERISK_STABLE
      if ( bc->pres ) {
	if (misdn_debug > 0) chan_misdn_log(" --> Number Not Screened\n");
	chan->cid.cid_pres=AST_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED;
      }  else {
	if (misdn_debug > 0) chan_misdn_log(" --> Number Screened\n");
	chan->cid.cid_pres=AST_PRES_ALLOWED_USER_NUMBER_PASSED_SCREEN;
      }
#endif
      
      ch->ast = chan;


      if (bc->digital) {
        ast_set_flag(chan,AST_FLAG_DIGITAL);
        if (misdn_debug >0 ) chan_misdn_log(" --> Setting Digital\n");
        pbx_builtin_setvar_helper(chan,"CALLTYPE","DIGITAL");
      } else {
        pbx_builtin_setvar_helper(chan,"CALLTYPE","SPEECH");
      }
      
      
      /** queue new chan **/
      cl_queue_chan(&cl_te, ch) ;
      


#ifndef ASTERISK_STABLE
      //Check for Pickup Request first
      if (!strcmp(chan->exten, ast_pickup_ext())) {
	int ret;/** Sending SETUP_ACK**/
	ret = manager_te_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
	
	if (ast_pickup_call(chan)) {
	  if (misdn_debug>0)chan_misdn_log("Nothing to pick up\n");
	  
	  ast_hangup(chan);
	} else {
	  
	  if (misdn_debug>0)chan_misdn_log("I picked Up, senbing ACK to phone port:%d\n",bc->stack->port);
	  //ret = manager_te_send_event(mgr, bc, EVENT_ALERTING);
	  ch->state = MISDN_CALLING_ACKNOWLEDGE;
	  
	  ch->ast=NULL;
	  
	  ast_setstate(chan, AST_STATE_DOWN);
	  ast_hangup(chan); 
	  
	  break;
	}
      }
#endif
      /*
	added support for s extension hope it will help those poor cretains
	which haven't overlap dial.
      */
      {
	config_list_t *config = find_config_by_port(bc->stack->port);
	if ( config) {
	  strcpy(chan->language, get_language(config));
	  if ( config->immediate) {
	    if (misdn_debug>0)chan_misdn_log(" --> Doing immediate Setup\n");
	    do_immediate_setup(bc, ch , chan);
	    break;
	  }
	}
      }

      //chan_misdn_log("Can Match Extension: Ast->exten:%s Ast->context:%s Context:%s dad:%s oad:%s\n", ch->ast->exten, ch->ast->context, get_context(bc), bc->dad,bc->oad);
      
      if(!ast_canmatch_extension(ch->ast, get_context(bc), bc->dad, 1, bc->oad)) {
	chan_misdn_log("Extension can never match, so disconnecting\n");
	manager_send_tone(mgr,bc,TONE_BUSY);
	ch->state=MISDN_EXTCANTMATCH;
	
	if (bc->stack->mode == NT_MODE)
	  manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	else
	  manager_te_send_event(mgr, bc, EVENT_DISCONNECT );
	break;
      }
      
      if (ast_exists_extension(ch->ast, get_context(bc), bc->dad, 1, bc->oad)) {
	ch->state=MISDN_DIALING;
	
	if (bc->stack->mode == NT_MODE) {
	  int ret; 
	  ret = manager_te_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
	} else {
	  int ret;
	  ret= manager_te_send_event(mgr, bc, EVENT_PROCEEDING );
	}
	
	if (misdn_debug >0) chan_misdn_log(" --> * Starting Ast ctx:%s dad:%s oad:%s\n", chan->context, chan->exten, AST_CID_P(chan));
	if (ast_pbx_start(chan)<0) {
	  chan_misdn_log("ast_pbx_start returned <0 in SETUP\n");
	  chan=NULL;
	  manager_send_tone(mgr,bc,TONE_BUSY);
	  if (bc->stack->mode == NT_MODE)
	    manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	  else
	    manager_te_send_event(mgr, bc, EVENT_DISCONNECT );
	}
      } else {
	int ret= manager_te_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
	if ( check_ret (mgr, bc, EVENT_SETUP_ACKNOWLEDGE, ret)) {
	  manager_te_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
	}
	// send tone to phone :)
	
	if ( (strlen(bc->dad) > 0) && stop_tone_after_first_digit ) 
	  manager_send_tone(mgr, bc,TONE_NONE);
	else
	  manager_send_tone(mgr, bc,TONE_DIAL);
	
	ch->state=MISDN_WAITING4DIGS;
      }
      
    }
    break;
  case EVENT_SETUP_ACKNOWLEDGE:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (bc->stack->mode == TE_MODE ) {
	
	if (ch) {
	  //should i do something here?
	}
      }
      
      ch->state = MISDN_CALLING_ACKNOWLEDGE;
      if (strlen(bc->infos_pending)) {
	//TX Pending Infos
	strcat(bc->dad,bc->infos_pending);
	strcpy(ch->ast->exten, bc->dad);
	strcpy(bc->info_dad, bc->infos_pending);
	strcpy(bc->infos_pending,"");

	manager_te_send_event(mgr_te, bc, EVENT_INFORMATION);
      }
    }
    break;
  case EVENT_PROCEEDING:
    break;
  case EVENT_PROGRESS:
    if (bc->stack->mode == TE_MODE ) {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch && ch->ast) {
	start_bc_tones(ch);  // Only
	if (misdn_debug >0) chan_misdn_log(" --> * SEND: Queue Progress pid:%d\n", bc?bc->pid:-1);

	ast_queue_control(ch->ast, AST_CONTROL_PROGRESS);

	
      }
    }
    break;
    
  case EVENT_RELEASE:
    if (misdn_debug > 0) chan_misdn_log(" --> cause %d\n",bc->cause);
    
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      
      if (ch ){
	
	if (ch && ch->ast) {
	  switch ( bc->cause) {
	  case 17: //user busy

	    if (misdn_debug >0) chan_misdn_log(" --> * SEND: Queue Busy pid:%d\n", bc?bc->pid:-1);
	    break;
	  case -1:
	    /*
	      OK, it really sucks, this is a RELEASE from NT-Stack So we take
	      it and return easylie, It seems that we've send a DISCONNECT
	      before, so we should RELEASE_COMPLETE after that Disconnect
	      (looks like ALERTING State at misdn_hangup !!
	    */
	    return RESPONSE_OK;
	    break;
	  }
	}
	
	if (ch && ch->ast) {
	  ch->ast->hangupcause=bc->cause;
	}
	
	bc->cause=16;
	
	stop_bc_tones(ch);
	release_chan(bc);
      }
    }
    break;
  case EVENT_RELEASE_COMPLETE:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);

      if (ch && ch->ast) {
	ch->ast->hangupcause=bc->cause;
      }
      
      if (ch ){
	if (misdn_debug > 0) chan_misdn_log("--> cause %d\n",bc->cause);
	
	stop_bc_tones(ch);
	release_chan(bc);
      } else {
	chan_misdn_log("Didn't RELEAS_COMPLETED cause there was no chan_list ??\n");
      }
      
    }
    break;
  case EVENT_ALERTING:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      
      if (ch) ch->state = MISDN_ALERTING;
      
      if (ch && ch->ast) {
	if (misdn_debug>0) chan_misdn_log(" --> * SEND: Queue Frame Ringing pid:%d\n", bc?bc->pid:-1);
	
	ast_queue_control(ch->ast, AST_CONTROL_RINGING);
	ast_setstate(ch->ast, AST_STATE_RINGING);
	
      }
    }
    break;
  case EVENT_CONNECT:
    manager_te_send_event(mgr,bc,EVENT_CONNECT_ACKNOWLEDGE);
  case EVENT_CONNECT_ACKNOWLEDGE:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      bc->state=STATE_CONNECTED;
      
      ch->l3id=bc->l3_id;
      
      start_bc_tones(ch);
      if (ch && ch->ast) {
	ch->state = MISDN_CONNECTED;
	if (misdn_debug > 0) chan_misdn_log(" --> * SEND: Queue Answer pid:%d\n", bc?bc->pid:-1);

	ast_queue_control(ch->ast, AST_CONTROL_ANSWER);

      }
    }
    break;
  case EVENT_DISCONNECT:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      chan_list_t *holded_ch = cl_find_chan_by_bc_te_holded(cl_te, bc);
      
      if (ch && (misdn_debug > 1)) {
	chan_misdn_log (" --> ch->state %s, bc->holded %d\n",misdn_get_ch_state(ch),bc->holded );
      }
      if (holded_ch && ch->state == MISDN_CONNECTED) {
	if (misdn_debug > 0) chan_misdn_log("Found holded_bc so transfering\n");
	misdn_transfer_bc(ch, holded_ch) ;
	bc->l3_id = ch->l3id;
	manager_te_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
	break;
      }
      //lets have a try
      if (ch) stop_bc_tones(ch);
      if (misdn_debug > 0) chan_misdn_log(" --> cause %d\n",bc->cause);

      if (ch && ch->ast && MISDN_ASTERISK_PVT(ch->ast)) {
	switch ( bc->cause) {
	case 17: //user busy
	  if (misdn_debug > 0) chan_misdn_log(" --> * SEND: Queue Busy pid:%d\n", bc?bc->pid:-1);
	  
	  ast_queue_control(ch->ast, AST_CONTROL_BUSY);
	  
	  break;
	}
      }
      
      if (ch && ch->ast) {
	ch->ast->hangupcause=bc->cause;
      }
      
      bc->cause=16;
      manager_te_send_event(mgr,bc,EVENT_RELEASE);
    }
    break;
  case EVENT_BCHAN_DATA:
    {
      chan_list_t *tmp=cl_find_chan_by_bc_te(cl_te, bc);
#if 1
      //if (bc->stack->pri) printf("Got Bchan Data %x!\n",bc->addr);
      if (tmp) {
	if ( !tmp->bc_te->digital ) 
	  manager_flip_buf_bits(bc->bframe, bc->bframe_len);
	chan_misdn_write_frm(tmp, bc->bframe, bc->bframe_len );
      }
#else
      len=bc->bframe_len;
      
      if (bc->bframe_len > ibuf_freecount(bc->astbuf)) {
	printf("sbuf overflow!\n");
	len=ibuf_freecount(bc->astbuf);
      }
      
      ibuf_memcpy_w(bc->astbuf, bc->bframe, len);
      if (bc->astbuf->rsem) sem_post(bc->astbuf->rsem);
#endif
      
    }
    break;
  case EVENT_TIMEOUT:
    break; //Ignore now ..
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch)
	switch (ch->state) {
	case MISDN_CALLING:
	  chan_misdn_log("GOT TIMOUT AT CALING pid:%d\n", bc?bc->pid:-1);
	  break;
	case MISDN_DIALING:
	  break;
	default:
	  manager_te_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
	}
    }
    break;
  case EVENT_CLEANUP:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch) stop_bc_tones(ch);
      release_chan(bc);
    }
    break;
    
    /***************************/
    /** Suplementary Services **/
    /***************************/
  case EVENT_RETRIEVE:
    {
      chan_list_t *ch=cl_find_chan_by_l3id(cl_te, bc->l3_id);

      if (ch ) {

	/** really ? **/
	ch->l3id=bc->l3_id;
	
	ch->state = MISDN_CONNECTED;
	ch->holded = 0;
	
	if (AST_BRIDGED_P(ch->ast)) {
	  ast_moh_stop(AST_BRIDGED_P(ch->ast));
	
	} else { //Other ASt End not there anymore ?
	  chan_misdn_log("Other ASt End not there anymore ?\n");
	}

	ch->bc_te = ch->holded_bc ;
	ch->holded_bc = NULL;

	if ( manager_te_send_event(mgr, bc, EVENT_RETRIEVE_ACKNOWLEDGE) < 0)
	  manager_te_send_event(mgr, bc, EVENT_RETRIEVE_REJECT);

	start_bc_tones(ch);
		
      } else {
	chan_misdn_log("Got Retrieve without holded Channel ? .. Rejecting\n");
	manager_te_send_event(mgr, bc, EVENT_RETRIEVE_REJECT);
      }
    }
    break;
    
  case EVENT_HOLD:
    {
      chan_list_t *ch=cl_find_chan_by_l3id(cl_te, bc->l3_id);
      if (ch && ch->ast) {
	config_list_t * config = find_config_by_port(bc->stack->port);

	if (!config || !config->hold_allowed) {
	  chan_misdn_log("Hold not allowed on port:%d\n", bc->stack->port);
	  manager_te_send_event(mgr, bc, EVENT_HOLD_REJECT);
	  break;
	}

	if (AST_BRIDGED_P(ch->ast)){
	  ast_moh_start(AST_BRIDGED_P(ch->ast), NULL);
	  manager_te_send_event(mgr, bc, EVENT_HOLD_ACKNOWLEDGE);
	  
	  ch->state = MISDN_HOLDED;
	  ch->holded = 1;
	  ch->holded_bc = bc ;
	  ch->bc_te=NULL;
	}
	else {
	  manager_te_send_event(mgr, bc, EVENT_HOLD_REJECT);
	  chan_misdn_log("We aren't bridged to anybody\n");
	}
      } else {
	manager_te_send_event(mgr, bc, EVENT_HOLD_REJECT);
	chan_misdn_log("No ast !\n");
      }
    } 
    break;
  default:
    chan_misdn_log("Got Unknown Event\n");
    break;
  }
  
  return RESPONSE_OK;
}

/** TE STUFF END **/

/******************************************
 *
 *   Asterisk Channel Endpoint END
 *
 *
*******************************************/

static msn_list_t *build_msn_list(char *msns)
{ //create MSN - Matrix
  char *tok, *tokb;
  msn_list_t *msn=NULL, *list;
  char *mymsns=strdup(msns);

  if (msns) {
    for (tok=strtok_r(mymsns,",\n",&tokb);
	 tok;
	 tok=strtok_r(NULL,",\n",&tokb)) {

      if (!msn) {
	msn=malloc(sizeof(msn_list_t));
	list=msn;
	if (!msn) {perror("mallocing msn list"); exit(1); }

	msn->msn=strdup(tok);
	msn->next=NULL;
      } else {
	msn->next=malloc(sizeof(msn_list_t));
	msn->next->msn=strdup(tok);
	msn->next->next=NULL;
	msn=msn->next;
      }
    }
  } else {
    free(mymsns);
    return NULL;
  }
  
  free(mymsns);
  return list;
}

static void my_build_config(char *cat, struct ast_variable *v)
{
  char ports[255]="";
  char context[255]="";
  char language[255]="";
  char callerid[255]="";
  char natprefix[255]="";
  char internatprefix[255]="";
  int dialplan=-1;
  int volume=0;
  int te_choose_channel=0;
  
#ifndef ASTERISK_STABLE
  ast_group_t callgroup=0;
  ast_group_t pickupgroup=0;
#endif
  char msns[255]="";
  char *prt, *tokb;
  int immediate = 0;
  int hold_allowed = 0;
  
  if (!v || !cat) return ;
  
  while(v) {
    if (!strcasecmp(v->name, "ports"))
      strncpy(ports,v->value,sizeof(ports)-1);
    if (!strcasecmp(v->name, "context"))
      strncpy(context,v->value,sizeof(context)-1);

    if (!strcasecmp(v->name, "diaplan"))
      dialplan=atoi(v->value);
    
    if (!strcasecmp(v->name, "nationalprefix"))
      strncpy(natprefix,v->value,sizeof(natprefix)-1);
    
    if (!strcasecmp(v->name, "internationalprefix"))
      strncpy(internatprefix,v->value,sizeof(internatprefix)-1);
    
    if (!strcasecmp(v->name, "language"))
      strncpy(language,v->value,sizeof(language)-1);

    if (!strcasecmp(v->name, "volume"))
      volume=atoi(v->value);

    if (!strcasecmp(v->name, "te_choose_channel"))
      te_choose_channel=ast_true(v->value);
    
    if (!strcasecmp(v->name, "msns"))
      strncpy(msns,v->value,sizeof(msns)-1);
    if (!strcasecmp(v->name, "immediate") && ast_true(v->value))
      immediate=1;
    if (!strcasecmp(v->name, "hold_allowed") && ast_true(v->value))
      hold_allowed=1;
    if (!strcasecmp(v->name, "callerid"))
      strncpy(callerid,v->value,sizeof(callerid)-1);

#ifndef ASTERISK_STABLE
    if (!strcasecmp(v->name, "callgroup"))
      callgroup=ast_get_group(v->value);
    if (!strcasecmp(v->name, "pickupgroup"))
      pickupgroup=ast_get_group(v->value);
#endif
    
    v=v->next;
  }
  
  if (!strlen(ports)) {
    chan_misdn_log("MISDN ERROR: you've forgot to set the ports in the misdn configfile @%s\n",cat);
    exit(1);
  }

  
  {
    for (prt=strtok_r(ports,",\n",&tokb);
	 prt;
	 prt=strtok_r(NULL,",\n",&tokb)) {
      config_list_t *element, *help ; 
      
      if (config_list) {
	for (help=config_list;
	     help->next;
	     help=help->next);
	help->next=malloc(sizeof(config_list_t));
	element=help->next;
      } else {
	element=malloc(sizeof(config_list_t));
	config_list=element;
      }
      
      if (!element) { perror("malloc in build_config"); exit(1); }

      element->group=strdup(cat);
      element->port=atoi(prt);
      if (strstr(prt,"ptp")) element->ptp=1; else element->ptp=0;
      element->context=strdup(context);
      element->language=strdup(language);
      element->volume=volume;
      element->callerid=strdup(callerid);
      element->immediate=immediate;
      element->hold_allowed=hold_allowed;

      element->dialplan=dialplan; 
      element->natprefix=strdup(natprefix);
      element->internatprefix=strdup(internatprefix);

      element->te_choose_channel=te_choose_channel; 
      
#ifndef ASTERISK_STABLE
      element->callgroup=callgroup;
      element->pickupgroup=pickupgroup;
#endif
      
      if (strlen(msns)){
	element->msn_list=build_msn_list(msns);
      } else {
	element->msn_list=NULL;
      }
      element->next=NULL;
    }
  }
  
  
  
}

static void get_ports_from_cfg(char *ports)
{
  config_list_t *help;
  
  strcpy(ports,"");

  //if (misdn_debug > 2) chan_misdn_log("Locking Config Mutex\n");
  pthread_mutex_lock(&config_mutex);
  if (config_list) {
    for (help=config_list;
	 help;
	 help=help->next) {
      char prt[16]="";
      if (help->ptp)
	sprintf(prt,"%dptp,",help->port);
      else
	sprintf(prt,"%d,",help->port);
      
      strcat(ports,prt);
    }
  }
  //if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
  pthread_mutex_unlock(&config_mutex);
  
  ports[strlen(ports)-1]=0;
}


static void destroy_config()
{
  config_list_t *help, *t1;

  for (help=config_list;
       help; ) {
    t1=help->next;
    
    if (help->msn_list) {
      msn_list_t *msn_help;

      for (msn_help=help->msn_list;
	   msn_help; ) {
	msn_list_t *t=msn_help->next;
	if (msn_help->msn) free(msn_help->msn);
	free (msn_help);
	msn_help=t;
      }
    }

    if (help->context) free(help->context);
    free(help);
    help=t1;
  }
}

void misdn_reload_config () {
  char ports[256];
  struct ast_config *cfg;
  struct ast_variable *v;
  char config[]="misdn.conf";


  //if (misdn_debug>2) chan_misdn_log("Locking config_mutex\n");
  pthread_mutex_lock(&config_mutex);
  
  
  if (config_list) destroy_config();
  config_list=NULL;

  if ((cfg = AST_LOAD_CFG(config))) {
    v = ast_variable_browse(cfg, "general");
    while(v) {
      if (!strcasecmp(v->name, "ports"))
	strncpy(ports,v->value,sizeof(ports)-1);
      if (!strcasecmp(v->name, "context"))
	strncpy(default_context,v->value,sizeof(default_context)-1);

      if (!strcasecmp(v->name, "dialplan"))
	default_dialplan=atoi(v->value);
      
      
      if (!strcasecmp(v->name, "nationalprefix"))
	strncpy(default_natprefix,v->value,sizeof(default_natprefix)-1);
      
      if (!strcasecmp(v->name, "internationalprefix"))
	strncpy(default_internatprefix,v->value,sizeof(default_internatprefix)-1);
      
      if (!strcasecmp(v->name, "language"))
	strncpy(default_language,v->value,sizeof(default_language)-1);

      if (!strcasecmp(v->name, "volume"))
	default_volume=atoi(v->value);

      if (!strcasecmp(v->name, "te_choose_channel"))
	default_te_choose_channel=ast_true(v->value);
      
      if (!strcasecmp(v->name, "msns"))
	strncpy(default_msns,v->value,sizeof(default_msns)-1);
      if (!strcasecmp(v->name, "debug"))
	misdn_debug=atoi(v->value);

      if (!strcasecmp(v->name, "dynamic_crypting") ) {
	if (ast_true(v->value)) 
	  misdn_dynamic_crypting=1;
	else
	  misdn_dynamic_crypting=0;
      }
      if (!strcasecmp(v->name, "crypt_prefix"))
	misdn_crypt_prefix=strdup(v->value);
      
      if (!strcasecmp(v->name, "crypt_keys")) {
	int i=0;
	char *tok=NULL;
	  
	misdn_crypt_keys=strdup(v->value);
	
	if (misdn_key_vector) 
	  free(misdn_key_vector);
	
	for (tok=strchr(misdn_crypt_keys, ','), i=0;
	     tok;
	     tok=strchr(++tok, ','), i++) ;
	
	misdn_key_vector_size=i;
	misdn_key_vector=malloc ( (i+1)*sizeof(char*) );
	
	for (tok=strtok(misdn_crypt_keys,","), misdn_key_vector[(i=0)]=tok;
	     tok;
	     tok=strtok(NULL,","),misdn_key_vector[++i]=tok);
	
      }
    
    
    
      if (!strcasecmp(v->name, "stop_tone_after_first_digit") ) {
	if (ast_true(v->value)) 
	  stop_tone_after_first_digit=1;
	else
	  stop_tone_after_first_digit=0;
      }
      
      if (!strcasecmp(v->name, "append_digits2exten") ) {
	if (ast_true(v->value))
	  append_digits2exten=1;
	else
	  append_digits2exten=0;
      }
      
      if (!strcasecmp(v->name, "tracefile")) {
	tracing=1;
	strncpy(tracefile,v->value,sizeof(tracefile)-1);
      }
      
      v=v->next;
    }
  } 
  
  {
    char *cat;
    cat = ast_category_browse(cfg, NULL);
    while(cat) {
      if (strcasecmp(cat,"general")) {
	struct ast_variable *v=ast_variable_browse(cfg,cat);
	my_build_config (cat,v );
      }
      cat=ast_category_browse(cfg,cat);
    }
  }
  AST_DESTROY_CFG(cfg);
  
  //if (misdn_debug>2) chan_misdn_log("UnLocking config_mutex\n");
  pthread_mutex_unlock(&config_mutex);
  
}




int load_module()
{
  char ports[256]="";
  int i ;

  i = mISDN_open();
  if (i<0) {
    perror("Open of mISDN Failed\n");
    exit(1);
  }
  mISDN_close(i);
  
  misdn_reload_config();

#ifndef ASTERISK_STABLE 
  {
    if (ast_channel_register(&misdn_tech)) {
      ast_log(LOG_ERROR, "Unable to register channel class %s\n", type);
      unload_module();
      return -1;
  }
  }
#else
  if (ast_channel_register(type, tdesc, prefformat, misdn_request)) {
    ast_log(LOG_ERROR, "Unable to register channel class %s\n", type);
    unload_module();
    return -1;
  }
#endif
  
  ast_cli_register(&cli_send_display);
  ast_cli_register(&cli_send_digit);
  ast_cli_register(&cli_toggle_echocancel);
  ast_cli_register(&cli_set_tics);

  ast_cli_register(&cli_show_cls);
  ast_cli_register(&cli_show_cl);
  ast_cli_register(&cli_show_config);
  ast_cli_register(&cli_show_port);
  ast_cli_register(&cli_show_stacks);
  ast_cli_register(&cli_show_fullstacks);
  ast_cli_register(&cli_flush_stack);
  ast_cli_register(&cli_restart_port);
  ast_cli_register(&cli_set_debug);
  ast_cli_register(&cli_set_crypt_debug);
  ast_cli_register(&cli_reload);

  
  ast_register_application("misdn_set_opt", misdn_set_opt_exec, "misdn_set_flags",
			   "misdn_set_opt(:<opt><optarg>:<opt><optarg>..):\n"
			   "Sets mISDN opts. and optargs\n"
			   "\n"
			   );
  
  pthread_mutex_init(&cl_te_lock, NULL);

  
  msg_init();
  debug_init(0 , NULL, NULL, NULL);
  //debug_init( 0xffffffff & ~DBGM_MSG, NULL, NULL, NULL);
  
  get_ports_from_cfg(ports);
  {
    mgr_te=manager_te_init(ports, cb_te_events, NULL);
    if (!mgr_te) chan_misdn_log("No te ports initialized\n");
  }

  if (misdn_dynamic_crypting)
    crypt_init();
  
  return 0;
}



int unload_module()
{
  /* First, take us out of the channel loop */
  chan_misdn_log("-- Unregistering mISDN Channel Driver --\n");

  ast_cli_unregister(&cli_send_display);
  ast_cli_unregister(&cli_send_digit);
  ast_cli_unregister(&cli_toggle_echocancel);
  ast_cli_unregister(&cli_set_tics);
  
  ast_cli_unregister(&cli_show_cls);
  ast_cli_unregister(&cli_show_cl);
  ast_cli_unregister(&cli_show_config);
  ast_cli_unregister(&cli_show_port);
  ast_cli_unregister(&cli_show_stacks);
  ast_cli_unregister(&cli_show_fullstacks);
  ast_cli_unregister(&cli_flush_stack);
  ast_cli_unregister(&cli_restart_port);
  ast_cli_unregister(&cli_set_debug);
  ast_cli_unregister(&cli_set_crypt_debug);
  ast_cli_unregister(&cli_reload);
  //ast_unregister_application("misdn_crypt");
  ast_unregister_application("misdn_set_opt");
  
#ifndef ASTERISK_STABLE
  ast_channel_unregister(&misdn_tech);
#else
  ast_channel_unregister(type);
#endif
  
  destroy_config();

  manager_te_destroy(mgr_te);
  
  return 0;
}

int usecount()
{
  int res;
  ast_mutex_lock(&usecnt_lock);
  res = usecnt;
  ast_mutex_unlock(&usecnt_lock);
  return res;
}

char *description()
{
  return desc;
}

char *key()
{
  return ASTERISK_GPL_KEY;
}

void chan_misdn_log(char *tmpl, ...)
{
  va_list ap;
  char buf[1024]; //maxbuflog
  
  va_start(ap, tmpl);
  
  vsprintf( buf, tmpl, ap );
  
  va_end(ap);

  ast_console_puts(buf);
  
  if (tracing) {
    time_t tm = time(NULL);
    char *tmp=ctime(&tm),*p;
    FILE *fp= fopen(tracefile, "a");

    p=strchr(tmp,'\n');
    if (p) *p=':';
    
    if (!fp) {
      ast_console_puts("Error opening Tracefile");
      ast_console_puts(strerror(errno));
      ast_console_puts("\n");
      return ;
    }
    
    fputs(tmp,fp);
    fputs(" ", fp);
    fputs(buf, fp);

    fclose(fp);
  }
}



int reload(void) {

  chan_misdn_log("Reloading mISDN Config\n");
  chan_misdn_log("Dynamic Crypting Activation is not support during reload at the moment\n");
  misdn_reload_config();

  
  
  return 0;
}



/*** SOME APPS ;)***/

static int misdn_set_opt_exec(struct ast_channel *chan, void *data)
{
  chan_list_t *ch = MISDN_ASTERISK_TECH_PVT(chan);
  char *tok,*tokb;
  int volume=0, keyidx=0;
  if (!data || !strlen((char *)data)) {
    ast_log(LOG_WARNING, "misdn_volume_change Requires arguments\n");
    return -1;
  }

  for (tok=strtok_r((char*)data, ":",&tokb);
       tok;
       tok=strtok_r(NULL,":",&tokb) ) {
    
    switch(tok[0]) {

    case 'd' :
      strncpy(ch->bc_te->display,++tok,84);
      if (misdn_debug>0) chan_misdn_log("SETOPT: Display:%s\n",ch->bc_te->display);
      break;
      
    case 'n':
      if (misdn_debug>0) chan_misdn_log("SETOPT: No DTMF\n");
      ch->bc_te->nodtmf=1;
      break;
      
    case 'v':
      volume=atoi(++tok);
      if (volume<-8) volume=-8;
      if (volume>8) volume=8;
      ch->bc_te->volume=volume;
      if (misdn_debug>0) chan_misdn_log("SETOPT: Volume:%d\n",volume);
      break;
      
    case 'c':
      keyidx=atoi(++tok);
      
      if (keyidx > misdn_key_vector_size  || keyidx < 0 ) {
	ast_log(LOG_WARNING, "You entered the keyidx: %d but we have only %d keys\n",keyidx, misdn_key_vector_size );
	continue; 
      }
      
      strcpy(ch->bc_te->crypt_key, misdn_key_vector[keyidx]);
      chan_misdn_log("SETOPT: crypt with key:%s\n",misdn_key_vector[keyidx]);
      break;

    case 'e':
      if (misdn_debug >0) chan_misdn_log("SETOPT: EchoCancel\n");
      ch->bc_te->ec_enable=1;
      ch->bc_te->orig=ch->orginator;
      break;
      
    case 'h':
      if (misdn_debug >0) chan_misdn_log("SETOPT: Digital\n");
      ch->bc_te->digital=1;
      
    default:
      break;
    }
  }
  
  return 0;
}
     

