/*
 * Asterisk -- A telephony toolkit for Linux.
 *
 * Trivial application to playback a sound file
 * 
 * Copyright (C) 1999, Mark Spencer
 *
 * Mark Spencer <markster@linux-support.net>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License
 */

#include <asterisk/lock.h>
#include <asterisk/file.h>
#include <asterisk/logger.h>
#include <asterisk/channel.h>
#include <asterisk/pbx.h>
#include <asterisk/module.h>
#include <asterisk/translate.h>
#include <asterisk/utils.h>
#include <asterisk/cli.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>
#include <semaphore.h>
#include <pty.h>
#include <unistd.h>
#include <errno.h>

static char *tdesc = "Waits for overlap dialed digits";

static char *app = "waitfordigits";

static char *synopsis = "Wait for digits";

static char *descrip =
" waitfordigits([milliseconds]|[maxnum]|addexten|<control>):\n"
" Waits for given milliseconds for overlapdialed digits\n" 
" If you pass maxnum, it waits only till maxnum digits is dialed, then\n"
" exits immediately\n"
" The argument 'addexten' indicates wether the dialed digits should be added\n"
" to the given channel\n"
" The channel Variable ALREADY_WAITED indicates wether we have already waited\n"
" If this Variable is set, the application exits immediatly\n"
" The control option allows you to send the given control Number in terms\n"
" of AST_CONTROL_XX (asterisk/include/asterisk/frame.h) Numbers to this \n"
" channel before extiting. In q.931 it makes a lot of sense to send a \n"
" Proceeding in this case which means control=15\n"
;


STANDARD_LOCAL_USER;

LOCAL_USER_DECL;



#define CHANNEL_INFO " (%s,%s)"
#ifdef ASTERISK_STATBLE
#define CHANNEL_INFO_PARTS(a) , a->callerid, a->exten
#else
#define CHANNEL_INFO_PARTS(a) , a->cid.cid_num, a->exten
#endif



static int waitfordigits_exec(struct ast_channel *chan, void *data)
{
  int res = 0;
  struct localuser *u;

  /** Default values **/
  int timeout=3500;
  int maxnum=20;
  int addexten=0;
  
  int control=0;
  
  char *tmp=pbx_builtin_getvar_helper(chan,"ALREADY_WAITED") ;
  char aw=0;
  
  if (tmp) aw=atoi(tmp);
  
  
  if (!data || !strlen((char *)data)) {
	  ast_log(LOG_WARNING, "waitfordigits Requires arguments\n");
    return -1;
  }

  LOCAL_USER_ADD(u);
  
  /** Parsing cmd line args **/
  {
    char *tok=data;

    timeout=atoi(tok);
    
    tok=strchr(tok,'|');
    
    if (tok) {
      maxnum=atoi(++tok);
      
      tok=strchr(tok,'|');
      
      if (tok) {
	
	if (strstr(tok,"addexten"))
	  addexten=1;
	
	tok=strchr(tok,'|');
	
	if (tok) control=atoi(tok);
	
      }
    }
  }
  
  ast_verbose("You passed timeout:%d maxnum:%d addexten:%d control:%d\n",
	      timeout, maxnum, addexten, control);
  
  
  
  /** Check wether we have something to do **/
  
  if (chan->_state == AST_STATE_UP || aw ) {
	  LOCAL_USER_REMOVE(u);
	  return 0;
  }
  
  pbx_builtin_setvar_helper(chan,"ALREADY_WAITED","1");
  
  {
	  char dig=0;
	  char numsubst[255];
	  
	  /** Saving predialed Extension **/
	  strcpy(numsubst, chan->exten);
	  while ( (strlen(numsubst)< maxnum) && (dig=ast_waitfordigit(chan, timeout))) {
		  int l=strlen(numsubst);
		  
		  if (dig<=0) {
			  ast_log(LOG_NOTICE, "Error at adding dig: %c!\n",dig);
			  res=-1;
			  break;
		  }
		  
		  numsubst[l]=dig;
		  numsubst[l+1]=0;
		  //ast_log(LOG_NOTICE, "Adding Dig to Chan: %c\n",dig);
	  }
	  
	  /** Restoring Extension if requested **/
	  if (addexten) { 
		  ast_verbose("Overwriting extension:%s with new Number: %s\n",chan->exten, numsubst);
		  strcpy(chan->exten, numsubst);
	  } else {
		  ast_verbose( "Not Overwriting extension:%s with new Number: %s\n",chan->exten, numsubst);
	  }
	  
	  /** Posting Exten to Var: NEWEXTEN **/
	  pbx_builtin_setvar_helper(chan,"NEWEXTEN",numsubst);
	  
	  
	  if (chan->_state != AST_STATE_UP &&
	      chan->_state != AST_STATE_RINGING && (control>0) ) {
		  ast_verbose( "Sending CONTROL: %d  to Channel %s\n",control, chan->exten);
		  
		  ast_indicate(chan, control);
	  } else {
		  ast_verbose( "Not Sending any control  to Channel %s\n",chan->exten);
	  }
	  
  }
  
  LOCAL_USER_REMOVE(u);
  return res;
}

int unload_module(void)
{
  STANDARD_HANGUP_LOCALUSERS;
  return ast_unregister_application(app);
}



int load_module(void)
{
	return ast_register_application(app, waitfordigits_exec, synopsis, descrip);
}

char *description(void)
{
  return tdesc;
}

int usecount(void)
{
  int res;
  STANDARD_USECOUNT(res);
  return res;
}

char *key()
{
  return ASTERISK_GPL_KEY;
}
