# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
"Wrapper" to LSF, PBS, SunGE batch schedulers.
Only for submit function yet.
"""


import os
import re
from asrun.common_func import get_tmpname
from asrun.i18n        import _


class BatchError(Exception): pass


class BatchSystem(object):

   batch_sub = '%(batch_sub)s < %(btc_file)s'

   def __init__(self, run, prof, script):
      """
      Initialization
      run    : ASTER_RUN object,
      prof   : ASTER_PROFIL object,
      script : shell script to submit.
      """
      self.run    = run
      self.prof   = prof
      self.script = script
      self.jn = os.getpid()
      self.btc_file = get_tmpname(self.run, self.run['tmp_user'], basename='btc_file')


   def build_dict_info(self):
      """
      Add infos into dict_info.
      """
      dico = {
         'nomjob'    : self.prof['nomjob'][0],
         'tpsjob'    : self.prof['tpsjob'][0],        # in minutes
         'tpmax'     : self.prof.args['tpmax'],    # in seconds
         'memjob'    : self.prof['memjob'][0],
         'clasgr'    : self.prof['classe'][0],
         'depart'    : self.prof['depart'][0],
         'batch_ini' : self.run.get('batch_ini'),
         'batch_sub' : self.run.get('batch_sub'),
      }
      if dico['clasgr'] != '' and dico['clasgr'] in self.run.get('batch_queue_group', '').split():
         dico['classe'] = self.run.get('batch_queue_%s' % dico['clasgr'], '')
      dico['btc_file'] = self.btc_file
      self.dict_info = dico


   def change_script(self):
      """
      Add header to 'script' file.
      """
      if not os.path.isfile(self.script):
         raise BatchError, 'file not found : %s' % self.script
      txt = open(self.script, 'r').read().splitlines()
      return txt


   def parse_output(self, output):
      """
      Extract jobid and queue from output of submission.
      """
      return '', ''


   def submit(self):
      """
      Submit the script.
      Returns a tuple : (exitcode, jobid, queue)
      """
      cmd = ''
      if self.dict_info['batch_ini'] != '':
         cmd = '. %(batch_ini)s ; '
      cmd += self.batch_sub
      iret, out = self.run.Shell(cmd % self.dict_info)
      self.run.DBG(u'Output of submitting :', out, all=True)
      if iret != 0:
         self.run.Mess(_(u'Failure during submitting'),)
         if os.path.isfile(self.btc_file):
            self.run.DBG('submitted script :', open(self.btc_file, 'r').read(), all=True)
      jobid, queue = self.parse_output(out)
      return iret, jobid, queue


   def start(self):
      """
      Go
      """
      self.build_dict_info()
      txt = os.linesep.join(self.change_script())
      content = txt % self.dict_info
      open(self.btc_file, 'w').write(content)
      iret, jobid, queue = self.submit()
      os.remove(self.btc_file)
      return iret, jobid, queue



class LSF(BatchSystem):
   """
   for LSF batch system.
   """
   def build_dict_info(self):
      """
      Add infos into dict_info.
      """
      super(LSF, self).build_dict_info()
      self.dict_info.update({
         'error'     : os.path.join(self.run['flasheur'], '%s.e%%J' % self.dict_info['nomjob']),
         'output'    : os.path.join(self.run['flasheur'], '%s.o%%J' % self.dict_info['nomjob']),
      })


   def change_script(self):
      """
      Modify submission 'script' file.
      """
      txt = ["#BSUB -J %(nomjob)s",
             "#BSUB -c %(tpsjob)s",
             "#BSUB -M %(memjob)s",
             "#BSUB -e %(error)s",
             "#BSUB -o %(output)s"]
      if self.dict_info.get('classe'):
         txt.append('''#BSUB -q "%(classe)s"''')
      if self.dict_info.get('depart'):
         txt.append("""#BSUB -b %(depart)s""")
      # core script
      core = super(LSF, self).change_script()
      txt.extend(core)
      return txt


   def parse_output(self, output):
      """
      Extract jobid and queue from output of submission.
      """
      mat = re.search('Job *<([^\s]+)> *is submitted to .*queue *<([^\s]+)>', output)
      jobid, queue = '', ''
      if mat is not None:
         jobid, queue = mat.groups()
      return jobid, queue



class PBS(BatchSystem):
   """
   for PBS batch system.
   """
   def build_dict_info(self):
      """
      Add infos into dict_info.
      """
      super(LSF, self).build_dict_info()
      # format of start time : [[MM]DD]HHmm
      self.dict_info['depart'] = self.dict_info['depart'].replace(':', '')


   def change_script(self):
      """
      Modify submission 'script' file.
      """
      core = super(PBS, self).change_script()
      txt = [core.pop(0),]
      txt.extend(["#PBS -N %(nomjob)s",
                  "#PBS -l cput=%(tpmax)s",
                  "#PBS -m mem=%(memjob)skb",])
      # output/error will be named %(nomjob).o%(jobid) in the directory where qsub has been run
      #            "#PBS -e %(error)s",
      #            "#PBS -o %(output)s"
      if self.dict_info.get('classe'):
         txt.append('''#PBS -q "%(classe)s"''')
      if self.dict_info.get('depart'):
         txt.append("""#PBS -a %(depart)s""")
      # core script
      txt.extend(core)
      return txt


   def parse_output(self, output):
      """
      Extract jobid and queue from output of submission.
      """
      queue = 'unknown'
      jobid = output.split('.')[0]
      return jobid, queue


   def start(self):
      prev = os.getcwd()
      os.chdir()
      iret, jobid, queue = super(PBS, self).start()
      os.chdir(prev)
      return iret, jobid, queue


class SGE(BatchSystem):
   """
   for Sun Grid Engine batch system.
   """
   batch_sub = '%(batch_sub)s -S /bin/sh < %(btc_file)s'
   
   def build_dict_info(self):
      """Add infos into dict_info.
      """
      super(LSF, self).build_dict_info()
      self.dict_info.update({
         'error'     : os.path.join(self.run['flasheur'], '%s.e$JOB_ID' % self.dict_info['nomjob']),
         'output'    : os.path.join(self.run['flasheur'], '%s.o$JOB_ID' % self.dict_info['nomjob']),
      })
      # format of start time : [[MM]DD]HHmm (MM/DD compulsory ?)
      self.dict_info['depart'] = self.dict_info['depart'].replace(':', '')


   def change_script(self):
      """Modify submission 'script' file.
      """
      txt = ["#$ -N %(nomjob)s",
             "#$ -l cpu=%(tpmax)s,s_rss=%(memjob)sK",
             "#$ -e %(error)s",
             "#$ -o %(output)s"]
      if self.dict_info.get('classe'):
         txt.append('''#$ -q "%(classe)s"''')
      if self.dict_info.get('depart'):
         txt.append("""#$ -a %(depart)s""")
      # core script
      core = super(SGE, self).change_script()
      txt.extend(core)
      return txt


   def parse_output(self, output):
      """
      Extract jobid and queue from output of submission.
      """
      # Ex.: Your job 789 ("my_job") has been submitted
      queue = 'unknown'
      jobid = output.split(' ')[0]
      return jobid, queue



def BatchSystemFactory(run, prof, script):
   name = run.get('batch_nom', '').lower()
   if   name == 'lsf':
      return LSF(run, prof, script)
   elif name == 'pbs':
      return PBS(run, prof, script)
   elif name == 'SunGE':
      return SGE(run, prof, script)
   else:
      raise BatchError, "unknown batch scheduler : '%s'" % name

